import axios, { AxiosError, AxiosRequestConfig, AxiosResponse } from 'axios';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:5000/api/v1';

// Define types for the API responses
interface ApiResponse<T = any> {
  success: boolean;
  data: T;
  message?: string;
  error?: string;
  token?: string;
  accessToken?: string;
}

interface RefreshTokenResponse {
  accessToken: string;
  refreshToken: string;
}

// Extend AxiosRequestConfig to include _retry flag
interface RetryableAxiosRequestConfig extends AxiosRequestConfig {
  _retry?: boolean;
}

// Create axios instance
const api = axios.create({
  baseURL: API_URL,
  withCredentials: true,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Helper function to get authenticated image URL
export const getAuthenticatedImageUrl = (path: string): string => {
  // Extract just the filename from the path
  const filename = path.split('/').pop();
  // Return the full URL to the authenticated image endpoint
  return `/prescriptions/image/${filename}`;
};

// Helper function to get authenticated image blob
export const getAuthenticatedImage = async (path: string): Promise<string> => {
  try {
    const response = await api.get(`/prescriptions/image/${path.split('/').pop()}`, {
      responseType: 'blob'
    });
    return URL.createObjectURL(response.data);
  } catch (error) {
    console.error('Error loading image:', error);
    return '';
  }
};

// Helper function to get the current token
const getAuthToken = (): string | null => {
  try {
    // Try to get token from localStorage
    let token = localStorage.getItem('token');
    
    // If in development and no token in localStorage, check sessionStorage
    if ((!token || token === 'undefined') && import.meta.env.DEV) {
      const sessionToken = sessionStorage.getItem('token');
      if (sessionToken) {
        console.log('Found token in sessionStorage, moving to localStorage');
        localStorage.setItem('token', sessionToken);
        token = sessionToken;
      }
    }
    
    // Ensure token is properly formatted
    if (token && token.startsWith('Bearer ')) {
      token = token.replace('Bearer ', '').trim();
    }
    
    console.log('Current auth token:', token ? 'Token exists' : 'No token found');
    return token;
  } catch (error) {
    console.error('Error getting auth token:', error);
    return null;
  }
};

// Add request interceptor to include auth token for all requests
api.interceptors.request.use((config) => {
  try {
    const token = getAuthToken();
    
    if (token) {
      // Ensure headers object exists
      config.headers = config.headers || {};
      
      // Log the token being set (without logging the actual token for security)
      console.log(`Setting Authorization header for ${config.method?.toUpperCase()} ${config.url}`, {
        hasToken: !!token,
        tokenLength: token?.length,
        tokenPrefix: token?.substring(0, 10) + '...'
      });
      
      // Ensure we don't have double 'Bearer ' prefix
      const authHeader = token.startsWith('Bearer ') ? token : `Bearer ${token}`;
      
      // Set the Authorization header
      config.headers.Authorization = authHeader;
      
      // Debug log the full request headers
      console.log('Request headers:', {
        ...config.headers,
        // Don't log the full auth token
        Authorization: config.headers.Authorization 
          ? `${config.headers.Authorization.substring(0, 15)}...` 
          : 'No Authorization header'
      });
    } else {
      console.warn('No auth token found for request to:', config.url);
    }
  } catch (error) {
    console.error('Error setting auth token:', error);
  }

  // Don't set Content-Type for FormData - let browser set it automatically
  if (config.data instanceof FormData) {
    delete config.headers?.['Content-Type'];
  }

  // Log the request for debugging (without sensitive data)
  if (import.meta.env.DEV) {
    const logData = { ...config.data };
    if (logData.password) logData.password = '***';
    if (logData.newPassword) logData.newPassword = '***';
    
    console.log(`%cAPI ${config.method?.toUpperCase()} ${config.url}`, 
      'color: #4CAF50; font-weight: bold', 
      { headers: config.headers, data: logData }
    );
  }

  return config;
}, (error) => {
  console.error('Request Error:', error);
  return Promise.reject(error);
});

// Add response interceptor to handle token refresh and errors
api.interceptors.response.use(
  (response: AxiosResponse<ApiResponse>) => {
    // Log successful responses for debugging
    if (import.meta.env.DEV) {
      console.log(`%cAPI ${response.status} ${response.config.url}`, 
        'color: #4CAF50; font-weight: bold', 
        { data: response.data }
      );
    }
    return response;
  },
  async (error: unknown) => {
    // Type guard for AxiosError
    if (!axios.isAxiosError(error)) {
      console.error('Non-Axios error occurred:', error);
      return Promise.reject(error);
    }
    
    const axiosError = error as AxiosError<ApiResponse>;
    const originalRequest = axiosError.config as RetryableAxiosRequestConfig;
    if (!originalRequest) {
      return Promise.reject(axiosError);
    }
    
    const isAuthEndpoint = originalRequest.url?.includes('/auth/');
    
    // Log the error for debugging
    console.error(`%cAPI Error ${error.response?.status || 'NO_RESPONSE'} ${originalRequest?.url}`, 
      'color: #F44336; font-weight: bold', 
      {
        status: error.response?.status,
        data: error.response?.data,
        message: error.message,
        config: {
          method: originalRequest?.method,
          headers: originalRequest?.headers,
          data: originalRequest?.data
        }
      }
    );
    
    // Handle 401 Unauthorized (token expired or invalid)
    if (axiosError.response?.status === 401 && !originalRequest._retry && !isAuthEndpoint) {
      // Don't retry if this is a retry attempt that failed
      if (originalRequest._retry) {
        console.log('Already attempted to refresh token, not retrying');
        return Promise.reject(axiosError);
      }
      
      originalRequest._retry = true;
      
      try {
        console.log('Attempting to refresh token...');
        const refreshToken = document.cookie
          .split('; ')
          .find(row => row.startsWith('refreshToken='))
          ?.split('=')[1];
        
        if (!refreshToken) {
          console.warn('No refresh token found in cookies');
          throw new Error('No refresh token available');
        }
        
        const response = await axios.post(`${API_URL}/auth/refresh-token`, 
          { refreshToken },
          {
            withCredentials: true,
            headers: {
              'Content-Type': 'application/json',
              'X-Requested-With': 'XMLHttpRequest'
            },
            timeout: 5000 // Add timeout to prevent hanging
          }
        );
        
        if (response.data.accessToken) {
          const { accessToken } = response.data;
          console.log('Token refreshed successfully');
          
          // Store the new token
          localStorage.setItem('token', accessToken);
          
          // Update the original request with the new token
          originalRequest.headers = originalRequest.headers || {};
          originalRequest.headers.Authorization = `Bearer ${accessToken}`;
          
          // Only retry if the request method is idempotent
          const idempotentMethods = ['get', 'head', 'options'];
          if (originalRequest.method && idempotentMethods.includes(originalRequest.method.toLowerCase())) {
            console.log(`Retrying ${originalRequest.method} ${originalRequest.url}`);
            return api(originalRequest);
          }
          
          // For non-idempotent methods, just return the response
          return Promise.resolve(response);
        }
      } catch (refreshError) {
        console.error('Token refresh failed:', refreshError);
        
        // Don't clear tokens or redirect in development mode
        if (import.meta.env.DEV) {
          console.warn('Token refresh failed in development, but continuing with current session');
          return Promise.reject(axiosError);
        }
        
        // Clear any invalid tokens
        localStorage.removeItem('token');
        sessionStorage.removeItem('token');
        
        // Check if this is an authentication error (401 or 403)
        const isAuthError = (refreshError.response?.status === 401 || refreshError.response?.status === 403);
        
        if (isAuthError) {
          // Clear user data for auth errors
          localStorage.removeItem('token');
          localStorage.removeItem('user');
          
          // Only redirect to login if we're not already on the login page
          if (!window.location.pathname.includes('/login')) {
            // Use setTimeout to prevent React update during render
            setTimeout(() => {
              window.location.href = '/login';
            }, 100);
          }
        }
        
        return Promise.reject(refreshError);
      }
    }
    
    // Handle 403 Forbidden (insufficient permissions)
    if (error.response?.status === 403) {
      console.error('Access forbidden - insufficient permissions');
      // You might want to show a notification to the user here
    }
    
    return Promise.reject(error);
  }
);

export default api;

export const categoryService = {
  /**
   * Fetch active categories
   * @param limit Number of categories to fetch (default 10)
   * @returns Array of active categories
   */
  getActiveCategories: async (limit = 10) => {
    try {
      const response = await api.get(`/categories/list/active?limit=${limit}`);
      return response.data;
    } catch (error) {
      console.error('Error fetching active categories:', error);
      throw error;
    }
  },

  /**
   * Fetch category by ID
   * @param id Category ID
   */
  getCategoryById: async (id: string) => {
    try {
      const response = await api.get(`/categories/${id}`);
      return response.data;
    } catch (error) {
      console.error(`Error fetching category ${id}:`, error);
      throw error;
    }
  },

  /**
   * Search categories by name
   * @param query Search query
   */
  searchCategories: async (query: string) => {
    try {
      const response = await api.get(`/categories/search?q=${encodeURIComponent(query)}`);
      return response.data;
    } catch (error) {
      console.error('Error searching categories:', error);
      throw error;
    }
  },

  /**
   * Get category statistics
   */
  getCategoryStats: async () => {
    try {
      const response = await api.get('/categories/statistics');
      return response.data;
    } catch (error) {
      console.error('Error fetching category statistics:', error);
      throw error;
    }
  }
};


export const deliveryBoysService = {
  /**
   * Update delivery boy availability
   * @param id Delivery boy ID
   * @param isAvailable New availability status
   */
  updateAvailability: (id: string, isAvailable: boolean): Promise<AxiosResponse<ApiResponse<{ isAvailable: boolean }>>> => {
    return api.patch(`/delivery-boys/${id}/availability`, { isAvailable });
  },

  /**
   * Get all delivery boys
   */
  getAll: (params?: any): Promise<AxiosResponse<ApiResponse<Array<{
    _id: string;
    name: string;
    email: string;
    phone: string;
    isAvailable: boolean;
    status: string;
    totalDeliveries: number;
    completedDeliveries: number;
    cancelledDeliveries: number;
    rating?: number;
  }>>>> => {
    return api.get('/delivery-boys', { params });
  },

  /**
   * Get delivery boy by ID
   * @param id Delivery boy ID
   */
  getById: (id: string): Promise<AxiosResponse<ApiResponse<any>>> => {
    return api.get(`/delivery-boys/${id}`);
  },
};

export const itemService = {
  /**
   * Fetch items with optional pagination and category filter
   * @param page Page number for pagination
   * @param limit Number of items per page
   * @param categoryId Optional category ID to filter items
   * @param sortBy Optional sorting parameter
   */
  getItems: async (page = 1, limit = 16, queryString = '') => {
    try {
      const params = new URLSearchParams({
        page: page.toString(),
        limit: limit.toString(),
        status: 'active' // Only fetch active products
      });

      // Parse the query string to extract additional parameters
      const queryParams = new URLSearchParams(queryString);
      
      // Add all query parameters to the request
      queryParams.forEach((value, key) => {
        if (key === 'ids') {
          // Handle comma-separated list of IDs
          const ids = value.split(',');
          ids.forEach(id => params.append('ids', id));
        } else if (key === 'populate') {
          // Handle populate parameter
          params.append('populate', value);
        } else {
          params.append(key, value);
        }
      });

      const response = await api.get(`/items?${params.toString()}`);
      return response.data;
    } catch (error) {
      console.error('Error fetching items:', error);
      throw error;
    }
  },

  /**
   * Fetch categories for filtering
   */
  getCategories: async () => {
    try {
      const response = await api.get('/categories');
      return response.data;
    } catch (error) {
      console.error('Error fetching categories:', error);
      throw error;
    }
  },

  /**
   * Fetch a single item by its ID
   * @param id Item ID
   */
  getItemById: async (id: string) => {
    try {
      // Use a separate axios instance without auth for public endpoints
      const publicApi = axios.create({
        baseURL: API_URL,
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      const response = await publicApi.get(`/items/${id}`);
      return response.data;
    } catch (error) {
      console.error(`Error fetching item ${id}:`, error);
      throw error;
    }
  }
};

export const orderService = {
  /**
   * Fetch orders with optional filtering and pagination
   * @param params Query parameters for filtering and pagination
   */
  getOrders: (params: any = {}) => {
    // For store managers, use the store endpoint with their store ID
    if (params.storeManagerId) {
      return api.get<ApiResponse<{
        count: number;
        pagination: any;
        data: any[];
      }>>(`/orders/store/${params.storeManagerId}`, { 
        params: {
          search: params.search,
          status: params.status,
          page: params.page,
          limit: params.limit
        }
      });
    }
    
    // For admins or when no specific store is specified
    return api.get<ApiResponse<any>>('/orders', { 
      params: {
        search: params.search,
        status: params.status,
        page: params.page,
        limit: params.limit
      } 
    });
  },

  /**
   * Get order by ID
   * @param orderId ID of the order to fetch
   */
  getOrderById: (orderId: string) => {
    return api.get<ApiResponse<any>>(`/orders/${orderId}`);
  },

  /**
   * Update order status
   * @param orderId ID of the order to update
   * @param status New status for the order
   * @param notes Optional notes about the status update
   */
  updateOrderStatus: (orderId: string, status: string, notes?: string) => {
    return api.put<ApiResponse<any>>(`/orders/${orderId}/status`, { status, notes });
  },

  /**
   * Assign delivery boy to an order
   * @param orderId ID of the order
   * @param deliveryBoyId ID of the delivery boy to assign
   */
  assignDeliveryBoy: (orderId: string, deliveryBoyId: string) => {
    return api.put<ApiResponse<any>>(`/orders/${orderId}/assign-delivery`, { deliveryBoyId });
  },

  /**
   * Get order statistics
   */
  getOrderStats: () => {
    return api.get<ApiResponse<{
      totalOrders: number;
      pending: number;
      confirmed: number;
      processing: number;
      outForDelivery: number;
      delivered: number;
      cancelled: number;
      thisMonth: number;
      lastMonth: number;
      revenue: {
        current: number;
        previous: number;
        growth: number;
      };
    }>>('/orders/stats');
  },

  /**
   * Search orders by order number, customer name, or phone
   * @param query Search query string
   */
  searchOrders: (query: string) => {
    return api.get<ApiResponse<any[]>>('/orders/search', { params: { q: query } });
  }
};

export const prescriptionService = {
  /**
   * Fetch prescriptions by status (pending/approved/rejected)
   * @param status Status to filter by ('pending', 'approved', 'rejected')
   * @param page Page number for pagination
   * @param limit Number of items per page
   */
  getPrescriptionsByStatus: (status: 'pending' | 'approved' | 'rejected' = 'pending', page = 1, limit = 20) => {
    return api.get<ApiResponse<{
      prescriptions: Array<{
        _id: string;
        userId: string | { _id: string; name: string; email: string; phone?: string };
        orderId?: string | { _id: string; orderNumber: string };
        storeId?: string | { _id: string; name: string; address: string };
        doctor_name?: string;
        patient_name: string;
        patient_age?: number;
        patient_gender?: 'male' | 'female' | 'other';
        diagnosis?: string;
        notes?: string;
        status: 'pending' | 'approved' | 'rejected';
        rejectionReason?: string;
        pharmacistNotes?: string;
        reviewedBy?: string | { _id: string; name: string };
        reviewedAt?: Date;
        images: Array<{
          path: string;
          original_name: string;
          mimetype: string;
          size: number;
          is_primary?: boolean;
          uploaded_at?: Date;
        }>;
        verification_history?: Array<{
          status: string;
          changedAt: Date;
          comment?: string;
          changedBy: string | { _id: string; name: string };
        }>;
        createdAt: Date;
        uploaded_at?: Date;
      }>;
      total: number;
      page: number;
      totalPages: number;
    }>>(`/prescriptions/status?status=${status}&page=${page}&limit=${limit}`);
  },

  /**
   * @deprecated Use getPrescriptionsByStatus('pending') instead
   */
  getPendingPrescriptions: (page = 1, limit = 20) => {
    return prescriptionService.getPrescriptionsByStatus('pending', page, limit);
  },

  /**
   * Update prescription status
   * @param id Prescription ID
   * @param status New status ('approved' or 'rejected')
   * @param rejectionReason Required if status is 'rejected'
   * @param pharmacistNotes Optional notes from the pharmacist
   */
  updatePrescriptionStatus: (
    id: string,
    status: 'approved' | 'rejected',
    rejectionReason?: string,
    pharmacistNotes?: string
  ) => {
    return api.put<ApiResponse<{
      _id: string;
      status: string;
      reviewedBy: string;
      reviewedAt: Date;
      rejectionReason?: string;
      pharmacistNotes?: string;
    }>>(`/prescriptions/${id}/status`, {
      status,
      rejectionReason,
      pharmacistNotes
    });
  },

  /**
   * Get prescription by ID
   * @param id Prescription ID
   */
  getPrescriptionById: (id: string) => {
    return api.get<ApiResponse<{
      _id: string;
      userId: string | { _id: string; name: string; email: string };
      orderId?: string | { _id: string; orderNumber: string };
      doctor_name?: string;
      patient_name: string;
      patient_age?: number;
      patient_gender?: string;
      diagnosis?: string;
      notes?: string;
      status: string;
      images: Array<{
        path: string;
        original_name: string;
        mimetype: string;
        size: number;
      }>;
      verification_history?: Array<{
        status: string;
        changedAt: Date;
        comment?: string;
        changedBy: string | { _id: string; name: string };
      }>;
      createdAt: Date;
      updatedAt: Date;
    }>>(`/prescriptions/${id}`);
  }
};

export const homepageSectionService = {
  /**
   * Fetch homepage sections for display
   * @param query Optional query string for filtering (e.g., '?lat=123&lng=456&storeId=123')
   */
  getDisplaySections: async (query: string = '') => {
    try {
      const url = `/homepage-sections/display${query}`;
      console.log('Fetching homepage sections from:', url);
      const response = await api.get(url);
      console.log('Homepage sections response:', response.data);
      return response.data;
    } catch (error) {
      console.error('Error fetching homepage sections:', error);
      throw error;
    }
  },

  /**
   * Fetch a single homepage section by ID
   * @param sectionId ID of the section to fetch
   */
  getSectionById: async (sectionId: string): Promise<{ success: boolean; data: any }> => {
    try {
      const response = await api.get(`/homepage-sections/${sectionId}`);
      return response.data;
    } catch (error) {
      console.error('Error fetching homepage section:', error);
      throw error;
    }
  },

  /**
   * Create a new homepage section (admin only)
   * @param sectionData Section data to create
   */
  createSection: async (sectionData: any) => {
    try {
      const response = await api.post('/homepage-sections', sectionData);
      return response.data;
    } catch (error) {
      console.error('Error creating homepage section:', error);
      throw error;
    }
  },

  /**
   * Update an existing homepage section (admin only)
   * @param sectionId ID of the section to update
   * @param sectionData Updated section data
   */
  updateSection: async (sectionId: string, sectionData: any) => {
    try {
      const response = await api.put(`/homepage-sections/${sectionId}`, sectionData);
      return response.data;
    } catch (error) {
      console.error('Error updating homepage section:', error);
      throw error;
    }
  },

  /**
   * Delete a homepage section (admin only)
   * @param sectionId ID of the section to delete
   */
  deleteSection: async (sectionId: string) => {
    try {
      const response = await api.delete(`/homepage-sections/${sectionId}`);
      return response.data;
    } catch (error) {
      console.error('Error deleting homepage section:', error);
      throw error;
    }
  },

  /**
   * Reorder homepage sections (admin only)
   * @param sections Array of sections with their new order
   */
  reorderSections: async (sections: any[]) => {
    try {
      const response = await api.put('/homepage-sections/reorder', { sections });
      return response.data;
    } catch (error) {
      console.error('Error reordering homepage sections:', error);
      throw error;
    }
  }
};
