import React, { useState, useEffect, useMemo, Suspense, lazy } from 'react';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '@/context/AuthContext';
import api from '@/services/api';
import { toast } from 'react-hot-toast';
import { LayoutDashboard, Package, ShoppingCart, Users, Settings, Box, Tag, Percent, MapPin, FileText, Bell, Download } from 'lucide-react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';


import { Plus, Search, Filter, X, Edit, Trash2, Eye, ChevronLeft, ChevronRight } from 'lucide-react';

// Helper function to safely get stock quantity from product
const getStockQuantity = (product: any): number => {
  // Check various possible locations for stock/quantity
  if (typeof product.quantity === 'number') return product.quantity;
  if (typeof product.stock === 'number') return product.stock;
  if (product.stockInfo && typeof product.stockInfo.current === 'number') return product.stockInfo.current;
  if (typeof product.quantity === 'string') return parseInt(product.quantity, 10) || 0;
  if (typeof product.stock === 'string') return parseInt(product.stock, 10) || 0;
  return 0;
};

// Helper function to format stock quantity with proper localization
const formatStockQuantity = (product: any): string => {
  const quantity = getStockQuantity(product);
  return quantity.toLocaleString();
};

// Helper function to format date for date input (yyyy-MM-dd)
const formatDateForInput = (dateString: string | null | undefined): string => {
  if (!dateString) return '';
  try {
    const date = new Date(dateString);
    // Check if date is valid
    if (isNaN(date.getTime())) return '';
    // Format as yyyy-MM-dd
    return date.toISOString().split('T')[0];
  } catch (error) {
    console.error('Error formatting date:', error);
    return '';
  }
};

// Helper function to format date for display in view modal
const formatDate = (dateString: string | null | undefined): string => {
  if (!dateString) return 'N/A';
  try {
    const date = new Date(dateString);
    return isNaN(date.getTime()) ? 'Invalid Date' : date.toLocaleDateString();
  } catch (error) {
    console.error('Error formatting date:', error);
    return 'Invalid Date';
  }
};

// Helper function to get unit short code
const getUnitShortCode = (product: any): string => {
  // Check various possible locations for unit information
  if (product.unit_id && typeof product.unit_id === 'object') {
    if (product.unit_id.shortCode) return product.unit_id.shortCode;
    if (product.unit_id.name) return product.unit_id.name;
  }
  if (product.unit && typeof product.unit === 'object') {
    if (product.unit.shortCode) return product.unit.shortCode;
    if (product.unit.name) return product.unit.name;
  }
  return '';
};

// Lazy load ReactQuill for better performance
const ReactQuill = lazy(() => import('react-quill'));
import 'react-quill/dist/quill.snow.css';

// Loading component for the editor
const EditorLoading = () => (
  <div className="h-32 bg-gray-100 rounded-lg animate-pulse"></div>
);

// Quill modules and formats
const modules = {
  toolbar: [
    [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
    ['bold', 'italic', 'underline', 'strike', 'blockquote'],
    [{'list': 'ordered'}, {'list': 'bullet'}, {'indent': '-1'}, {'indent': '+1'}],
    ['link', 'image'],
    ['clean']
  ],
};

const formats = [
  'header',
  'bold', 'italic', 'underline', 'strike', 'blockquote',
  'list', 'bullet', 'indent',
  'link', 'image'
];

interface Brand {
  _id: string;
  name: string;
  status: boolean;
  createdAt?: string;
  updatedAt?: string;
}

interface CategoryOption {
  _id: string;
  name: string;
}

interface UnitOption {
  _id: string;
  name: string;
  shortCode?: string;
}

interface TaxOption {
  _id: string;
  name: string;
  percentage: number;
  inclusive?: boolean; // Whether the tax is included in the price
}

interface DiscountOption {
  _id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
}

interface StoreOption {
  _id: string;
  name: string;
  storeName?: string;
}

interface ItemImage {
  _id?: string;
  url: string;
  is_primary: boolean;
  position: number;
}

interface Product {
  _id: string;
  name: string;
  sku: string;
  hsn?: string;
  barcode?: string;
  description?: string;
  images?: ItemImage[];
  price: number;
  sales_price?: number;
  expire_date?: string | null;
  quantity: number;
  status: boolean | string;
  brand_id: string | Brand;
  brand_name?: string;
  brandName?: string;
  stock?: number;
  stockInfo?: any;
  category_id: string | {
    _id: string;
    name: string;
  };
  unit_id?: string | {
    _id: string;
    name: string;
    shortCode?: string;
  };
  tax_id?: string | {
    _id: string;
    name: string;
    percentage: number;
  };
  tax_type?: 'inclusive' | 'exclusive';
  discount_id?: string | {
    _id: string;
    name: string;
    type: 'percentage' | 'fixed';
    value: number;
  };
  store_id: string | {
    _id: string;
    name: string;
    storeName?: string;
  };
  image?: string;
  brand?: Brand;
  category?: {
    _id: string;
    name: string;
  };
  store?: {
    _id: string;
    name: string;
    storeName?: string;
  };
  createdAt?: string;
  updatedAt?: string;
}

type ProductImage = string | File | ItemImage;

interface ItemFormData {
  name: string;
  sku: string;
  price: string;
  brand_id: string;
  category_id: string;
  unit_id: string;
  description: string;
  hsn: string;
  barcode: string;
  images?: ProductImage[];
  expire_date: string;
  sales_price: string;
  tax_id: string;
  tax_type: 'inclusive' | 'exclusive';
  discount_id: string;
  store_id: string;
  status: boolean;
  quantity: string;
  image?: string; // For backward compatibility
}

const Products: React.FC = () => {
  // Initialize auth first
  const { user } = useAuth();
  const navigate = useNavigate();
  
  // State for data fetching and UI
  const [products, setProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [totalPages, setTotalPages] = useState<number>(1);
  
  // Modal states
  const [selectedProduct, setSelectedProduct] = useState<Product | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState<boolean>(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState<boolean>(false);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState<boolean>(false);
  
  // Image handling
  const [imagePreviews, setImagePreviews] = useState<string[]>([]);
  const [selectedImages, setSelectedImages] = useState<File[]>([]);
  
  // Data states
  const [brands, setBrands] = useState<Brand[]>([]);
  const [categories, setCategories] = useState<CategoryOption[]>([]);
  const [units, setUnits] = useState<UnitOption[]>([]);
  const [taxes, setTaxes] = useState<TaxOption[]>([]);
  const [discounts, setDiscounts] = useState<DiscountOption[]>([]);
  const [stores, setStores] = useState<StoreOption[]>([]);
  const [selectedDiscount, setSelectedDiscount] = useState<DiscountOption | null>(null);
  
  // Loading states
  const [loadingBrands, setLoadingBrands] = useState<boolean>(false);
  const [loadingCategories, setLoadingCategories] = useState<boolean>(false);
  const [loadingUnits, setLoadingUnits] = useState<boolean>(false);
  const [loadingTaxes, setLoadingTaxes] = useState<boolean>(false);
  const [loadingDiscounts, setLoadingDiscounts] = useState<boolean>(false);
  const [loadingStores, setLoadingStores] = useState<boolean>(false);
  const [isSubmitting, setIsSubmitting] = useState<boolean>(false);
  
  // Form errors
  const [errors, setErrors] = useState<Record<string, string>>({});
  
  // Initialize form data with default values
  const getInitialFormData = (): ItemFormData => ({
    name: '',
    sku: '',
    price: '0',
    brand_id: '',
    category_id: '',
    unit_id: '',
    description: '',
    hsn: '',
    barcode: '',
    images: [],
    expire_date: '',
    sales_price: '0',
    tax_id: '',
    tax_type: 'inclusive',
    discount_id: '',
    store_id: user?.store?._id || user?.store_id || '',
    status: true,
    quantity: '0',
  });

  // Reset form function
  const resetForm = () => {
    setFormData(getInitialFormData());
    setImagePreviews([]);
    setSelectedImages([]);
    setSelectedDiscount(null);
    setErrors({});
  };

  // Initialize form data
  const [formData, setFormData] = useState<ItemFormData>(getInitialFormData());

  // Reset form when switching between create and edit modes
  useEffect(() => {
    if (isCreateModalOpen || isEditModalOpen) {
      // Only reset if we're opening a modal
      if (!isCreateModalOpen && !isEditModalOpen) {
        setFormData(getInitialFormData());
      }
    }
  }, [isCreateModalOpen, isEditModalOpen]);
  
  // Get current store name for display
  const currentStoreName = useMemo(() => {
    if (!user) return 'Loading...';
    return user?.store?.name || user?.store?.storeName || 'My Store';
  }, [user]);

  // Handle image upload
  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (!files || files.length === 0) return;

    const newImages = Array.from(files).map(file => {
      // Validate file type
      if (!file.type.startsWith('image/')) {
        console.warn(`File ${file.name} is not an image`);
        return null;
      }
      // Validate file size (max 5MB)
      if (file.size > 5 * 1024 * 1024) {
        console.warn(`File ${file.name} is too large (max 5MB)`);
        return null;
      }
      return file;
    }).filter(Boolean) as File[]; // Filter out null values

    setFormData(prev => {
      const currentImages = Array.isArray(prev.images) ? [...prev.images] : [];
      return {
        ...prev,
        images: [...currentImages, ...newImages].slice(0, 10) // Limit to 10 images max
      };
    });
  };



  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (isSubmitting) return;
    
    setIsSubmitting(true);
    
    try {
      const formDataToSend = new FormData();
      
      // Create a clean copy of form data with proper string conversion
      const cleanFormData: Record<string, any> = {};
      
      // Process each field in formData
      Object.entries(formData).forEach(([key, value]) => {
        // Skip internal fields that shouldn't be sent to the API
        if (['_id', 'createdAt', 'updatedAt', 'brand', 'category', 'unit', 'tax', 'discount', 'store'].includes(key)) return;
        
        // Handle special cases
        if (key === 'images') {
          // Images are handled separately below
          return;
        } else if (['quantity', 'price', 'sales_price'].includes(key)) {
          // Ensure numeric fields are properly formatted
          cleanFormData[key] = value === '' ? '0' : String(Number(value) || 0);
        } else if (key === 'status') {
          // Convert status to number (0 or 1) for the backend
          cleanFormData[key] = (value === true || value === 'true' || value === '1') ? 1 : 0;
        } else if (key === 'expire_date') {
          // Format date to ISO string if it exists
          cleanFormData[key] = value ? new Date(String(value)).toISOString().split('T')[0] : '';
        } else if (key.endsWith('_id') && value) {
          // Handle all _id fields (brand_id, category_id, etc.)
          cleanFormData[key] = typeof value === 'object' ? value._id : value;
        } else if (key === 'sku') {
          // Always include SKU in the form data
          cleanFormData[key] = String(value).trim();
          
          // If we're in edit mode and the SKU hasn't changed, 
          // we'll add a flag to let the backend know not to validate SKU uniqueness
          if (isEditModalOpen && selectedProduct && value === selectedProduct.sku) {
            // Add the flag to both cleanFormData and formDataToSend
            cleanFormData['skipSkuValidation'] = 'true';
            formDataToSend.append('skipSkuValidation', 'true');
          }
        } else if (value !== null && value !== undefined) {
          // Default case - convert to string and trim whitespace
          cleanFormData[key] = String(value).trim();
        }
      });
      
      // Log the data being sent for debugging
      console.log('Submitting form data:', cleanFormData);
      
      // Include the product ID in the form data for updates
      if (isEditModalOpen && selectedProduct?._id) {
        formDataToSend.append('_id', selectedProduct._id);
        // Add skipSkuValidation flag to both formDataToSend and cleanFormData
        formDataToSend.append('skipSkuValidation', 'true');
        cleanFormData.skipSkuValidation = 'true';
        
        // Debug log to verify the flag is set
        console.log('Setting skipSkuValidation to true for product update');
      }
      
      // Append all form data to FormData
      Object.entries(cleanFormData).forEach(([key, value]) => {
        // Skip appending _id here as we've already handled it
        if (value !== null && value !== undefined && key !== '_id') {
          // Convert boolean to string if needed
          const finalValue = typeof value === 'boolean' ? String(value) : value;
          formDataToSend.append(key, finalValue);
        }
      });
      
      // Handle image uploads
      if (formData.images && formData.images.length > 0) {
        formData.images.forEach((image: any, index: number) => {
          if (image instanceof File) {
            formDataToSend.append('images', image);
          } else if (typeof image === 'string') {
            formDataToSend.append('existing_images[]', image);
          } else if (image && typeof image === 'object' && 'url' in image) {
            formDataToSend.append('existing_images[]', image.url);
          }
        });
      }
      
      if (isEditModalOpen && selectedProduct?._id) {
        // Update existing product - use PATCH instead of PUT for partial updates
        await api.patch(`/items/${selectedProduct._id}`, formDataToSend, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Product updated successfully', { duration: 2000, position: 'top-right' });
      } else {
        // Create new product
        await api.post('/items', formDataToSend, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Product created successfully', { duration: 2000, position: 'top-right' });
      }
      
      // Refresh products list
      fetchProducts();
      
      // Close modal and reset form
      setIsCreateModalOpen(false);
      setIsEditModalOpen(false);
      const resetForm = () => {
        setFormData({
          name: '',
          sku: '',
          price: '0',
          brand_id: '',
          category_id: '',
          unit_id: '',
          description: '',
          hsn: '',
          barcode: '',
          images: [],
          expire_date: '',
          sales_price: '0',
          tax_id: '',
          tax_type: 'inclusive',
          discount_id: '',
          store_id: user?.store?._id || user?.store_id || '',
          status: true,
          quantity: '0',
        });
      }
      resetForm();
    } catch (error: any) {
      console.error('Error saving product:', error);
      // Log the full error response if available
      if (error.response) {
        console.error('Error response data:', error.response.data);
        console.error('Error response status:', error.response.status);
        console.error('Error response headers:', error.response.headers);
      }
      const errorMessage = error.response?.data?.message || 'Failed to save product. Please check the console for details.';
      toast.error(errorMessage, { duration: 5000, position: 'top-right' });
      console.error('Save product error details:', {
        message: error.message,
        code: error.code,
        config: error.config,
        response: error.response?.data
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  // Calculate final price after applying discount
  const calculateDiscountedPrice = (price: string, discount: DiscountOption | null): number => {
    const basePrice = parseFloat(price) || 0;
    
    // If no discount is selected or price is not a valid number, return the base price
    if (!discount || isNaN(basePrice)) {
      return basePrice;
    }

    // Calculate discount based on type
    if (discount.type === 'percentage') {
      const discountAmount = (basePrice * discount.value) / 100;
      return Math.max(0, basePrice - discountAmount);
    } else if (discount.type === 'fixed') {
      return Math.max(0, basePrice - discount.value);
    }
    
    return basePrice;
  };

  // Calculate tax amount based on price and tax rate
  const calculateTaxAmount = (price: number, taxRate: number, isInclusive: boolean): number => {
    if (isInclusive) {
      // For inclusive tax, calculate the tax amount from the price that already includes tax
      return price - (price / (1 + (taxRate / 100)));
    } else {
      // For exclusive tax, calculate tax as a percentage of the price
      return (price * taxRate) / 100;
    }
  };

  // Calculate final price including tax and discount
  const calculateFinalPrice = (price: string, discount: DiscountOption | null, taxId: string, taxType: 'inclusive' | 'exclusive', taxList: TaxOption[]): { finalPrice: number, taxAmount: number } => {
    const basePrice = parseFloat(price) || 0;
    
    // Apply discount first
    const priceAfterDiscount = calculateDiscountedPrice(price, discount);
    
    // Find the tax
    const tax = taxList.find(t => t._id === taxId);
    
    if (!tax) {
      return { finalPrice: priceAfterDiscount, taxAmount: 0 };
    }
    
    const taxAmount = calculateTaxAmount(priceAfterDiscount, tax.percentage, taxType === 'inclusive');
    
    // Calculate final price based on tax type
    const finalPrice = taxType === 'inclusive' 
      ? priceAfterDiscount // Price already includes tax
      : priceAfterDiscount + taxAmount; // Add tax to the price
    
    return { 
      finalPrice: parseFloat(finalPrice.toFixed(2)), 
      taxAmount: parseFloat(taxAmount.toFixed(2)) 
    };
  };

  // Calculate final price whenever relevant values change
  const { finalPrice, taxAmount } = useMemo(() => {
    if (!formData.price) {
      return { finalPrice: 0, taxAmount: 0 };
    }
    
    return calculateFinalPrice(
      formData.price, 
      selectedDiscount, 
      formData.tax_id, 
      formData.tax_type, 
      taxes || []
    );
  }, [formData.price, formData.tax_id, formData.tax_type, selectedDiscount, taxes]);
  
  // Log tax calculation for debugging
  useEffect(() => {
    console.log('Tax Calculation Debug:', {
      price: formData.price,
      tax_id: formData.tax_id,
      tax_type: formData.tax_type,
      selectedDiscount,
      taxes,
      finalPrice,
      taxAmount
    });
  }, [formData.price, formData.tax_id, formData.tax_type, selectedDiscount, taxes, finalPrice, taxAmount]);

  // Update sales price when price, discount, tax, or tax type changes
  useEffect(() => {
    if (formData.price) {
      const newSalesPrice = parseFloat(finalPrice.toFixed(2));
      if (!isNaN(newSalesPrice) && newSalesPrice !== parseFloat(formData.sales_price || '0')) {
        setFormData(prev => ({
          ...prev,
          sales_price: newSalesPrice.toString()
        }));
      }
    }
  }, [finalPrice, formData.price]);
  
  // Handle tax type change
  const handleTaxTypeChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const newTaxType = e.target.value as 'inclusive' | 'exclusive';
    setFormData(prev => ({
      ...prev,
      tax_type: newTaxType
    }));
  };
  
  // Handle tax selection change
  const handleTaxChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const taxId = e.target.value;
    const selectedTax = taxes.find(tax => tax._id === taxId);
    
    setFormData(prev => ({
      ...prev,
      tax_id: taxId
    }));
    
    // If we have a selected tax, update the tax type to match the tax's default if not set
    if (selectedTax && !formData.tax_type) {
      setFormData(prev => ({
        ...prev,
        tax_type: selectedTax.inclusive ? 'inclusive' : 'exclusive'
      }));
    }
  };

  // Fetch brands with their names
  const fetchBrandsWithNames = async () => {
    try {
      const response = await api.get('/brands?status=true&limit=1000');
      return response.data.data || [];
    } catch (error) {
      // Don't show error toast for initial load to prevent false alerts
      if (page > 1) {
        console.warn('Failed to load more products');
      }
    } finally {
      setLoading(false);
    }
  };

  // Fetch units
  const fetchUnits = async () => {
    try {
      setLoadingUnits(true);
      const response = await api.get('/units?status=true&limit=1000');
      setUnits(response.data.data || []);
    } catch (error) {
      console.warn('Could not load units:', error);
      // Silently handle the error without showing a toast
    } finally {
      setLoadingUnits(false);
    }
  };

  // Fetch taxes
  const fetchTaxes = async () => {
    try {
      setLoadingTaxes(true);
      const response = await api.get('/taxes?status=true&limit=1000');
      setTaxes(response.data.data || []);
    } catch (error) {
      console.warn('Could not load taxes:', error);
      // Silently handle the error without showing a toast
    } finally {
      setLoadingTaxes(false);
    }
  };

  // Fetch discounts
  const fetchDiscounts = async () => {
    try {
      setLoadingDiscounts(true);
      const response = await api.get('/discounts?status=true&limit=1000');
      setDiscounts(response.data.data || []);
    } catch (error) {
      console.warn('Could not load discounts:', error);
      // Silently handle the error without showing a toast
    } finally {
      setLoadingDiscounts(false);
    }
  };

  // Fetch categories and brands for form
  const fetchCategoriesAndBrands = async () => {
    try {
      console.log('Fetching categories and brands...');
      
      // Fetch categories and brands in parallel
      const [categoriesRes, brandsRes] = await Promise.all([
        api.get('/categories?status=true&limit=1000'),
        api.get('/brands?status=true&limit=1000')
      ]);
      
      console.log('Categories response:', categoriesRes);
      console.log('Brands response:', brandsRes);
      
      // Handle categories response
      const categoriesData = categoriesRes.data?.data || [];
      console.log('Categories data:', categoriesData);
      setCategories(categoriesData);
      
      // Handle brands response - check both possible response structures
      let brandsData = [];
      if (Array.isArray(brandsRes.data)) {
        // If response is directly an array
        brandsData = brandsRes.data;
      } else if (brandsRes.data && Array.isArray(brandsRes.data.data)) {
        // If response has a data property that's an array
        brandsData = brandsRes.data.data;
      }
      
      // Map brandName to name for compatibility with the rest of the application
      brandsData = brandsData.map(brand => ({
        ...brand,
        name: brand.brandName || brand.name // Use brandName if available, fallback to name
      }));
      
      console.log('Brands data:', brandsData);
      setBrands(brandsData);
      
    } catch (error: unknown) {
      // Type guard to check if error is an instance of Error
      const errorMessage = error instanceof Error ? error.message : 'Unknown error occurred';
      const axiosError = error as any; // Type assertion to access axios error properties
      
      console.error('Error fetching form data:', error);
      console.error('Error details:', {
        message: errorMessage,
        response: axiosError.response?.data,
        status: axiosError.response?.status,
        config: {
          url: axiosError.config?.url,
          method: axiosError.config?.method,
          params: axiosError.config?.params
        }
      });
      
      // Set empty arrays to prevent UI errors
      setCategories([]);
      setBrands([]);
      
      // Show error toast to user
      toast.error('Failed to load brands and categories. Please try again later.', {
        duration: 5000,
        position: 'top-right'
      });
    }
  };

  // Fetch all required data
  const fetchAllData = async () => {
    try {
      // First fetch categories and brands together
      await fetchCategoriesAndBrands();
      
      // Then fetch units, taxes, and discounts in parallel
      await Promise.all([
        fetchUnits(),
        fetchTaxes(),
        fetchDiscounts()
      ]);
    } catch (error) {
      console.warn('Error loading some data:', error);
      // Silently handle the error without showing a toast
    }
  };

  useEffect(() => {
    console.log('Current User Object:', user);
    console.log('User ID:', user?._id || user?.id || 'No user ID found');
    console.log('Store ID:', user?.store_id || user?.storeId || user?.store?._id || 'No store ID found');

    if (!user) {
      console.warn('No user object found in AuthContext');
      return;
    }

    fetchProducts();
    fetchAllData();
  }, [user]);

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    fetchProducts(1, value, statusFilter);
  };

  // Handle filter change
  const handleFilterChange = (value: string) => {
    setStatusFilter(value);
    fetchProducts(1, searchTerm, value);
  };

  // Handle page change
  const handlePageChange = (page: number) => {
    fetchProducts(page, searchTerm, statusFilter);
  };

  // Handle view product
  const handleViewProduct = (product: Product) => {
    setSelectedProduct(product);
    setIsViewModalOpen(true);
  };

  // Handle delete product
  const handleDeleteProduct = async (product: Product) => {
    if (!product._id) {
      console.error('Cannot delete product: No product ID');
      return;
    }

    const confirmDelete = window.confirm(`Are you sure you want to delete "${product.name}"?`);
    if (!confirmDelete) return;

    try {
      setLoading(true);
      await api.delete(`/items/${product._id}`);
      
      // Remove the deleted product from the local state
      setProducts(prevProducts => prevProducts.filter(p => p._id !== product._id));
      
      toast.success('Product deleted successfully', { duration: 2000, position: 'top-right' });
    } catch (error) {
      console.error('Error deleting product:', error);
      toast.error('Failed to delete product. Please try again.', { duration: 3000, position: 'top-right' });
      console.error('Delete product error');
    } finally {
      setLoading(false);
    }
  };

  // Handle edit product
  const handleEditProduct = (product: Product) => {
    console.log('Editing product:', product);
    
    // Extract brand ID, handling both object and string cases
    let brandId = '';
    if (product.brand_id) {
      brandId = typeof product.brand_id === 'object' ? product.brand_id._id : product.brand_id;
    } else if (product.brand) {
      // Handle case where brand is populated as 'brand' instead of 'brand_id'
      brandId = product.brand._id;
    }

    // Set form data with product details
    setFormData({
      ...formData,
      _id: product._id,
      name: product.name,
      sku: product.sku || '',
      price: product.price.toString(),
      brand_id: brandId,
      category_id: typeof product.category_id === 'object' ? product.category_id._id : product.category_id || '',
      unit_id: typeof product.unit_id === 'object' ? product.unit_id._id : product.unit_id || '',
      description: product.description || '',
      hsn: product.hsn || '',
      barcode: product.barcode || '',
      expire_date: formatDateForInput(product.expire_date),
      sales_price: product.sales_price ? product.sales_price.toString() : '',
      tax_id: typeof product.tax_id === 'object' ? product.tax_id._id : product.tax_id || '',
      tax_type: product.tax_type || 'inclusive',
      discount_id: typeof product.discount_id === 'object' ? product.discount_id._id : product.discount_id || '',
      store_id: typeof product.store_id === 'object' ? product.store_id._id : product.store_id || '',
      status: product.status as boolean,
      quantity: product.quantity ? product.quantity.toString() : '0',
      image: product.image || '',
    });

    // Set image previews if product has images
    if (product.images && product.images.length > 0) {
      setImagePreviews(
        product.images.map((img) => ({
          url: img.url,
          is_primary: img.is_primary,
          position: img.position,
        }))
      );
    } else if (product.image) {
      setImagePreviews([{ url: product.image, is_primary: true, position: 0 }]);
    } else {
      setImagePreviews([]);
    }

    // Set selected discount if exists
    if (product.discount_id) {
      const discount = discounts.find(
        (d) => d._id === (typeof product.discount_id === 'object' ? product.discount_id._id : product.discount_id)
      );
      if (discount) {
        setSelectedDiscount(discount);
      }
    } else {
      setSelectedDiscount(null);
    }

    // Open edit modal
    setIsEditModalOpen(true);
  };

  // Handle create product
  const handleCreateProduct = async (e: React.FormEvent) => {
    e.preventDefault();
    
    try {
      setIsSubmitting(true);
      
      // Make sure we have the current user's store ID
      const currentStoreId = user?.store?._id || user?.store_id;
      if (!currentStoreId) {
        throw new Error('No store associated with your account. Please contact support.');
      }
      
      const formDataToSend = new FormData();
      
      // Prepare the data to be sent
      const formDataCopy = { ...formData };
      
      // Ensure all ID fields are properly formatted
      const idFields = ['brand_id', 'category_id', 'unit_id', 'tax_id', 'discount_id', 'store_id'];
      idFields.forEach(field => {
        if (formDataCopy[field as keyof ItemFormData]) {
          const value = formDataCopy[field as keyof ItemFormData];
          if (typeof value === 'object' && value !== null && '_id' in value) {
            // If the value is an object with an _id, use that
            formDataCopy[field as keyof ItemFormData] = (value as any)._id;
          }
        }
      });
      
      // Set the store ID
      formDataCopy.store_id = currentStoreId;
      
      // Include the SKU in the form data
      if (formData.sku) {
        formDataToSend.append('sku', formData.sku);
      }
      
      // Append other fields
      Object.entries(formDataCopy).forEach(([key, value]) => {
        if (key !== 'images' && value !== null && value !== undefined) {
          if (key === 'status') {
            formDataToSend.append(key, String(value));
          } else if (key === 'price' || key === 'sales_price' || key === 'quantity') {
            formDataToSend.append(key, String(Number(value) || 0));
          } else if (key === 'expire_date' && value) {
            // Format date as YYYY-MM-DD
            const date = new Date(value);
            if (!isNaN(date.getTime())) {
              formDataToSend.append(key, date.toISOString().split('T')[0]);
            }
          } else if (key === 'sku') {
            // Skip if already added
            if (!formDataToSend.has('sku')) {
              formDataToSend.append('sku', String(value));
            }
          } else {
            formDataToSend.append(key, String(value));
          }
        }
      });
      
      // Append images
      selectedImages.forEach((file) => {
        formDataToSend.append('images', file);
      });
      
      const response = await api.post('/products', formDataToSend, {
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });
      
      toast.success('Product created successfully');
      setIsCreateModalOpen(false);
      fetchProducts();
      
      // Reset form
      setFormData({
        name: '',
        sku: '',
        price: '0',
        brand_id: '',
        category_id: '',
        unit_id: '',
        description: '',
        hsn: '',
        barcode: '',
        images: [],
        minimum_qty: '1',
        expire_date: '',
        sales_price: '0',
        tax_id: '',
        tax_type: 'inclusive',
        discount_id: '',
        store_id: user?.store?._id || '',
        status: true,
        quantity: '0',
      });
      
      setSelectedImages([]);
      setImagePreviews([]);
      setSelectedDiscount(null);
      
    } catch (error: any) {
      console.error('Error creating product:', error);
      const errorMessage = error.response?.data?.message || 'Failed to create product';
      toast.error(errorMessage, { duration: 3000, position: 'top-right' });
      console.error('Create product error:', errorMessage);
    } finally {
      setIsSubmitting(false);
    }
  };

  // Handle field changes
  const handleFieldChange = (field: string, value: any) => {
    // Special handling for date fields to ensure proper format
    if (field === 'expire_date') {
      // If the date is empty, set it to an empty string
      if (!value) {
        setFormData(prev => ({
          ...prev,
          [field]: ''
        }));
        return;
      }
      
      // If it's already in the correct format, use it as is
      if (typeof value === 'string' && /^\d{4}-\d{2}-\d{2}$/.test(value)) {
        setFormData(prev => ({
          ...prev,
          [field]: value
        }));
        return;
      }
      
      // Otherwise, try to format it
      try {
        const date = new Date(value);
        if (!isNaN(date.getTime())) {
          const formattedDate = date.toISOString().split('T')[0];
          setFormData(prev => ({
            ...prev,
            [field]: formattedDate
          }));
          return;
        }
      } catch (error) {
        console.error('Error formatting date:', error);
      }
    }
    
    // For tax_id field, update tax_type based on the selected tax's inclusive property
    if (field === 'tax_id') {
      const tax = taxes.find(t => t._id === value);
      if (tax) {
        setFormData(prev => ({
          ...prev,
          tax_id: value,
          tax_type: tax.inclusive ? 'inclusive' : 'exclusive'
        }));
        return;
      }
    }
    
    // For all other fields, update normally
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));

    // Update selected discount when discount_id changes
    if (field === 'discount_id') {
      const discount = discounts.find(d => d._id === value) || null;
      setSelectedDiscount(discount);
    }
  };

  // Handle image upload
  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files.length > 0) {
      const files = Array.from(e.target.files);
      const newPreviews = files.map(file => ({
        url: URL.createObjectURL(file),
        is_primary: false,
        position: 0
      }));
      
      setSelectedImages(prev => [...prev, ...files]);
      setImagePreviews(prev => [...prev, ...newPreviews]);
      
      // Update form data with the new files
      setFormData(prev => ({
        ...prev,
        images: [...(prev.images || []), ...files]
      }));
    }
  };

  // Remove image from preview and form data
  const removeImage = (index: number) => {
    setImagePreviews(prev => prev.filter((_, i) => i !== index));
    setFormData(prev => {
      const currentImages = Array.isArray(prev.images) ? [...prev.images] : [];
      const newImages = [...currentImages];
      newImages.splice(index, 1);
      return { ...prev, images: newImages };
    });
  };


  // Handle image previews
  const handleImagePreviews = (files: FileList) => {
    const newPreviews = Array.from(files).map(file => URL.createObjectURL(file));
    setImagePreviews(prev => [...prev, ...newPreviews].slice(0, 10));
    setSelectedImages(prev => [...prev, ...Array.from(files)]);
    
    // Update form data with new files
    setFormData(prev => ({
      ...prev,
      images: [...(prev.images || []), ...Array.from(files)].slice(0, 10)
    }));
  };

  // Set primary image (moves image to first position in array)
  const setPrimaryImage = (index: number) => {
    setFormData(prev => {
      const currentImages = Array.isArray(prev.images) ? [...prev.images] : [];
      if (index < 0 || index >= currentImages.length) return prev;

      const newImages = [...currentImages];
      const [selectedImage] = newImages.splice(index, 1);
      return { ...prev, images: [selectedImage, ...newImages] };
    });
  };

  // Fetch brands
  const fetchBrands = async () => {
    try {
      setLoadingBrands(true);
      const response = await api.get('/brands?status=true&limit=1000');
      setBrands(response.data.data || []);
    } catch (error) {
      console.warn('Could not load brands:', error);
      // Silently handle the error without showing a toast
    } finally {
      setLoadingBrands(false);
    }
  };

  // Fetch categories
  const fetchCategories = async () => {
    try {
      setLoadingCategories(true);
      const response = await api.get('/categories?status=true&limit=1000');
      setCategories(response.data.data || []);
    } catch (error) {
      console.warn('Could not load categories:', error);
      // Silently handle the error without showing a toast
    } finally {
      setLoadingCategories(false);
    }
  };

  // Fetch stores
  const fetchStores = async () => {
    try {
      setLoadingStores(true);
      const response = await api.get('/stores?status=true&limit=1000');
      setStores(response.data.data || []);

      // Auto-select the first store if only one exists
      if (response.data.data?.length === 1) {
        setFormData(prev => ({
          ...prev,
          store_id: response.data.data[0]._id,
        }));
      }
    } catch (error) {
      console.error('Error fetching stores:', error);
      toast.error('Failed to load stores');
    } finally {
      setLoadingStores(false);
    }
  };

  // State to store brand names
  const [brandNames, setBrandNames] = useState<Record<string, string>>({});
  
  // Effect to update brand names when brands array changes
  useEffect(() => {
    const newBrandNames: Record<string, string> = {};
    brands.forEach(brand => {
      if (brand && typeof brand === 'object') {
        newBrandNames[brand._id] = brand.name ;
      }
    });
    setBrandNames(prev => ({ ...prev, ...newBrandNames }));
  }, [brands]);

  // Helper function to get brand name with fallbacks
  const getBrandName = (product: Product) => {
    // If we already have the brand name in state, return it
    if (product.brand_id && brandNames[product.brand_id]) {
      return brandNames[product.brand_id];
    }
    
    // If brand is a populated object with name
    if (product.brand && typeof product.brand === 'object') {
      const brandName = product.brand.name || 
                       (product.brand as any).brandName || 
                       (product.brand as any).companyName;
      
      if (brandName) {
        // Update the brand names state
        setBrandNames(prev => ({
          ...prev,
          [product.brand_id as string]: brandName
        }));
        return brandName;
      }
    }
    
    // If we have a brand_id but no name yet, trigger a fetch
    if (product.brand_id && !brandNames[product.brand_id]) {
      fetchBrandName(product.brand_id);
      return 'Loading...';
    }
    
    // Fallback to brand_name or default
    return product.brand_name || 'No Brand';
  };
  
  // Function to fetch brand name by ID
  const fetchBrandName = async (brandId: string) => {
    try {
      // Skip if already loading
      if (brandNames[brandId] === 'Loading...') return;
      
      // Set loading state
      setBrandNames(prev => ({
        ...prev,
        [brandId]: 'Loading...'
      }));
      
      const response = await api.get(`/brands/${brandId}`);
      if (response.data?.data) {
        const brand = response.data.data;
        const brandName = brand.name || brand.brandName || brand.companyName || 'No Brand';
        
        // Update the brands array with the fetched brand
        setBrands(prev => {
          if (!prev.some(b => b._id === brand._id)) {
            return [...prev, brand];
          }
          return prev;
        });
        
        // Update the brand name in state
        setBrandNames(prev => ({
          ...prev,
          [brandId]: brandName
        }));
      }
    } catch (error) {
      console.error('Error fetching brand:', error);
      // Set error state
      setBrandNames(prev => ({
        ...prev,
        [brandId]: 'Error loading brand'
      }));
    }
  };

  // Fetch all products for the current store (both active and inactive)
  const fetchProducts = async () => {
    const storeId = user?.store?._id;

    if (!storeId) {
      console.error('No store ID found');
      setLoading(false);
      return;
    }

    setLoading(true);
    try {
      // First, ensure we have brands loaded
      if (brands.length === 0) {
        await fetchBrands();
      }

      // Get all items for the current store (both active and inactive)
      const response = await api.get('/items', {
        params: {
          limit: 1000, // Increase limit to get all items
          store_id: storeId,
          populate: 'category_id,brand_id,unit_id,store_id',
          status: 'all' // Explicitly fetch all statuses
        }
      });
      
      if (response.data?.data) {
        // Process and enrich products with brand names and ensure consistent data structure
        const processedProducts = response.data.data.map((product: any) => {
          const status = typeof product.status === 'string' 
            ? product.status === 'true' 
            : Boolean(product.status);
          
          // Handle brand data - it might be a populated object or just an ID
          let brandData = null;
          if (product.brand_id && typeof product.brand_id === 'object') {
            // If brand_id is a populated object, use it
            brandData = product.brand_id;
          } else if (product.brand_id && typeof product.brand_id === 'string') {
            // If brand_id is just an ID, try to find it in our brands array
            brandData = brands.find(b => b._id === product.brand_id) || null;
          }
          
          return {
            ...product,
            status,
            // Ensure brand_id is always an ID string
            brand_id: brandData?._id || product.brand_id || '',
            // Add brand object for easy access
            brand: brandData || null,
            // For backward compatibility
            brand_name: brandData?.name || brandData?.brandName || brandData?.companyName || product.brand_name || ''
          };
        });
        
        setProducts(processedProducts);
        setCurrentPage(response.data.pagination?.currentPage || page);
        setTotalPages(response.data.pagination?.totalPages || 1);
      }
    } catch (error) {
      console.error('Error fetching products:', error);
      toast.error('Failed to load products');
    } finally {
      setLoading(false);
    }
  };

  // Initial data fetch
useEffect(() => {
  fetchAllData();
}, []);

return (
  <Layout title="Products" sidebar={<Sidebar items={storeSidebarItems} />}>
    <div className="space-y-6">
      {/* Header */}
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <h1 className="text-2xl font-bold">Products</h1>
        <button
          onClick={() => setIsCreateModalOpen(true)}
          className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
        >
          <Plus className="w-4 h-4 mr-2" />
          Add New Product
        </button>
      </div>

      {/* Filters */}
      <div className="flex flex-col sm:flex-row gap-4">
        <div className="relative flex-grow">
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
          <input
            type="text"
            placeholder="Search products..."
            className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            value={searchTerm}
            onChange={(e) => handleSearch(e.target.value)}
          />
        </div>
        <select
          value={statusFilter}
          onChange={(e) => handleFilterChange(e.target.value)}
          className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
        >
          <option value="all">All Status</option>
          <option value="active">Active</option>
          <option value="inactive">Inactive</option>
          <option value="out_of_stock">Out of Stock</option>
        </select>
        <button
          onClick={() => {
            setSearchTerm('');
            setStatusFilter('all');
            fetchProducts(1, '', 'all');
          }}
          className="flex items-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
        >
          <Filter className="w-4 h-4 mr-2" />
          Reset Filters
        </button>
        <button
          onClick={() => toast('Export functionality coming soon', { duration: 2000, position: 'top-right' })}
          className="flex items-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
        >
          <Download className="w-4 h-4 mr-2" />
          Export
        </button>
      </div>

      {/* Products Table */}
      <div className="bg-white shadow-sm border rounded-lg overflow-hidden">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Product
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  SKU
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Category
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Price
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Stock
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="divide-y divide-gray-200">
              {loading ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
                    <div className="flex justify-center">
                      <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                    </div>
                    <p className="mt-2">Loading products...</p>
                  </td>
                </tr>
              ) : products.length === 0 ? (
                <tr>
                  <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
                    No products found
                  </td>
                </tr>
              ) : (
                products.map((product: Product) => {
                  // Log brand data for debugging
                  console.log('Product ID:', product._id, 'Brand Data:', {
                    brand: product.brand,
                    brand_id: product.brand_id,
                    brand_name: product.brand_name,
                    brandName: (product as any).brandName
                  });
                  
                  return (
                    <tr key={product._id}>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <div className="h-10 w-10 flex-shrink-0">
                            {product.image ? (
                              <img
                                className="h-10 w-10 rounded-full object-cover"
                                src={product.image}
                                alt={product.name}
                              />
                            ) : (
                              <div className="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                <Package className="h-5 w-5 text-gray-500" />
                              </div>
                            )}
                          </div>
                          <div className="ml-4">
                            <div className="text-sm font-medium text-gray-900">{product.name}</div>
                            <div className="text-sm text-gray-500">
                              {getBrandName(product)}
                            </div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {product.sku || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {product.category_id?.name || product.category?.name || 'Uncategorized'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">₹{product.price.toFixed(2)}</div>
                        {product.sales_price && product.sales_price < product.price && (
                          <div className="text-sm text-green-600">₹{product.sales_price.toFixed(2)}</div>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          {/* Debug info - will be removed later */}
                          {process.env.NODE_ENV === 'development' && (
                            <div className="hidden">
                              {JSON.stringify({
                                quantity: product.quantity,
                                unit: product.unit_id,
                                stock: product.stock,
                                stockInfo: product.stockInfo
                              })}
                            </div>
                          )}
                          
                          {/* Stock display - quantity only */}
                          <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                            getStockQuantity(product) > 10
                              ? 'bg-green-100 text-green-800'
                              : getStockQuantity(product) > 0
                              ? 'bg-yellow-100 text-yellow-800'
                              : 'bg-red-100 text-red-800'
                          }`}>
                            {formatStockQuantity(product)}
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                            product.status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                          }`}>
                            {product.status ? 'Active' : 'Inactive'}
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                        <div className="flex space-x-2">
                          <button
                            onClick={() => handleViewProduct(product)}
                            className="text-blue-600 hover:text-blue-900"
                            title="View Product"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button
                            onClick={() => handleEditProduct(product)}
                            className="text-indigo-600 hover:text-indigo-900"
                            title="Edit Product"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button
                            onClick={() => handleDeleteProduct(product)}
                            className="text-red-600 hover:text-red-900"
                            title="Delete Product"
                          >
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  );
                })
              )}
              </tbody>
            </table>
          </div>

          {/* Pagination */}
          {totalPages > 1 && (
            <div className="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <p className="text-sm text-gray-700">
                    Page <span className="font-medium">{currentPage}</span> of{' '}
                    <span className="font-medium">{totalPages}</span>
                  </p>
                </div>
                <div className="flex items-center space-x-2">
                  <button
                    onClick={() => handlePageChange(currentPage - 1)}
                    disabled={currentPage === 1}
                    className={`px-3 py-1 rounded-md ${
                      currentPage === 1
                        ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                        : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-300'
                    }`}
                  >
                    Previous
                  </button>
                  <button
                    onClick={() => handlePageChange(currentPage + 1)}
                    disabled={currentPage === totalPages}
                    className={`px-3 py-1 rounded-md ${
                      currentPage === totalPages
                        ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                        : 'bg-white text-gray-700 hover:bg-gray-50 border border-gray-300'
                    }`}
                  >
                    Next
                  </button>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>

      {/* View Product Modal */}
      {isViewModalOpen && selectedProduct && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg p-6 w-full max-w-4xl max-h-[90vh] overflow-y-auto">
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-bold">Product Details</h2>
              <button
                onClick={() => setIsViewModalOpen(false)}
                className="text-gray-500 hover:text-gray-700"
              >
                ✕
              </button>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Product Images */}
              <div>
                <h3 className="font-medium mb-2">Images</h3>
                {selectedProduct.images && selectedProduct.images.length > 0 ? (
                  <div className="grid grid-cols-2 gap-4">
                    {selectedProduct.images.map((img, index) => (
                      <div key={index} className="border rounded-lg overflow-hidden">
                        <img
                          src={img.url}
                          alt={`${selectedProduct.name} ${index + 1}`}
                          className="w-full h-40 object-cover"
                        />
                        {img.is_primary && (
                          <div className="bg-blue-100 text-blue-800 text-xs px-2 py-1 text-center">
                            Primary
                          </div>
                        )}
                      </div>
                    ))}
                  </div>
                ) : selectedProduct.image ? (
                  <div className="border rounded-lg overflow-hidden">
                    <img
                      src={selectedProduct.image}
                      alt={selectedProduct.name}
                      className="w-full h-64 object-cover"
                    />
                  </div>
                ) : (
                  <div className="w-full h-40 bg-gray-100 rounded-lg flex items-center justify-center">
                    <Package className="h-12 w-12 text-gray-400" />
                    <span className="text-gray-500 ml-2">No images</span>
                  </div>
                )}
              </div>

              {/* Product Details */}
              <div className="space-y-4">
                <div>
                  <h1 className="text-2xl font-bold">{selectedProduct.name}</h1>
                  <p className="text-gray-600">{selectedProduct.sku}</p>
                </div>

                <div className="bg-gray-50 p-4 rounded-lg">
                  <div className="flex items-baseline gap-2">
                    <span className="text-2xl font-bold">
                      ₹{selectedProduct.price.toFixed(2)}
                    </span>
                    {selectedProduct.sales_price && selectedProduct.sales_price < selectedProduct.price && (
                      <span className="text-gray-500 line-through">
                        ₹{selectedProduct.price.toFixed(2)}
                      </span>
                    )}
                    {selectedProduct.discount_id && (
                      <span className="ml-2 bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                        {discounts.find(d => d._id === formData.discount_id)?.name || 'Discount Applied'}
                      </span>
                    )}
                  </div>

                  {selectedProduct.tax_id && (
                    <div className="mt-1 text-sm text-gray-500">
                      {formData.tax_type === 'inclusive' ? 'Incl.' : 'Excl.'}{' '}
                      {taxes.find(t => t._id === formData.tax_id)?.percentage || 0}% Tax
                    </div>
                  )}
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <p className="text-sm text-gray-500">HSN Code</p>
                    <p className="font-medium">{selectedProduct.hsn || 'N/A'}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Barcode</p>
                    <p className="font-medium">{selectedProduct.barcode || 'N/A'}</p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Brand</p>
                    <p className="font-medium">
                      {typeof selectedProduct.brand_id === 'object'
                        ? selectedProduct.brand_id.name || selectedProduct.brand_id.brandName
                        : brands.find(b => b._id === selectedProduct.brand_id)?.name || 'N/A'}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Category</p>
                    <p className="font-medium">
                      {typeof selectedProduct.category_id === 'object'
                        ? selectedProduct.category_id.name
                        : categories.find(c => c._id === selectedProduct.category_id)?.name || 'N/A'}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Unit</p>
                    <p className="font-medium">
                      {selectedProduct.unit_id
                        ? (typeof selectedProduct.unit_id === 'object'
                            ? selectedProduct.unit_id.name
                            : units.find(u => u._id === selectedProduct.unit_id)?.name || 'N/A')
                        : 'N/A'}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Expiry Date</p>
                    <p className="font-medium">
                      {selectedProduct.expire_date
                        ? new Date(selectedProduct.expire_date).toLocaleDateString()
                        : 'N/A'}
                    </p>
                  </div>
                  <div>
                    <p className="text-sm text-gray-500">Status</p>
                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                      selectedProduct.status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                    }`}>
                      {selectedProduct.status ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                </div>

                {selectedProduct.description && (
                  <div className="mt-4">
                    <h3 className="text-sm font-medium text-gray-700 mb-1">Description</h3>
                    <div
                      className="prose max-w-none text-sm text-gray-600"
                      dangerouslySetInnerHTML={{ __html: selectedProduct.description }}
                    />
                  </div>
                )}
              </div>
            </div>

            <div className="mt-6 flex justify-end space-x-3">
              <button
                onClick={() => {
                  setIsViewModalOpen(false);
                  handleEditProduct(selectedProduct);
                }}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 flex items-center"
              >
                <Edit className="w-4 h-4 mr-2" />
                Edit Product
              </button>
              <button
                onClick={() => setIsViewModalOpen(false)}
                className="px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Create/Edit Product Modal */}
      {(isCreateModalOpen || isEditModalOpen) && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg p-6 w-full max-w-5xl max-h-[90vh] overflow-y-auto">
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-medium text-gray-900">
                {isEditModalOpen ? 'Edit Product' : 'Add New Product'}
              </h3>
              <button
                onClick={() => {
                  isEditModalOpen ? setIsEditModalOpen(false) : setIsCreateModalOpen(false);
                  resetForm();
                }}
                className="text-gray-400 hover:text-gray-500"
              >
                <X className="h-6 w-6" />
              </button>
            </div>

            <form onSubmit={handleSubmit} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Basic Information */}
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Product Name <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="text"
                      name="name"
                      value={formData.name}
                      onChange={(e) => handleFieldChange('name', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    />
                    {errors.name && <p className="mt-1 text-sm text-red-600">{errors.name}</p>}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">SKU</label>
                    <input
                      type="text"
                      name="sku"
                      value={formData.sku}
                      onChange={(e) => handleFieldChange('sku', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="e.g., SKU-001"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      HSN Code
                    </label>
                    <input
                      type="text"
                      name="hsn"
                      value={formData.hsn}
                      onChange={(e) => handleFieldChange('hsn', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="e.g., 123456"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Barcode
                    </label>
                    <input
                      type="text"
                      name="barcode"
                      value={formData.barcode}
                      onChange={(e) => handleFieldChange('barcode', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="e.g., 123456789012"
                    />
                  </div>
                </div>

                {/* Pricing */}
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Price <span className="text-red-500">*</span>
                    </label>
                    <div className="relative rounded-md shadow-sm">
                      <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <span className="text-gray-500 sm:text-sm">₹</span>
                      </div>
                      <input
                        type="number"
                        name="price"
                        min="0"
                        step="0.01"
                        value={formData.price}
                        onChange={(e) => handleFieldChange('price', e.target.value)}
                        className="pl-7 w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="0.00"
                        required
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Sales Price
                    </label>
                    <div className="relative rounded-md shadow-sm">
                      <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <span className="text-gray-500 sm:text-sm">₹</span>
                      </div>
                      <input
                        type="number"
                        name="sales_price"
                        min="0"
                        step="0.01"
                        value={formData.sales_price}
                        onChange={(e) => handleFieldChange('sales_price', e.target.value)}
                        className="pl-7 w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        placeholder="0.00"
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Quantity
                    </label>
                    <input
                      type="number"
                      name="quantity"
                      min="0"
                      value={formData.quantity}
                      onChange={(e) => handleFieldChange('quantity', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="0"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Expiry Date
                    </label>
                    <input
                      type="date"
                      name="expire_date"
                      value={formData.expire_date || ''}
                      onChange={(e) => handleFieldChange('expire_date', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    />
                  </div>
                </div>

                {/* Category & Brand */}
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Category <span className="text-red-500">*</span>
                    </label>
                    <select
                      name="category_id"
                      value={formData.category_id}
                      onChange={(e) => handleFieldChange('category_id', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    >
                      <option value="">Select Category</option>
                      {categories.map((category) => (
                        <option key={category._id} value={category._id}>
                          {category.name}
                        </option>
                      ))}
                    </select>
                    {errors.category_id && (
                      <p className="mt-1 text-sm text-red-600">{errors.category_id}</p>
                    )}
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Brand
                    </label>
                    <select
                      name="brand_id"
                      value={formData.brand_id}
                      onChange={(e) => handleFieldChange('brand_id', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    >
                      <option value="">Select Brand</option>
                      {brands.map((brand) => (
                        <option key={brand._id} value={brand._id}>
                          {brand.name || brand.brandName}
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Unit <span className="text-red-500">*</span>
                    </label>
                    <select
                      name="unit_id"
                      value={formData.unit_id}
                      onChange={(e) => handleFieldChange('unit_id', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    >
                      <option value="">Select Unit</option>
                      {units.map((unit) => (
                        <option key={unit._id} value={unit._id}>
                          {unit.name} {unit.shortCode ? `(${unit.shortCode})` : ''}
                        </option>
                      ))}
                    </select>
                    {errors.unit_id && (
                      <p className="mt-1 text-sm text-red-600">{errors.unit_id}</p>
                    )}
                  </div>
                </div>

                {/* Tax & Discount */}
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Tax
                    </label>
                    <div className="flex space-x-2">
                      <select
                        name="tax_id"
                        value={formData.tax_id}
                        onChange={(e) => handleFieldChange('tax_id', e.target.value)}
                        className="flex-1 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      >
                        <option value="">Select Tax</option>
                        {taxes.map((tax) => (
                          <option key={tax._id} value={tax._id}>
                            {tax.name} ({tax.percentage}%)
                          </option>
                        ))}
                      </select>
                      <select
                        name="tax_type"
                        value={formData.tax_type}
                        onChange={(e) =>
                          handleFieldChange('tax_type', e.target.value as 'inclusive' | 'exclusive')
                        }
                        className="w-32 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      >
                        <option value="inclusive">Inclusive</option>
                        <option value="exclusive">Exclusive</option>
                      </select>
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Discount
                    </label>
                    <select
                      name="discount_id"
                      value={formData.discount_id}
                      onChange={(e) => handleFieldChange('discount_id', e.target.value)}
                      className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    >
                      <option value="">Select Discount</option>
                      {discounts.map((discount) => (
                        <option key={discount._id} value={discount._id}>
                          {discount.name} (
                          {discount.type === 'percentage'
                            ? `${discount.value}%`
                            : `₹${discount.value}`}
                          )
                        </option>
                      ))}
                    </select>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Store
                    </label>
                    <div className="mt-1">
                      <input
                        type="text"
                        readOnly
                        value={currentStoreName}
                        className="w-full px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-gray-700 cursor-not-allowed"
                      />
                      <input
                        type="hidden"
                        name="store_id"
                        value={formData.store_id}
                      />
                    </div>
                    <p className="mt-1 text-xs text-gray-500">
                      Products are automatically associated with your store.
                    </p>
                  </div>
                </div>

                {/* Images */}
                <div className="md:col-span-2">
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Images
                  </label>
                  <div className="mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-gray-300 border-dashed rounded-md">
                    <div className="space-y-1 text-center">
                      <svg
                        className="mx-auto h-12 w-12 text-gray-400"
                        stroke="currentColor"
                        fill="none"
                        viewBox="0 0 48 48"
                        aria-hidden="true"
                      >
                        <path
                          d="M28 8H12a4 4 0 00-4 4v20m32-12v8m0 0v8a4 4 0 01-4 4H12a4 4 0 01-4-4v-4m32-4l-3.172-3.172a4 4 0 00-5.656 0L28 28M8 32l9.172-9.172a4 4 0 015.656 0L28 28m0 0l4 4m4-24h8m-4-4v8m-12 4h.02"
                          strokeWidth="2"
                          strokeLinecap="round"
                          strokeLinejoin="round"
                        />
                      </svg>
                      <div className="flex text-sm text-gray-600">
                        <label
                          htmlFor="file-upload"
                          className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none focus-within:ring-2 focus-within:ring-offset-2 focus-within:ring-blue-500"
                        >
                          <span>Upload files</span>
                          <input
                            id="file-upload"
                            name="file-upload"
                            type="file"
                            className="sr-only"
                            multiple
                            onChange={handleImageUpload}
                            accept="image/*"
                          />
                        </label>
                        <p className="pl-1">or drag and drop</p>
                      </div>
                      <p className="text-xs text-gray-500">PNG, JPG, GIF up to 10MB</p>
                    </div>
                  </div>

                  {/* Image Previews */}
                  {formData.images && formData.images.length > 0 && (
                    <div className="mt-4">
                      <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-6 gap-4">
                        {formData.images.map((image, index) => (
                          <div
                            key={index}
                            className="relative group rounded-md overflow-hidden border border-gray-200"
                          >
                            <img
                              src={typeof image === 'string' 
                                ? image 
                                : (image instanceof Blob || image instanceof File) 
                                  ? URL.createObjectURL(image) 
                                  : ''}
                              alt={`Preview ${index + 1}`}
                              className="h-24 w-full object-cover"
                              onError={(e) => {
                                // Fallback to a placeholder if the image fails to load
                                const target = e.target as HTMLImageElement;
                                target.src = 'https://via.placeholder.com/150?text=Image+Not+Available';
                                target.onerror = null; // Prevent infinite loop if placeholder also fails
                              }}
                            />
                            <div className="absolute inset-0 bg-black bg-opacity-50 flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity">
                              <button
                                type="button"
                                onClick={() => setPrimaryImage(index)}
                                className={`p-1 rounded-full ${
                                  (typeof image !== 'string' && image.is_primary) ||
                                  (typeof image === 'string' &&
                                    formData.images?.[0] === image)
                                    ? 'bg-blue-500 text-white'
                                    : 'bg-white text-gray-700 hover:bg-gray-100'
                                }`}
                                title="Set as primary"
                              >
                                <svg
                                  xmlns="http://www.w3.org/2000/svg"
                                  className="h-4 w-4"
                                  viewBox="0 0 20 20"
                                  fill="currentColor"
                                >
                                  <path
                                    fillRule="evenodd"
                                    d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                    clipRule="evenodd"
                                  />
                                </svg>
                              </button>
                              <button
                                type="button"
                                onClick={() => removeImage(index)}
                                className="ml-1 p-1 rounded-full bg-white text-red-600 hover:bg-red-50"
                                title="Remove image"
                              >
                                <svg
                                  xmlns="http://www.w3.org/2000/svg"
                                  className="h-4 w-4"
                                  viewBox="0 0 20 20"
                                  fill="currentColor"
                                >
                                  <path
                                    fillRule="evenodd"
                                    d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z"
                                    clipRule="evenodd"
                                  />
                                </svg>
                              </button>
                            </div>
                            {((typeof image !== 'string' && image.is_primary) ||
                              (typeof image === 'string' && formData.images?.[0] === image)) && (
                              <div className="absolute top-1 right-1 bg-blue-500 text-white text-xs px-1 rounded">
                                Primary
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    </div>
                  )}
                </div>

                {/* Description */}
                <div className="md:col-span-2">
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Description
                  </label>
                  <Suspense fallback={<EditorLoading />}>
                    <ReactQuill
                      theme="snow"
                      value={formData.description}
                      onChange={(value) => handleFieldChange('description', value)}
                      modules={modules}
                      formats={formats}
                      placeholder="Enter product description..."
                      className="h-48 mb-12"
                    />
                  </Suspense>
                </div>
              </div>

              <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200 mt-6">
                <button
                  type="button"
                  onClick={() => {
                    isEditModalOpen ? setIsEditModalOpen(false) : setIsCreateModalOpen(false);
                    // Only reset form if not in edit mode
                    if (!isEditModalOpen) {
                      resetForm();
                    }
                  }}
                  className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  disabled={loading}
                  className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {loading ? (
                    <span className="flex items-center">
                      <svg
                        className="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                        xmlns="http://www.w3.org/2000/svg"
                        fill="none"
                        viewBox="0 0 24 24"
                      >
                        <circle
                          className="opacity-25"
                          cx="12"
                          cy="12"
                          r="10"
                          stroke="currentColor"
                          strokeWidth="4"
                        ></circle>
                        <path
                          className="opacity-75"
                          fill="currentColor"
                          d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                        ></path>
                      </svg>
                      {isEditModalOpen ? 'Updating...' : 'Creating...'}
                    </span>
                  ) : isEditModalOpen ? (
                    'Update Product'
                  ) : (
                    'Create Product'
                  )}
                </button>
              </div>
            </form>
          </div>
        </div>
      )}

      {/* Delete Confirmation Modal */}
      {isDeleteModalOpen && selectedProduct && (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
          <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div className="mt-3 text-center">
              <div className="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                <Trash2 className="h-6 w-6 text-red-600" />
              </div>
              <h3 className="text-lg leading-6 font-medium text-gray-900 mt-4">Delete Product</h3>
              <div className="mt-2 px-7 py-3">
                <p className="text-sm text-gray-500">
                  Are you sure you want to delete {selectedProduct.name}? This action cannot be undone.
                </p>
              </div>
              <div className="flex justify-center mt-4 space-x-3">
                <button
                  onClick={() => setIsDeleteModalOpen(false)}
                  className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
                >
                  Cancel
                </button>
                <button
                  onClick={() => {
                    // Delete product logic here
                  }}
                  className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700"
                >
                  Delete
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
};

export default Products;