import React, { useState, useEffect } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, Plus, Trash2, Eye, Building, MapPin, BarChart3 } from 'lucide-react';
import { toast } from "react-hot-toast";
import api from "@/services/api";
import { useNavigate } from 'react-router-dom';

interface Store {
  _id: string;
  name: string;
  email: string;
  mobile: string;
  store: {
    storeName: string;
    panNumber: string;
    commission: number;
    logo?: string;
    description?: string;
  };
  location: {
    city: string;
    state?: string;
    street?: string;
    latitude: number;
    longitude: number;
  };
  bank: {
    bankName: string;
    accountNumber: string;
    ifscCode: string;
    accountHolderName: string;
  };
  status: 'Active' | 'Inactive' | 'Pending' | 'Suspended';
  isVerified: boolean;
  createdAt: string;
  updatedAt: string;
  displayName?: string;
  fullAddress?: string;
}

interface Statistics {
  total: number;
  active: number;
  pending: number;
  inactive: number;
  suspended: number;
  verified: number;
  avgCommission: number;
}

const ManageStoreManagers: React.FC = () => {
  const navigate = useNavigate();
  const [searchTerm, setSearchTerm] = useState('');
  const [showViewModal, setShowViewModal] = useState(false);
  const [showStatsModal, setShowStatsModal] = useState(false);
  const [stores, setStores] = useState<Store[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedStore, setSelectedStore] = useState<Store | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [statistics, setStatistics] = useState<Statistics | null>(null);
  const [statusFilter, setStatusFilter] = useState('');
  const [cityFilter, setCityFilter] = useState('');

  // Fetch stores from API
  const fetchStores = async (page = 1, search = '', status = '', city = '') => {
    try {
      setLoading(true);
      const response = await api.get('/stores', {
        params: {
          page,
          limit: 10,
          search: search || undefined,
          status: status || undefined,
          city: city || undefined
        }
      });
      
      const { data, pagination } = response.data;
      setStores(data);
      setTotalPages(pagination?.totalPages || 1);
      setCurrentPage(page);
    } catch (error: any) {
      console.error('Error fetching stores:', error);
      toast.error(error.response?.data?.msg || 'Failed to fetch stores');
    } finally {
      setLoading(false);
    }
  };

  // Fetch statistics
  const fetchStatistics = async () => {
    try {
      const response = await api.get('/stores/analytics/statistics');
      setStatistics(response.data.data);
    } catch (error: any) {
      console.error('Error fetching statistics:', error);
      toast.error('Failed to fetch statistics');
    }
  };

  // Toggle store status
  const toggleStoreStatus = async (storeId: string, newStatus: string) => {
    try {
      await api.patch(`/stores/${storeId}/toggle-status`, { status: newStatus });
      toast.success(`Store status updated to ${newStatus} successfully`);
      fetchStores(currentPage, searchTerm, statusFilter, cityFilter);
    } catch (error: any) {
      console.error('Error updating store status:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to update store status';
      toast.error(errorMessage);
    }
  };

  // Delete store
  const deleteStore = async (id: string) => {
    if (!confirm('Are you sure you want to permanently delete this store? This action cannot be undone.')) {
      return;
    }

    try {
      await api.delete(`/stores/${id}`);
      toast.success('Store permanently deleted successfully');
      fetchStores(currentPage, searchTerm, statusFilter, cityFilter);
    } catch (error: any) {
      console.error('Error deleting store:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to delete store';
      toast.error(errorMessage);
    }
  };

  const getStatusBadge = (status: string) => {
    const statusColors = {
      'Active': 'bg-green-100 text-green-800',
      'Inactive': 'bg-red-100 text-red-800',
      'Pending': 'bg-yellow-100 text-yellow-800',
      'Suspended': 'bg-purple-100 text-purple-800'
    };
    return statusColors[status as keyof typeof statusColors] || 'bg-gray-100 text-gray-800';
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
    fetchStores(1, value, statusFilter, cityFilter);
  };

  // Handle filter changes
  const handleFilterChange = (type: 'status' | 'city', value: string) => {
    if (type === 'status') {
      setStatusFilter(value);
      fetchStores(1, searchTerm, value, cityFilter);
    } else {
      setCityFilter(value);
      fetchStores(1, searchTerm, statusFilter, value);
    }
    setCurrentPage(1);
  };

  // Handle view store
  const handleView = (store: Store) => {
    setSelectedStore(store);
    setShowViewModal(true);
  };

  // Handle pagination
  const handlePageChange = (page: number) => {
    setCurrentPage(page);
    fetchStores(page, searchTerm, statusFilter, cityFilter);
  };

  // Load stores on component mount
  useEffect(() => {
    fetchStores();
    fetchStatistics();
  }, []);

  return (
    <Layout title="Store Managers" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex items-center gap-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search stores..."
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={searchTerm}
                onChange={(e) => handleSearch(e.target.value)}
              />
            </div>
            
            {/* Filters */}
            <select
              value={statusFilter}
              onChange={(e) => handleFilterChange('status', e.target.value)}
              className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="">All Status</option>
              <option value="Active">Active</option>
              <option value="Pending">Pending</option>
              <option value="Inactive">Inactive</option>
              <option value="Suspended">Suspended</option>
            </select>
            
            <input
              type="text"
              placeholder="Filter by city..."
              className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={cityFilter}
              onChange={(e) => handleFilterChange('city', e.target.value)}
            />
          </div>
          
          <div className="flex items-center gap-2">
            <button 
              onClick={() => setShowStatsModal(true)}
              className="flex items-center px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700"
            >
              <BarChart3 className="w-4 h-4 mr-2" />
              Statistics
            </button>
            
            <button 
              onClick={() => navigate('/admin/store-managers/create')}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              <Plus className="w-4 h-4 mr-2" />
              Add Store Manager
            </button>
          </div>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        )}

        {/* Stores Table */}
        {!loading && (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Store
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Owner
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Location
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Commission
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Created
                    </th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {stores.length === 0 ? (
                    <tr>
                      <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
                        <Building className="mx-auto h-12 w-12 text-gray-400 mb-4" />
                        <p>No stores found</p>
                      </td>
                    </tr>
                  ) : (
                    stores.map((store) => (
                      <tr key={store._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="flex items-center">
                            {store.store.logo ? (
                              <img
                                src={store.store.logo}
                                alt={store.store.storeName}
                                className="h-10 w-10 rounded-lg object-cover mr-3"
                              />
                            ) : (
                              <div className="h-10 w-10 rounded-lg bg-gray-200 flex items-center justify-center mr-3">
                                <Building className="h-5 w-5 text-gray-500" />
                              </div>
                            )}
                            <div>
                              <div className="text-sm font-medium text-gray-900">
                                {store.store.storeName}
                              </div>
                              <div className="text-sm text-gray-500">
                                PAN: {store.store.panNumber}
                              </div>
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div>
                            <div className="text-sm font-medium text-gray-900">{store.name}</div>
                            <div className="text-sm text-gray-500">{store.email}</div>
                            <div className="text-sm text-gray-500">{store.mobile}</div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="flex items-center text-sm text-gray-900">
                            <MapPin className="h-4 w-4 mr-1 text-gray-400" />
                            <div>
                              <div>{store.location.city}</div>
                              {store.location.state && (
                                <div className="text-xs text-gray-500">{store.location.state}</div>
                              )}
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {store.store.commission}%
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="flex flex-col gap-1">
                            <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusBadge(store.status)}`}>
                              {store.status}
                            </span>
                            {store.isVerified && (
                              <span className="inline-flex px-2 py-1 text-xs font-semibold rounded-full bg-blue-100 text-blue-800">
                                Verified
                              </span>
                            )}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {formatDate(store.createdAt)}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                          <div className="flex items-center justify-end gap-2">
                            <button
                              onClick={() => handleView(store)}
                              className="text-blue-600 hover:text-blue-900 p-1"
                              title="View Details"
                            >
                              <Eye className="h-4 w-4" />
                            </button>
                            
                            <select
                              value={store.status}
                              onChange={(e) => toggleStoreStatus(store._id, e.target.value)}
                              className="text-xs border border-gray-300 rounded px-2 py-1 focus:ring-1 focus:ring-blue-500 focus:border-blue-500"
                              title="Change Status"
                            >
                              <option value="Active">Active</option>
                              <option value="Pending">Pending</option>
                              <option value="Inactive">Inactive</option>
                              <option value="Suspended">Suspended</option>
                            </select>
                            
                            <button
                              onClick={() => deleteStore(store._id)}
                              className="text-red-600 hover:text-red-900 p-1"
                              title="Delete Store"
                            >
                              <Trash2 className="h-4 w-4" />
                            </button>
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>

            {/* Pagination */}
            {totalPages > 1 && (
              <div className="bg-white px-4 py-3 border-t border-gray-200 sm:px-6">
                <div className="flex items-center justify-between">
                  <div className="flex items-center">
                    <p className="text-sm text-gray-700">
                      Page <span className="font-medium">{currentPage}</span> of{' '}
                      <span className="font-medium">{totalPages}</span>
                    </p>
                  </div>
                  <div className="flex items-center space-x-2">
                    <button
                      onClick={() => handlePageChange(currentPage - 1)}
                      disabled={currentPage === 1}
                      className="px-3 py-1 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      Previous
                    </button>
                    
                    <select
                      value={currentPage}
                      onChange={(e) => handlePageChange(Number(e.target.value))}
                      className="px-3 py-1 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white focus:ring-1 focus:ring-blue-500 focus:border-blue-500"
                    >
                      {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                        <option key={page} value={page}>
                          {page}
                        </option>
                      ))}
                    </select>
                    
                    <button
                      onClick={() => handlePageChange(currentPage + 1)}
                      disabled={currentPage === totalPages}
                      className="px-3 py-1 border border-gray-300 rounded-md text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      Next
                    </button>
                  </div>
                </div>
              </div>
            )}
          </div>
        )}

        {/* View Store Modal */}
        {showViewModal && selectedStore && (
          <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-20 mx-auto p-5 border w-11/12 max-w-4xl shadow-lg rounded-md bg-white">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-gray-900">Store Details</h3>
                <button
                  onClick={() => setShowViewModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ×
                </button>
              </div>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Store Information */}
                <div className="space-y-4">
                  <div>
                    <h4 className="font-medium text-gray-900 mb-2">Store Information</h4>
                    <div className="space-y-2 text-sm">
                      <div><span className="font-medium">Name:</span> {selectedStore.store.storeName}</div>
                      <div><span className="font-medium">PAN:</span> {selectedStore.store.panNumber}</div>
                      <div><span className="font-medium">Commission:</span> {selectedStore.store.commission}%</div>
                      {selectedStore.store.description && (
                        <div><span className="font-medium">Description:</span> {selectedStore.store.description}</div>
                      )}
                    </div>
                  </div>
                  
                  <div>
                    <h4 className="font-medium text-gray-900 mb-2">Owner Information</h4>
                    <div className="space-y-2 text-sm">
                      <div><span className="font-medium">Name:</span> {selectedStore.name}</div>
                      <div><span className="font-medium">Email:</span> {selectedStore.email}</div>
                      <div><span className="font-medium">Mobile:</span> {selectedStore.mobile}</div>
                    </div>
                  </div>
                </div>
                
                {/* Location & Bank Information */}
                <div className="space-y-4">
                  <div>
                    <h4 className="font-medium text-gray-900 mb-2">Location</h4>
                    <div className="space-y-2 text-sm">
                      <div><span className="font-medium">City:</span> {selectedStore.location.city}</div>
                      {selectedStore.location.state && (
                        <div><span className="font-medium">State:</span> {selectedStore.location.state}</div>
                      )}
                      {selectedStore.location.street && (
                        <div><span className="font-medium">Street:</span> {selectedStore.location.street}</div>
                      )}
                      <div><span className="font-medium">Coordinates:</span> {selectedStore.location.latitude}, {selectedStore.location.longitude}</div>
                    </div>
                  </div>
                  
                  {selectedStore.bank ? (
                    <div>
                      <h4 className="font-medium text-gray-900 mb-2">Bank Information</h4>
                      <div className="space-y-2 text-sm">
                        <div><span className="font-medium">Bank:</span> {selectedStore.bank.bankName || 'N/A'}</div>
                        <div><span className="font-medium">Account:</span> {selectedStore.bank.accountNumber || 'N/A'}</div>
                        <div><span className="font-medium">IFSC:</span> {selectedStore.bank.ifscCode || 'N/A'}</div>
                        <div><span className="font-medium">Holder:</span> {selectedStore.bank.accountHolderName || 'N/A'}</div>
                      </div>
                    </div>
                  ) : (
                    <div className="text-gray-500">No bank information available</div>
                  )}
                </div>
              </div>
              
              <div className="mt-6 flex justify-end">
                <button
                  onClick={() => setShowViewModal(false)}
                  className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        )}

        {/* Statistics Modal */}
        {showStatsModal && statistics && (
          <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
            <div className="relative top-20 mx-auto p-5 border w-11/12 max-w-2xl shadow-lg rounded-md bg-white">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-lg font-semibold text-gray-900">Store Statistics</h3>
                <button
                  onClick={() => setShowStatsModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ×
                </button>
              </div>
              
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div className="bg-blue-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-blue-600">{statistics.total}</div>
                  <div className="text-sm text-blue-600">Total Stores</div>
                </div>
                <div className="bg-green-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-green-600">{statistics.active}</div>
                  <div className="text-sm text-green-600">Active</div>
                </div>
                <div className="bg-yellow-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-yellow-600">{statistics.pending}</div>
                  <div className="text-sm text-yellow-600">Pending</div>
                </div>
                <div className="bg-red-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-red-600">{statistics.inactive}</div>
                  <div className="text-sm text-red-600">Inactive</div>
                </div>
                <div className="bg-purple-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-purple-600">{statistics.suspended}</div>
                  <div className="text-sm text-purple-600">Suspended</div>
                </div>
                <div className="bg-indigo-50 p-4 rounded-lg">
                  <div className="text-2xl font-bold text-indigo-600">{statistics.verified}</div>
                  <div className="text-sm text-indigo-600">Verified</div>
                </div>
                <div className="bg-gray-50 p-4 rounded-lg col-span-2">
                  <div className="text-2xl font-bold text-gray-600">{statistics.avgCommission.toFixed(1)}%</div>
                  <div className="text-sm text-gray-600">Average Commission</div>
                </div>
              </div>
              
              <div className="mt-6 flex justify-end">
                <button
                  onClick={() => setShowStatsModal(false)}
                  className="px-4 py-2 bg-gray-600 text-white rounded-lg hover:bg-gray-700"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default ManageStoreManagers;