import React, { useState, useEffect, useRef, useCallback } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { ArrowLeft, Upload, MapPin, Building, User, Settings, Loader2 } from 'lucide-react';
import { toast } from "react-hot-toast";
import api from "@/services/api";
import { useNavigate } from 'react-router-dom';

declare global {
  interface Window {
    initMap: () => void;
    google: typeof google;
  }
}

// Google Maps API Key
const GOOGLE_MAPS_API_KEY = 'AIzaSyCJKwxaSS0glDtxXMX37uHX_KHUEleCMk0';

interface StoreFormData {
  // Seller Credentials
  name: string;
  email: string;
  mobile: string;
  password: string;
  
  // Store Information
  store: {
    storeName: string;
    categoryIds: string[];
    productStatus: 'Active' | 'Inactive';
    taxName: string;
    taxNumber: string;
    panNumber: string;
    commission: number;
    description: string;
    // Location data for geospatial queries
    location: {
      type: 'Point';
      coordinates: [number, number]; // [longitude, latitude]
      address: string;
      city: string;
      state: string;
      country: string;
      pincode: string;
      formattedAddress?: string;
    };
    deliveryRadius: number; // in meters
    isActive: boolean;
  };
  
  // Location (for backward compatibility)
  location: {
    street: string;
    city: string;
    state: string;
    country: string;
    pincode: string;
    landmark?: string;
  };
  
  // Settings
  settings: {
    requireProductApproval: boolean;
    viewCustomerDetails: boolean;
  };
  
  status: 'Active' | 'Inactive' | 'Pending' | 'Suspended';
}

const CreateStoreManagers: React.FC = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [mapLoaded, setMapLoaded] = useState(false);
  const [mapInitialized, setMapInitialized] = useState(false);
  const [mapError, setMapError] = useState<string | null>(null);
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstance = useRef<google.maps.Map | null>(null);
  const markerInstance = useRef<google.maps.Marker | null>(null);
  const autocompleteInputRef = useRef<HTMLInputElement>(null);
  const autocompleteRef = useRef<google.maps.places.Autocomplete | null>(null);
  const [selectedFiles, setSelectedFiles] = useState<{
    logo?: File;
    nationalIdCard?: File;
    addressProof?: File;
  }>({});
  
  const [formData, setFormData] = useState<StoreFormData>({
    name: '',
    email: '',
    mobile: '',
    password: '',
    store: {
      storeName: '',
      categoryIds: [],
      productStatus: 'Active',
      taxName: '',
      taxNumber: '',
      panNumber: '',
      commission: 10,
      description: '',
      location: {
        type: 'Point',
        coordinates: [0, 0], // [longitude, latitude]
        address: '',
        city: '',
        state: '',
        country: 'India',
        pincode: '',
        formattedAddress: ''
      },
      deliveryRadius: 3000, // 3km default
      isActive: true
    },
    location: {
      street: '',
      city: '',
      state: '',
      country: 'India',
      pincode: '',
      landmark: ''
    },
    settings: {
      requireProductApproval: false,
      viewCustomerDetails: false
    },
    status: 'Pending'
  });

  // Initialize Google Places Autocomplete
  useEffect(() => {
    // Load Google Maps API script if not already loaded
    if (!window.google) {
      const script = document.createElement('script');
      script.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyCJKwxaSS0glDtxXMX37uHX_KHUEleCMk0&libraries=places';
      script.async = true;
      script.onload = () => initAutocomplete();
      document.head.appendChild(script);
    } else if (window.google.maps && window.google.maps.places) {
      initAutocomplete();
    }

    return () => {
      // Cleanup autocomplete instance
      if (autocompleteRef.current) {
        google.maps.event.clearInstanceListeners(autocompleteRef.current);
      }
    };
  }, []);

  const initAutocomplete = () => {
    if (!autocompleteInputRef.current) return;

    autocompleteRef.current = new google.maps.places.Autocomplete(
      autocompleteInputRef.current,
      {
        types: ['address'],
        componentRestrictions: { country: 'in' }, // Restrict to India
        fields: ['address_components', 'formatted_address', 'geometry']
      }
    );

    autocompleteRef.current.addListener('place_changed', () => {
      const place = autocompleteRef.current?.getPlace();
      if (!place || !place.address_components) return;

      const addressComponents: Record<string, string> = {
        street: '',
        city: '',
        state: '',
        country: 'India',
        pincode: '',
        landmark: ''
      };

      // Parse address components
      for (const component of place.address_components) {
        const componentType = component.types[0];
        
        switch (componentType) {
          case 'street_number':
            addressComponents.street = `${component.long_name} `;
            break;
          case 'route':
            addressComponents.street = (addressComponents.street || '') + component.long_name;
            break;
          case 'locality':
          case 'sublocality_level_1':
            addressComponents.city = component.long_name;
            break;
          case 'administrative_area_level_1':
            addressComponents.state = component.long_name;
            break;
          case 'postal_code':
            addressComponents.pincode = component.long_name;
            break;
          case 'sublocality_level_2':
          case 'sublocality_level_3':
            addressComponents.landmark = component.long_name;
            break;
        }
      }

      const formattedAddress = [
        addressComponents.street,
        addressComponents.landmark,
        addressComponents.city,
        addressComponents.state,
        addressComponents.pincode,
        addressComponents.country
      ].filter(Boolean).join(', ');

      // Update form data with the parsed address
      setFormData(prev => ({
        ...prev,
        location: {
          ...prev.location,
          ...addressComponents
        },
        store: {
          ...prev.store,
          location: {
            type: 'Point',
            coordinates: [
              place.geometry?.location?.lng() || 0,
              place.geometry?.location?.lat() || 0
            ],
            address: addressComponents.street,
            city: addressComponents.city,
            state: addressComponents.state,
            country: addressComponents.country,
            pincode: addressComponents.pincode,
            formattedAddress
          }
        }
      }));
    });
  };

  // Derived address and map URLs
  const addressParts = [
    formData.location.street,
    formData.location.landmark,
    formData.location.city,
    formData.location.state,
    formData.location.pincode,
    formData.location.country
  ].filter(Boolean).join(', ');

  const isAddressReady = Boolean(
    formData.location.street &&
    formData.location.city &&
    formData.location.state &&
    formData.location.pincode &&
    formData.location.pincode.length >= 6
  );

  const mapQuery = encodeURIComponent(addressParts);
  const mapSearchUrl = `https://www.google.com/maps/search/?api=1&query=${mapQuery}`;


  // Track if the script is loaded
  const scriptLoaded = useRef(false);
  const scriptLoading = useRef(false);

  // Load Google Maps API script
  useEffect(() => {
    if (!GOOGLE_MAPS_API_KEY) {
      const errorMsg = 'Google Maps API key is not configured';
      console.error(errorMsg);
      setMapError(errorMsg);
      return;
    }

    // Check if already loaded
    if (scriptLoaded.current || window.google?.maps) {
      setMapLoaded(true);
      return;
    }

    // Check if script is already being loaded
    if (scriptLoading.current) {
      return;
    }

    scriptLoading.current = true;
    console.log('Loading Google Maps script...');

    // Create script element
    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_API_KEY}&libraries=places`;
    script.async = true;
    script.defer = true;
    
    // Handle script load
    script.onload = () => {
      console.log('Google Maps script loaded');
      scriptLoaded.current = true;
      scriptLoading.current = false;
      setMapLoaded(true);
    };

    // Handle script error
    script.onerror = (error) => {
      const errorMsg = `Error loading Google Maps: ${error?.message || 'Unknown error'}`;
      console.error(errorMsg);
      scriptLoading.current = false;
      setMapError(errorMsg);
    };

    // Add to document
    document.head.appendChild(script);

    // Cleanup function
    return () => {
      // We don't remove the script as it's better to keep it cached
      script.onload = null;
      script.onerror = null;
    };
  }, []);

  // Initialize map when Google Maps is loaded and container is ready
  useEffect(() => {
    if (!mapLoaded || !mapRef.current) return;

    console.log('Initializing map...');
    
    const initializeMap = () => {
      try {
        const defaultLocation = {
          lat: 22.5726, // Default to Kolkata
          lng: 88.3639
        };

        // Ensure the map container has proper dimensions
        if (mapRef.current) {
          mapRef.current.style.height = '400px';
          mapRef.current.style.width = '100%';
        }

        const mapOptions: google.maps.MapOptions = {
          center: defaultLocation,
          zoom: 15,
          mapTypeId: 'roadmap',
          streetViewControl: false,
          fullscreenControl: true,
          zoomControl: true,
          mapTypeControl: false,
          styles: [
            {
              featureType: 'poi',
              elementType: 'labels',
              stylers: [{ visibility: 'off' }]
            }
          ]
        };

        // Initialize the map
        mapInstance.current = new window.google.maps.Map(mapRef.current, mapOptions);
        
        // Add click listener to map
        mapInstance.current.addListener('click', (event: google.maps.MapMouseEvent) => {
          if (event.latLng) {
            const location = {
              lat: event.latLng.lat(),
              lng: event.latLng.lng()
            };
            updateMarker(location);
            reverseGeocode(location);
          }
        });

        // Try to get user's current location
        if (navigator.geolocation) {
          navigator.geolocation.getCurrentPosition(
            (position) => {
              const pos = {
                lat: position.coords.latitude,
                lng: position.coords.longitude
              };
              updateMarker(pos);
              mapInstance.current?.setCenter(pos);
              mapInstance.current?.setZoom(15);
              reverseGeocode(pos);
            },
            () => {
              // Use default location if geolocation fails
              updateMarker(defaultLocation);
              mapInstance.current?.setCenter(defaultLocation);
              mapInstance.current?.setZoom(15);
            }
          );
        } else {
          // Browser doesn't support Geolocation
          updateMarker(defaultLocation);
          mapInstance.current?.setCenter(defaultLocation);
          mapInstance.current?.setZoom(15);
        }

        // Trigger resize to ensure proper rendering
        setTimeout(() => {
          if (mapInstance.current) {
            google.maps.event.trigger(mapInstance.current, 'resize');
          }
        }, 100);

        setMapInitialized(true);
        console.log('Map initialized successfully');
      } catch (error) {
        console.error('Error initializing map:', error);
        setMapError(`Failed to initialize the map: ${error.message}`);
      }
    };

    // Small delay to ensure the container is ready
    const timer = setTimeout(initializeMap, 100);
    
    return () => {
      clearTimeout(timer);
      // Clean up map instance if needed
      if (mapInstance.current) {
        // Remove all event listeners
        if (window.google?.maps?.event) {
          window.google.maps.event.clearInstanceListeners(mapInstance.current);
        }
        // Clear the map container
        const mapContainer = mapRef.current;
        if (mapContainer) {
          // Only clear if we're the ones who initialized it
          if (mapInitialized) {
            mapContainer.innerHTML = '';
          }
        }
      }
      setMapInitialized(false);
    };
  }, [mapLoaded]);

  // Update marker position
  const updateMarker = (position: { lat: number; lng: number }) => {
    if (!mapInstance.current) return;

    if (markerInstance.current) {
      markerInstance.current.setPosition(position);
    } else {
      markerInstance.current = new window.google.maps.Marker({
        position,
        map: mapInstance.current,
        draggable: true,
        title: 'Store Location'
      });

      // Add dragend listener to update address when marker is moved
      markerInstance.current.addListener('dragend', (event: google.maps.MapMouseEvent) => {
        if (event.latLng) {
          const location = {
            lat: event.latLng.lat(),
            lng: event.latLng.lng()
          };
          reverseGeocode(location);
        }
      });
    }
  };

  // Reverse geocode function to get address from coordinates
  const reverseGeocode = useCallback(async (location: { lat: number; lng: number }) => {
    if (!window.google || !window.google.maps) return;

    try {
      const geocoder = new window.google.maps.Geocoder();
      
      geocoder.geocode({ location }, (results, status) => {
        if (status === 'OK' && results && results[0]) {
          const addressComponents = results[0].address_components;
          let street = '';
          let city = '';
          let state = '';
          let country = '';
          let pincode = '';

          // Parse address components
          addressComponents.forEach(component => {
            const types = component.types;
            if (types.includes('route') || types.includes('sublocality')) {
              street = component.long_name;
            } else if (types.includes('locality')) {
              city = component.long_name;
            } else if (types.includes('administrative_area_level_1')) {
              state = component.long_name;
            } else if (types.includes('country')) {
              country = component.long_name;
            } else if (types.includes('postal_code')) {
              pincode = component.long_name;
            }
          });

          // Update form data with the new address
          setFormData(prev => ({
            ...prev,
            location: {
              ...prev.location,
              street: street || prev.location.street,
              city: city || prev.location.city,
              state: state || prev.location.state,
              country: country || prev.location.country,
              pincode: pincode || prev.location.pincode
            }
          }));
        }
      });
    } catch (error) {
      console.error('Reverse geocoding error:', error);
      setMapError('Failed to get address from selected location');
    }
  }, []);

  // Handle input changes for nested form data
  const handleInputChange = <T extends keyof StoreFormData>(
    section: T,
    field: keyof StoreFormData[T],
    value: any
  ) => {
    setFormData(prev => ({
      ...prev,
      [section]: {
        ...prev[section],
        [field]: value
      }
    }));
  };
  
  // Handle direct form field changes (for top-level fields)
  const handleDirectChange = <K extends keyof Omit<StoreFormData, 'store' | 'location' | 'settings'>>(
    field: K,
    value: StoreFormData[K]
  ) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  // Handle file uploads
  const handleFileChange = (fileType: keyof typeof selectedFiles, file: File | null) => {
    if (file) {
      setSelectedFiles(prev => ({
        ...prev,
        [fileType]: file
      }));
    }
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    
    // Basic validation
    if (!formData.name.trim()) {
      toast.error('Store owner name is required');
      return;
    }
    
    if (!formData.email.trim()) {
      toast.error('Email is required');
      return;
    }
    
    if (!formData.store.storeName.trim()) {
      toast.error('Store name is required');
      return;
    }

    // Validate location data
    const requiredLocationFields = ['street', 'city', 'state', 'country', 'pincode'];
    const missingFields = requiredLocationFields.filter(field => !formData.location[field as keyof typeof formData.location]);
    
    if (missingFields.length > 0) {
      toast.error(`Missing required location fields: ${missingFields.join(', ')}`);
      return;
    }
    
    // Validate map marker position
    if (!markerInstance.current) {
      toast.error('Please set the store location by clicking on the map');
      return;
    }

    try {
      setLoading(true);
      
      // Get the current marker position
      if (!markerInstance.current) {
        toast.error('Please set the store location by clicking on the map');
        setLoading(false);
        return;
      }
      const currentLocation = markerInstance.current.getPosition();

      // Prepare store manager data
      const storeManagerData: any = {
        name: formData.name.trim(),
        email: formData.email.trim().toLowerCase(),
        mobile: formData.mobile.trim(),
        status: formData.status || 'Active',
        store: {
          storeName: formData.store.storeName.trim(),
          categoryIds: formData.store.categoryIds || [],
          productStatus: formData.store.productStatus || 'Active',
          taxName: formData.store.taxName || '',
          taxNumber: formData.store.taxNumber || '',
          panNumber: formData.store.panNumber || '',
          commission: formData.store.commission || 0,
          description: formData.store.description || '',
          // Add the required location data for the store
          location: {
            type: 'Point',
            coordinates: [
              currentLocation.lng(), // longitude
              currentLocation.lat()  // latitude
            ],
            address: formData.location.street.trim(),
            city: formData.location.city.trim(),
            state: formData.location.state.trim(),
            country: formData.location.country.trim() || 'India',
            pincode: formData.location.pincode.trim(),
            formattedAddress: `${formData.location.street}, ${formData.location.city}, ${formData.location.state}, ${formData.location.pincode}, ${formData.location.country}`
          },
          deliveryRadius: 5000, // Default 5km radius in meters
          isActive: formData.status === 'Active' // Set isActive based on status
        },
        location: {
          street: formData.location.street.trim(),
          city: formData.location.city.trim(),
          state: formData.location.state.trim(),
          country: formData.location.country.trim() || 'India',
          pincode: formData.location.pincode.trim(),
          landmark: formData.location.landmark || ''
        },
        settings: {
          requireProductApproval: formData.settings.requireProductApproval || false,
          viewCustomerDetails: formData.settings.viewCustomerDetails || false
        },
        role: 'store_manager',
        password: formData.password
      };

      // Create FormData instance for file uploads
      const formDataToSend = new FormData();
      
      // Append all fields to FormData
      Object.entries(storeManagerData).forEach(([key, value]) => {
        if (value !== null && value !== undefined) {
          if (typeof value === 'object' && !(value instanceof File)) {
            formDataToSend.append(key, JSON.stringify(value));
          } else {
            formDataToSend.append(key, value as string | Blob);
          }
        }
      });

      // Handle file uploads
      if (selectedFiles.logo) {
        formDataToSend.append('logo', selectedFiles.logo);
      }
      if (selectedFiles.nationalIdCard) {
        formDataToSend.append('nationalIdCard', selectedFiles.nationalIdCard);
      }
      if (selectedFiles.addressProof) {
        formDataToSend.append('addressProof', selectedFiles.addressProof);
      }


      // Add auth header if available
      const config = {
        headers: {
          'Content-Type': 'multipart/form-data',
          'Accept': 'application/json'
        },
        timeout: 30000 // 30 seconds timeout
      };
      
      // Add auth token if available
      const token = localStorage.getItem('token');
      if (token) {
        config.headers['Authorization'] = `Bearer ${token}`;
      }
      
      // Create the store
      const response = await api.post('/stores', formDataToSend, config);
      
      if (response.data && response.data.success) {
        toast.success('Store created successfully');
        // Redirect to store managers list after a short delay
        setTimeout(() => {
          navigate('/admin/store-managers');
        }, 1500);
      } else {
        throw new Error(response.data?.msg || 'Failed to create store');
      }
    } catch (error: any) {
      console.error('Error creating store:', {
        message: error.message,
        response: error.response?.data,
        status: error.response?.status,
        code: error.code,
        config: error.config,
        request: error.request
      });
      
      let errorMessage = 'Failed to create store';
      
      if (error.response) {
        // The request was made and the server responded with a status code
        // that falls out of the range of 2xx
        errorMessage = error.response.data?.msg || 
                      error.response.data?.message || 
                      error.response.statusText ||
                      `Server responded with status ${error.response.status}`;
      } else if (error.request) {
        // The request was made but no response was received
        console.error('No response received:', error.request);
        errorMessage = 'No response from server. Please check your connection.';
      } else if (error.code === 'ECONNABORTED') {
        errorMessage = 'Request timeout. Please try again.';
      } else if (error.message === 'Network Error') {
        errorMessage = 'Network error. Please check your internet connection.';
      } else {
        // Something happened in setting up the request that triggered an Error
        errorMessage = error.message || 'Failed to create store';
      }
      
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  return (
    <Layout title="Create Store Manager" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex items-center gap-4">
          <button
            onClick={() => navigate('/admin/store-managers')}
            className="flex items-center text-gray-600 hover:text-gray-800"
          >
            <ArrowLeft className="w-5 h-5 mr-2" />
            Back to Store Managers
          </button>
        </div>

        <form onSubmit={handleSubmit} className="space-y-8">
          {/* Seller Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <User className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Seller Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Owner Name *
                </label>
                <input
                  type="text"
                  value={formData.name}
                  onChange={(e) => handleDirectChange('name', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Email *
                </label>
                <input
                  type="email"
                  value={formData.email}
                  onChange={(e) => handleDirectChange('email', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Mobile Number *
                </label>
                <input
                  type="tel"
                  value={formData.mobile}
                  onChange={(e) => handleDirectChange('mobile', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  pattern="[0-9]{10}"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Password *
                </label>
                <input
                  type="password"
                  value={formData.password}
                  onChange={(e) => handleDirectChange('password', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  minLength={6}
                  required
                />
              </div>
            </div>

            
          </div>

          {/* Store Location */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <MapPin className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Store Location</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Address Line 1 *
                  </label>
                  <input
                    type="text"
                    value={formData.location.street}
                    onChange={(e) => 
                      setFormData(prev => ({
                        ...prev,
                        location: {
                          ...prev.location,
                          street: e.target.value
                        }
                      }))
                    }
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    City *
                  </label>
                  <input
                    type="text"
                    value={formData.location.city}
                    onChange={(e) => 
                      setFormData(prev => ({
                        ...prev,
                        location: {
                          ...prev.location,
                          city: e.target.value
                        }
                      }))
                    }
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      State *
                    </label>
                    <input
                      type="text"
                      value={formData.location.state}
                      onChange={(e) => 
                        setFormData(prev => ({
                          ...prev,
                          location: {
                            ...prev.location,
                            state: e.target.value
                          }
                        }))
                      }
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    />
                  </div>
                  
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-2">
                      Pincode *
                    </label>
                    <input
                      type="text"
                      value={formData.location.pincode}
                      onChange={(e) => 
                        setFormData(prev => ({
                          ...prev,
                          location: {
                            ...prev.location,
                            pincode: e.target.value
                          }
                        }))
                      }
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      required
                    />
                  </div>
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Landmark (Optional)
                  </label>
                  <input
                    type="text"
                    value={formData.location.landmark || ''}
                    onChange={(e) => 
                      setFormData(prev => ({
                        ...prev,
                        location: {
                          ...prev.location,
                          landmark: e.target.value
                        }
                      }))
                    }
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Delivery Radius (meters) *
                  </label>
                  <div className="flex items-center space-x-2">
                    <input
                      type="range"
                      min="500"
                      max="10000"
                      step="100"
                      value={formData.store.deliveryRadius}
                      onChange={(e) => 
                        setFormData(prev => ({
                          ...prev,
                          store: {
                            ...prev.store,
                            deliveryRadius: parseInt(e.target.value, 10)
                          }
                        }))
                      }
                      className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer"
                    />
                    <span className="text-sm font-medium text-gray-700 w-20">
                      {formData.store.deliveryRadius.toLocaleString()} m
                    </span>
                  </div>
                  <div className="text-xs text-gray-500 mt-1">
                    {formData.store.deliveryRadius < 1000 
                      ? `Delivery radius: ${formData.store.deliveryRadius} meters` 
                      : `Delivery radius: ${(formData.store.deliveryRadius / 1000).toFixed(1)} km`}
                  </div>
                </div>
              </div>
              
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Search Location
                  </label>
                  <input
                    ref={autocompleteInputRef}
                    type="text"
                    placeholder="Search for an address"
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                </div>
                
                <div className="bg-gray-100 rounded-lg overflow-hidden" style={{ minHeight: '400px' }}>
                  <div 
                    ref={mapRef} 
                    className="w-full h-full"
                    style={{ 
                      height: '400px', 
                      width: '100%',
                      minHeight: '400px',
                      position: 'relative',
                      backgroundColor: '#e5e7eb' // Light gray background while loading
                    }}
                  >
                    <div className="absolute inset-0">
                      <div className="h-full w-full bg-gray-100 flex items-center justify-center">
                        <p className="text-gray-500">Map will be displayed here</p>
                      </div>
                    </div>
                  </div>
                </div>
                
                <div className="text-xs text-gray-500">
                  <p>Click on the map to set the exact location of your store.</p>
                  <p>Or search for an address above to find your location.</p>
                </div>
              </div>
            </div>
          </div>

          {/* Store Information */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Building className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Store Information</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Store Name *
                </label>
                <input
                  type="text"
                  value={formData.store.storeName}
                  onChange={(e) => handleInputChange('store', 'storeName', e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  PAN Number *
                </label>
                <input
                  type="text"
                  value={formData.store.panNumber}
                  onChange={(e) => handleInputChange('store', 'panNumber', e.target.value.toUpperCase())}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  pattern="[A-Z]{5}[0-9]{4}[A-Z]{1}"
                  placeholder="ABCDE1234F"
                  required
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Commission (%)
                </label>
                <input
                  type="number"
                  value={formData.store.commission}
                  onChange={(e) => handleInputChange('store', 'commission', Number(e.target.value))}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  min="0"
                  max="100"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Status
                </label>
                <select
                  value={formData.status}
                  onChange={(e) => handleDirectChange('status', e.target.value as StoreFormData['status'])}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                >
                  <option value="Active">Active</option>
                  <option value="Inactive">Inactive</option>
                  <option value="Pending">Pending</option>
                  <option value="Suspended">Suspended</option>
                </select>
              </div>
            </div>

          </div>
          
          {/* File Uploads */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Upload className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Documents</h3>
            </div>
            
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Store Logo
                </label>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('logo', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  National ID Card
                </label>
                <input
                  type="file"
                  accept="image/*,application/pdf"
                  onChange={(e) => handleFileChange('nationalIdCard', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Address Proof
                </label>
                <input
                  type="file"
                  accept="image/*,application/pdf"
                  onChange={(e) => handleFileChange('addressProof', e.target.files?.[0] || null)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
            </div>
          </div>

          {/* Settings */}
          <div className="bg-white rounded-lg shadow-sm border p-6">
            <div className="flex items-center mb-6">
              <Settings className="w-5 h-5 mr-2 text-blue-600" />
              <h3 className="text-lg font-semibold">Settings</h3>
            </div>
            
            <div className="space-y-4">
              <div className="flex items-center">
                <input
                  type="checkbox"
                  id="requireApproval"
                  checked={formData.settings.requireProductApproval}
                  onChange={(e) => handleInputChange('settings', 'requireProductApproval', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <label htmlFor="requireApproval" className="ml-2 block text-sm text-gray-700">
                  Require product approval before listing
                </label>
              </div>
              
              <div className="flex items-center">
                <input
                  type="checkbox"
                  id="viewCustomers"
                  checked={formData.settings.viewCustomerDetails}
                  onChange={(e) => handleInputChange('settings', 'viewCustomerDetails', e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <label htmlFor="viewCustomers" className="ml-2 block text-sm text-gray-700">
                  Allow viewing customer details
                </label>
              </div>
            </div>
          </div>

          {/* Submit Button */}
          <div className="flex justify-end space-x-4">
            <button
              type="button"
              onClick={() => navigate('/admin/store-managers')}
              className="px-6 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
            >
              Cancel
            </button>
            <button
              type="submit"
              disabled={loading}
              className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
            >
              {loading ? 'Creating...' : 'Create Store Manager'}
            </button>
          </div>
        </form>
      </div>
    </Layout>
  );
};

export default CreateStoreManagers;