import React, { useState, useEffect, useMemo, useCallback } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Input, Button, Table, DatePicker, Select, Card, Typography, Tooltip, Spin, Alert } from 'antd';
// Import icons individually to avoid the jsx/global attribute warnings
import { Search as SearchIcon } from 'lucide-react';
import { Download } from 'lucide-react';
import { RefreshCw } from 'lucide-react';
import { BarChart2 } from 'lucide-react';
import { PieChart as PieChartIcon } from 'lucide-react';
import { LineChart as LineChartIcon } from 'lucide-react';
import { Filter } from 'lucide-react';
import { Calendar as CalendarIcon } from 'lucide-react';
import { DollarSign } from 'lucide-react';
import { ShoppingCart } from 'lucide-react';
import { Users } from 'lucide-react';
import { Tag } from 'lucide-react';
import { ChevronDown } from 'lucide-react';
import { ChevronUp } from 'lucide-react';
import { Package } from 'lucide-react';
import { AlertCircle } from 'lucide-react';
import { 
  LineChart, 
  Line, 
  BarChart, 
  Bar, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip as RechartsTooltip, 
  Legend, 
  ResponsiveContainer,
  PieChart,
  Pie,
  Cell
} from 'recharts';
import dayjs from 'dayjs';
import { RangePickerProps } from 'antd/es/date-picker';
import { toast } from 'react-hot-toast';
import { reportService } from '../../../services/reportService';
import { AxiosError } from 'axios';

const { RangePicker } = DatePicker;
const { Title, Text } = Typography;
const { Option: SelectOption } = Select;

// Time period options for the report
type Timeframe = 'daily' | 'weekly' | 'monthly' | 'yearly';

// Category filter options
const categories = [
  { value: 'all', label: 'All Categories' },
  { value: 'supplements', label: 'Supplements' },
  { value: 'vitamins', label: 'Vitamins' },
  { value: 'protein', label: 'Protein' },
  { value: 'herbs', label: 'Herbs' },
];

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8'];

interface SalesData {
  date: string;
  sales: number;
  revenue: number;
  orders: number;
  customers: number;
}

interface TopProduct {
  id: string;
  name: string;
  sales: number;
  revenue: number;
  image?: string;
}

const SalesReport: React.FC = () => {
  const [loading, setLoading] = useState(true);
  const [apiError, setApiError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [salesData, setSalesData] = useState<SalesData[]>([]);
  const [topProducts, setTopProducts] = useState<TopProduct[]>([]);
  const [dateRange, setDateRange] = useState<[dayjs.Dayjs, dayjs.Dayjs]>([
    dayjs().subtract(30, 'days'),
    dayjs(),
  ]);
  type Timeframe = 'daily' | 'weekly' | 'monthly' | 'yearly';
  const [timeframe, setTimeframe] = useState<Timeframe>('daily');
  const [category, setCategory] = useState('all');
  const [showFilters, setShowFilters] = useState(false);
  const [summary, setSummary] = useState({
    totalSales: 0,
    totalRevenue: 0,
    totalOrders: 0,
    totalCustomers: 0,
    averageOrderValue: 0,
    conversionRate: 0,
  });
  // Toggle filters visibility
  const toggleFilters = () => setShowFilters(!showFilters);

  // Fetch sales report data
  const fetchSalesReport = useCallback(async (params: {
    startDate?: string;
    endDate?: string;
    period?: 'daily' | 'weekly' | 'monthly' | 'yearly';
  } = {}) => {
    setLoading(true);
    setApiError(null);
    try {
      const requestParams = {
        startDate: params.startDate || dateRange[0]?.format('YYYY-MM-DD'),
        endDate: params.endDate || dateRange[1]?.format('YYYY-MM-DD'),
        period: params.period || timeframe,
      };
      
      console.log('Fetching sales report with params:', requestParams);
      const response = await reportService.getSalesReport(requestParams);
      
      console.log('Sales report response:', response);
      
      if (response && response.success && response.data) {
        // Ensure we have the data we need
        const { salesData, topProducts, summary } = response.data;
        
        if (!salesData || !topProducts || !summary) {
          throw new Error('Invalid response format from server');
        }
        
        setSalesData(salesData);
        setTopProducts(topProducts);
        setSummary({
          totalSales: summary.totalSales || 0,
          totalRevenue: summary.totalRevenue || 0,
          totalOrders: summary.totalOrders || 0,
          totalCustomers: summary.totalCustomers || 0,
          averageOrderValue: summary.averageOrderValue || 0,
          conversionRate: summary.conversionRate || 0,
        });
      }
    } catch (err) {
      const axiosError = err as AxiosError<{ message?: string }>;
      console.error('Error fetching sales report:', axiosError);
      const errorMessage = axiosError.response?.data?.message || axiosError.message || 'Failed to load sales report data';
      setApiError(errorMessage);
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  }, [dateRange, timeframe]);

  // Initial data fetch
  useEffect(() => {
    fetchSalesReport({
      startDate: dateRange[0]?.format('YYYY-MM-DD'),
      endDate: dateRange[1]?.format('YYYY-MM-DD'),
      period: timeframe,
    });
  }, [dateRange, timeframe]);

  // Filter data based on search term
  const filteredData = useMemo(() => {
    if (!searchTerm) return salesData;
    const term = searchTerm.toLowerCase();
    return salesData.filter(item => 
      item.date.toLowerCase().includes(term) ||
      item.sales.toString().includes(term) ||
      item.revenue.toString().includes(term) ||
      item.orders.toString().includes(term) ||
      item.customers.toString().includes(term)
    );
  }, [salesData, searchTerm]);

  // Format currency
  const formatCurrency = (value: number): string => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(value);
  };

  // Format tooltip label
  const formatTooltipLabel = (value: string): string => {
    return dayjs(value).format('MMM D, YYYY');
  };

  // Format tooltip value
  const formatTooltipValue = (value: number, name: string): [string, string] => {
    if (name === 'Revenue') {
      return [formatCurrency(value), name];
    }
    return [value.toString(), name];
  };

  // Format date based on timeframe
  const formatDate = (dateString: string) => {
    const date = dayjs(dateString);
    switch (timeframe) {
      case 'daily':
        return date.format('MMM D, YYYY');
      case 'weekly':
        return `Week of ${date.startOf('week').format('MMM D')}`;
      case 'monthly':
        return date.format('MMM YYYY');
      case 'yearly':
        return date.format('YYYY');
      default:
        return date.format('MMM D, YYYY');
    }
  };

  // Format percentage
  const formatPercentage = (value: number) => {
    return `${(value * 100).toFixed(1)}%`;
  };

  // Disable future dates
  const disabledDate: RangePickerProps['disabledDate'] = (current) => {
    return current && current > dayjs().endOf('day');
  };

  const handleDateChange = (dates: [dayjs.Dayjs, dayjs.Dayjs] | null) => {
    if (dates && dates[0] && dates[1]) {
      setDateRange([dates[0], dates[1]]);
      // Trigger a new data fetch when date range changes
      fetchSalesReport({
        startDate: dates[0].format('YYYY-MM-DD'),
        endDate: dates[1].format('YYYY-MM-DD'),
        period: timeframe,
      });
    }
  };

  // Handle timeframe change
  const handleTimeframeChange = (value: Timeframe) => {
    setTimeframe(value);
    fetchSalesReport({
      startDate: dateRange[0]?.format('YYYY-MM-DD'),
      endDate: dateRange[1]?.format('YYYY-MM-DD'),
      period: value,
    });
  };

  // Handle category filter change
  const handleCategoryChange = (value: string) => {
    setCategory(value);
    // You can add additional filtering logic here if needed
  };

  const handleRefresh = async () => {
    await fetchSalesReport();
    toast.success('Sales data refreshed');
  };

  const handleExport = async () => {
    try {
      await reportService.exportSalesReport('excel', {
        startDate: dateRange[0]?.format('YYYY-MM-DD'),
        endDate: dateRange[1]?.format('YYYY-MM-DD'),
        period: timeframe,
      });
      toast.success('Export started');
    } catch (error) {
      console.error('Export failed:', error);
      toast.error('Failed to export report');
    }
  };

  // Render loading state
  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <Spin size="large" />
      </div>
    );
  }

  // Render error state
  if (apiError) {
    return (
      <div className="p-4">
        <Alert
          message="Error Loading Report"
          description={apiError}
          type="error"
          showIcon
          action={
            <Button type="primary" onClick={() => fetchSalesReport()}>
              Retry
            </Button>
          }
        />
      </div>
    );
  }

  const columns = [
    {
      title: 'Date',
      dataIndex: 'date',
      key: 'date',
      render: (text: string) => (
        <div className="flex items-center">
          <CalendarIcon className="w-4 h-4 mr-2 text-gray-500" />
          <span>{dayjs(text).format('MMM D, YYYY')}</span>
        </div>
      ),
      sorter: (a: SalesData, b: SalesData) => new Date(a.date).getTime() - new Date(b.date).getTime(),
    },
    {
      title: 'Sales',
      dataIndex: 'sales',
      key: 'sales',
      render: (value: number) => (
        <div className="flex items-center">
          <ShoppingCart className="w-4 h-4 mr-2 text-blue-500" />
          <span>{value}</span>
        </div>
      ),
      sorter: (a: SalesData, b: SalesData) => a.sales - b.sales,
    },
    {
      title: 'Revenue',
      dataIndex: 'revenue',
      key: 'revenue',
      render: (value: number) => (
        <div className="flex items-center">
          <DollarSign className="w-4 h-4 mr-2 text-green-500" />
          <span>${value.toLocaleString()}</span>
        </div>
      ),
      sorter: (a: SalesData, b: SalesData) => a.revenue - b.revenue,
    },
    {
      title: 'Orders',
      dataIndex: 'orders',
      key: 'orders',
      render: (value: number) => (
        <div className="flex items-center">
          <ShoppingCart className="w-4 h-4 mr-2 text-purple-500" />
          <span>{value}</span>
        </div>
      ),
      sorter: (a: SalesData, b: SalesData) => a.orders - b.orders,
    },
    {
      title: 'Customers',
      dataIndex: 'customers',
      key: 'customers',
      render: (value: number) => (
        <div className="flex items-center">
          <Users className="w-4 h-4 mr-2 text-orange-500" />
          <span>{value}</span>
        </div>
      ),
      sorter: (a: SalesData, b: SalesData) => a.customers - b.customers,
    },
  ];

  // Render loading state if no data
  if (loading && salesData.length === 0) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <Spin size="large" />
      </div>
    );
  }

  // Render the main component
  return (
    <div className="flex h-screen bg-gray-50">
      <Sidebar items={adminSidebarItems} />
      <div className="flex-1 flex flex-col overflow-hidden">
        <div className="flex-1 p-6 overflow-auto">
          {/* Header with title and actions */}
          <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
            <div>
              <h1 className="text-2xl font-semibold text-gray-800">Sales Report</h1>
              <p className="text-gray-500">Analyze your sales performance and trends</p>
            </div>
            <div className="mt-4 md:mt-0 flex space-x-2">
              <Button 
                type="default" 
                icon={<RefreshCw className="w-4 h-4" />} 
                onClick={handleRefresh}
                loading={loading}
                className="flex items-center"
              >
                <span className="ml-1">Refresh</span>
              </Button>
              <Button 
                type="primary" 
                icon={<Download className="w-4 h-4" />} 
                onClick={handleExport}
                className="flex items-center bg-blue-600 hover:bg-blue-700"
              >
                <span className="ml-1">Export</span>
              </Button>
            </div>
          </div>
          {/* Search and Filters */}
          <Card className="mb-6">
            <div className="flex flex-col space-y-4">
              <div className="flex flex-col md:flex-row md:items-center md:justify-between">
                <div className="w-full md:w-1/3 mb-4 md:mb-0">
                  <Input
                    placeholder="Search sales..."
                    prefix={<SearchIcon className="w-4 h-4 text-gray-400" />}
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full"
                    size="large"
                  />
                </div>
                <Button 
                  type="text" 
                  icon={<Filter className="w-4 h-4" />} 
                  onClick={toggleFilters}
                  className="flex items-center text-gray-600"
                >
                  {showFilters ? 'Hide Filters' : 'Show Filters'}
                  {showFilters ? (
                    <ChevronUp className="w-4 h-4 ml-1" />
                  ) : (
                    <ChevronDown className="w-4 h-4 ml-1" />
                  )}
                </Button>
              </div>

              {showFilters && (
                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 pt-4 border-t border-gray-200">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Date Range</label>
                    <RangePicker
                      value={dateRange as any}
                      onChange={handleDateChange as any}
                      disabledDate={disabledDate}
                      className="w-full"
                      size="middle"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Timeframe</label>
                    <Select
                      value={timeframe}
                      onChange={setTimeframe}
                      className="w-full"
                      size="middle"
                      suffixIcon={<ChevronDown className="w-4 h-4 text-gray-400" />}
                    >
                      <SelectOption value="day">Daily</SelectOption>
                      <SelectOption value="week">Weekly</SelectOption>
                      <SelectOption value="month">Monthly</SelectOption>
                    </Select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Category</label>
                    <Select
                      value={category}
                      onChange={setCategory}
                      className="w-full"
                      size="middle"
                      suffixIcon={<ChevronDown className="w-4 h-4 text-gray-400" />}
                    >
                      <SelectOption value="all">All Categories</SelectOption>
                      <SelectOption value="electronics">Electronics</SelectOption>
                      <SelectOption value="clothing">Clothing</SelectOption>
                      <SelectOption value="food">Food</SelectOption>
                    </Select>
                  </div>
                  <div className="flex items-end">
                    <Button 
                      type="primary" 
                      icon={<Filter className="w-4 h-4" />} 
                      onClick={handleRefresh}
                      loading={loading}
                      className="w-full"
                    >
                      Apply Filters
                    </Button>
                  </div>
                </div>
              )}
            </div>
          </Card>

          {/* Summary Cards */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
            <Card className="shadow-sm hover:shadow-md transition-shadow">
              <div className="flex items-center">
                <div className="p-3 rounded-lg bg-blue-50">
                  <CalendarIcon className="w-6 h-6 text-blue-600" />
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-500">Total Sales</p>
                  <p className="text-2xl font-semibold text-gray-900">{summary.totalSales}</p>
                  <p className="text-xs text-green-600">+12% from last month</p>
                </div>
              </div>
            </Card>
            <Card className="shadow-sm hover:shadow-md transition-shadow">
              <div className="flex items-center">
                <div className="p-3 rounded-lg bg-green-50">
                  <DollarSign className="w-6 h-6 text-green-600" />
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-500">Total Revenue</p>
                  <p className="text-2xl font-semibold text-gray-900">${summary.totalRevenue.toLocaleString()}</p>
                  <p className="text-xs text-green-600">+8.5% from last month</p>
                </div>
              </div>
            </Card>
            <Card className="shadow-sm hover:shadow-md transition-shadow">
              <div className="flex items-center">
                <div className="p-3 rounded-lg bg-purple-50">
                  <Tag className="w-6 h-6 text-purple-600" />
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-500">Total Orders</p>
                  <p className="text-2xl font-semibold text-gray-900">{summary.totalOrders}</p>
                  <p className="text-xs text-green-600">+5.2% from last month</p>
                </div>
              </div>
            </Card>
            <Card className="shadow-sm hover:shadow-md transition-shadow">
              <div className="flex items-center">
                <div className="p-3 rounded-lg bg-orange-50">
                  <Users className="w-6 h-6 text-orange-600" />
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-500">Total Customers</p>
                  <p className="text-2xl font-semibold text-gray-900">{summary.totalCustomers}</p>
                  <p className="text-xs text-green-600">+3.7% from last month</p>
                </div>
              </div>
            </Card>
          </div>

          {/* Charts Section */}
          <div className="mb-6">
            <div className="flex items-center justify-between mb-4">
              <h2 className="text-lg font-medium text-gray-900">Sales Analytics</h2>
              <div className="flex space-x-2">
                <Button type="text" icon={<LineChartIcon className="w-4 h-4" />} size="small">
                  trends
                </Button>
                <Button type="text" icon={<BarChart2 className="w-4 h-4" />} size="small">
                  Compare
                </Button>
                <Button type="text" icon={<PieChartIcon className="w-4 h-4" />} size="small">
                  Distribution
                </Button>
              </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
              <Card className="shadow-sm hover:shadow-md transition-shadow">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="font-medium text-gray-900">Sales Trend</h3>
                  <Select 
                    defaultValue="week" 
                    size="small"
                    suffixIcon={<ChevronDown className="w-4 h-4 text-gray-400" />}
                    className="w-32"
                  >
                    <SelectOption value="week">This Week</SelectOption>
                    <SelectOption value="month">This Month</SelectOption>
                    <SelectOption value="year">This Year</SelectOption>
                  </Select>
                </div>
                <div style={{ width: '100%', height: 300 }}>
                  <ResponsiveContainer>
                    <LineChart data={salesData}>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis 
                        dataKey="date" 
                        tickFormatter={(value) => dayjs(value).format('MMM D')}
                        tick={{ fill: '#6b7280', fontSize: 12 }}
                        axisLine={false}
                      />
                      <YAxis 
                        tick={{ fill: '#6b7280', fontSize: 12 }}
                        axisLine={false}
                        tickFormatter={(value) => `$${value / 1000}k`}
                      />
                      <Tooltip 
                        labelFormatter={(value) => dayjs(value).format('MMM D, YYYY')}
                        formatter={(value: any) => [`${value} sales`, 'Sales']}
                        contentStyle={{
                          borderRadius: '8px',
                          border: '1px solid #e5e7eb',
                          boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)'
                        }}
                      />
                      <Legend />
                      <Line 
                        type="monotone" 
                        dataKey="sales" 
                        name="Sales" 
                        stroke="#3b82f6" 
                        strokeWidth={2}
                        dot={false}
                        activeDot={{ r: 6, stroke: '#2563eb', strokeWidth: 2, fill: '#fff' }}
                      />
                    </LineChart>
                  </ResponsiveContainer>
                </div>
              </Card>

              <Card className="shadow-sm hover:shadow-md transition-shadow">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="font-medium text-gray-900">Revenue by Day</h3>
                  <Select 
                    defaultValue="revenue" 
                    size="small"
                    suffixIcon={<ChevronDown className="w-4 h-4 text-gray-400" />}
                    className="w-32"
                  >
                    <SelectOption value="revenue">Revenue</SelectOption>
                    <SelectOption value="orders">Orders</SelectOption>
                    <SelectOption value="customers">Customers</SelectOption>
                  </Select>
                </div>
                <div style={{ width: '100%', height: 300 }}>
                  <ResponsiveContainer>
                    <BarChart data={salesData}>
                      <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                      <XAxis 
                        dataKey="date" 
                        tickFormatter={(value) => dayjs(value).format('ddd')}
                        tick={{ fill: '#6b7280', fontSize: 12 }}
                        axisLine={false}
                      />
                      <YAxis 
                        tick={{ fill: '#6b7280', fontSize: 12 }}
                        axisLine={false}
                        tickFormatter={(value) => `$${value / 1000}k`}
                      />
                      <Tooltip 
                        labelFormatter={(value) => dayjs(value).format('MMM D, YYYY')}
                        formatter={(value: any) => [`$${value.toLocaleString()}`, 'Revenue']}
                        contentStyle={{
                          borderRadius: '8px',
                          border: '1px solid #e5e7eb',
                          boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)'
                        }}
                      />
                      <Bar 
                        dataKey="revenue" 
                        name="Revenue" 
                        fill="#10b981"
                        radius={[4, 4, 0, 0]}
                      />
                    </BarChart>
                  </ResponsiveContainer>
                </div>
              </Card>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              <Card className="shadow-sm hover:shadow-md transition-shadow">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="font-medium text-gray-900">Top Selling Products</h3>
                  <Button type="text" size="small">View All</Button>
                </div>
                <div style={{ width: '100%', height: 300 }} className="flex items-center">
                  <ResponsiveContainer>
                    <PieChart>
                      <Pie
                        data={topProducts}
                        cx="50%"
                        cy="50%"
                        innerRadius={60}
                        outerRadius={80}
                        paddingAngle={5}
                        dataKey="sales"
                        nameKey="name"
                        label={({ percent }) => `${(percent * 100).toFixed(0)}%`}
                        labelLine={false}
                      >
                        {topProducts.map((entry, index) => (
                          <Cell 
                            key={`cell-${index}`} 
                            fill={COLORS[index % COLORS.length]} 
                          />
                        ))}
                      </Pie>
                      <Tooltip 
                        formatter={(value: any, name: any, props: any) => [
                          `$${props.payload.revenue.toLocaleString()}`,
                          props.payload.name
                        ]}
                        contentStyle={{
                          borderRadius: '8px',
                          border: '1px solid #e5e7eb',
                          boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06)'
                        }}
                      />
                      <Legend 
                        layout="vertical"
                        verticalAlign="middle"
                        align="right"
                        formatter={(value, entry: any, index) => (
                          <div className="flex items-center text-sm text-gray-600">
                            <div 
                              className="w-3 h-3 rounded-full mr-2" 
                              style={{ backgroundColor: COLORS[index % COLORS.length] }}
                            />
                            {value}
                          </div>
                        )}
                      />
                    </PieChart>
                  </ResponsiveContainer>
                </div>
              </Card>

              <Card className="shadow-sm hover:shadow-md transition-shadow">
                <div className="flex justify-between items-center mb-4">
                  <h3 className="font-medium text-gray-900">Top Products by Revenue</h3>
                  <Button type="text" size="small">View All</Button>
                </div>
                <Table
                  dataSource={topProducts}
                  columns={[
                    {
                      title: 'Product',
                      dataIndex: 'name',
                      key: 'name',
                      render: (text: string) => (
                        <div className="flex items-center">
                          <div className="w-8 h-8 bg-gray-100 rounded-md mr-3 flex items-center justify-center">
                            <Package className="w-4 h-4 text-gray-500" />
                          </div>
                          <span className="font-medium">{text}</span>
                        </div>
                      ),
                    },
                    {
                      title: 'Sales',
                      dataIndex: 'sales',
                      key: 'sales',
                      sorter: (a: TopProduct, b: TopProduct) => a.sales - b.sales,
                      render: (value: number) => (
                        <div className="flex items-center">
                          <span className="font-medium">{value}</span>
                          <span className="text-xs text-gray-500 ml-1">units</span>
                        </div>
                      ),
                    },
                    {
                      title: 'Revenue',
                      dataIndex: 'revenue',
                      key: 'revenue',
                      sorter: (a: TopProduct, b: TopProduct) => a.revenue - b.revenue,
                      render: (value: number) => (
                        <div className="flex items-center">
                          <DollarSign className="w-3.5 h-3.5 text-green-600 mr-1" />
                          <span className="font-medium">{value.toLocaleString()}</span>
                        </div>
                      ),
                    },
                  ]}
                  pagination={false}
                  size="middle"
                  rowKey="id"
                  className="custom-table"
                />
              </Card>
            </div>
          </div>

          {/* Sales Data Table */}
          <Card className="shadow-sm hover:shadow-md transition-shadow">
            <div className="flex justify-between items-center mb-4">
              <div>
                <h3 className="font-medium text-gray-900">Sales Data</h3>
                <p className="text-sm text-gray-500">Detailed view of all sales transactions</p>
              </div>
              <div className="flex space-x-2">
                <Button type="text" size="small" icon={<Download className="w-4 h-4" />}>
                  Export
                </Button>
                <Button type="text" size="small" icon={<RefreshCw className="w-4 h-4" />} onClick={handleRefresh}>
                  Refresh
                </Button>
              </div>
            </div>
            
            <Table
              columns={columns as any}
              dataSource={filteredData}
              rowKey="date"
              loading={loading}
              size="middle"
              scroll={{ x: true }}
              pagination={{
                showSizeChanger: true,
                showQuickJumper: true,
                showTotal: (total, range) => `Showing ${range[0]}-${range[1]} of ${total} records`,
                pageSizeOptions: ['5', '10', '20', '50'],
                defaultPageSize: 10,
                className: 'pagination-custom'
              }}
              className="custom-table"
            />
          </Card>
        </div>
        <style jsx global>{`
        .ant-table-thead > tr > th {
          background-color: #f9fafb !important;
          font-weight: 500 !important;
          color: #374151 !important;
          text-transform: uppercase;
          font-size: 0.75rem;
          letter-spacing: 0.05em;
        }
        
        .ant-table-tbody > tr > td {
          border-bottom: 1px solid #f3f4f6 !important;
        }
        
        .ant-table-tbody > tr:hover > td {
          background-color: #f9fafb !important;
        }
        
        .ant-pagination-item-active {
          border-color: #3b82f6 !important;
        }
        
        .ant-pagination-item-active a {
          color: #3b82f6 !important;
        }
        
        .ant-select-selector {
          border-radius: 0.375rem !important;
        }
        
        .ant-btn > span {
          display: inline-flex;
          align-items: center;
        }
        
        .ant-card-head {
          border-bottom: 1px solid #f3f4f6 !important;
        }
        
        .ant-card-head-title {
          padding: 16px 0 !important;
        }
        
        .ant-table-pagination.ant-pagination {
          margin: 16px 0 0 0 !important;
          padding: 12px 16px;
          border-top: 1px solid #f3f4f6;
        }
        
        .custom-table .ant-table-thead > tr > th {
          background-color: #f9fafb !important;
        }
        
        .pagination-custom .ant-pagination-options {
          margin-left: auto;
        }
          `}
        </style>
      </div>
    </div>
  );
};

export default SalesReport;
