import React, { useState, useRef } from 'react';
import { Upload, FileText, Download, X, Check, AlertCircle } from 'lucide-react';
import { toast } from 'react-hot-toast';
import axios from 'axios';
import api from '@/services/api';

interface UpdateError {
  row: number;
  sku: string;
  error: string;
  rawData?: any;
}

interface UpdateResult {
  success: boolean;
  updatedCount?: number;
  errorCount: number;
  errors?: UpdateError[];
  totalCount?: number;
  msg?: string;
}

interface BulkUpdateProps {
  onBackToList: () => void;
}

const BulkUpdate: React.FC<BulkUpdateProps> = ({ onBackToList }) => {
  const handleBackToList = onBackToList || (() => window.history.back());
  const [file, setFile] = useState<File | null>(null);
  const [isUpdating, setIsUpdating] = useState(false);
  const [progress, setProgress] = useState(0);
  const [updateResult, setUpdateResult] = useState<UpdateResult | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files.length > 0) {
      setFile(e.target.files[0]);
      setUpdateResult(null);
    }
  };

  const handleUpdate = async () => {
    if (!file) {
      toast.error('Please select a file to update products');
      return;
    }

    console.log('File to update:', {
      name: file.name,
      type: file.type,
      size: file.size,
      lastModified: file.lastModified,
      isFile: file instanceof File
    });

    // Create a new FormData instance
    const formData = new FormData();
    
    // Create a new file with the correct MIME type
    const fileToUpload = new File(
      [file], 
      file.name.replace(/[^\w\d\.-]/g, '_'), // Sanitize filename
      { 
        type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        lastModified: file.lastModified
      }
    );
    
    // Append the file with a simple field name
    formData.append('upload', fileToUpload);
    
    // Log FormData contents for debugging
    console.log('FormData contents:');
    for (let pair of (formData as any).entries()) {
      if (pair[1] instanceof File) {
        console.log(pair[0], {
          name: pair[1].name,
          type: pair[1].type,
          size: pair[1].size,
          lastModified: pair[1].lastModified
        });
      } else {
        console.log(pair[0], pair[1]);
      }
    }

    setIsUpdating(true);
    setProgress(30);
    setUpdateResult(null);

    try {
      console.log('Sending bulk update request...');
      // Use a direct fetch request to avoid any axios issues with FormData
      const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:5000/api/v1';
      const token = localStorage.getItem('token');
      
      // Create a new FormData instance for the fetch request
      const fetchFormData = new FormData();
      fetchFormData.append('file', file);
      
      // Make the request using fetch
      const response = await fetch(`${API_URL}/items/bulk-update`, {
        method: 'PUT',
        headers: {
          'Authorization': token ? `Bearer ${token}` : '',
          // Let the browser set the Content-Type with the correct boundary
        },
        body: fetchFormData,
        credentials: 'include', // equivalent to withCredentials: true in axios
      });
      
      // Parse the response
      const responseData = await response.json();
      
      // Create a response object similar to axios for consistency
      const formattedResponse = {
        data: responseData,
        status: response.status,
        statusText: response.statusText,
        headers: response.headers,
        config: {},
      };
      
      // Throw an error for non-2xx responses
      if (!response.ok) {
        const error = new Error(response.statusText);
        (error as any).response = formattedResponse;
        throw error;
      }
      
      return formattedResponse;
      
      console.log('Update response:', response);

      setProgress(100);
      
      const result = {
        success: true,
        updatedCount: response.data.updatedCount || 0,
        errorCount: response.data.errorCount || 0,
        errors: response.data.errors || [],
      };
      
      setUpdateResult(result);

      if (result.errorCount > 0) {
        toast.error(`Updated ${result.updatedCount || 0} products. ${result.errorCount} products had errors.`);
      } else {
        toast.success(`Successfully updated ${result.updatedCount || 0} products.`);
      }
    } catch (error: any) {
      console.error('Error updating products:', error);
      
      const errorResponse = error.response?.data;
      const errorMessage = errorResponse?.msg || 'Error updating products';
      const errors = Array.isArray(errorResponse?.errors) 
        ? errorResponse.errors 
        : [{ row: 0, sku: 'N/A', error: errorMessage }];
      
      const result = {
        success: false,
        errorCount: errors.length,
        errors: errors,
      };
      
      setUpdateResult(result);
      toast.error(errorMessage);
    } finally {
      setIsUpdating(false);
    }
  };

  const handleDownloadTemplate = async () => {
    try {
      const response = await api.get('/items/bulk-update/template', {
        responseType: 'blob',
      });

      const url = window.URL.createObjectURL(new Blob([response.data]));
      const link = document.createElement('a');
      link.href = url;
      link.setAttribute('download', 'bulk-update-template.xlsx');
      document.body.appendChild(link);
      link.click();
      link.remove();
    } catch (error) {
      console.error('Error downloading update template:', error);
      toast.error('Failed to download update template');
    }
  };

  const handleRemoveFile = () => {
    setFile(null);
    setUpdateResult(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  // UI Components
  const Button: React.FC<{
    variant?: 'default' | 'outline' | 'ghost' | 'link';
    size?: 'sm' | 'default' | 'lg';
    className?: string;
    onClick?: () => void;
    disabled?: boolean;
    children: React.ReactNode;
  }> = ({
    variant = 'default',
    size = 'default',
    className = '',
    onClick,
    disabled = false,
    children,
    ...props
  }) => {
    const baseStyles = 'inline-flex items-center justify-center rounded-md font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:opacity-50 disabled:pointer-events-none';
    const variantStyles = {
      default: 'bg-primary text-primary-foreground hover:bg-primary/90',
      outline: 'border border-input hover:bg-accent hover:text-accent-foreground',
      ghost: 'hover:bg-accent hover:text-accent-foreground',
      link: 'underline-offset-4 hover:underline text-primary',
    };
    const sizeStyles = {
      sm: 'h-9 px-3 text-xs',
      default: 'h-10 py-2 px-4',
      lg: 'h-11 px-8',
    };

    return (
      <button
        className={`${baseStyles} ${variantStyles[variant]} ${sizeStyles[size]} ${className}`}
        onClick={onClick}
        disabled={disabled}
        {...props}
      >
        {children}
      </button>
    );
  };

  const Card: React.FC<{ className?: string; children: React.ReactNode }> = ({ className = '', children }) => (
    <div className={`rounded-lg border bg-card text-card-foreground shadow-sm ${className}`}>
      {children}
    </div>
  );

  const CardHeader: React.FC<{ children: React.ReactNode }> = ({ children }) => (
    <div className="flex flex-col space-y-1.5 p-6">
      {children}
    </div>
  );

  const CardTitle: React.FC<{ children: React.ReactNode }> = ({ children }) => (
    <h3 className="text-2xl font-semibold leading-none tracking-tight">{children}</h3>
  );

  const CardDescription: React.FC<{ children: React.ReactNode }> = ({ children }) => (
    <p className="text-sm text-muted-foreground">{children}</p>
  );

  const CardContent: React.FC<{ className?: string; children: React.ReactNode }> = ({ className = '', children }) => (
    <div className={`p-6 pt-0 ${className}`}>
      {children}
    </div>
  );

  const Progress: React.FC<{ value: number; className?: string }> = ({ value, className = '' }) => (
    <div className={`relative h-2 w-full overflow-hidden rounded-full bg-secondary ${className}`}>
      <div
        className="h-full w-full flex-1 bg-primary transition-all"
        style={{ transform: `translateX(-${100 - (value || 0)}%)` }}
      />
    </div>
  );

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold tracking-tight">Bulk Update Products</h2>
          <p className="text-muted-foreground">
            Upload a file to update multiple products at once
          </p>
        </div>
        <Button onClick={handleDownloadTemplate} variant="outline">
          <Download className="mr-2 h-4 w-4" />
          Download Update Template
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Update File</CardTitle>
          <CardDescription>
            Upload a CSV or Excel file with product updates. Make sure to include product IDs or SKUs.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-6">
            <div
              className={`flex flex-col items-center justify-center border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
                file ? 'border-primary/50' : 'border-border hover:border-primary/50'
              }`}
              onDragOver={(e) => {
                e.preventDefault();
                e.stopPropagation();
              }}
              onDrop={(e) => {
                e.preventDefault();
                e.stopPropagation();
                if (e.dataTransfer.files && e.dataTransfer.files.length > 0) {
                  setFile(e.dataTransfer.files[0]);
                  setUpdateResult(null);
                }
              }}
            >
              <input
                type="file"
                ref={fileInputRef}
                onChange={handleFileChange}
                accept=".xlsx,.xls,.csv"
                className="hidden"
                disabled={isUpdating}
              />
              
              {!file ? (
                <div className="flex flex-col items-center space-y-2">
                  <Upload className="h-10 w-10 text-muted-foreground" />
                  <div className="text-sm text-muted-foreground">
                    <span className="font-medium text-primary">Click to upload</span> or drag and drop
                  </div>
                  <p className="text-xs text-muted-foreground">CSV or Excel files only (max. 10MB)</p>
                </div>
              ) : (
                <div className="flex flex-col items-center space-y-2 w-full">
                  <FileText className="h-10 w-10 text-primary" />
                  <div className="flex items-center space-x-2">
                    <span className="font-medium truncate max-w-xs">{file.name}</span>
                    <button
                      type="button"
                      onClick={handleRemoveFile}
                      className="text-muted-foreground hover:text-destructive"
                      disabled={isUpdating}
                    >
                      <X className="h-4 w-4" />
                    </button>
                  </div>
                  <span className="text-xs text-muted-foreground">
                    {(file.size / 1024 / 1024).toFixed(2)} MB
                  </span>
                </div>
              )}
            </div>

            {isUpdating && (
              <div className="space-y-2">
                <div className="flex justify-between text-sm text-muted-foreground">
                  <span>Updating products...</span>
                  <span>{progress}%</span>
                </div>
                <Progress value={progress} className="h-2" />
              </div>
            )}

            <div className="flex justify-end">
              <Button
                onClick={() => fileInputRef.current?.click()}
                variant="outline"
                className="mr-2"
                disabled={isUpdating}
              >
                {file ? 'Change File' : 'Select File'}
              </Button>
              <Button
                onClick={handleUpdate}
                disabled={!file || isUpdating}
                className="min-w-[120px]"
              >
                {isUpdating ? 'Updating...' : 'Update Products'}
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {updateResult && (
        <Card className={updateResult.success ? 'border-green-200 dark:border-green-900' : 'border-destructive/50'}>
          <CardHeader className="pb-3">
            <div className="flex items-center space-x-2">
              {updateResult.success ? (
                <>
                  <Check className="h-5 w-5 text-green-600" />
                  <CardTitle className="text-green-600">
                    {updateResult.errorCount && updateResult.errorCount > 0 ? 'Update Completed with Issues' : 'Update Complete'}
                  </CardTitle>
                </>
              ) : (
                <>
                  <AlertCircle className="h-5 w-5 text-destructive" />
                  <CardTitle className="text-destructive">
                    {updateResult.errorCount && updateResult.errorCount > 1 ? 'Validation Errors' : 'Update Failed'}
                  </CardTitle>
                </>
              )}
            </div>
            <CardDescription>
              {updateResult.success ? (
                <>
                  Successfully updated {updateResult.updatedCount || 0} products.
                  {updateResult.errorCount !== undefined && updateResult.errorCount > 0 && (
                    <span className="text-yellow-600 dark:text-yellow-400">
                      {' '}{updateResult.errorCount} product{updateResult.errorCount !== 1 ? 's had' : ' had'} errors.
                    </span>
                  )}
                </>
              ) : (
                <>
                  {updateResult.errorCount !== undefined && updateResult.errorCount > 1 
                    ? `Found ${updateResult.errorCount} validation errors in your file.`
                    : 'There was an error processing your update file.'
                  }
                </>
              )}
            </CardDescription>
          </CardHeader>
          
          {updateResult.errors && updateResult.errors.length > 0 && (
            <CardContent>
              <div className="rounded-md border border-border overflow-hidden">
                <div className="bg-muted/50 p-2 flex text-sm font-medium">
                  <div className="w-16">Row</div>
                  <div className="w-32">SKU</div>
                  <div>Error</div>
                </div>
                <div className="max-h-96 overflow-y-auto">
                  {updateResult.errors.map((error, index) => (
                    <div 
                      key={index} 
                      className="border-t border-border hover:bg-muted/10"
                    >
                      <div className="p-2 flex text-sm items-start">
                        <div className="w-16 font-medium">{error.row}</div>
                        <div className="w-32 font-mono text-sm break-words">
                          {error.sku || 'N/A'}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div className="text-destructive font-medium">{error.error}</div>
                          {error.rawData && (
                            <details className="mt-1 text-xs text-muted-foreground">
                              <summary className="cursor-pointer hover:underline">View row data</summary>
                              <pre className="mt-1 p-2 bg-muted/30 rounded overflow-x-auto">
                                {typeof error.rawData === 'string' 
                                  ? error.rawData 
                                  : JSON.stringify(error.rawData, null, 2)}
                              </pre>
                            </details>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          )}
        </Card>
      )}

      <Card>
        <CardHeader>
          <CardTitle>Update Instructions</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div>
            <div className="flex justify-between items-center mb-2">
              <h3 className="font-medium">Preparing Your Update File</h3>
              <Button 
                variant="outline" 
                size="sm" 
                onClick={handleBackToList}
                className="text-sm"
              >
                Back to List
              </Button>
            </div>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>Download and use the provided update template</li>
              <li>Only include the fields you want to update</li>
              <li>Always include the product ID or SKU to identify the product</li>
              <li>Leave fields blank to keep their current values</li>
              <li>Required fields for identification: <code>id</code> or <code>sku</code></li>
            </ul>
          </div>
          
          <div>
            <h3 className="font-medium mb-2">File Requirements</h3>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>File size: Maximum 10MB</li>
              <li>File formats: .xlsx, .xls, .csv</li>
              <li>Maximum rows: 10,000 per file</li>
              <li>Date format: YYYY-MM-DD</li>
            </ul>
          </div>
          
          <div>
            <h3 className="font-medium mb-2">Tips for Successful Updates</h3>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>Test with a small number of products first</li>
              <li>Verify IDs/SKUs exist in the system</li>
              <li>Check that all values match the expected format</li>
              <li>Review the error report after updating</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default BulkUpdate;