import React, { useEffect, useMemo, useState, useRef, useCallback } from 'react';
import { Search, Plus, Edit, Trash2, User, Users, MapPin, Store, Phone, Mail, Lock, Clock, Briefcase as BriefcaseIcon, Save, Eye, X, Star, CheckCircle, XCircle, FileText, Loader2, AlertCircle } from 'lucide-react';
import { deliveryBoysService } from '@/services/api';
import { toast } from 'react-hot-toast';
import api from '@/services/api';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';

interface DeliveryBoy {
  _id?: string;
  id?: string;
  name: string;
  email: string;
  phone: string;
  password?: string;
  store: string;
  isAvailable: boolean;
  status: 'active' | 'inactive' | 'on_leave' | 'pending' | 'approved' | 'rejected';
  rating?: number;
  totalDeliveries: number;
  completedDeliveries: number;
  cancelledDeliveries: number;
  location?: {
    type: string;
    coordinates: [number, number];
  };
  isVerified: boolean;
  lastOnline?: string;
  documents?: Array<{
    type: string;
    url: string;
    verified: boolean;
  }>;
  createdAt?: string;
  updatedAt?: string;
}

const DeliveryBoysList: React.FC = () => {
  const [deliveryBoys, setDeliveryBoys] = useState<DeliveryBoy[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState<DeliveryBoy | null>(null);
  const [showViewModal, setShowViewModal] = useState(false);
  const [viewingBoy, setViewingBoy] = useState<DeliveryBoy | null>(null);
  const [form, setForm] = useState<Partial<DeliveryBoy>>({
    name: '',
    email: '',
    phone: '',
    password: '',
    store: '',
    isAvailable: true,
    status: 'active',
    documents: [],
  });

  const [location, setLocation] = useState<{ longitude: string; latitude: string }>({ longitude: '', latitude: '' });
  const [stores, setStores] = useState<Array<{ _id: string; name: string }>>([]);

  // Map related state
  const [mapLoaded, setMapLoaded] = useState(false);
  const [mapError, setMapError] = useState<string | null>(null);
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstance = useRef<google.maps.Map | null>(null);
  const markerInstance = useRef<google.maps.Marker | null>(null);
  const GOOGLE_MAPS_API_KEY = import.meta.env.VITE_GOOGLE_MAPS_API_KEY || 'AIzaSyCJKwxaSS0glDtxXMX37uHX_KHUEleCMk0';

  const hasRows = useMemo(() => deliveryBoys.length > 0, [deliveryBoys]);

  const fetchDeliveryBoys = async () => {
    try {
      setLoading(true);
      const res = await api.get('/delivery-boys', { params: { search: searchTerm } });
      const rows = res.data?.data || res.data || [];
      setDeliveryBoys(rows);
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to load delivery boys');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    const fetchStores = async () => {
      try {
        const res = await api.get('/stores?limit=100'); // Adjust limit as needed
        setStores(res.data?.data || []);
      } catch (error) {
        console.error('Failed to fetch stores', error);
      }
    };

    fetchDeliveryBoys();
    fetchStores();
  }, []);

  // Load Google Maps API script
  useEffect(() => {
    if (!GOOGLE_MAPS_API_KEY) {
      setMapError('Google Maps API key is not configured');
      return;
    }

    // Check if already loaded
    if (window.google && window.google.maps) {
      console.log('Google Maps already loaded');
      setMapLoaded(true);
      return;
    }

    // Check if script is already added
    const existingScript = document.querySelector(`script[src*="maps.googleapis.com/maps/api/js"]`);
    if (existingScript) {
      console.log('Google Maps script already added, waiting for load...');
      // If script is added but not loaded yet, we'll wait for the callback
      if (!window.google || !window.google.maps) {
        (window as any).initMap = () => {
          console.log('Google Maps script loaded successfully (existing script)');
          setMapLoaded(true);
        };
      }
      return;
    }

    console.log('Loading Google Maps script...');
    
    // Create script element
    const script = document.createElement('script');
    script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_API_KEY}&libraries=places&callback=initMap`;
    script.async = true;
    script.defer = true;
    
    // Add global init function
    (window as any).initMap = () => {
      console.log('Google Maps script loaded successfully');
      setMapLoaded(true);
      // Force a resize event to ensure the map renders correctly
      setTimeout(() => {
        window.dispatchEvent(new Event('resize'));
      }, 100);
    };

    script.onerror = () => {
      console.error('Error loading Google Maps script');
      setMapError('Failed to load Google Maps. Please check your internet connection and API key.');
    };

    document.head.appendChild(script);

    // Cleanup
    return () => {
      if ((window as any).initMap) {
        delete (window as any).initMap;
      }
    };
  }, []);

  // Initialize map when modal opens
  useEffect(() => {
    if (!mapLoaded || !showModal || !mapRef.current) return;

    // Ensure the map container has proper dimensions
    mapRef.current.style.minHeight = '256px';
    mapRef.current.style.width = '100%';

    try {
      const defaultLocation = {
        lat: 22.5726, // Default to Kolkata
        lng: 88.3639,
      };

      // If we have location from props, use that
      let initialLocation = defaultLocation;
      if (editing?.location?.coordinates) {
        initialLocation = {
          lat: editing.location.coordinates[1],
          lng: editing.location.coordinates[0],
        };
      } else if (location.latitude && location.longitude) {
        initialLocation = {
          lat: parseFloat(location.latitude),
          lng: parseFloat(location.longitude),
        };
      }

      const mapOptions: google.maps.MapOptions = {
        center: initialLocation,
        zoom: 15,
        mapTypeId: 'roadmap',
        streetViewControl: false,
        fullscreenControl: true,
        zoomControl: true,
        mapTypeControl: false,
        backgroundColor: '#f8fafc', // Light gray background
        styles: [
          {
            featureType: 'poi',
            elementType: 'labels',
            stylers: [{ visibility: 'off' }]
          }
        ]
      };

      // Initialize the map
      mapInstance.current = new google.maps.Map(mapRef.current, mapOptions);

      // Add marker if we have a location
      if (initialLocation.lat !== defaultLocation.lat || initialLocation.lng !== defaultLocation.lng) {
        updateMarker(initialLocation);
      }

      // Add click listener to map
      mapInstance.current.addListener('click', (event: google.maps.MapMouseEvent) => {
        if (event.latLng) {
          const location = {
            lat: event.latLng.lat(),
            lng: event.latLng.lng(),
          };
          updateMarker(location);
          updateLocationFromCoords(location);
        }
      });

      // Try to get user's current location if no location is set
      if (!editing?.location?.coordinates && !location.latitude && !location.longitude && navigator.geolocation) {
        navigator.geolocation.getCurrentPosition(
          (position) => {
            const pos = {
              lat: position.coords.latitude,
              lng: position.coords.longitude,
            };
            updateMarker(pos);
            mapInstance.current?.setCenter(pos);
            updateLocationFromCoords(pos);
          },
          () => {
            // Use default location if geolocation fails
            updateMarker(defaultLocation);
          },
        );
      }
    } catch (error) {
      console.error('Error initializing map:', error);
      setMapError('Failed to initialize the map. Please try again.');
    }

    // Cleanup
    return () => {
      if (mapInstance.current) {
        mapInstance.current = null;
      }
      if (markerInstance.current) {
        markerInstance.current = null;
      }
    };
  }, [mapLoaded, showModal, editing]);

  // Update marker position
  const updateMarker = (position: { lat: number; lng: number }) => {
    if (!mapInstance.current) return;

    if (markerInstance.current) {
      markerInstance.current.setPosition(position);
    } else {
      markerInstance.current = new google.maps.Marker({
        position,
        map: mapInstance.current,
        draggable: true,
        title: 'Delivery Location',
      });

      // Add dragend listener to update location when marker is moved
      markerInstance.current.addListener('dragend', (event: google.maps.MapMouseEvent) => {
        if (event.latLng) {
          const location = {
            lat: event.latLng.lat(),
            lng: event.latLng.lng(),
          };
          updateLocationFromCoords(location);
        }
      });
    }
  };

  // Update location state from coordinates
  const updateLocationFromCoords = (location: { lat: number; lng: number }) => {
    // Update the location state
    setLocation({
      latitude: location.lat.toString(),
      longitude: location.lng.toString(),
    });

    // Reverse geocode to get address
    reverseGeocode(location);
  };

  // Reverse geocode function to get address from coordinates
  const reverseGeocode = useCallback(async (location: { lat: number; lng: number }) => {
    if (!window.google || !window.google.maps) return;

    try {
      const geocoder = new google.maps.Geocoder();

      geocoder.geocode({ location }, (results, status) => {
        if (status === 'OK' && results && results[0]) {
          // You can update any address fields here if needed
          console.log('Address from coordinates:', results[0].formatted_address);
        }
      });
    } catch (error) {
      console.error('Reverse geocoding error:', error);
    }
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    console.log('Form submission started');
    console.log('Form data:', form);
    console.log('Editing mode:', !!editing);

    // Basic validation
    if (!form.name?.trim() || !form.email?.trim() || !form.phone?.trim()) {
      const errorMsg = 'Name, email, and phone are required';
      console.error('Validation error:', errorMsg);
      toast.error(errorMsg);
      return;
    }

    if (!editing && !form.password) {
      const errorMsg = 'Password is required for new delivery boy';
      console.error('Validation error:', errorMsg);
      toast.error(errorMsg);
      return;
    }

    try {
      const deliveryBoyData: any = {
        name: form.name.trim(),
        email: form.email.trim().toLowerCase(),
        phone: form.phone.trim(),
        status: form.status || 'active',
        isAvailable: form.isAvailable !== false,
      };

      console.log('Prepared delivery boy data:', deliveryBoyData);

      // Add store only if selected
      if (form.store && form.store.trim()) {
        deliveryBoyData.store = form.store;
      }

      // Add password only for new delivery boys
      if (!editing && form.password) {
        deliveryBoyData.password = form.password;
      }

      // Add location if provided
      if (location.longitude && location.latitude) {
        deliveryBoyData.longitude = parseFloat(location.longitude);
        deliveryBoyData.latitude = parseFloat(location.latitude);
      }

      console.log('Final delivery boy data to send:', deliveryBoyData);

      if (editing) {
        const updateUrl = `/delivery-boys/${editing.id || editing._id}`;
        console.log('Sending PUT request to:', updateUrl, 'with data:', deliveryBoyData);

        try {
          const response = await api.put(updateUrl, deliveryBoyData);
          console.log('Update response:', response);

          if (response.data) {
            console.log('Successfully updated delivery boy:', response.data);
            toast.success('Delivery boy updated successfully');
          } else {
            console.error('Unexpected response format:', response);
            toast.error('Received unexpected response from server');
          }
        } catch (error: any) {
          console.error('Error updating delivery boy:', error);
          console.error('Error response:', error.response);
          throw error; // Re-throw to be caught by the outer catch
        }
      } else {
        console.log('Sending POST request to /delivery-boys with data:', deliveryBoyData);
        const response = await api.post('/delivery-boys', deliveryBoyData);
        console.log('Create response:', response);
        toast.success('Delivery boy created successfully');
      }

      // Reset form and fetch updated list
      console.log('Resetting form and refreshing delivery boys list');
      setShowModal(false);
      setEditing(null);
      setForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        store: '',
        status: 'active',
        isAvailable: true,
      });
      setLocation({ longitude: '', latitude: '' });
      await fetchDeliveryBoys();
    } catch (e: any) {
      console.error('Error in handleSubmit:', e);

      // Log detailed error information
      if (e.response) {
        console.error('Error response data:', e.response.data);
        console.error('Error status:', e.response.status);
        console.error('Error headers:', e.response.headers);

        // More specific error messages based on status code
        if (e.response.status === 401) {
          toast.error('Session expired. Please log in again.');
        } else if (e.response.status === 403) {
          toast.error('You do not have permission to perform this action.');
        } else if (e.response.status === 400) {
          toast.error(e.response.data.message || 'Invalid data. Please check your input.');
        } else if (e.response.status === 404) {
          toast.error('The requested resource was not found.');
        } else if (e.response.status >= 500) {
          toast.error('Server error. Please try again later.');
        } else {
          toast.error(e.response.data?.message || e.response.data?.msg || 'An error occurred');
        }
      } else if (e.request) {
        console.error('No response received:', e.request);
        toast.error('No response from server. Please check your connection.');
      } else {
        console.error('Error setting up request:', e.message);
        toast.error('Failed to process request. Please try again.');
      }

      // If it's a validation error with multiple messages
      if (e.response?.data?.errors) {
        console.error('Validation errors:', e.response.data.errors);
        Object.values(e.response.data.errors).forEach((errorMsg: any) => {
          toast.error(Array.isArray(errorMsg) ? errorMsg[0] : errorMsg);
        });
      } else if (e.message) {
        toast.error(e.message);
      }
    }
  };

  const handleEdit = (deliveryBoy: DeliveryBoy) => {
    setEditing(deliveryBoy);
    setForm({
      ...deliveryBoy,
      status: deliveryBoy.status || 'active',
      isAvailable: deliveryBoy.isAvailable !== false,
    });

    // Set location if available
    if (deliveryBoy.location?.coordinates) {
      setLocation({
        longitude: deliveryBoy.location.coordinates[0]?.toString() || '',
        latitude: deliveryBoy.location.coordinates[1]?.toString() || '',
      });
    } else {
      setLocation({ longitude: '', latitude: '' });
    }

    setShowModal(true);
  };

  const [updatingAvailability, setUpdatingAvailability] = useState<Record<string, boolean>>({});

  const toggleAvailability = async (id: string, currentStatus: boolean) => {
    const deliveryBoyId = id || '';
    if (!deliveryBoyId) return;

    setUpdatingAvailability((prev) => ({ ...prev, [deliveryBoyId]: true }));

    try {
      // Update the UI optimistically first
      const newStatus = !currentStatus;
      setDeliveryBoys((prev) =>
        prev.map((boy) =>
          (boy.id === deliveryBoyId || boy._id === deliveryBoyId)
            ? { ...boy, isAvailable: newStatus }
            : boy,
        ),
      );

      // Call the API to update the backend
      await deliveryBoysService.updateAvailability(deliveryBoyId, newStatus);

      toast.success(`Delivery boy marked as ${newStatus ? 'available' : 'unavailable'}`);
    } catch (e: any) {
      console.error('Error updating availability:', e);

      // Revert the UI on error
      setDeliveryBoys((prev) =>
        prev.map((boy) =>
          (boy.id === deliveryBoyId || boy._id === deliveryBoyId)
            ? { ...boy, isAvailable: currentStatus }
            : boy,
        ),
      );

      const errorMessage = e.response?.data?.message ||
        e.response?.data?.error ||
        'Failed to update availability. Please try again.';

      toast.error(errorMessage);
    } finally {
      setUpdatingAvailability((prev) => ({ ...prev, [deliveryBoyId]: false }));
    }
  };

  const handleDelete = async (id: string) => {
    if (!window.confirm('Are you sure you want to permanently delete this delivery boy? This action cannot be undone.')) {
      return;
    }

    try {
      console.log('Attempting to delete delivery boy with ID:', id);
      const response = await api.delete(`/delivery-boys/${id}`);
      console.log('Delete response:', response);

      // If we get here, the deletion was successful
      toast.success('Delivery boy deleted successfully');
      // Update the UI by removing the deleted item from the list
      setDeliveryBoys((prev) => prev.filter((boy) => (boy.id || boy._id) !== id));
    } catch (error: any) {
      console.error('Error deleting delivery boy:', error);

      // If the error is 404, the delivery boy was already deleted or never existed
      if (error.response?.status === 404) {
        // Remove the item from the UI anyway since it doesn't exist on the server
        setDeliveryBoys((prev) => prev.filter((boy) => (boy.id || boy._id) !== id));
        toast.success('Delivery boy was already removed');
        return;
      }

      // If the error is not 404, show a generic error message
      const errorMessage = error.response?.data?.message ||
        error.response?.data?.error ||
        error.message ||
        'Failed to delete delivery boy. Please try again.';

      toast.error(errorMessage);
    }
  };

  const renderDeliveryBoyRow = (deliveryBoy: DeliveryBoy) => {
    const deliveryBoyId = deliveryBoy.id || deliveryBoy._id || '';
    const isUpdating = updatingAvailability[deliveryBoyId] || false;

    const statusStyles = {
      active: 'bg-green-100 text-green-800',
      approved: 'bg-blue-100 text-blue-800',
      pending: 'bg-yellow-100 text-yellow-800',
      rejected: 'bg-red-100 text-red-800',
      inactive: 'bg-gray-100 text-gray-800',
      on_leave: 'bg-purple-100 text-purple-800',
    };

    return (
      <tr className="hover:bg-gray-50 transition-colors">
        <td className="px-6 py-4 whitespace-nowrap">
          <div className="flex items-center">
            <div className="flex-shrink-0 h-11 w-11 rounded-full bg-blue-100 flex items-center justify-center ring-2 ring-blue-200">
              <User className="h-6 w-6 text-blue-600" />
            </div>
            <div className="ml-4">
              <div className="text-sm font-semibold text-gray-900">{deliveryBoy.name}</div>
              <div className="text-xs text-gray-500">{deliveryBoy.email}</div>
            </div>
          </div>
        </td>
        <td className="px-6 py-4 whitespace-nowrap">
          <div className="text-sm text-gray-900 flex items-center">
            <Phone className="h-4 w-4 mr-2 text-gray-400" />
            {deliveryBoy.phone}
          </div>
          <div className="text-xs text-gray-500 flex items-center mt-1">
            <Store className="h-4 w-4 mr-2 text-gray-400" />
            {deliveryBoy.store ? stores.find((s: { _id: string; name: string }) => s._id === deliveryBoy.store)?.name || 'N/A' : 'N/A'}
          </div>
        </td>
        <td className="px-6 py-4 whitespace-nowrap">
          <div className="flex flex-col space-y-2">
            <span
              className={`px-3 py-1 inline-flex text-xs leading-5 font-bold rounded-full ${statusStyles[deliveryBoy.status] || 'bg-gray-100 text-gray-800'}`}
            >
              {deliveryBoy.status.charAt(0).toUpperCase() + deliveryBoy.status.slice(1).replace('_', ' ')}
            </span>
            <button
              type="button"
              onClick={() => (deliveryBoy.id || deliveryBoy._id) && toggleAvailability((deliveryBoy.id || deliveryBoy._id) as string, deliveryBoy.isAvailable !== false)}
              disabled={isUpdating}
              className={`px-3 py-1 text-xs font-bold rounded-full transition-all flex items-center space-x-1 ${deliveryBoy.isAvailable ? 'bg-green-100 text-green-800 hover:bg-green-200' : 'bg-gray-200 text-gray-700 hover:bg-gray-300'} ${isUpdating ? 'opacity-70 cursor-not-allowed' : ''}`}
              title={deliveryBoy.isAvailable ? 'Mark as unavailable' : 'Mark as available'}
            >
              {isUpdating ? (
                <span className="inline-block w-3 h-3 border-2 border-current border-t-transparent rounded-full animate-spin mr-1"></span>
              ) : (
                <span className="w-2 h-2 rounded-full inline-block mr-1" style={{ backgroundColor: deliveryBoy.isAvailable ? '#10B981' : '#6B7280' }}></span>
              )}
              <span>{deliveryBoy.isAvailable ? 'Available' : 'Unavailable'}</span>
            </button>
          </div>
        </td>
        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-600">
          <div className="flex items-center">
            <BriefcaseIcon className="h-4 w-4 mr-2 text-gray-400" />
            <div>
              <div>
                <span className="font-semibold">Success:</span> {deliveryBoy.completedDeliveries}/{deliveryBoy.totalDeliveries}
              </div>
              <div className="text-xs">Cancelled: {deliveryBoy.cancelledDeliveries}</div>
            </div>
          </div>
          {deliveryBoy.rating !== undefined && (
            <div className="flex items-center mt-1 text-xs">
              <span className="font-semibold mr-1">Rating:</span> {deliveryBoy.rating.toFixed(1)}/5.0
            </div>
          )}
        </td>
        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
          <div className="flex items-center justify-end space-x-1">
            <button
              onClick={() => {
                setViewingBoy(deliveryBoy);
                setShowViewModal(true);
              }}
              className="text-gray-500 hover:text-gray-700 p-2 rounded-full hover:bg-gray-100 transition-colors"
              title="View Details"
            >
              <Eye className="h-5 w-5" />
            </button>
            <button
              onClick={() => handleEdit(deliveryBoy)}
              className="text-blue-600 hover:text-blue-800 p-2 rounded-full hover:bg-blue-50 transition-colors"
              title="Edit Delivery Boy"
            >
              <Edit className="h-5 w-5" />
            </button>
            <button
              onClick={() => handleDelete((deliveryBoy.id || deliveryBoy._id)!)}
              className="text-red-600 hover:text-red-800 p-2 rounded-full hover:bg-red-50 transition-colors"
              title="Delete Delivery Boy"
            >
              <Trash2 className="h-5 w-5" />
            </button>
          </div>
        </td>
      </tr>
    );
  };

  const renderMapSection = () => (
    <div className="mt-4">
      <label className="block text-sm font-medium text-gray-700 mb-1">Delivery Location</label>
      <div style={{ height: '300px' }} className="rounded-md overflow-hidden relative border border-gray-300 bg-gray-50">
        {!mapLoaded ? (
          <div className="h-full flex items-center justify-center">
            <Loader2 className="w-8 h-8 animate-spin text-blue-600" />
            <span className="ml-2">Loading map...</span>
          </div>
        ) : mapError ? (
          <div className="h-full flex flex-col items-center justify-center text-red-600 p-4 text-center">
            <AlertCircle className="w-8 h-8 mb-2" />
            <p>{mapError}</p>
            <button
              onClick={() => window.location.reload()}
              className="mt-2 px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm"
            >
              Reload
            </button>
          </div>
        ) : (
          <>
            <div 
              ref={mapRef} 
              className="w-full h-full"
              style={{ minHeight: '256px' }}
            ></div>
            <div className="absolute top-2 left-2 bg-white/90 px-2 py-1 rounded shadow text-xs text-gray-700 z-10">
              Click on the map to set the delivery location
            </div>
          </>
        )}
      </div>
      <p className="mt-1 text-xs text-gray-500">
        Click on the map to set the delivery location. Drag the marker to adjust the position.
      </p>

      <div className="grid grid-cols-2 gap-4 mt-4">
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">Latitude</label>
          <input
            type="text"
            value={location.latitude}
            onChange={(e) => setLocation((prev) => ({ ...prev, latitude: e.target.value }))}
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            placeholder="e.g. 22.5726"
          />
        </div>
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-1">Longitude</label>
          <input
            type="text"
            value={location.longitude}
            onChange={(e) => setLocation((prev) => ({ ...prev, longitude: e.target.value }))}
            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            placeholder="e.g. 88.3639"
          />
        </div>
      </div>
    </div>
  );

  const renderModal = () => (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
      <div className="bg-white rounded-lg p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <div className="flex justify-between items-center mb-4">
          <h2 className="text-2xl font-bold text-gray-800">
            {editing ? 'Edit Delivery Boy' : 'Add New Delivery Boy'}
          </h2>
          <button
            onClick={() => {
              setShowModal(false);
              setEditing(null);
              setForm({ 
                name: '', 
                email: '', 
                phone: '', 
                password: '',
                store: '',
                status: 'active',
                isAvailable: true 
              });
              setLocation({ longitude: '', latitude: '' });
            }}
            className="text-gray-500 hover:text-gray-700"
          >
            ✕
          </button>
        </div>
        
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Personal Information */}
            <div className="space-y-4">
              <h3 className="text-lg font-medium text-gray-900 border-b pb-2">
                <User className="inline-block mr-2 h-5 w-5" />
                Personal Information
              </h3>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Full Name *</label>
                <div className="relative">
                  <input
                    type="text"
                    value={form.name || ''}
                    onChange={(e) => setForm({ ...form, name: e.target.value })}
                    className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                    placeholder="John Doe"
                    required
                  />
                  <User className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Email *</label>
                <div className="relative">
                  <input
                    type="email"
                    value={form.email || ''}
                    onChange={(e) => setForm({ ...form, email: e.target.value })}
                    className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                    placeholder="john@example.com"
                    required
                    disabled={!!editing}
                  />
                  <Mail className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Phone *</label>
                <div className="relative">
                  <input
                    type="tel"
                    value={form.phone || ''}
                    onChange={(e) => setForm({ ...form, phone: e.target.value })}
                    className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                    placeholder="+1234567890"
                    required
                  />
                  <Phone className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              {!editing && (
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Password *</label>
                  <div className="relative">
                    <input
                      type="password"
                      value={form.password || ''}
                      onChange={(e) => setForm({ ...form, password: e.target.value })}
                      className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                      placeholder="••••••••"
                      required={!editing}
                      minLength={6}
                    />
                    <Lock className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                  </div>
                </div>
              )}
            </div>
            
            {/* Work Information */}
            <div className="space-y-4">
              <h3 className="text-lg font-medium text-gray-900 border-b pb-2">
                <BriefcaseIcon className="inline-block mr-2 h-5 w-5" />
                Work Information
              </h3>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Store (Optional)</label>
                <div className="relative">
                  <select
                    value={form.store || ''}
                    onChange={(e) => setForm({ ...form, store: e.target.value })}
                    className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                  >
                    <option value="">Select a store</option>
                    {stores.map((store) => (
                      <option key={`store-${store._id}`} value={store._id}>
                        {store.name}
                      </option>
                    ))}
                  </select>
                  <Store className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Status *</label>
                <div className="relative">
                  <select
                    value={form.status || 'pending'}
                    onChange={(e) => setForm({ ...form, status: e.target.value as 'pending' | 'approved' | 'rejected' })}
                    className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                  >
                    <option value="pending">Pending</option>
                    <option value="approved">Approved</option>
                    <option value="rejected">Rejected</option>
                  </select>
                  <Clock className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                </div>
              </div>
              
              <div className="flex items-center">
                <div className="relative flex items-start">
                  <div className="flex items-center h-5">
                    <input
                      id="isAvailable"
                      name="isAvailable"
                      type="checkbox"
                      checked={form.isAvailable !== false}
                      onChange={(e) => setForm({ ...form, isAvailable: e.target.checked })}
                      className="focus:ring-blue-500 h-4 w-4 text-blue-600 border-gray-300 rounded"
                    />
                  </div>
                  <div className="ml-3 text-sm">
                    <label htmlFor="isAvailable" className="font-medium text-gray-700">
                      Currently Available for Deliveries
                    </label>
                    <p className="text-gray-500">This will make the delivery boy visible for new orders</p>
                  </div>
                </div>
              </div>
              
              <div className="space-y-2">
                <label className="block text-sm font-medium text-gray-700">Location (Optional)</label>
                <div className="grid grid-cols-2 gap-3">
                  <div className="relative">
                    <input
                      type="number"
                      step="0.000001"
                      value={location.longitude}
                      onChange={(e) => setLocation({ ...location, longitude: e.target.value })}
                      className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                      placeholder="Longitude"
                    />
                    <MapPin className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                  </div>
                  <div className="relative">
                    <input
                      type="number"
                      step="0.000001"
                      value={location.latitude}
                      onChange={(e) => setLocation({ ...location, latitude: e.target.value })}
                      className="block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 pl-10"
                      placeholder="Latitude"
                    />
                    <MapPin className="absolute left-3 top-2.5 h-5 w-5 text-gray-400" />
                  </div>
                </div>
                <p className="text-xs text-gray-500">
                  Enter coordinates for better delivery assignment (e.g., 77.1025, 28.7041)
                </p>
              </div>
            </div>
          </div>
          
          <div className="flex justify-end space-x-3 pt-6 border-t mt-6">
            <button
              type="button"
              onClick={() => {
                setShowModal(false);
                setEditing(null);
                setForm({ 
                  name: '', 
                  email: '', 
                  phone: '', 
                  password: '',
                  store: '',
                  status: 'active',
                  isAvailable: true 
                });
                setLocation({ longitude: '', latitude: '' });
              }}
              className="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              Cancel
            </button>
            <button
              type="submit"
              className="inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              {editing ? (
                <>
                  <Save className="mr-2 h-4 w-4" />
                  Update Delivery Boy
                </>
              ) : (
                <>
                  <Plus className="mr-2 h-4 w-4" />
                  Create Delivery Boy
                </>
              )}
            </button>
          </div>
        </form>
      </div>
    </div>
  );

  const renderViewModal = (boy: DeliveryBoy, stores: Array<{_id: string, name: string}>) => {
    const DetailItem = ({ icon, label, value }: { icon: React.ReactNode, label: string, value: React.ReactNode }) => (
      <div className="flex items-start py-3 px-2 rounded-md hover:bg-gray-50">
        <div className="text-gray-500 mr-3 mt-1">{icon}</div>
        <div>
          <p className="text-xs font-medium text-gray-500">{label}</p>
          <p className="text-sm font-semibold text-gray-800">{value}</p>
        </div>
      </div>
    );

    return (
      <div className="fixed inset-0 bg-black bg-opacity-60 flex items-center justify-center z-50 p-4">
        <div className="bg-white rounded-xl shadow-2xl p-6 w-full max-w-3xl max-h-[95vh] overflow-y-auto relative animate-fade-in-up">
          <button
            onClick={() => setShowViewModal(false)}
            className="absolute top-4 right-4 text-gray-400 hover:text-gray-600 hover:bg-gray-100 rounded-full p-2 transition-colors"
          >
            <X className="h-6 w-6" />
          </button>

          <div className="flex items-center mb-6">
            <div className="flex-shrink-0 h-16 w-16 rounded-full bg-blue-100 flex items-center justify-center ring-4 ring-blue-200">
              <User className="h-9 w-9 text-blue-600" />
            </div>
            <div className="ml-5">
              <h2 className="text-3xl font-bold text-gray-900">{boy.name}</h2>
              <p className="text-sm text-gray-500">Delivery Partner</p>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {/* Left Column */}
            <div className="md:col-span-1 space-y-2">
              <DetailItem icon={<Mail className="h-5 w-5" />} label="Email" value={boy.email} />
              <DetailItem icon={<Phone className="h-5 w-5" />} label="Phone" value={boy.phone} />
              <DetailItem icon={<Store className="h-5 w-5" />} label="Store" value={stores.find((s: { _id: string; name: string }) => s._id === boy.store)?.name || 'N/A'} />
              {boy.location && (
                <DetailItem 
                  icon={<MapPin className="h-5 w-5" />} 
                  label="Location" 
                  value={`${boy.location.coordinates[0].toFixed(4)}, ${boy.location.coordinates[1].toFixed(4)}`}
                />
              )}
            </div>

            {/* Middle Column */}
            <div className="md:col-span-1 space-y-2">
              <DetailItem 
                icon={<Clock className="h-5 w-5" />} 
                label="Status" 
                value={<span className={`px-2 py-1 text-xs font-bold rounded-full`}>{boy.status}</span>}
              />
              <DetailItem 
                icon={boy.isAvailable ? <CheckCircle className="h-5 w-5 text-green-500" /> : <XCircle className="h-5 w-5 text-red-500" />}
                label="Availability"
                value={boy.isAvailable ? 'Available' : 'Unavailable'}
              />
              <DetailItem 
                icon={boy.isVerified ? <CheckCircle className="h-5 w-5 text-green-500" /> : <XCircle className="h-5 w-5 text-red-500" />}
                label="Verification"
                value={boy.isVerified ? 'Verified' : 'Not Verified'}
              />
              {boy.lastOnline && (
                <DetailItem 
                  icon={<Clock className="h-5 w-5" />} 
                  label="Last Online"
                  value={new Date(boy.lastOnline).toLocaleString()}
                />
              )}
            </div>

            {/* Right Column */}
            <div className="md:col-span-1 space-y-2">
              <DetailItem icon={<Star className="h-5 w-5" />} label="Rating" value={`${boy.rating?.toFixed(1) || 'N/A'} / 5.0`} />
              <DetailItem icon={<BriefcaseIcon className="h-5 w-5" />} label="Total Deliveries" value={boy.totalDeliveries} />
              <DetailItem icon={<CheckCircle className="h-5 w-5" />} label="Completed" value={boy.completedDeliveries} />
              <DetailItem icon={<XCircle className="h-5 w-5" />} label="Cancelled" value={boy.cancelledDeliveries} />
            </div>
          </div>

          {boy.documents && boy.documents.length > 0 && (
            <div className="mt-8">
              <h3 className="text-lg font-bold text-gray-800 mb-3 border-b pb-2">Documents</h3>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                {boy.documents.map((doc, index) => (
                  <a href={doc.url} target="_blank" rel="noopener noreferrer" key={index} className="block p-3 bg-gray-50 rounded-lg hover:bg-gray-100 transition-colors">
                    <div className="flex items-center">
                      <FileText className="h-6 w-6 text-gray-500" />
                      <div className="ml-3">
                        <p className="text-sm font-semibold capitalize">{doc.type}</p>
                        <p className={`text-xs font-bold ${doc.verified ? 'text-green-600' : 'text-yellow-600'}`}>
                          {doc.verified ? 'Verified' : 'Pending'}
                        </p>
                      </div>
                    </div>
                  </a>
                ))}
              </div>
            </div>
          )}

        </div>
      </div>
    );
  };

  return (
    <Layout title="Delivery Boys" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Search + Add button */}
        <div className="flex items-center justify-between">
          <div className="relative w-full max-w-sm">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && fetchDeliveryBoys()}
              placeholder="Search delivery boys..."
              className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 w-full"
            />
          </div>
          <button
            onClick={() => {
              setEditing(null);
              setForm({ name: '', email: '', phone: '', status: 'active', password: '' });
              setShowModal(true);
            }}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors ml-4"
          >
            <Plus className="w-4 h-4 mr-2" /> Add Delivery Boy
          </button>
        </div>

        {/* Table / Empty / Loading */}
        {loading ? (
          <div className="py-12 text-center">Loading...</div>
        ) : !hasRows ? (
          <div className="text-center py-12">
            <Users className="mx-auto h-12 w-12 text-gray-400" />
            <div className="text-gray-600">No delivery boys found</div>
          </div>
        ) : (
          <div className="bg-white border rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Contact</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Deliveries</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {deliveryBoys.map((boy) => (
                    <React.Fragment key={boy.id || boy._id}>
                      {renderDeliveryBoyRow(boy)}
                    </React.Fragment>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
        {/* Modals */}
        {showModal && renderModal()}
        {showViewModal && viewingBoy && renderViewModal(viewingBoy, stores)}
      </div>
  </Layout>
  );
  };
  
  export default DeliveryBoysList;
