import React from 'react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import StatsCard from '../../components/shared/StatsCard';
import { pharmacistSidebarItems } from './PharmacistLayout';
import { 
  ClipboardList, 
  Pill, 
  Calendar, 
  MessageSquare, 
  AlertCircle,
  Activity,
  FileText,
  CheckCircle,
  User,
  Clock,
  TrendingUp,
  Users as UsersIcon
} from 'lucide-react';

interface ActivityType {
  id: number;
  type: string;
  patient: string;
  time: string;
  status: string;
}

interface PerformanceMetric {
  metric: string;
  value: string;
  category: string;
}

interface StatType {
  title: string;
  value: string;
  icon: any;
  trend: { value: number; isPositive: boolean };
  color: 'blue' | 'green' | 'purple' | 'red' | 'orange';
}

const Dashboard: React.FC = () => {
  const stats: StatType[] = [
    {
      title: 'Pending Prescriptions',
      value: '12',
      icon: ClipboardList,
      trend: { value: 5, isPositive: false },
      color: 'blue'
    },
    {
      title: 'Medications to Verify',
      value: '8',
      icon: Pill,
      trend: { value: 2, isPositive: true },
      color: 'green'
    },
    {
      title: 'Scheduled Consultations',
      value: '5',
      icon: Calendar,
      trend: { value: 1, isPositive: false },
      color: 'purple'
    },
    {
      title: 'Patient Messages',
      value: '3',
      icon: MessageSquare,
      trend: { value: 0, isPositive: true },
      color: 'orange'
    }
  ];

  const recentActivities: ActivityType[] = [
    { id: 1, type: 'prescription', patient: 'John Doe', time: '10:30 AM', status: 'verified' },
    { id: 2, type: 'consultation', patient: 'Jane Smith', time: '11:45 AM', status: 'completed' },
    { id: 3, type: 'medication', patient: 'Robert Johnson', time: '01:15 PM', status: 'pending' },
    { id: 4, type: 'message', patient: 'Emily Davis', time: '02:30 PM', status: 'replied' },
  ];

  const performanceMetrics: PerformanceMetric[] = [
    { metric: 'Prescription Accuracy', value: '98%', category: 'Quality' },
    { metric: 'Patient Satisfaction', value: '4.8/5', category: 'Rating' },
    { metric: 'Average Response Time', value: '15 min', category: 'Efficiency' },
    { metric: 'Medication Reviews', value: '47', category: 'This Week' }
  ];

  const getStatusBadge = (status: string) => {
    const statusColors: Record<string, string> = {
      verified: 'bg-green-100 text-green-800',
      completed: 'bg-blue-100 text-blue-800',
      pending: 'bg-yellow-100 text-yellow-800',
      replied: 'bg-purple-100 text-purple-800',
      default: 'bg-gray-100 text-gray-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status] || statusColors.default}`;
  };

  const getActivityIcon = (type: string) => {
    const icons: Record<string, JSX.Element> = {
      prescription: <FileText className="h-5 w-5 text-blue-500" />,
      consultation: <User className="h-5 w-5 text-purple-500" />,
      medication: <Pill className="h-5 w-5 text-green-500" />,
      message: <MessageSquare className="h-5 w-5 text-yellow-500" />
    };
    return icons[type] || <Activity className="h-5 w-5 text-gray-500" />;
  };

  return (
    <Layout title="Pharmacist Dashboard" sidebar={<Sidebar items={pharmacistSidebarItems} />}>
      <div className="space-y-6">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <StatsCard
              key={index}
              title={stat.title}
              value={stat.value}
              icon={stat.icon}
              trend={stat.trend}
              color={stat.color}
            />
          ))}
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Activities */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Recent Activities</h3>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Activity
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Patient
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Time
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {recentActivities.map((activity) => (
                    <tr key={activity.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <div className="flex-shrink-0">
                            {getActivityIcon(activity.type)}
                          </div>
                          <div className="ml-2">
                            <div className="text-sm font-medium text-gray-900 capitalize">{activity.type}</div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {activity.patient}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getStatusBadge(activity.status)}>
                          {activity.status}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {activity.time}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {/* Performance Metrics */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Performance Metrics</h3>
            </div>
            <div className="p-6">
              <div className="space-y-4">
                {performanceMetrics.map((metric, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-blue-50 border border-blue-200 rounded-lg">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-gray-900">{metric.metric}</p>
                      <p className="text-xs text-gray-500">{metric.category}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-blue-600">
                        {metric.value}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
              <button className="w-full mt-4 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                View Full Analytics
              </button>
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Quick Actions</h3>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {[
                { label: 'Process Prescription', icon: ClipboardList, count: '12 pending' },
                { label: 'Verify Medication', icon: CheckCircle, count: '8 to review' },
                { label: 'Schedule Consultation', icon: Calendar, count: '5 upcoming' },
                { label: 'View Alerts', icon: AlertCircle, count: '3 new' }
              ].map((action, index) => {
                const Icon = action.icon;
                return (
                  <button
                    key={index}
                    className="p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-all text-left"
                  >
                    <div className="flex items-center mb-2">
                      <Icon className="w-5 h-5 text-blue-600 mr-2" />
                      <span className="text-sm font-medium text-gray-900">{action.label}</span>
                    </div>
                    <p className="text-xs text-gray-500">{action.count}</p>
                  </button>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </Layout>
  );
};

export default Dashboard;