import React, { useEffect, useState, useCallback } from 'react';
import { toast } from 'react-hot-toast';
import Header from '../../components/HomepageComponent/Header';
import Hero from '../../components/HomepageComponent/Hero';
import CategorySection from '../../components/HomepageComponent/CategorySection';
import api from '../../services/api';
import ProductSection from '../../components/HomepageComponent/ProductSection';
import AppDownload from '../../components/HomepageComponent/AppDownload';
import WhyChoose from '../../components/HomepageComponent/WhyChoose';
import Footer from '../../components/HomepageComponent/Footer';
import { 
  homepageSectionService, 
  categoryService, 
  itemService 
} from '../../services/api';

// Helper function to calculate distance between two coordinates (Haversine formula)
const calculateDistance = (lat1: number, lon1: number, lat2: number, lon2: number): number => {
  const R = 6371e3; // Earth's radius in meters
  const φ1 = (lat1 * Math.PI) / 180;
  const φ2 = (lat2 * Math.PI) / 180;
  const Δφ = ((lat2 - lat1) * Math.PI) / 180;
  const Δλ = ((lon2 - lon1) * Math.PI) / 180;

  const a =
    Math.sin(Δφ / 2) * Math.sin(Δφ / 2) +
    Math.cos(φ1) * Math.cos(φ2) * Math.sin(Δλ / 2) * Math.sin(Δλ / 2);
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));

  return R * c; // Distance in meters
};

interface Image {
  url: string;
  is_primary?: boolean;
  _id?: string;
}

interface Product {
  _id: string;
  id?: string;
  name: string;
  title?: string;
  description?: string;
  price: number;
  sale_price?: number;
  sales_price?: number; // Some APIs might use sales_price instead of sale_price
  images?: (Image | Image[])[]; // Can be an array of Image objects or arrays of Image objects
  image?: string | Image | Image[];
  [key: string]: any;
}

// Helper function to get the first available image URL from a product
const getProductImageUrl = (product: Product, baseUrl: string = ''): string => {
  if (!product) return '/placeholder-product.png';
  
  // Helper to process a single image (string, Image object, or array of Images)
  const processImage = (img: string | Image | Image[] | undefined): string | null => {
    if (!img) return null;
    
    // Handle string URL
    if (typeof img === 'string') {
      return img.startsWith('http') || img.startsWith('/') 
        ? img 
        : `${baseUrl}/${img.replace(/^\/+/, '')}`;
    }
    
    // Handle array of images
    if (Array.isArray(img)) {
      const firstImg = img[0];
      if (typeof firstImg === 'string') {
        return firstImg.startsWith('http') || firstImg.startsWith('/')
          ? firstImg
          : `${baseUrl}/${firstImg.replace(/^\/+/, '')}`;
      }
      // If it's an array of Image objects, get the first one's URL
      return firstImg?.url || '';
    }
    
    // Handle single Image object
    return img.url || '';
  };
  
  // Check images array first
  if (Array.isArray(product.images) && product.images.length > 0) {
    const firstImage = product.images[0];
    const url = processImage(firstImage);
    if (url) return url;
  }
  
  // Then check the image property
  if (product.image) {
    const url = processImage(product.image);
    if (url) return url;
  }
  
  // Check for other possible image fields
  const possibleImageFields = ['thumbnail', 'coverImage', 'imageUrl', 'imagePath'];
  for (const field of possibleImageFields) {
    if (product[field]) {
      const url = processImage(product[field]);
      if (url) return url;
    }
  }
  
  return '/placeholder-product.png';
};

interface Category {
  _id: string;
  name: string;
  image?: string;
  slug?: string;
}

interface HomepageSection {
  _id: string;
  title: string;
  subtitle?: string;
  productType: string;
  products: Product[];
  sortOrder: number;
  status: 'active' | 'inactive';
}

const Home: React.FC = () => {
  const [sections, setSections] = useState<HomepageSection[]>([]);
  const [categories, setCategories] = useState<Category[]>([]);
  const [loading, setLoading] = useState(true);
  const [categoriesLoading, setCategoriesLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedStore, setSelectedStore] = useState<any>(() => {
    // Initialize from localStorage if available
    try {
      const savedStore = localStorage.getItem('selectedStore');
      return savedStore ? JSON.parse(savedStore) : null;
    } catch (error) {
      console.error('Error parsing saved store:', error);
      return null;
    }
  });
  const [userLocation, setUserLocation] = useState<{lat: number; lng: number} | null>(() => {
    // Initialize from localStorage if available
    try {
      const savedLocation = localStorage.getItem('userLocation');
      return savedLocation ? JSON.parse(savedLocation) : null;
    } catch (error) {
      console.error('Error parsing saved location:', error);
      return null;
    }
  });

  // Update localStorage when selectedStore or userLocation changes
  useEffect(() => {
    if (selectedStore) {
      try {
        localStorage.setItem('selectedStore', JSON.stringify(selectedStore));
      } catch (error) {
        console.error('Error saving selected store to localStorage:', error);
      }
    }
  }, [selectedStore]);

  useEffect(() => {
    if (userLocation) {
      try {
        localStorage.setItem('userLocation', JSON.stringify(userLocation));
      } catch (error) {
        console.error('Error saving user location to localStorage:', error);
      }
    }
  }, [userLocation]);

  // Listen for storage events to update when location changes in other tabs/windows
  useEffect(() => {
    const handleStorageChange = (e: StorageEvent) => {
      if (e.key === 'selectedStore') {
        try {
          if (e.newValue) {
            setSelectedStore(JSON.parse(e.newValue));
          } else {
            setSelectedStore(null);
          }
        } catch (error) {
          console.error('Error parsing updated store:', error);
        }
      } else if (e.key === 'userLocation') {
        try {
          if (e.newValue) {
            setUserLocation(JSON.parse(e.newValue));
          } else {
            setUserLocation(null);
          }
        } catch (error) {
          console.error('Error parsing updated location:', error);
        }
      }
    };

    window.addEventListener('storage', handleStorageChange);
    return () => window.removeEventListener('storage', handleStorageChange);
  }, []);

  // Fetch complete product data including images and filter by location
  const fetchProductDetails = async (productIds: string[]): Promise<Product[]> => {
    try {
      if (!productIds.length) return [];
      
      // Fetch products with their store information and location data
      const idsParam = productIds.join(',');
      console.log('Fetching products with store data for IDs:', productIds);
      
      // Ensure we're properly populating the store data
      const response = await itemService.getItems(1, productIds.length, `ids=${idsParam}&populate=store`);
      
      // Log the raw response to debug
      console.log('Products API response:', response);
      
      // Handle both response formats: response.data.data and response.data
      let products = [];
      if (response?.data && Array.isArray(response.data)) {
        products = response.data;
      } else if (response?.data?.data && Array.isArray(response.data.data)) {
        products = response.data.data;
      } else if (Array.isArray(response)) {
        products = response;
      }
      
      console.log('Processed products with store data:', products);
      
      // If we don't have a selected store or user location, show all products
      if (!selectedStore || !userLocation) {
        console.log('No store selected or location not available, showing all products');
        return products;
      }
      
      console.log('Filtering products for store:', selectedStore.name);
      console.log('User location:', userLocation);
      
      // Get the store's location if available
      const storeLocation = selectedStore.store?.location || selectedStore.location;
      
      // If we don't have store location, just show all products
      if (!storeLocation?.coordinates) {
        console.warn('Selected store has no location data, showing all products');
        return products;
      }
      
      const [storeLng, storeLat] = Array.isArray(storeLocation.coordinates) 
        ? storeLocation.coordinates 
        : [storeLocation.coordinates.lng, storeLocation.coordinates.lat];
      
      const deliveryRadius = selectedStore.deliveryRadius || 5000; // Default to 5km
      
      console.log(`Filtering products within ${deliveryRadius}m of store location (${storeLat}, ${storeLng})`);
      
      // Filter products based on store's location and delivery radius
      return products.filter(product => {
        // If product has no store data, include it
        if (!product.store) {
          console.log('Product has no store data, including:', product.name);
          return true;
        }
        
        // Get store location from either store.location or directly from store
        const productStoreLocation = product.store.location || product.store;
        
        // If product has no location data, include it
        if (!productStoreLocation?.coordinates) {
          console.log('Product has no location data, including:', product.name);
          return true;
        }
        
        try {
          let productLng, productLat;
          
          // Log the store location for debugging
          console.log('Store location data:', {
            storeId: product.store._id,
            storeName: product.store.name || product.store.storeName,
            coordinates: storeLocation.coordinates,
            coordinatesType: Array.isArray(storeLocation.coordinates) ? 'array' : typeof storeLocation.coordinates
          });
          
          // Handle different coordinate formats
          if (Array.isArray(storeLocation.coordinates)) {
            // MongoDB GeoJSON format: [longitude, latitude]
            [productLng, productLat] = storeLocation.coordinates;
          } else if (storeLocation.coordinates.lng !== undefined && storeLocation.coordinates.lat !== undefined) {
            // Object format: { lng: number, lat: number }
            productLng = storeLocation.coordinates.lng;
            productLat = storeLocation.coordinates.lat;
          } else if (storeLocation.coordinates[0] !== undefined && storeLocation.coordinates[1] !== undefined) {
            // Array-like object: { 0: lng, 1: lat }
            productLng = storeLocation.coordinates[0];
            productLat = storeLocation.coordinates[1];
          } else {
            console.log('Unsupported coordinate format for product:', product.name, storeLocation.coordinates);
            return false;
          }
          
          // If we couldn't get valid coordinates, exclude the product
          if (productLat === undefined || productLng === undefined) {
            console.log('Product has invalid coordinates, excluding:', product.name);
            return false;
          }
          
          const distance = calculateDistance(
            storeLat,
            storeLng,
            productLat,
            productLng
          );
          
          const isWithinRange = distance <= deliveryRadius;
          if (!isWithinRange) {
            console.log(`Product '${product.name}' is ${Math.round(distance)}m away (beyond ${deliveryRadius}m radius)`);
          }
          return isWithinRange;
        } catch (error) {
          console.error('Error calculating distance for product:', product._id, error);
          return false; // Exclude product if there's an error
        }
      });
    } catch (error) {
      console.error('Error fetching product details:', error);
      return [];
    }
  };

  // Track if we've already loaded data for the current store
  const [lastFetchedStore, setLastFetchedStore] = useState<string | null>(null);

  // Fetch homepage sections with complete product data
  useEffect(() => {
    const fetchSections = async () => {
      // Skip if we don't have a selected store yet or already fetched for this store
      if (!selectedStore) {
        console.log('No store selected, skipping section fetch');
        return;
      }
      
      // Skip if we've already fetched for this store
      if (lastFetchedStore === selectedStore._id) {
        console.log('Already fetched sections for store:', selectedStore.name);
        return;
      }

      try {
        setLoading(true);
        console.log('Fetching homepage sections for store:', selectedStore.name);
        
        // Build query parameters for the request
        const queryParams: Record<string, string> = {
          storeId: selectedStore._id
        };
        const headers: Record<string, string> = {};
        
        // Use the current user location or fallback to localStorage
        let locationToUse = userLocation;
        if (!locationToUse) {
          try {
            const savedLocation = localStorage.getItem('userLocation');
            if (savedLocation) {
              locationToUse = JSON.parse(savedLocation);
              console.log('Using location from localStorage:', locationToUse);
            }
          } catch (e) {
            console.error('Error parsing location from localStorage:', e);
          }
        }
        
        // Always include storeId if available
        if (selectedStore?._id) {
          queryParams.storeId = selectedStore._id;
          console.log('Using store ID for filtering:', selectedStore._id);
        }
        
        // Include location in query params only (no headers to avoid CORS preflight)
        if (locationToUse) {
          queryParams.lat = locationToUse.lat.toString();
          queryParams.lng = locationToUse.lng.toString();
          
          console.log('Using location for filtering:', locationToUse);
        }
        
        // Convert query params to string
        const queryString = new URLSearchParams(queryParams).toString();
        
        // Make the API request with query parameters
        console.log('Fetching sections with query params:', queryString);
        
        // Ensure we have either a store ID or location for filtering
        if (!queryParams.storeId && !queryParams.lat) {
          console.warn('No store ID or location provided for filtering. Using default settings.');
        }
        
        // Make the API request with query parameters and headers
        const response = await api.get(
          `/homepage-sections/display${queryString ? `?${queryString}` : ''}`,
          { headers }
        );
        
        console.log('API Response:', response);
        
        console.log('Sections response:', response);
        
        // The response should have a data property with the sections array
        let sectionsData: HomepageSection[] = [];
        
        if (response?.data?.data && Array.isArray(response.data.data)) {
          // If response has data.data array, use that (this matches the backend response structure)
          sectionsData = response.data.data;
        } else if (Array.isArray(response?.data)) {
          // Fallback: if data is directly an array
          sectionsData = response.data;
        } else if (Array.isArray(response)) {
          // Fallback: if response is directly an array
          sectionsData = response;
        }
        
        console.log('Raw sections data from API:', response?.data);
        
        console.log('Processed sections data:', sectionsData);
        
        if (sectionsData.length === 0) {
          console.warn('No sections found in the response');
          setSections([]);
          setLoading(false);
          return;
        }
        
        // Collect all product IDs from all sections
        const allProductIds: string[] = [];
        const productIdToSectionMap: Record<string, string[]> = {};
        
        sectionsData.forEach((section: HomepageSection) => {
          if (section.products?.length) {
            section.products.forEach((product: Product) => {
              const productId = product._id || product.id;
              if (productId) {
                allProductIds.push(productId);
                if (!productIdToSectionMap[productId]) {
                  productIdToSectionMap[productId] = [];
                }
                productIdToSectionMap[productId].push(section._id || section.title || '');
              }
            });
          }
        });
        
        // Fetch complete product data for all products
        const productsData = await fetchProductDetails(allProductIds);
        
        // Create a map of product ID to complete product data
        const productMap = productsData.reduce<Record<string, Product>>((acc: Record<string, Product>, product: Product) => {
          const id = product._id || product.id;
          if (id) {
            acc[id] = product;
          }
          return acc;
        }, {});
        
        // Update sections with complete product data
        const updatedSections = sectionsData
          .map((section: HomepageSection) => {
            if (!section.products || section.products.length === 0) {
              console.log(`Section '${section.title}' has no products`);
              return null;
            }
            
            // Log the raw products for debugging
            console.log(`Section '${section.title}' raw products:`, section.products);
            
            // Handle both object and string product references
            const productIds = section.products.map((p: any) => {
              if (!p) {
                console.log('Found null/undefined product reference');
                return null;
              }
              if (typeof p === 'string') return p;
              if (p._id) return p._id;
              if (p.$oid) return p.$oid;
              console.log('Product reference has no recognizable ID:', p);
              return null;
            }).filter((id): id is string => {
              if (!id) return false;
              if (!productMap[id]) {
                console.log(`Product with ID ${id} not found in productMap`);
                return false;
              }
              return true;
            });
            
            console.log(`Section '${section.title}' product IDs (${productIds.length}):`, productIds);
            
            // Get products that exist in our product map and belong to the selected store
            const filteredProducts = productIds
              .map(productId => {
                const product = productMap[productId];
                if (!product) {
                  console.warn(`Product with ID ${productId} not found in productMap`);
                  return null;
                }
                return product;
              })
              .filter((product): product is Product => {
                if (!product) return false;
                
                // Debug product store info
                const storeId = product.store_id?._id || product.store?._id || product.storeId || 
                              (typeof product.store_id === 'string' ? product.store_id : null);
                
                // Log detailed store information
                console.log(`Product ${product.name} (${product._id}) store info:`, {
                  productStoreId: storeId,
                  selectedStoreId: selectedStore?._id,
                  store: product.store || product.store_id,
                  hasStore: !!product.store || !!product.store_id,
                  store_id: product.store_id,
                  store_id_type: typeof product.store_id,
                  store_id_value: product.store_id,
                  store_type: typeof product.store,
                  store_value: product.store,
                  storeId_value: product.storeId
                });
                
                // If we have a selected store, check if the product belongs to it
                if (selectedStore?._id) {
                  // If product has no store information, include it
                  if (!product.store && !product.store_id && !product.storeId) {
                    console.log(`Product ${product.name} (${product._id}) has no store info, including`);
                    return true;
                  }
                  
                  // Check all possible store ID locations
                  const productStoreId = product.store_id?._id || 
                                      product.store?._id || 
                                      product.storeId || 
                                      (typeof product.store_id === 'string' ? product.store_id : null);
                  
                  if (productStoreId === selectedStore._id) {
                    console.log(`Product ${product.name} (${product._id}) included - matches selected store ${selectedStore._id}`);
                    return true;
                  }
                  
                  // If we can't determine the store ID, include the product
                  if (!productStoreId) {
                    console.log(`Product ${product.name} (${product._id}) has no valid store ID, including by default`);
                    return true;
                  }
                  
                  // Only exclude if we're certain it's from a different store
                  console.log(`Product ${product.name} (${product._id}) excluded - from different store`);
                  return false;
                }
                
                // If no store is selected, include all products
                return true;
              });
            
            console.log(`Section '${section.title}' filtered products (${filteredProducts.length}):`, filteredProducts);
            
            if (filteredProducts.length === 0) {
              console.log(`No products remaining in section '${section.title}' after store filtering`);
              return null;
            }
            
            return {
              ...section,
              products: filteredProducts
            } as HomepageSection;
          })
          .filter((section): section is HomepageSection => section !== null); // Type guard to filter out nulls
        
        console.log('Updated sections with filtered product data:', updatedSections);
        setSections(updatedSections);
        
        if (selectedStore) {
          console.log(`Showing ${updatedSections.reduce((acc, section) => acc + section.products.length, 0)} products for store: ${selectedStore.name}`);
        }
        
      } catch (err: any) {
        console.error('Error fetching homepage sections:', err);
        setError('Failed to load homepage sections');
        toast.error('Failed to load homepage content');
      } finally {
        setLoading(false);
      }
    };

    // Fetch categories
    const fetchCategories = async () => {
      try {
        setCategoriesLoading(true);
        console.log('Fetching categories...');
        const response = await categoryService.getActiveCategories(10);
        console.log('Categories response:', response);
        
        // The response should have a data property containing the categories array
        const categoriesData = response.data || [];
        
        if (categoriesData.length === 0) {
          console.warn('No categories found in the response');
        } else {
          console.log(`Found ${categoriesData.length} categories`);
        }
        
        setCategories(categoriesData);
      } catch (err: any) {
        console.error('Error fetching categories:', err);
        toast.error('Failed to load categories');
      } finally {
        setCategoriesLoading(false);
      }
    };

    fetchSections();
    fetchCategories();
    
    // Refresh sections when selected store or user location changes
    const handleLocationChange = () => {
      console.log('Location or store changed, refreshing sections...');
      fetchSections();
    };
    
    // Add event listener for location changes
    window.addEventListener('locationChanged', handleLocationChange);
    
    return () => {
      window.removeEventListener('locationChanged', handleLocationChange);
    };
  }, [selectedStore, userLocation]);

  if (loading) {
    return (
      <div className="min-h-screen bg-white flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
      </div>
    );
  }
  
  // Render error state
  if (error) {
    return (
      <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
        <div className="text-red-500 text-lg mb-4">{error}</div>
        <button 
          onClick={() => window.location.reload()} 
          className="px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600"
        >
          Retry
        </button>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white">
      <Header />
      <Hero />
      <CategorySection 
        categories={categories} 
        loading={categoriesLoading}
        onCategoryClick={(categoryId) => {
          // Handle category click if needed
          console.log('Category clicked:', categoryId);
        }}
      />
      
      {/* Render dynamic product sections */}
      {sections.length > 0 ? (
        sections.map((section) => {
          // Check if products array exists and has items
          if (!section.products || section.products.length === 0) {
            console.log(`Skipping section ${section.title} - no products`);
            return null;
          }
          
          console.log(`Rendering section "${section.title}" with ${section.products.length} products`, section.products);
          
          // Transform products to ensure they have the expected structure
          const transformedProducts = section.products.map(product => {
            // Ensure we have a valid price (default to 0 if not provided)
            const price = Number(product.price) || 0;
            
            // Check for sale price in different possible fields
            const sales_price = product.sales_price || product.sale_price || product.discounted_price;
            
            // Create the base transformed product
            const transformed = {
              ...product,
              // Ensure we have an id field (some APIs use _id)
              id: product.id || product._id,
              // Map name to title if title is not present
              title: product.title || product.name,
              // Ensure we have an images array
              images: Array.isArray(product.images) 
                ? product.images 
                : product.image 
                  ? [typeof product.image === 'string' 
                      ? { url: product.image, is_primary: true }
                      : product.image]
                  : [],
              // Set price and sales_price
              price: price,
              // Only set sales_price if it's less than the original price
              sales_price: sales_price && sales_price < price ? Number(sales_price) : undefined,
              // Calculate discount percentage if there's a sale
              discount: sales_price && sales_price < price 
                ? Math.round(((price - sales_price) / price) * 100)
                : 0
            };
            
            return transformed;
          });
          
          console.log('Transformed products:', transformedProducts);
          
          return (
            <ProductSection
              key={section._id || section.id || section.title}
              title={section.title || 'Featured Products'}
              subtitle={section.description}
              products={transformedProducts}
            />
          );
        })
      ) : (
        <div className="py-12 text-center text-gray-500">
          No sections found. Please add some sections from the admin panel.
        </div>
      )}

      <AppDownload />
      <WhyChoose />
      <Footer />
    </div>
  );
};

export default Home;
