import { useState, useEffect } from 'react';

const BROWSE_HISTORY_KEY = 'browsing_history';
const MAX_HISTORY_ITEMS = 5;

interface BrowsedItem {
  id: string;
  name: string;
  price: number;
  image: string;
  viewedAt: number;
  slug?: string;
}

export const useBrowsingHistory = () => {
  const [browsingHistory, setBrowsingHistory] = useState<BrowsedItem[]>([]);

  // Load browsing history from localStorage on component mount
  useEffect(() => {
    const savedHistory = localStorage.getItem(BROWSE_HISTORY_KEY);
    if (savedHistory) {
      try {
        const parsedHistory = JSON.parse(savedHistory);
        setBrowsingHistory(Array.isArray(parsedHistory) ? parsedHistory : []);
      } catch (e) {
        console.error('Failed to parse browsing history', e);
      }
    }
  }, []);

  // Add current product to browsing history
  const addToBrowsingHistory = (product: Omit<BrowsedItem, 'viewedAt'>) => {
    if (!product.id) return;

    setBrowsingHistory(prevHistory => {
      // Filter out the current product if it's already in history
      const updatedHistory = [
        { ...product, viewedAt: Date.now() },
        ...prevHistory.filter(item => item.id !== product.id)
      ].slice(0, MAX_HISTORY_ITEMS);

      // Save to localStorage
      localStorage.setItem(BROWSE_HISTORY_KEY, JSON.stringify(updatedHistory));
      
      return updatedHistory;
    });
  };

  // Get recently viewed items (excluding current product if provided)
  const getRecentItems = (excludeId?: string) => {
    return browsingHistory
      .filter(item => item.id !== excludeId)
      .sort((a, b) => b.viewedAt - a.viewedAt)
      .slice(0, 4); // Show max 4 items
  };

  return {
    browsingHistory,
    addToBrowsingHistory,
    getRecentItems,
  };
};
