import React, { createContext, useContext, useReducer, ReactNode, useCallback } from 'react';
import { toast } from 'react-hot-toast';

type CartItem = {
  id: string;
  _id?: string;
  name: string;
  price: number;
  image?: string;
  quantity: number;
  originalPrice?: number;
  inStock?: boolean;
  brand?: string;
  category?: string;
  storeId?: string;
  storeName?: string;
  store?: string;
  discount?: number | { _id: string; name: string; type: 'percentage' | 'fixed'; value: number };
  unit?: string;
  taxAmount?: number;
  discountAmount?: number;
  variantId?: string | null;
  requiresPrescription?: boolean;
  // Add prescription data directly to cart item for easier access
  prescription?: PrescriptionData;
};

interface PrescriptionFile {
  _id?: string;
  name: string;
  path: string;
  url: string;
  type: string;
  size: number;
  is_primary?: boolean;
  uploaded_at?: string;
  mimetype?: string;
  originalName?: string;
  filename?: string;
  dataUrl?: string;
}

type PrescriptionData = {
  _id?: string;
  id?: string;
  doctor_name: string;
  patient_name: string;
  patient_age: number | string;
  patient_gender: string;
  diagnosis?: string;
  notes?: string;
  prescription_date: string;
  prescriptionDate?: string;
  next_visit_date?: string | null;
  status?: string;
  is_verified?: boolean;
  verification_method?: string;
  reviewed_by?: string | null;
  reviewed_at?: string | null;
  rejection_reason?: string | null;
  files?: PrescriptionFile[];
  images?: PrescriptionFile[];
  fileUrls?: string[];
  verification_history?: Array<{
    status: string;
    changed_by: string;
    notes: string;
    changed_at: string | Date;
  }>;
  createdAt?: string;
  updatedAt?: string;
};

type CartState = {
  items: CartItem[];
  totalItems: number;
  totalPrice: number;
  requiresPrescription: boolean;
  prescription?: PrescriptionData;
  prescriptionUrl?: string;
  prescriptionItems: Array<{
    id: string;
    name: string;
    price: number;
    quantity: number;
    prescription: PrescriptionData;
  }>;
};

type CartAction =
  | { type: 'ADD_ITEM'; payload: Omit<CartItem, 'quantity'> }
  | { type: 'REMOVE_ITEM'; payload: { id: string } }
  | { type: 'UPDATE_QUANTITY'; payload: { id: string; quantity: number } }
  | { type: 'CLEAR_CART' }
  | { type: 'LOAD_CART'; payload: CartState }
  | { type: 'SET_PRESCRIPTION_URL'; payload: string | undefined }
  | { type: 'SET_PRESCRIPTION_DATA'; payload: PrescriptionData | undefined };

const CartContext = createContext<{
  state: CartState;
  addToCart: (item: Omit<CartItem, 'quantity'>) => void;
  removeFromCart: (id: string) => void;
  updateQuantity: (id: string, quantity: number) => void;
  clearCart: () => void;
  getItemQuantity: (id: string) => number;
  setPrescriptionUrl: (url: string | undefined) => void;
  setPrescriptionData: (data: PrescriptionData | undefined) => void;
} | undefined>(undefined);

// Function to save cart to localStorage
const saveCartToLocalStorage = (state: CartState) => {
  try {
    localStorage.setItem('cart', JSON.stringify(state));
  } catch (error) {
    console.error('Failed to save cart to localStorage', error);
  }
};

// Define default cart state
const defaultCartState: CartState = {
  items: [],
  totalItems: 0,
  totalPrice: 0,
  requiresPrescription: false,
  prescription: undefined,
  prescriptionItems: []
};

// Function to load cart from localStorage
const loadCartFromLocalStorage = (): CartState => {

  if (typeof window === 'undefined') {
    return defaultCartState;
  }
  
  try {
    const savedCart = localStorage.getItem('cart');
    if (savedCart) {
      const parsedCart = JSON.parse(savedCart);
      
      // Migrate from old format if needed
      if (!parsedCart.items) {
        return defaultCartState;
      }
      
      // Load prescription data if it exists
      let prescriptionData;
      try {
        const savedPrescription = localStorage.getItem('prescriptionData');
        if (savedPrescription) {
          prescriptionData = JSON.parse(savedPrescription);
        }
      } catch (e) {
        console.error('Failed to load prescription data from localStorage', e);
      }
      
      return {
        ...defaultCartState,
        items: parsedCart.items || [],
        totalItems: parsedCart.totalItems || 0,
        totalPrice: parsedCart.totalPrice || 0,
        requiresPrescription: parsedCart.requiresPrescription || false,
        prescription: parsedCart.prescription || prescriptionData,
        prescriptionUrl: parsedCart.prescriptionUrl,
        prescriptionItems: parsedCart.prescriptionItems || []
      };
    }

    return defaultCartState;
  } catch (error) {
    console.error('Error loading cart from localStorage', error);
    return defaultCartState;
  }
};

const cartReducer = (state: CartState, action: CartAction): CartState => {
  // Helper to check if cart requires prescription
  const checkPrescriptionRequired = (items: CartItem[]) => 
    items.some(item => item.requiresPrescription);
    
  // Helper to normalize prescription data
  const normalizePrescriptionData = (prescription: PrescriptionData): PrescriptionData => {
    const images = [
      ...(prescription.images || []).map(img => ({
        ...img,
        is_primary: img.is_primary || false,
        uploaded_at: img.uploaded_at || new Date().toISOString(),
        mimetype: img.mimetype || 'image/jpeg',
        size: img.size || 0,
        originalName: img.originalName || img.filename || img.name || 'prescription.jpg',
        path: img.path || img.url || '',
        url: img.url || img.path || ''
      })),
      ...(prescription.files || []).map(file => ({
        ...file,
        is_primary: false,
        uploaded_at: new Date().toISOString(),
        mimetype: file.mimetype || 'image/jpeg',
        size: file.size || 0,
        originalName: file.originalName || file.filename || file.name || 'prescription.jpg',
        path: file.path || file.url || '',
        url: file.url || file.path || ''
      })),
      ...(prescription.fileUrls || []).map(url => ({
        _id: `file-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        name: url.split('/').pop() || 'prescription.jpg',
        path: url,
        url: url,
        type: 'image/jpeg',
        size: 0,
        is_primary: false,
        uploaded_at: new Date().toISOString(),
        originalName: url.split('/').pop() || 'prescription.jpg'
      }))
    ];

    return {
      ...prescription,
      _id: prescription._id || `presc-${Date.now()}`,
      id: prescription.id || prescription._id || `presc-${Date.now()}`,
      patient_name: prescription.patient_name || 'Patient Name',
      doctor_name: prescription.doctor_name || 'Dr. Smith',
      patient_age: prescription.patient_age || 30,
      patient_gender: (prescription.patient_gender || 'other').toLowerCase(),
      prescription_date: prescription.prescription_date || prescription.prescriptionDate || new Date().toISOString(),
      status: prescription.status || 'pending',
      is_verified: prescription.is_verified || false,
      verification_method: prescription.verification_method || 'manual',
      images,
      files: undefined, // Clear files array to avoid duplication
      fileUrls: undefined // Clear fileUrls to avoid duplication
    };
  };

  switch (action.type) {
    case 'ADD_ITEM': {
      // For prescription items, handle specially
      if (action.payload.requiresPrescription && action.payload.prescription) {
        const normalizedPrescription = normalizePrescriptionData(action.payload.prescription);
        const prescriptionId = normalizedPrescription.id || `presc-${Date.now()}`;
        
        const prescriptionItem: CartItem = {
          id: action.payload.id || `presc-item-${Date.now()}`,
          name: action.payload.name || `Prescription - ${normalizedPrescription.patient_name}`,
          price: typeof action.payload.price === 'number' ? action.payload.price : 0,
          quantity: 1,
          requiresPrescription: true,
          prescription: normalizedPrescription,
          // Include other necessary fields
          _id: prescriptionId,
          image: normalizedPrescription.images?.[0]?.url || '',
          originalPrice: 0,
          discountAmount: 0,
          taxAmount: 0,
          storeId: action.payload.storeId,
          storeName: action.payload.storeName,
          category: 'Prescription',
          brand: 'Medical',
          inStock: true
        };

        const updatedPrescriptionItems = [
          ...(state.prescriptionItems || []),
          prescriptionItem
        ];

        const updatedState = {
          ...state,
          prescriptionItems: updatedPrescriptionItems,
          totalItems: state.totalItems + 1,
          totalPrice: state.totalPrice + (prescriptionItem.price * prescriptionItem.quantity),
          requiresPrescription: true,
          // Update the main prescription data if this is the first prescription
          prescription: state.prescription || normalizedPrescription
        };

        saveCartToLocalStorage(updatedState);
        return updatedState;
      }

      // For regular items, check if already in cart
      const existingItemIndex = state.items.findIndex(item => 
        item.id === action.payload.id && 
        item.variantId === action.payload.variantId &&
        !item.requiresPrescription // Only match non-prescription items
      );

      let updatedItems;
      if (existingItemIndex >= 0) {
        // If item exists, update its quantity
        updatedItems = state.items.map((item, index) => 
          index === existingItemIndex 
            ? { 
                ...item, 
                quantity: item.quantity + 1
              }
            : item
        );
      } else {
        // If item doesn't exist, add it with quantity 1
        updatedItems = [
          ...state.items, 
          { 
            ...action.payload, 
            quantity: 1,
            // Ensure price is a number and default to 0 if not provided
            price: typeof action.payload.price === 'number' ? action.payload.price : 0
          }
        ];
      }

      const newState = {
        ...state,
        items: updatedItems,
        totalItems: updatedItems.reduce((total, item) => total + item.quantity, 0),
        totalPrice: updatedItems.reduce((total, item) => total + (item.price * item.quantity), 0),
        requiresPrescription: updatedItems.some(item => item.requiresPrescription)
      };

      saveCartToLocalStorage(newState);
      return newState;
    }
    case 'REMOVE_ITEM': {
      // Check if it's a prescription item
      const isPrescriptionItem = action.payload.id.startsWith('prescription-');
      
      if (isPrescriptionItem) {
        const updatedPrescriptionItems = state.prescriptionItems?.filter(
          item => item.id !== action.payload.id
        ) || [];
        
        const removedPrescription = state.prescriptionItems?.find(
          item => item.id === action.payload.id
        );
        
        const priceToDeduct = removedPrescription 
          ? removedPrescription.price * removedPrescription.quantity 
          : 0;
        
        const updatedState = {
          ...state,
          prescriptionItems: updatedPrescriptionItems,
          totalItems: Math.max(0, state.totalItems - 1),
          totalPrice: Math.max(0, state.totalPrice - priceToDeduct),
          requiresPrescription: checkPrescriptionRequired(state.items) || updatedPrescriptionItems.length > 0
        };
        
        saveCartToLocalStorage(updatedState);
        return updatedState;
      }
      
      // Handle regular items
      const itemToRemove = state.items.find(item => item.id === action.payload.id);
      if (!itemToRemove) return state;
      
      const filteredItems = state.items.filter(item => item.id !== action.payload.id);
      const updatedStateAfterRemove = {
        ...state,
        items: filteredItems,
        totalItems: filteredItems.reduce((total, item) => total + item.quantity, 0),
        totalPrice: filteredItems.reduce((total, item) => total + (item.price * item.quantity), 0),
        requiresPrescription: checkPrescriptionRequired(filteredItems) || (state.prescriptionItems?.length ?? 0) > 0
      };
      saveCartToLocalStorage(updatedStateAfterRemove);
      return updatedStateAfterRemove;
    }
      
    case 'UPDATE_QUANTITY':
      const itemToUpdate = state.items.find(item => item.id === action.payload.id);
      if (!itemToUpdate) return state;
      
      const updatedItems = state.items.map(item => 
        item.id === action.payload.id 
          ? { ...item, quantity: action.payload.quantity }
          : item
      );
      
      const updatedState = {
        ...state,
        items: updatedItems,
        totalItems: updatedItems.reduce((total, item) => total + item.quantity, 0),
        totalPrice: updatedItems.reduce((total, item) => total + (item.price * item.quantity), 0),
        requiresPrescription: updatedItems.some(item => item.requiresPrescription)
      };
      
      saveCartToLocalStorage(updatedState);
      return updatedState;
      
    case 'CLEAR_CART':
      return defaultCartState;
      
    case 'LOAD_CART':
      return action.payload;
      
    case 'SET_PRESCRIPTION_URL':
      return {
        ...state,
        prescriptionUrl: action.payload,
      };
    case 'SET_PRESCRIPTION_DATA': {
      const hasPrescriptionItems = state.items.some(item => item.requiresPrescription);
      const hasPrescriptionData = action.payload !== undefined && action.payload !== null;
      
      const updatedStateWithPrescription = {
        ...state,
        prescription: action.payload,
        requiresPrescription: hasPrescriptionItems || hasPrescriptionData
      };
      
      // Save prescription data to localStorage separately for persistence
      if (hasPrescriptionData) {
        try {
          localStorage.setItem('prescriptionData', JSON.stringify(action.payload));
        } catch (error) {
          console.error('Failed to save prescription to localStorage', error);
        }
      } else {
        localStorage.removeItem('prescriptionData');
      }
      
      saveCartToLocalStorage(updatedStateWithPrescription);
      return updatedStateWithPrescription;
    }
      
    default:
      return state;
  }
};

export const CartProvider: React.FC<{ children: ReactNode }> = ({ children }) => {
  const [isInitialized, setIsInitialized] = React.useState(false);
  const [state, dispatch] = useReducer(cartReducer, loadCartFromLocalStorage());

  // Load cart from localStorage on initial render
  React.useEffect(() => {
    if (!isInitialized) {
      const savedCart = loadCartFromLocalStorage();
      dispatch({ type: 'LOAD_CART', payload: savedCart });
      setIsInitialized(true);
    }
  }, [isInitialized]);

  // Save cart to localStorage whenever it changes
  React.useEffect(() => {
    if (isInitialized) {
      saveCartToLocalStorage(state);
    }
  }, [state, isInitialized]);

  const addToCart = useCallback((item: Omit<CartItem, 'quantity'>) => {
    // Ensure the item has an ID
    if (!item.id && !(item as any)._id) {
      console.error('Cannot add item to cart: missing ID', item);
      toast.error('Cannot add item to cart: missing ID');
      return;
    }
    
    // Use _id as id if id is not present
    const itemWithId: Omit<CartItem, 'quantity'> = {
      ...item,
      id: item.id || (item as any)._id,
      requiresPrescription: item.requiresPrescription || false
    };
    
    dispatch({ type: 'ADD_ITEM', payload: itemWithId });
    toast.success(`${item.name} added to cart`);
  }, []);

  const removeFromCart = useCallback((id: string) => {
    dispatch({ type: 'REMOVE_ITEM', payload: { id } });
    toast.success('Item removed from cart');
  }, []);

  const updateQuantity = useCallback((id: string, quantity: number) => {
    if (quantity < 1) {
      removeFromCart(id);
      return;
    }
    dispatch({ type: 'UPDATE_QUANTITY', payload: { id, quantity } });
  }, []);

  const clearCart = useCallback(() => {
    dispatch({ type: 'CLEAR_CART' });
  }, []);

  const getItemQuantity = useCallback((id: string) => {
    const item = state.items.find(item => item.id === id);
    return item ? item.quantity : 0;
  }, [state.items]);

  const setPrescriptionUrl = useCallback((url: string | undefined) => {
    dispatch({ type: 'SET_PRESCRIPTION_URL', payload: url });
  }, []);

  const setPrescriptionData = useCallback((data: PrescriptionData | undefined) => {
    dispatch({ type: 'SET_PRESCRIPTION_DATA', payload: data });
  }, []);

  // Only provide the context value after initialization to prevent flash of empty cart
  const value = React.useMemo(() => ({
    state: isInitialized ? state : loadCartFromLocalStorage(),
    addToCart,
    removeFromCart,
    updateQuantity,
    clearCart,
    getItemQuantity,
    setPrescriptionUrl,
    setPrescriptionData,
  }), [state, isInitialized, addToCart, removeFromCart, updateQuantity, getItemQuantity, setPrescriptionUrl, setPrescriptionData]);

  return (
    <CartContext.Provider value={value}>
      {children}
    </CartContext.Provider>
  );
};

export const useCart = () => {
  const context = useContext(CartContext);
  if (context === undefined) {
    throw new Error('useCart must be used within a CartProvider');
  }
  return context;
};

export type { PrescriptionData };
export default CartContext;
