import React, { useEffect } from 'react';
import { Navigate, Outlet, useLocation } from 'react-router-dom';
import { useAuth } from '../context/AuthContext';

interface ProtectedRouteProps {
  allowedRoles?: string[];
  children?: React.ReactNode;
}

const ProtectedRoute: React.FC<ProtectedRouteProps> = ({ 
  allowedRoles = [], 
  children 
}) => {
  const { user, loading } = useAuth();
  const location = useLocation();
  
  // Debug logs
  useEffect(() => {
    console.log('ProtectedRoute - User:', user);
    console.log('ProtectedRoute - Allowed Roles:', allowedRoles);
    console.log('ProtectedRoute - Current Path:', location.pathname);
  }, [user, allowedRoles, location.pathname]);
  
  // Show loading state while checking authentication
  if (loading) {
    return <div>Loading...</div>;
  }
  
  // If no user is logged in, redirect to login
  if (!user) {
    console.log('ProtectedRoute - No user found, redirecting to login');
    return <Navigate to="/login" state={{ from: location.pathname }} replace />;
  }

  // If no specific roles required, render the children
  if (allowedRoles.length === 0) {
    console.log('ProtectedRoute - No role restriction, allowing access');
    return children ? <>{children}</> : <Outlet />;
  }

  // Check if user has any of the required roles
  const userRole = user.role?.name || '';
  const hasRequiredRole = userRole && allowedRoles.includes(userRole);
  
  if (!hasRequiredRole) {
    console.log('ProtectedRoute - Access denied. User role:', userRole, 'Allowed roles:', allowedRoles);
    // User is authenticated but doesn't have required role
    return <Navigate to="/unauthorized" state={{ from: location.pathname }} replace />;
  }

  // User is authenticated and has required role
  console.log('ProtectedRoute - Access granted for role:', userRole);
  return children ? <>{children}</> : <Outlet />;
};

export default ProtectedRoute;