import React, { useState } from 'react';
import { UploadOutlined, FileImageOutlined, FilePdfOutlined, DeleteOutlined } from '@ant-design/icons';
import { useCart } from '@/context/CartContext';
import { Button, Form, Input, Select, Upload, message, Modal, Typography } from 'antd';
import type { UploadFile, UploadProps } from 'antd/es/upload/interface';
import api from '@/services/api';

const { TextArea } = Input;
const { Option } = Select;
const { Title, Text } = Typography;

interface PrescriptionData {
  doctor_name: string;
  patient_name: string;
  patient_age: string | number;
  patient_gender: string;
  diagnosis?: string;
  notes?: string;
  prescription_date: string;
}

interface UploadPrescriptionProps {
  onUpload?: (files: File[], prescriptionData: PrescriptionData) => void;
  onUploadSuccess?: (files: File[], prescriptionData: PrescriptionData) => void;
  onClose?: () => void;
  onCancel?: () => void;
  initialData?: Partial<PrescriptionData>;
}

const UploadPrescription: React.FC<UploadPrescriptionProps> = ({
  onUpload,
  onUploadSuccess,
  onClose,
  onCancel,
  initialData = {},
}) => {
  const [form] = Form.useForm();
  const [fileList, setFileList] = useState<UploadFile[]>([]);
  const [isDragging, setIsDragging] = useState(false);
  const [loading, setLoading] = useState(false);
  const { addToCart } = useCart();
  const [previewFile, setPreviewFile] = useState<{
    visible: boolean;
    title: string;
    url: string;
    type: 'image' | 'pdf' | 'other';
  } | null>(null);

  const handleDragOver = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    if (!isDragging) setIsDragging(true);
  };

  const handleDragLeave = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
  };

  const processFiles = (files: File[]) => {
    const validFiles = files.filter(
      (file) => file.type.startsWith('image/') || file.type === 'application/pdf'
    );

    if (validFiles.length !== files.length) {
      message.warning('Only image and PDF files are allowed');
    }

    if (validFiles.length > 0) {
      const newFiles = validFiles.map(file => ({
        uid: `file-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        name: file.name,
        status: 'done',
        originFileObj: file,
      })) as unknown as UploadFile[];
      
      setFileList(prev => [...prev, ...newFiles]);
    }
  };

  const handleDrop = (e: React.DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
    
    const files = Array.from(e.dataTransfer.files);
    processFiles(files);
  };

  const handleRemove = (file: UploadFile) => {
    setFileList(prev => prev.filter(item => item.uid !== file.uid));
  };

  const handlePreview = async (file: UploadFile) => {
    if (!file.originFileObj) return;

    const fileType = file.type?.startsWith('image/') ? 'image' : 
                    file.type === 'application/pdf' ? 'pdf' : 'other';
    
    if (fileType === 'other') {
      message.info('Preview not available for this file type');
      return;
    }

    const url = URL.createObjectURL(file.originFileObj);
    setPreviewFile({
      visible: true,
      title: file.name || 'Preview',
      url,
      type: fileType as 'image' | 'pdf' | 'other'
    });
  };

  const handlePreviewCancel = () => {
    if (previewFile) {
      URL.revokeObjectURL(previewFile.url);
      setPreviewFile(null);
    }
  };

  const handleSubmit = async () => {
    try {
      setLoading(true);
      const values = await form.validateFields();
      
      if (fileList.length === 0) {
        message.error('Please upload at least one prescription file');
        setLoading(false);
        return;
      }

      // Create FormData for file upload
      const formData = new FormData();
      
      // Add files to form data
      fileList.forEach(file => {
        formData.append('prescription', file.originFileObj as File);
      });

      // Add form data
      const formValues = {
        patientName: values.patient_name,
        patientAge: values.patient_age,
        patientGender: values.patient_gender,
        doctorName: values.doctor_name || '',
        diagnosis: values.diagnosis || '',
        notes: values.notes || '',
        prescriptionDate: values.prescription_date || new Date().toISOString().split('T')[0]
      };

      // Append form values to FormData
      Object.entries(formValues).forEach(([key, value]) => {
        formData.append(key, value);
      });

      // Upload to server - use the prescriptions endpoint
      const response = await api.post('/prescriptions/upload', formData, {
        headers: {
          'Content-Type': 'multipart/form-data',
        },
      });

      // Get the server-uploaded file paths from the response
      const uploadedFiles = response.data.files || [];
      
      // Add prescription to cart context with server-uploaded file paths
      const prescriptionData = {
        id: `prescription-${Date.now()}`,
        name: `Prescription - ${values.patient_name}`,
        price: 0, // Set appropriate price if needed
        quantity: 1,
        requiresPrescription: true,
        prescription: {
          ...values,
          // Use server-uploaded file paths if available, otherwise fallback to local URLs
          images: uploadedFiles.length > 0 
            ? uploadedFiles.map((file: any) => ({
                name: file.filename,
                path: file.path,
                type: file.mimetype || 'image/jpeg',
                size: file.size || 0,
                url: `/uploads/prescriptions/${file.filename}` // Construct the URL to access the file
              }))
            : fileList.map(file => ({
                name: file.name,
                type: file.type || 'image/jpeg',
                size: file.size || 0,
                url: URL.createObjectURL(file.originFileObj as File)
              }))
        }
      };
      
      addToCart(prescriptionData);
      
      // Call success handler if provided
      if (onUploadSuccess) {
        onUploadSuccess(fileList.map(f => f.originFileObj as File), values);
      }
      
      message.success('Prescription uploaded and added to cart successfully');
      const closeHandler = onClose || onCancel;
      if (closeHandler) closeHandler();
    } catch (error) {
      console.error('Upload failed:', error);
      const errorMessage = error.response?.data?.message || 'Failed to upload prescription. Please try again.';
      message.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const uploadProps: UploadProps = {
    onRemove: handleRemove,
    onPreview: handlePreview,
    beforeUpload: (file) => {
      const isImageOrPdf = file.type.startsWith('image/') || file.type === 'application/pdf';
      if (!isImageOrPdf) {
        message.error('You can only upload image or PDF files!');
        return Upload.LIST_IGNORE;
      }
      
      // Check file size (10MB limit)
      const isLt10M = file.size / 1024 / 1024 < 10;
      if (!isLt10M) {
        message.error('File must be smaller than 10MB!');
        return Upload.LIST_IGNORE;
      }
      
      // Add to file list
      const newFile: UploadFile = {
        uid: `file-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        name: file.name,
        status: 'done',
        originFileObj: file,
      };
      
      setFileList(prev => [...prev, newFile]);
      return false; // Prevent default upload
    },
    fileList,
    multiple: true,
    showUploadList: {
      showPreviewIcon: true,
      showRemoveIcon: true,
      showDownloadIcon: false,
    },
    listType: 'picture-card',
    accept: 'image/*,.pdf',
  };

  return (
    <Modal
      title="Upload Prescription"
      open={true}
      onCancel={onClose}
      footer={[
        <Button key="cancel" onClick={onClose}>
          Cancel
        </Button>,
        <Button
          key="submit"
          type="primary"
          loading={loading}
          onClick={handleSubmit}
          icon={<UploadOutlined />}
        >
          Upload Prescription
        </Button>,
      ]}
      width={800}
    >
      <Form
        form={form}
        layout="vertical"
        initialValues={{
          patient_gender: 'male',
          prescription_date: new Date().toISOString().split('T')[0],
          ...initialData
        }}
      >
        <Title level={4} style={{ marginBottom: 16 }}>Patient Information</Title>
        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '16px', marginBottom: 24 }}>
          <Form.Item
            name="patient_name"
            label="Patient Name"
            rules={[{ required: true, message: 'Please enter patient name' }]}
          >
            <Input placeholder="John Doe" />
          </Form.Item>
          
          <Form.Item
            name="patient_age"
            label="Age"
            rules={[
              { required: true, message: 'Please enter patient age' },
              { pattern: /^[0-9]+$/, message: 'Age must be a number' },
            ]}
          >
            <Input type="number" min={0} max={120} placeholder="30" />
          </Form.Item>
          
          <Form.Item
            name="patient_gender"
            label="Gender"
            rules={[{ required: true, message: 'Please select gender' }]}
          >
            <Select placeholder="Select gender">
              <Option value="male">Male</Option>
              <Option value="female">Female</Option>
              <Option value="other">Other</Option>
            </Select>
          </Form.Item>
          
          <Form.Item
            name="prescription_date"
            label="Prescription Date"
            rules={[{ required: true, message: 'Please select prescription date' }]}
          >
            <Input type="date" style={{ width: '100%' }} />
          </Form.Item>
        </div>

        <Title level={4} style={{ marginBottom: 16 }}>Prescription Information</Title>
        
        <div style={{ marginBottom: 24 }}>
          <div 
            onDragOver={handleDragOver}
            onDragLeave={handleDragLeave}
            onDrop={handleDrop}
            style={{
              border: `2px dashed ${isDragging ? '#1890ff' : '#d9d9d9'}`,
              borderRadius: 8,
              padding: 24,
              textAlign: 'center',
              backgroundColor: isDragging ? '#f0f9ff' : '#fafafa',
              cursor: 'pointer',
              transition: 'all 0.3s',
              marginBottom: 16
            }}
          >
            <Upload.Dragger
              {...uploadProps}
              style={{
                background: 'transparent',
                border: 'none',
                padding: 0,
                margin: 0
              }}
            >
              <p className="ant-upload-drag-icon">
                <FileImageOutlined style={{ fontSize: '32px', color: '#1890ff' }} />
              </p>
              <p className="ant-upload-text">
                Click or drag file to this area to upload
              </p>
              <p className="ant-upload-hint">
                Support for a single or bulk upload. Strictly prohibit from uploading company data or other band files
              </p>
            </Upload.Dragger>
          </div>
          
          {fileList.length > 0 && (
            <div style={{ marginTop: 16 }}>
              <h4>Selected Files:</h4>
              <div style={{ display: 'flex', flexWrap: 'wrap', gap: 8 }}>
                {fileList.map(file => (
                  <div
                    key={file.uid}
                    style={{
                      border: '1px solid #d9d9d9',
                      borderRadius: 4,
                      padding: 8,
                      display: 'flex',
                      alignItems: 'center',
                      gap: 8,
                      background: '#fff'
                    }}
                  >
                    {file.type?.startsWith('image/') ? (
                      <FileImageOutlined style={{ color: '#1890ff' }} />
                    ) : (
                      <FilePdfOutlined style={{ color: '#ff4d4f' }} />
                    )}
                    <span style={{ flex: 1, whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis' }}>
                      {file.name}
                    </span>
                    <Button
                      type="text"
                      danger
                      icon={<DeleteOutlined />}
                      onClick={() => handleRemove(file)}
                      size="small"
                    />
                  </div>
                ))}
              </div>
            </div>
          )}
        </div>

        <Title level={4} style={{ marginBottom: 16 }}>Doctor Information</Title>
        <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '16px', marginBottom: 24 }}>
          <Form.Item
            name="doctor_name"
            label="Doctor's Name"
          >
            <Input type="date" />
          </Form.Item>
        </div>

        <Title level={4} style={{ marginBottom: 16 }}>Doctor Information</Title>
        <div style={{ marginBottom: 24 }}>
          <Form.Item
            name="doctor_name"
            label="Doctor's Name"
          >
            <Input placeholder="Dr. Smith" />
          </Form.Item>
          
          <Form.Item
            name="diagnosis"
            label="Diagnosis"
          >
            <Input placeholder="e.g., Hypertension, Diabetes" />
          </Form.Item>
          
          <Form.Item
            name="notes"
            label="Additional Notes"
          >
            <TextArea rows={3} placeholder="Any additional instructions or notes..." />
          </Form.Item>
        </div>

        <Title level={4} style={{ marginBottom: 16 }}>Prescription Files</Title>
        <div
          onDragOver={handleDragOver}
          onDragLeave={handleDragLeave}
          onDrop={handleDrop}
          style={{
            border: `2px dashed ${isDragging ? '#1890ff' : '#d9d9d9'}`,
            borderRadius: 8,
            padding: 24,
            textAlign: 'center',
            marginBottom: 24,
            backgroundColor: isDragging ? '#f0f9ff' : '#fafafa',
            transition: 'all 0.3s',
          }}
        >
          <Upload.Dragger
            {...uploadProps}
            style={{ background: 'transparent', border: 'none' }}
            showUploadList={false}
          >
            <p className="ant-upload-drag-icon">
              <UploadOutlined style={{ fontSize: 32, color: '#1890ff' }} />
            </p>
            <p className="ant-upload-text">
              Click or drag file to this area to upload
            </p>
            <p className="ant-upload-hint">
              Support for a single or bulk upload. Strictly prohibit from uploading company data or other band files
            </p>
          </Upload.Dragger>
        </div>

        {fileList.length > 0 && (
          <div style={{ marginBottom: 24 }}>
            <Text strong>Selected Files:</Text>
            <div style={{ marginTop: 8 }}>
              {fileList.map(file => (
                <div
                  key={file.uid}
                  style={{
                    display: 'flex',
                    justifyContent: 'space-between',
                    alignItems: 'center',
                    padding: '8px 12px',
                    border: '1px solid #f0f0f0',
                    borderRadius: 4,
                    marginBottom: 8,
                  }}
                >
                  <div>
                    {file.type?.startsWith('image/') ? (
                      <FileImageOutlined style={{ marginRight: 8 }} />
                    ) : (
                      <FilePdfOutlined style={{ marginRight: 8 }} />
                    )}
                    {file.name}
                  </div>
                  <Button
                    type="text"
                    danger
                    icon={<DeleteOutlined />}
                    onClick={() => handleRemove(file)}
                  />
                </div>
              ))}
            </div>
          </div>
        )}
      </Form>

      {/* Preview Modal */}
      <Modal
        open={!!previewFile?.visible}
        title={previewFile?.title || 'Preview'}
        footer={null}
        onCancel={handlePreviewCancel}
        width={previewFile?.type === 'pdf' ? '80%' : 'auto'}
        style={{ top: 20 }}
      >
        {previewFile?.type === 'image' && (
          <img alt="preview" style={{ width: '100%' }} src={previewFile.url} />
        )}
        {previewFile?.type === 'pdf' && (
          <iframe
            src={previewFile.url}
            title={previewFile.title}
            style={{ width: '100%', height: '80vh', border: 'none' }}
          />
        )}
      </Modal>
    </Modal>
  );
};

export default UploadPrescription;
