import React, { useState, useEffect } from "react";
import { X, ShieldCheck, Clock, Star, ArrowLeft, Loader2 } from "lucide-react";
import { toast } from "react-hot-toast";
import { useAuth } from "@/context/AuthContext";

interface RegisterModalProps {
  isOpen: boolean;
  onClose: () => void;
}

const OTP_LENGTH = 6;
const OTP_RESEND_TIMEOUT = 30; // seconds

const RegisterModal: React.FC<RegisterModalProps> = ({ isOpen, onClose }) => {
  const { verifyOtp, sendOtp } = useAuth();
  const [mobile, setMobile] = useState("");
  const [otp, setOtp] = useState<string[]>(["", "", "", "", "", ""]);
  const [showOtpScreen, setShowOtpScreen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [resendTimer, setResendTimer] = useState(0);
  const [generatedOtp, setGeneratedOtp] = useState('');
  // Removed unused otpSent state as it's not being used
  const [formData] = useState({
    name: "",
    email: "",
  });

  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (resendTimer > 0) {
      interval = setInterval(() => {
        setResendTimer((prev) => prev - 1);
      }, 1000);
    }
    return () => clearInterval(interval);
  }, [resendTimer]);

  const startResendTimer = () => {
    setResendTimer(OTP_RESEND_TIMEOUT);
  };

  const handleSendOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!mobile || mobile.length !== 10) {
      toast.error('Please enter a valid 10-digit mobile number');
      return;
    }
    
    setIsLoading(true);
    
    try {
      // Call the sendOtp API
      const response = await sendOtp(mobile);
      
      // In development, auto-fill and verify OTP
      if (import.meta.env.DEV && response.otp) {
        console.log(`[DEV] OTP for ${mobile}: ${response.otp}`);
        // Auto-fill the OTP
        const otpArray = response.otp.split('');
        setOtp(otpArray);
        setGeneratedOtp(response.otp);
        
        // Show success message
        toast.success(`OTP sent to ${mobile}. Auto-verifying...`);
        
        // Auto-verify after a short delay
        setTimeout(async () => {
          try {
            setIsLoading(true);
            const { isNewUser } = await verifyOtp(mobile, response.otp, {
              name: formData.name || `User-${mobile.slice(-4) || 'New'}`,
              email: formData.email || `${mobile}@hittok.com`,
              phone: mobile
            });
            
            const successMessage = isNewUser 
              ? "Registration successful!" 
              : "Login successful!";
            
            toast.success(successMessage);
            
            // Reset form and close modal
            setOtp(["", "", "", "", "", ""]);
            setMobile("");
            setGeneratedOtp('');
            onClose();
            setShowOtpScreen(false);
            
          } catch (error: any) {
            console.error("Auto-verify error:", error);
            toast.error(error.message || "Auto-verification failed. Please try manually.");
          } finally {
            setIsLoading(false);
          }
        }, 1000); // 1 second delay before auto-verification
        
      } else {
        toast.success('OTP sent successfully');
      }
      
      // Show OTP screen after successful OTP send
      setShowOtpScreen(true);
      startResendTimer();
      
    } catch (error: any) {
      console.error('Error sending OTP:', error);
      toast.error(error.message || 'Failed to send OTP. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleOtpChange = (index: number, value: string) => {
    // Only allow numbers
    if (value && !/^\d*$/.test(value)) return;
    
    // Update the OTP array with the new value
    const newOtp = [...otp];
    newOtp[index] = value.slice(-1); // Only take the last character if multiple are pasted
    setOtp(newOtp);

    // Auto-focus next input if a digit was entered
    if (value && index < OTP_LENGTH - 1) {
      const nextInput = document.getElementById(`otp-${index + 1}`) as HTMLInputElement;
      if (nextInput) {
        nextInput.focus();
        nextInput.select();
      }
    }
  };

  const handleKeyDown = (index: number, e: React.KeyboardEvent<HTMLInputElement>) => {
    // Handle backspace to move to previous input
    if (e.key === 'Backspace' && !otp[index] && index > 0) {
      const prevInput = document.getElementById(`otp-${index - 1}`) as HTMLInputElement;
      if (prevInput) {
        prevInput.focus();
        prevInput.select();
      }
    }
    
    // Handle left arrow to move to previous input
    if (e.key === 'ArrowLeft' && index > 0) {
      e.preventDefault();
      const prevInput = document.getElementById(`otp-${index - 1}`) as HTMLInputElement;
      if (prevInput) {
        prevInput.focus();
        prevInput.select();
      }
    }
    
    // Handle right arrow to move to next input
    if (e.key === 'ArrowRight' && index < OTP_LENGTH - 1) {
      e.preventDefault();
      const nextInput = document.getElementById(`otp-${index + 1}`) as HTMLInputElement;
      if (nextInput) {
        nextInput.focus();
        nextInput.select();
      }
    }
  };
  
  // Handle paste event for OTP
  const handlePaste = (e: React.ClipboardEvent<HTMLInputElement>, index: number) => {
    e.preventDefault();
    const pasteData = e.clipboardData.getData('text/plain');
    const digits = pasteData.replace(/\D/g, ''); // Remove non-digit characters
    
    if (digits) {
      const newOtp = [...otp];
      // Fill the OTP fields with the pasted digits
      for (let i = 0; i < Math.min(digits.length, OTP_LENGTH - index); i++) {
        newOtp[index + i] = digits[i];
      }
      setOtp(newOtp);
      
      // Focus on the next empty field or the last field
      const nextIndex = Math.min(index + digits.length, OTP_LENGTH - 1);
      const nextInput = document.getElementById(`otp-${nextIndex}`) as HTMLInputElement;
      if (nextInput) {
        nextInput.focus();
        nextInput.select();
      }
    }
  };

  const handleVerifyOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    const otpString = otp.join('');
    
    if (otpString.length !== OTP_LENGTH) {
      toast.error("Please enter a valid 6-digit OTP");
      return;
    }
    
    setIsLoading(true);
    
    try {
      // In development mode, we'll still show the OTP in console for testing
      if (import.meta.env.DEV) {
        console.log('[DEV] Verifying OTP:', otpString);
      }
      
      // Call verifyOtp which will make the API call to verify the OTP
      const { isNewUser } = await verifyOtp(mobile, otpString, {
        name: formData.name || `User-${mobile.slice(-4) || 'New'}`,
        email: formData.email || `${mobile}@hittok.com`,
        phone: mobile
      });
      
      const successMessage = isNewUser 
        ? "Registration successful!" 
        : "Login successful!";
      
      toast.success(successMessage);
      
      // Reset form
      setOtp(["", "", "", "", "", ""]);
      setMobile("");
      setGeneratedOtp('');
      
      // Show success message for a moment before closing
      setTimeout(() => {
        onClose();
        setShowOtpScreen(false);
      }, 1000);
      
    } catch (error: any) {
      console.error("Error verifying OTP:", error);
      
      let errorMessage = "An error occurred. Please try again.";
      
      if (error.response) {
        // Server responded with error status
        errorMessage = error.response.data?.message || error.message;
      } else if (error.request) {
        // Request was made but no response received
        errorMessage = "No response from server. Please check your connection.";
      } else {
        // Other errors
        errorMessage = error.message || "OTP verification failed";
      }
      
      toast.error(errorMessage);
    } finally {
      setIsLoading(false);
    }
  };

  const handleResendOtp = async () => {
    if (resendTimer > 0) return;
    
    setIsLoading(true);
    try {
      const response = await sendOtp(mobile);
      startResendTimer();
      toast.success(response.message || "OTP resent successfully");
    } catch (error: any) {
      console.error("Error resending OTP:", error);
      toast.error(error.message || "Failed to resend OTP. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  const handleBackToPhone = () => {
    setShowOtpScreen(false);
    setOtp(["", "", "", "", "", ""]);
  };

  if (!isOpen) return null;

  // Prevent click events from propagating to the modal background
  const handleModalClick = (e: React.MouseEvent) => {
    e.stopPropagation();
  };

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center"
      onClick={onClose} // Close when clicking outside
    >
      <div 
        className="bg-white w-[90%] max-w-sm rounded-xl shadow-lg relative p-6"
        onClick={handleModalClick} // Prevent click from closing modal
      >
        {/* Close Button */}
        <button
          onClick={onClose}
          className="absolute top-3 right-3 text-gray-500 hover:text-gray-700"
        >
          <X className="h-5 w-5" />
        </button>

        {/* Header */}
        <h2 className="text-lg font-semibold mb-4 text-gray-800">
          Quickly Login/Register
        </h2>

        {/* Form */}
        {!showOtpScreen ? (
          <form onSubmit={handleSendOtp} className="space-y-3">
            <div className="relative">
              <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                <span className="text-gray-500">+91</span>
              </div>
              <input
                type="tel"
                placeholder="Enter mobile number"
                value={mobile}
                onChange={(e) => setMobile(e.target.value.replace(/\D/g, "").slice(0, 10))}
                className="w-full border border-gray-300 rounded-lg pl-12 pr-4 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                maxLength={10}
                required
                disabled={isLoading}
              />
            </div>
            <button
              type="submit"
              className="w-full bg-blue-600 hover:bg-blue-700 text-white rounded-lg py-2 font-medium flex items-center justify-center gap-2 disabled:opacity-70 disabled:cursor-not-allowed"
              disabled={isLoading || mobile.length !== 10}
            >
              {isLoading ? (
                <>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Sending...
                </>
              ) : (
                "Send OTP"
              )}
            </button>
          </form>
        ) : (
          <form onSubmit={handleVerifyOtp} className="space-y-4">
            <div className="flex justify-between items-center mb-2">
              <button
                type="button"
                onClick={handleBackToPhone}
                className="text-blue-600 hover:text-blue-800 flex items-center gap-1 text-sm"
              >
                <ArrowLeft className="h-4 w-4" />
                Back
              </button>
              <p className="text-sm text-gray-600">
                Enter OTP sent to +91 {mobile}
              </p>
            </div>
            
            <div className="flex justify-center gap-2 mb-4">
              {otp.map((digit, index) => (
                <input
                  key={index}
                  id={`otp-${index}`}
                  type="text"
                  inputMode="numeric"
                  autoComplete="one-time-code"
                  maxLength={1}
                  value={digit}
                  onChange={(e) => handleOtpChange(index, e.target.value)}
                  onKeyDown={(e) => handleKeyDown(index, e)}
                  onPaste={(e) => handlePaste(e, index)}
                  onClick={(e) => {
                    e.stopPropagation();
                    (e.target as HTMLInputElement).select();
                  }}
                  onFocus={(e) => e.target.select()}
                  className="w-10 h-12 text-center text-lg border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-70"
                  disabled={isLoading}
                  autoFocus={index === 0 && !otp[0]}
                />
              ))}
            </div>

            <button
              type="submit"
              className="w-full bg-blue-600 hover:bg-blue-700 text-white rounded-lg py-2 font-medium flex items-center justify-center gap-2 disabled:opacity-70 disabled:cursor-not-allowed"
              disabled={isLoading || otp.some(digit => !digit)}
            >
              {isLoading ? (
                <>
                  <Loader2 className="h-4 w-4 animate-spin" />
                  Verifying...
                </>
              ) : (
                "Verify OTP"
              )}
            </button>

            <div className="text-center text-sm">
              {resendTimer > 0 ? (
                <p className="text-gray-500">
                  Resend OTP in {resendTimer}s
                </p>
              ) : (
                <button
                  type="button"
                  onClick={handleResendOtp}
                  className="text-blue-600 hover:text-blue-800 disabled:opacity-50 disabled:cursor-not-allowed"
                  disabled={isLoading}
                >
                  Resend OTP
                </button>
              )}
            </div>
          </form>
        )}

        {/* Terms */}
        <p className="text-xs text-gray-500 mt-2">
          By signing up, I agree to the{" "}
          <a href="#" className="text-blue-600 underline">
            Privacy Policy
          </a>
          ,{" "}
          <a href="#" className="text-blue-600 underline">
            Terms and Conditions
          </a>{" "}
          of Apollo Pharmacy.
        </p>

        {/* Divider */}
        <div className="border-t my-4"></div>

        {/* Welcome Section */}
        <div>
          <h3 className="text-sm font-semibold text-gray-800 mb-2">
            Welcome to Healthsta Pharmacy
          </h3>
          <p className="text-xs text-gray-500 mb-4">
            Join millions of users who trust us for their healthcare needs
          </p>

          {/* Feature List */}
          <ul className="space-y-3 text-sm">
            <li className="flex items-start gap-3">
              <ShieldCheck className="h-5 w-5 text-blue-600 flex-shrink-0" />
              <span>
                <strong>100% Authentic Medicines</strong>
                <br />
                <span className="text-gray-500 text-xs">
                  All medicines sourced directly from licensed manufacturers
                </span>
              </span>
            </li>
            <li className="flex items-start gap-3">
              <Clock className="h-5 w-5 text-blue-600 flex-shrink-0" />
              <span>
                <strong>Express Delivery</strong>
                <br />
                <span className="text-gray-500 text-xs">
                  Get your medicines delivered within 2 hours
                </span>
              </span>
            </li>
            <li className="flex items-start gap-3">
              <Star className="h-5 w-5 text-blue-600 flex-shrink-0" />
              <span>
                <strong>Expert Care</strong>
                <br />
                <span className="text-gray-500 text-xs">
                  24/7 support from qualified pharmacists and doctors
                </span>
              </span>
            </li>
          </ul>
        </div>
      </div>
    </div>
  );
};

export default RegisterModal;
