import React, { useState } from "react";
import { ArrowLeft, Download, MessageCircle, MapPin, HelpCircle, Gift, LogOut, Home, Package } from "lucide-react";
import Header from "./Header";
import Footer from "./Footer";

type TabType = 'orders' | 'addresses' | 'help' | 'vouchers';

interface Address {
  id: string;
  name: string;
  address: string;
  city: string;
  state: string;
  pincode: string;
  isDefault: boolean;
  type: 'home' | 'work' | 'other';
}

interface Voucher {
  id: string;
  code: string;
  discount: number;
  type: 'percentage' | 'fixed';
  minOrder?: number;
  expiryDate: string;
  used: boolean;
}

const dummyAddresses: Address[] = [
  {
    id: '1',
    name: 'Home',
    address: '123 Main Street, Apartment 4B',
    city: 'Mumbai',
    state: 'Maharashtra',
    pincode: '400001',
    isDefault: true,
    type: 'home'
  },
  {
    id: '2',
    name: 'Work',
    address: '456 Business District, Tech Park',
    city: 'Mumbai',
    state: 'Maharashtra',
    pincode: '400002',
    isDefault: false,
    type: 'work'
  }
];

const dummyVouchers: Voucher[] = [
  {
    id: '1',
    code: 'WELCOME20',
    discount: 20,
    type: 'percentage',
    minOrder: 500,
    expiryDate: '2024-12-31',
    used: false
  },
  {
    id: '2',
    code: 'FLAT100',
    discount: 100,
    type: 'fixed',
    expiryDate: '2024-12-31',
    used: true
  }
];

const Profile: React.FC = () => {
  const [activeTab, setActiveTab] = useState<TabType>('addresses');
  const [addresses, setAddresses] = useState<Address[]>(dummyAddresses);
  const [vouchers] = useState<Voucher[]>(dummyVouchers);
  const [showAddForm, setShowAddForm] = useState(false);
  const [newAddress, setNewAddress] = useState<Omit<Address, 'id' | 'isDefault'>>({ 
    name: '',
    address: '',
    city: '',
    state: '',
    pincode: '',
    type: 'home'
  });
  
  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setNewAddress(prev => ({
      ...prev,
      [name]: value
    }));
  };
  
  const handleAddAddress = (e: React.FormEvent) => {
    e.preventDefault();
    // Basic validation
    if (!newAddress.name || !newAddress.address || !newAddress.city || !newAddress.state) {
      alert('Please fill in all required fields');
      return;
    }
    
    // In a real app, you would save this to your backend
    const newAddressWithId: Address = {
      ...newAddress,
      id: Date.now().toString(),
      isDefault: addresses.length === 0 // Set as default if it's the first address
    };
    
    setAddresses(prev => [...prev, newAddressWithId]);
    setNewAddress({ 
      name: '',
      address: '',
      city: '',
      state: '',
      pincode: '',
      type: 'home' 
    });
    setShowAddForm(false);
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'orders':
        return (
          <div>
            <h2 className="text-xl font-semibold text-gray-900 mb-6">My Orders</h2>
            <div className="bg-gray-50 rounded-lg p-6 text-center">
              <Package className="w-12 h-12 mx-auto text-gray-400 mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No orders yet</h3>
              <p className="text-gray-600 mb-4">You haven't placed any orders yet.</p>
              <button className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium">
                Start Shopping
              </button>
            </div>
          </div>
        );
      
      case 'addresses':
        return (
          <div>
            <div className="flex justify-between items-center mb-6">
              <h2 className="text-xl font-semibold text-gray-900">My Addresses</h2>
              {!showAddForm && (
                <button 
                  onClick={() => setShowAddForm(true)}
                  className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium"
                >
                  + Add New Address
                </button>
              )}
            </div>
            
            {showAddForm && (
              <div className="bg-white rounded-lg shadow-md p-6 mb-6">
                <h3 className="text-lg font-medium text-gray-900 mb-4">Add New Address</h3>
                <form onSubmit={handleAddAddress}>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Full Name *</label>
                      <input
                        type="text"
                        name="name"
                        value={newAddress.name}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Address *</label>
                      <input
                        type="text"
                        name="address"
                        value={newAddress.address}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">City *</label>
                      <input
                        type="text"
                        name="city"
                        value={newAddress.city}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">State *</label>
                      <input
                        type="text"
                        name="state"
                        value={newAddress.state}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Pincode</label>
                      <input
                        type="text"
                        name="pincode"
                        value={newAddress.pincode}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">Address Type</label>
                      <select
                        name="type"
                        value={newAddress.type}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      >
                        <option value="home">Home</option>
                        <option value="work">Work</option>
                        <option value="other">Other</option>
                      </select>
                    </div>
                  </div>
                  <div className="flex justify-end space-x-3">
                    <button
                      type="button"
                      onClick={() => setShowAddForm(false)}
                      className="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"
                    >
                      Cancel
                    </button>
                    <button
                      type="submit"
                      className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700"
                    >
                      Save Address
                    </button>
                  </div>
                </form>
              </div>
            )}
            
            <div className="grid md:grid-cols-2 gap-4">
              {addresses.map((address) => (
                <div key={address.id} className="border rounded-lg p-4 relative">
                  {address.isDefault && (
                    <span className="absolute top-2 right-2 bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">
                      Default
                    </span>
                  )}
                  <div className="flex items-start">
                    <MapPin className="w-5 h-5 text-blue-600 mt-0.5 mr-2 flex-shrink-0" />
                    <div>
                      <div className="flex items-center mb-1">
                        <h3 className="font-medium text-gray-900">{address.name}</h3>
                        <span className="ml-2 text-xs bg-gray-100 text-gray-600 px-2 py-0.5 rounded">
                          {address.type}
                        </span>
                      </div>
                      <p className="text-sm text-gray-600">{address.address}</p>
                      <p className="text-sm text-gray-600">{address.city}, {address.state} - {address.pincode}</p>
                      
                      <div className="mt-3 flex space-x-3">
                        <button className="text-blue-600 hover:text-blue-700 text-sm font-medium">
                          Edit
                        </button>
                        {!address.isDefault && (
                          <button className="text-red-600 hover:text-red-700 text-sm font-medium">
                            Delete
                          </button>
                        )}
                        {!address.isDefault && (
                          <button className="text-green-600 hover:text-green-700 text-sm font-medium">
                            Set as default
                          </button>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        );
      
      case 'help':
        return (
          <div>
            <h2 className="text-xl font-semibold text-gray-900 mb-6">Need Help?</h2>
            
            <div className="space-y-4">
              <div className="bg-gray-50 rounded-lg p-6">
                <h3 className="font-medium text-gray-900 mb-3">Frequently Asked Questions</h3>
                <div className="space-y-3">
                  {[
                    'How do I track my order?',
                    'What is your return policy?',
                    'How can I contact customer support?',
                    'How do I cancel my order?',
                    'What payment methods do you accept?'
                  ].map((question, index) => (
                    <div key={index} className="border-b border-gray-200 pb-3">
                      <button className="w-full text-left flex justify-between items-center">
                        <span className="text-gray-700">{question}</span>
                        <span className="text-gray-400">+</span>
                      </button>
                    </div>
                  ))}
                </div>
              </div>
              
              <div className="bg-blue-50 rounded-lg p-6">
                <div className="flex items-start">
                  <MessageCircle className="w-5 h-5 text-blue-600 mt-0.5 mr-3 flex-shrink-0" />
                  <div>
                    <h3 className="font-medium text-gray-900 mb-2">Still need help?</h3>
                    <p className="text-sm text-gray-600 mb-3">Our customer care team is available to help you with any questions or concerns.</p>
                    <button className="bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg text-sm font-medium">
                      Contact Support
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
        );
      
      case 'vouchers':
        return (
          <div>
            <div className="flex justify-between items-center mb-6">
              <h2 className="text-xl font-semibold text-gray-900">My Vouchers</h2>
              <div className="text-sm text-gray-500">{vouchers.length} vouchers</div>
            </div>
            
            <div className="space-y-4">
              {vouchers.length === 0 ? (
                <div className="bg-gray-50 rounded-lg p-8 text-center">
                  <Gift className="w-12 h-12 mx-auto text-gray-400 mb-4" />
                  <h3 className="text-lg font-medium text-gray-900 mb-2">No vouchers available</h3>
                  <p className="text-gray-600">You don't have any vouchers at the moment.</p>
                </div>
              ) : (
                vouchers.map((voucher) => (
                  <div key={voucher.id} className={`border rounded-lg p-4 ${voucher.used ? 'opacity-70' : ''}`}>
                    <div className="flex justify-between items-start">
                      <div className="flex items-start">
                        <Gift className={`w-10 h-10 ${voucher.used ? 'text-gray-400' : 'text-green-500'} mt-0.5 mr-3`} />
                        <div>
                          <div className="flex items-center">
                            <h3 className="font-medium text-gray-900">{voucher.code}</h3>
                            {voucher.used && (
                              <span className="ml-2 bg-gray-100 text-gray-600 text-xs px-2 py-0.5 rounded-full">
                                Used
                              </span>
                            )}
                          </div>
                          <p className="text-sm text-gray-600">
                            {voucher.type === 'percentage' 
                              ? `${voucher.discount}% off` 
                              : `₹${voucher.discount} off`}
                            {voucher.minOrder && ` on orders above ₹${voucher.minOrder}`}
                          </p>
                          <p className="text-xs text-gray-500 mt-1">Expires on {new Date(voucher.expiryDate).toLocaleDateString()}</p>
                        </div>
                      </div>
                      <button 
                        className={`px-4 py-2 rounded-lg text-sm font-medium ${
                          voucher.used
                            ? 'bg-gray-100 text-gray-400 cursor-not-allowed'
                            : 'bg-blue-600 hover:bg-blue-700 text-white'
                        }`}
                        disabled={voucher.used}
                      >
                        {voucher.used ? 'Used' : 'Apply'}
                      </button>
                    </div>
                  </div>
                ))
              )}
            </div>
          </div>
        );
      
      default:
        return null;
    }
  };
  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      {/* Header */}
      <Header />

      <main className="flex-1 max-w-7xl mx-auto w-full px-6 py-8">
        <div className="flex flex-col lg:flex-row gap-6">
          {/* Sidebar */}
          <aside className="w-full lg:w-1/4 bg-white border border-gray-200 rounded-xl p-4 h-fit">
            <div className="mb-4">
              <p className="text-gray-700 text-sm font-medium mb-2">
                My Profile
              </p>
              <p className="text-gray-900 font-semibold">+919748000111</p>
            </div>

            <nav className="flex flex-col space-y-1">
              <button 
                onClick={() => setActiveTab('orders')}
                className={`flex items-center px-4 py-2 rounded-lg font-medium text-left ${
                  activeTab === 'orders' 
                    ? 'bg-blue-50 text-blue-600' 
                    : 'text-gray-700 hover:bg-gray-100'
                }`}
              >
                <Package className="w-4 h-4 mr-2" />
                My Orders
              </button>
              <button 
                onClick={() => setActiveTab('addresses')}
                className={`flex items-center px-4 py-2 rounded-lg font-medium text-left ${
                  activeTab === 'addresses' 
                    ? 'bg-blue-50 text-blue-600' 
                    : 'text-gray-700 hover:bg-gray-100'
                }`}
              >
                <MapPin className="w-4 h-4 mr-2" />
                My Addresses
              </button>
              <button 
                onClick={() => setActiveTab('help')}
                className={`flex items-center px-4 py-2 rounded-lg font-medium text-left ${
                  activeTab === 'help' 
                    ? 'bg-blue-50 text-blue-600' 
                    : 'text-gray-700 hover:bg-gray-100'
                }`}
              >
                <HelpCircle className="w-4 h-4 mr-2" />
                Need Help?
              </button>
              <button 
                onClick={() => setActiveTab('vouchers')}
                className={`flex items-center px-4 py-2 rounded-lg font-medium text-left ${
                  activeTab === 'vouchers' 
                    ? 'bg-blue-50 text-blue-600' 
                    : 'text-gray-700 hover:bg-gray-100'
                }`}
              >
                <Gift className="w-4 h-4 mr-2" />
                Vouchers
              </button>
              <button 
                onClick={() => console.log('Logout clicked')}
                className="flex items-center px-4 py-2 text-red-600 hover:bg-red-50 rounded-lg font-medium text-left"
              >
                <LogOut className="w-4 h-4 mr-2" />
                Logout
              </button>
            </nav>
          </aside>

          {/* Main Content */}
          <section className="flex-1 bg-white border border-gray-200 rounded-xl p-6">
            {activeTab === 'orders' && (
              <>
                <button className="flex items-center text-gray-600 hover:text-gray-900 mb-4">
                  <ArrowLeft className="w-5 h-5 mr-1" />
                  Back
                </button>
                <h2 className="text-xl font-semibold text-gray-900 mb-6">Order Details</h2>
                <p className="text-sm text-gray-500 mb-3">Arrived at 8:20 pm</p>

                <button className="flex items-center text-sm text-blue-600 hover:text-blue-700 mb-6">
                  <Download className="w-4 h-4 mr-1" /> Download Invoice
                </button>

                <h3 className="text-sm font-medium text-gray-700 mb-2">
                  3 items in this order
                </h3>

                <div className="flex items-center border-b border-gray-200 pb-4 mb-6">
                  <img
                    src="https://res.cloudinary.com/demo/image/upload/sample.jpg"
                    alt="Product"
                    className="w-16 h-16 rounded-lg object-cover"
                  />
                  <div className="ml-4 flex-1">
                    <p className="text-sm font-medium text-gray-900">
                      Sebamed Anti-Dandruff Shampoo 200ml | pH 5.5 | Hair & Dandruff
                      Prone Scalp | 50% Reduction in 14 Days
                    </p>
                    <p className="text-sm text-gray-500 mt-1">4 × 300 ml ×1</p>
                  </div>
                  <p className="text-sm font-semibold text-gray-900">₹177</p>
                </div>

                <div className="mb-6">
                  <h3 className="text-sm font-medium text-gray-700 mb-2">
                    Bill details
                  </h3>
                  <div className="space-y-1 text-sm">
                    <div className="flex justify-between text-gray-700">
                      <span>MRP</span>
                      <span>₹187</span>
                    </div>
                    <div className="flex justify-between text-blue-600">
                      <span>Product discount</span>
                      <span>-₹14</span>
                    </div>
                    <div className="flex justify-between text-gray-700">
                      <span>Item total</span>
                      <span>₹173</span>
                    </div>
                    <div className="flex justify-between text-gray-700">
                      <span>Handling charge</span>
                      <span>+₹4</span>
                    </div>
                    <div className="flex justify-between text-gray-700">
                      <span>Delivery charges</span>
                      <span className="text-green-600">FREE</span>
                    </div>

                    <hr className="my-2" />
                    <div className="flex justify-between font-semibold text-gray-900">
                      <span>Bill total</span>
                      <span>₹177</span>
                    </div>
                  </div>
                </div>

                <div className="mb-6">
                  <h3 className="text-sm font-medium text-gray-700 mb-2">
                    Order details
                  </h3>
                  <div className="space-y-1 text-sm text-gray-700">
                    <p>
                      <span className="font-medium">Order ID:</span> ORD57243455206
                    </p>
                    <p>
                      <span className="font-medium">Payment:</span> Paid Online
                    </p>
                    <p>
                      <span className="font-medium">Deliver to:</span> Mollarpara,
                      Jagannathpur, Narendrapur, Kolkata - 700014
                    </p>
                    <p>
                      <span className="font-medium">Order placed:</span> Wed, 16 Jul
                      '25, 8:08 PM
                    </p>
                  </div>
                </div>

                <div className="border-t border-gray-200 pt-4">
                  <h3 className="text-sm font-medium text-gray-700 mb-2">
                    Need help with your order?
                  </h3>
                  <button 
                    onClick={() => setActiveTab('help')}
                    className="flex items-center justify-between w-full bg-gray-50 border border-gray-200 rounded-lg p-3 hover:bg-gray-100 transition"
                  >
                    <div className="flex items-center">
                      <MessageCircle className="w-5 h-5 text-blue-600 mr-3" />
                      <div className="text-left">
                        <p className="text-sm font-medium text-gray-900">
                          Chat with us
                        </p>
                        <p className="text-xs text-gray-500">
                          About any issues related to your order
                        </p>
                      </div>
                    </div>
                    <span className="text-gray-400">&gt;</span>
                  </button>
                </div>
              </>
            )}
            
            {activeTab !== 'orders' && (
              <div className="mb-6">
                <button 
                  onClick={() => setActiveTab('orders')} 
                  className="flex items-center text-gray-600 hover:text-gray-900 mb-4"
                >
                  <ArrowLeft className="w-5 h-5 mr-1" />
                  Back to Orders
                </button>
                {renderTabContent()}
              </div>
            )}
          </section>
        </div>
      </main>

      {/* Footer */}
      <Footer />
    </div>
  );
};

export default Profile;
