import React, { useEffect } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import { CheckCircle } from 'lucide-react';
import Header from './Header';
import Footer from './Footer';

const OrderConfirmation: React.FC = () => {
  const location = useLocation();
  const navigate = useNavigate();

  // Extract order details from navigation state
  const orderDetails = location.state || {};
  const { orderId, total, shippingAddress } = orderDetails;

  // Redirect if no order details
  useEffect(() => {
    if (!orderId) {
      navigate('/');
    }
  }, [orderId, navigate]);

  // If no order details, return null
  if (!orderId) return null;

  return (
    <div className="min-h-screen flex flex-col">
      <Header />
      
      <div className="flex-1 bg-gray-50 py-12 px-4 sm:px-6 lg:px-8">
        <div className="max-w-3xl mx-auto bg-white shadow-lg rounded-lg p-8">
          <div className="text-center">
            <CheckCircle className="mx-auto h-16 w-16 text-green-600 mb-4" />
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Order Confirmed!</h1>
            <p className="text-gray-600 mb-6">Thank you for your purchase.</p>
          </div>

          <div className="grid md:grid-cols-2 gap-6 mb-8">
            {/* Order Details */}
            <div>
              <h2 className="text-xl font-semibold mb-4">Order Details</h2>
              <div className="space-y-2 text-gray-700">
                <p>
                  <span className="font-medium">Order Number:</span> {orderId}
                </p>
                <p>
                  <span className="font-medium">Total Amount:</span> ₹{total.toFixed(2)}
                </p>
                <p>
                  <span className="font-medium">Payment Status:</span>{' '}
                  <span className="text-green-600 font-bold">Paid</span>
                </p>
              </div>
            </div>

            {/* Shipping Address */}
            <div>
              <h2 className="text-xl font-semibold mb-4">Shipping Address</h2>
              <div className="text-gray-700">
                <p>{shippingAddress?.fullName}</p>
                <p>{shippingAddress?.street}</p>
                <p>
                  {shippingAddress?.city}, {shippingAddress?.state}{' '}
                  {shippingAddress?.zipCode}
                </p>
                <p>Phone: {shippingAddress?.phone}</p>
              </div>
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex justify-center space-x-4">
            <button
              onClick={() => navigate('/products')}
              className="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors"
            >
              Continue Shopping
            </button>
            <button
              onClick={() => navigate('/orders')}
              className="bg-gray-100 text-gray-800 px-6 py-3 rounded-lg hover:bg-gray-200 transition-colors"
            >
              View Orders
            </button>
          </div>
        </div>
      </div>
      
      <Footer />
    </div>
  );
};

export default OrderConfirmation;
