import {
  MapPin, Bell, ShoppingCart, Menu, X, User, ChevronDown, Search, UserPlus
} from "lucide-react";
import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { useAuth } from "../../context/AuthContext";
import { useCart } from "../../context/CartContext";
import useLocation from "../../hooks/useLocation";
import useWebSettings from "../../hooks/useWebSettings";
import AccountModal from "./MyAccountModal";
import RegisterModal from "./RegisterModal";
import LocationModal from "../Location/LocationModal";

export default function Header() {
  const [isLocationModalOpen, setLocationModalOpen] = useState(false);
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [isRegisterModalOpen, setRegisterModalOpen] = useState(false);
  const [showAccountModal, setShowAccountModal] = useState(false);

  const navigate = useNavigate();
  const { user, logout } = useAuth();
  const { state: cartState } = useCart();
  const { settings } = useWebSettings();
  const { location, loading: isLocationLoading, selectedStore, detectCurrentLocation } = useLocation();

  const logoUrl = settings?.siteStoreLogo || "";
  const siteName = settings?.siteStoreName || "Healthsta";

  const [displayLocation, setDisplayLocation] = useState("Select delivery location");

  // Update display location when location or selectedStore changes
  useEffect(() => {
    const getLocationText = (loc: any) => {
      if (!loc) return 'Select delivery location';
      
      // Try different possible location name properties
      return (
        loc.displayName || 
        loc.formattedAddress || 
        loc.address ||
        (loc.storeName ? `${loc.storeName}${loc.address ? `, ${loc.address}` : ''}` : '') ||
        (loc.city ? `${loc.city}${loc.state ? `, ${loc.state}` : ''}` : '') ||
        'Selected location'
      );
    };

    if (selectedStore) {
      // Get the full address from the store's location data
      const address = selectedStore.address?.display || 
                     selectedStore.store?.location?.formattedAddress ||
                     selectedStore.storeName || 
                     selectedStore.name || 
                     'Selected store';
      setDisplayLocation(address);
    } else {
      const savedLocation = localStorage.getItem('userLocation');
      
      if (savedLocation) {
        try {
          const locationData = JSON.parse(savedLocation);
          setDisplayLocation(getLocationText(locationData));
        } catch (e) {
          console.error('Error parsing saved location:', e);
          setDisplayLocation('Select delivery location');
        }
      } else if (location) {
        setDisplayLocation(getLocationText(location));
      } else {
        setDisplayLocation('Select delivery location');
      }
    }
  }, [location, selectedStore]);

  // Auto-detect location on component mount if no location is set
  useEffect(() => {
    const savedLocation = localStorage.getItem("userLocation");
    const savedStore = localStorage.getItem("selectedStore");
    
    // If we don't have a saved location or store, try to detect location
    if (!savedLocation && !savedStore) {
      const detectLocation = async () => {
        try {
          await detectCurrentLocation();
        } catch (error) {
          console.error('Failed to detect location:', error);
          // Show location modal if auto-detection fails
          setLocationModalOpen(true);
        }
      };
      
      detectLocation();
    }
  }, [detectCurrentLocation]);

  return (
    <header className="sticky top-0 z-50 bg-white shadow-sm border-b border-gray-200">
      {/* Modals */}
      <LocationModal
        isOpen={isLocationModalOpen}
        onClose={() => {
          setLocationModalOpen(false);
        }}
        onLocationSelect={(location) => {
          // Close the modal only if we have a valid location
          if (location && (location.coordinates || location.filteredStores?.length > 0)) {
            setLocationModalOpen(false);
          }
        }}
      />
      <AccountModal
        isOpen={showAccountModal}
        onClose={() => setShowAccountModal(false)}
        user={user}
        onLogout={logout}
      />
      <RegisterModal
        isOpen={isRegisterModalOpen}
        onClose={() => setRegisterModalOpen(false)}
        onLoginClick={() => {
          setRegisterModalOpen(false);
          setShowAccountModal(true);
        }}
      />

      {/* Top Bar */}
      <div className="flex items-center justify-between px-4 py-2 max-w-7xl mx-auto">
        {/* Logo */}
        <div
          className="flex items-center cursor-pointer"
          onClick={() => navigate("/")}
        >
          {logoUrl ? (
            <img src={logoUrl} alt="logo" className="h-12 w-auto object-contain" />
          ) : (
            <div className="w-12 h-12 bg-blue-700 rounded-full flex items-center justify-center text-white font-bold text-xl">
              {siteName.charAt(0)}
            </div>
          )}
        </div>

        {/* Location */}
        <div 
          className="hidden md:flex items-center cursor-pointer group"
          onClick={(e) => {
            e.stopPropagation();
            setLocationModalOpen(true);
          }}
        >
          <div className="relative">
            <MapPin className={`w-5 h-5 mr-1 ${(location || selectedStore) ? 'text-blue-600' : 'text-gray-600'}`} />
            {(location || selectedStore) && (
              <span className="absolute -top-1 -right-1 w-2 h-2 bg-green-500 rounded-full border-2 border-white"></span>
            )}
          </div>
          <div className="flex flex-col">
            <span className="text-xs text-gray-500">
              {(location || selectedStore) ? 'Delivering to' : 'Select delivery location'}
            </span>
            <div className="flex items-center group-hover:text-blue-700 transition-colors">
              <span className="text-sm font-medium max-w-[180px] truncate">
                {isLocationLoading ? (
                  <span className="flex items-center">
                    <span className="animate-pulse">Detecting location...</span>
                  </span>
                ) : displayLocation ? (
                  <span className="truncate">{displayLocation}</span>
                ) : (
                  <span className="text-gray-600">Select location</span>
                )}
              </span>
              <ChevronDown className="ml-1 w-4 h-4 flex-shrink-0" />
            </div>
          </div>
        </div>

        {/* Search */}
        <div className="hidden md:flex flex-1 mx-6 max-w-lg">
          <div className="relative w-full">
            <input
              type="text"
              placeholder="Search for medicines..."
              className="w-full border border-gray-300 rounded-l-md px-4 py-2 focus:ring-2 focus:ring-blue-500 focus:outline-none"
            />
            <button className="absolute right-0 top-0 h-full px-4 bg-blue-700 text-white rounded-r-md hover:bg-blue-800">
              <Search size={20} />
            </button>
          </div>
        </div>

        {/* Right Icons */}
        <div className="flex items-center gap-4">
          <Bell className="w-6 h-6 text-gray-700 cursor-pointer hidden md:block" />
          <div className="relative">
            <ShoppingCart
              onClick={() => navigate("/cart")}
              className="w-6 h-6 text-gray-700 cursor-pointer"
            />
            {cartState.totalItems > 0 && (
              <span className="absolute -top-2 -right-2 bg-red-500 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center">
                {cartState.totalItems}
              </span>
            )}
          </div>

          {user ? (
            <button
              onClick={() => setShowAccountModal(true)}
              className="flex items-center space-x-1 text-gray-700 hover:text-blue-700"
            >
              <User className="w-5 h-5" />
              <span className="hidden sm:inline">My Account</span>
            </button>
          ) : (
            <div className="flex items-center gap-2">
              <button
                onClick={() => navigate("/login")}
                className="bg-blue-700 text-white px-4 py-2 rounded-full font-medium hover:bg-blue-800 text-sm"
              >
                Sign In
              </button>
              <button
                onClick={() => setRegisterModalOpen(true)}
                className="flex items-center justify-center bg-green-600 text-white w-10 h-10 rounded-full hover:bg-green-700"
              >
                <UserPlus className="w-5 h-5" />
              </button>
            </div>
          )}

          {/* Mobile menu toggle */}
          <button
            className="lg:hidden"
            onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
          >
            {mobileMenuOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
          </button>
        </div>
      </div>

      {/* Categories */}
      <div className="bg-gray-50 border-t border-gray-200">
        <div className="flex space-x-6 justify-center py-2 text-sm text-gray-700">
          {["All", "Medicines", "Healthcare", "Personal Care", "Offers"].map(
            (item) => (
              <a key={item} href="#" className="hover:text-blue-700">
                {item}
              </a>
            )
          )}
        </div>
      </div>

      {/* Mobile Search + Menu */}
      {mobileMenuOpen && (
        <div className="md:hidden bg-white border-t border-gray-200 p-4">
          <div className="relative mb-4">
            <Search className="absolute left-3 top-2.5 text-gray-400" />
            <input
              type="text"
              placeholder="Search for medicines..."
              className="w-full border border-gray-300 rounded-md px-4 py-2 pl-10 focus:ring-2 focus:ring-blue-500"
            />
          </div>
          <div className="space-y-2">
            <a href="#" className="block text-gray-700 hover:text-blue-700">
              Home
            </a>
            <a href="#" className="block text-gray-700 hover:text-blue-700">
              Categories
            </a>
            <a href="#" className="block text-gray-700 hover:text-blue-700">
              Offers
            </a>
            <a href="#" className="block text-gray-700 hover:text-blue-700">
              Contact Us
            </a>
          </div>
        </div>
      )}
    </header>
  );
}
