import React, { useState, useEffect, useCallback, useMemo } from 'react';
import { 
  CreditCard, 
  Shield, 
  Truck, 
  ChevronDown, 
  ChevronUp, 
  Tag,
  X,
  Check,
  MapPin,
  Loader2,
  Upload,
  FileText
} from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import { useAuth } from '../../context/AuthContext';
import { useCart } from '../../context/CartContext';
import Header from './Header';
import Footer from './Footer';
import axios from 'axios';
import { toast } from 'react-hot-toast';

// Define the Address interface
interface Address {
  id?: string;
  _id?: string;
  name: string;
  contactNumber: string;
  address: string;
  city: string;
  state: string;
  country?: string;
  postcode: string;
  landmark?: string;
  isDefault?: boolean;
  is_default?: boolean;
  type?: string;
  location?: {
    type: string;
    coordinates: number[];
  };
}

interface PrescriptionFile {
  file: File;
  preview: string;
  productId: string;
  productName: string;
}

const CheckoutProcess: React.FC = () => {
  const navigate = useNavigate();
  const { state: cartState, clearCart } = useCart();
  const { user } = useAuth();
  
  // State for prescription upload
  const [prescriptionFiles, setPrescriptionFiles] = useState<PrescriptionFile[]>([]);
  const [isUploading, setIsUploading] = useState(false);
  
  // Check if any product requires prescription
  const requiresPrescription = useMemo(() => {
    return cartState.items.some(item => item.requiresPrescription);
  }, [cartState.items]);
  
  // Handle file selection
  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>, productId: string, productName: string) => {
    const file = e.target.files?.[0];
    if (!file) return;
    
    // Check file type (only images and PDFs)
    if (!file.type.match('image/.*') && file.type !== 'application/pdf') {
      toast.error('Please upload an image or PDF file');
      return;
    }
    
    // Check file size (max 5MB)
    if (file.size > 5 * 1024 * 1024) {
      toast.error('File size should be less than 5MB');
      return;
    }
    
    const reader = new FileReader();
    reader.onloadend = () => {
      setPrescriptionFiles(prev => [
        ...prev.filter(pf => pf.productId !== productId),
        {
          file,
          preview: reader.result as string,
          productId,
          productName
        }
      ]);
    };
    reader.readAsDataURL(file);
  };
  
  // Remove prescription file
  const removePrescription = (productId: string) => {
    setPrescriptionFiles(prev => prev.filter(pf => pf.productId !== productId));
  };

  // States for payment, selected address, etc.
  const [active, setActive] = useState<'upi' | 'card' | 'cod'>('cod');
  const [isLoading, setIsLoading] = useState(false);
  const [savedAddresses, setSavedAddresses] = useState<Address[]>([]);
  const [isLoadingAddresses, setIsLoadingAddresses] = useState(false);
  const [selectedAddressId, setSelectedAddressId] = useState<string | null>(null);
  
  // Calculate subtotal and total with coupon discount
  const subtotal = cartState.items.reduce((sum, item) => sum + item.price * item.quantity, 0);
  
  // State for coupon (now optional)
  const [appliedCoupon, setAppliedCoupon] = useState<{
    code: string;
    discount: number;
    type: 'percentage' | 'fixed';
    maxDiscount?: number;
    appliedDiscount: number;
  } | null>(null);

  // Log cart items when they change
  useEffect(() => {
    console.log('Cart Items:', cartState.items);
    console.log('Cart Items Details:', JSON.stringify(cartState.items, null, 2));
  }, [cartState.items]);

  // Load coupon from localStorage on component mount
  useEffect(() => {
    const savedCoupon = localStorage.getItem('appliedCoupon');
    if (savedCoupon) {
      try {
        const parsedCoupon = JSON.parse(savedCoupon);
        // Check if coupon is expired
        if (parsedCoupon.expiryDate && new Date(parsedCoupon.expiryDate) < new Date()) {
          localStorage.removeItem('appliedCoupon');
          setAppliedCoupon(null);
          toast('Your applied coupon has expired.');
        } else {
          setAppliedCoupon(parsedCoupon);
        }
      } catch (error) {
        console.error('Failed to parse applied coupon from localStorage:', error);
        localStorage.removeItem('appliedCoupon');
      }
    }
  }, []);

  // Calculate coupon discount
  const couponDiscount = useMemo(() => {
    if (!appliedCoupon) return 0;
    
    // Check if coupon meets minimum cart value
    if (appliedCoupon.minCartValue && subtotal < appliedCoupon.minCartValue) {
      return 0;
    }
    
    let discount = 0;
    if (appliedCoupon.type === 'percentage') {
      discount = (subtotal * appliedCoupon.discount) / 100;
      // Apply max discount limit if specified
      if (appliedCoupon.maxDiscount !== undefined && discount > appliedCoupon.maxDiscount) {
        discount = appliedCoupon.maxDiscount;
      }
    } else {
      discount = appliedCoupon.discount;
    }
    
    // Ensure discount doesn't exceed cart total
    return Math.min(discount, subtotal);
  }, [appliedCoupon, subtotal]);
  
  const totalAmount = Math.max(0, subtotal - couponDiscount);

  // Accordion toggle for payment methods
  const toggle = (method: 'upi' | 'card' | 'cod') => setActive(active === method ? '' as any : method);

  // Fetch user's saved addresses
  const fetchAddresses = useCallback(async () => {
    if (!user) return;

    setIsLoadingAddresses(true);
    try {
      const token = localStorage.getItem('token');
      const response = await axios.get('http://localhost:5000/api/v1/addresses', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        withCredentials: true
      });

      let addresses: Address[] = [];
      if (Array.isArray(response.data)) {
        addresses = response.data;
      } else if (response.data && Array.isArray(response.data.data)) {
        addresses = response.data.data;
      } else if (response.data?.addresses) {
        addresses = response.data.addresses;
      }

      if (addresses.length === 0) {
        toast.error('No delivery addresses found. Please add an address to continue.');
        return;
      }

      // Ensure all addresses have both id and _id for backward compatibility
      const normalizedAddresses = addresses.map(addr => ({
        ...addr,
        id: addr.id || addr._id || '',
        _id: addr._id || addr.id || ''
      }));

      setSavedAddresses(normalizedAddresses);

      // Select default or first address
      const defaultAddress = normalizedAddresses.find(addr => addr.is_default || addr.isDefault) || normalizedAddresses[0];
      if (defaultAddress) {
        const addressId = defaultAddress._id || defaultAddress.id;
        if (addressId) {
          setSelectedAddressId(addressId);
        } else {
          console.error('Address ID is missing for default address:', defaultAddress);
          toast.error('Error loading address. Please try again.');
        }
      }
    } catch (error) {
      console.error('Error fetching addresses:', error);
      toast.error('Failed to load saved addresses');
    } finally {
      setIsLoadingAddresses(false);
    }
  }, [user]);

  // Handle address selection change
  const handleAddressChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const addressId = e.target.value;
    if (addressId) {
      setSelectedAddressId(addressId);
    } else {
      setSelectedAddressId(null);
    }
  };

  useEffect(() => {
    fetchAddresses();
    
    // Cleanup function to remove coupon when component unmounts if order not placed
    return () => {
      // Only remove if we're not on the order confirmation page
      if (!window.location.pathname.includes('order-confirmation')) {
        localStorage.removeItem('appliedCoupon');
      }
    };
  }, [fetchAddresses]);

  // Place order handler
  const handlePlaceOrder = async () => {
    try {
      // Validation checks
      if (!selectedAddressId) {
        toast.error('Please select a delivery address');
        return;
      }
      
      if (cartState.totalItems === 0) {
        toast.error('Your cart is empty');
        return;
      }

      // Get the full address object to ensure we have the correct _id
      const selectedAddress = savedAddresses.find(
        addr => (addr._id === selectedAddressId || addr.id === selectedAddressId)
      );

      if (!selectedAddress) {
        toast.error('Selected address not found. Please select a valid address.');
        return;
      }

      // Check if any prescription-required items are missing uploads
      const missingPrescriptions = cartState.items
        .filter(item => item.requiresPrescription && 
          !prescriptionFiles.some(pf => pf.productId === (item.id || item._id)));
      
      if (missingPrescriptions.length > 0) {
        const missingItems = missingPrescriptions.map(item => item.name).join(', ');
        toast.error(`Please upload prescriptions for: ${missingItems}`);
        // Scroll to the prescription upload section
        document.getElementById('prescription-upload-section')?.scrollIntoView({ 
          behavior: 'smooth',
          block: 'center'
        });
        return;
      }
      
      setIsLoading(true);
      
      try {
        // Calculate order totals with optional coupon
        const subtotal = cartState.items.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        const couponDiscount = appliedCoupon?.appliedDiscount || 0;
        const totalAmount = Math.max(0, subtotal - couponDiscount); // Ensure total doesn't go below 0
        
        // Get token from localStorage
        const token = localStorage.getItem('token');
        if (!token) {
          toast.error('Please log in to place an order');
          navigate('/login');
          return;
        }
        
        // Validate cart has items
        if (!cartState.items || cartState.items.length === 0) {
          toast.error('Your cart is empty');
          return;
        }

        // Get selected address
        const selectedAddress = savedAddresses.find(addr => addr._id === selectedAddressId || addr.id === selectedAddressId);
        if (!selectedAddress) {
          toast.error('Please select a delivery address');
          return;
        }

        // Prepare order items with proper formatting
        const orderItems = cartState.items.map(item => ({
          item: item.id || item._id,
          product: item.id || item._id, // Some backends expect 'product' instead of 'item'
          name: item.name,
          price: item.price,
          quantity: item.quantity,
          image: item.image || item.images?.[0] || '', // Include image URL if available
          requiresPrescription: item.requiresPrescription || false,
          total: item.price * item.quantity
        }));

        // Get store ID from the first item in cart (assuming all items are from the same store)
        const storeId = cartState.items[0]?.storeId || cartState.items[0]?.store || 'default-store';
        
        // Prepare order data
        const orderData = {
          items: orderItems.map(item => ({
            ...item,
            // Include store ID in each item
            storeId: item.storeId || storeId,
            store: item.storeId || storeId
          })),
          deliveryAddress: {
            _id: selectedAddress._id || selectedAddress.id,
            name: selectedAddress.name,
            address: selectedAddress.address,
            city: selectedAddress.city,
            state: selectedAddress.state,
            country: selectedAddress.country || 'India',
            postcode: selectedAddress.postcode || '',
            contactNumber: selectedAddress.contactNumber,
            landmark: selectedAddress.landmark || '',
            isDefault: selectedAddress.isDefault || selectedAddress.is_default || false,
            type: selectedAddress.type || 'home'
          },
          paymentMethod: active, // Use the active state for payment method
          itemsPrice: subtotal,
          taxPrice: 0,
          shippingPrice: 0,
          discountAmount: couponDiscount,
          totalPrice: totalAmount,
          status: 'pending_payment',
          store: storeId, // Use the store ID from the first item
          notes: '', // Add empty string as default for order notes
          // Add coupon data only if a coupon is applied
          ...(appliedCoupon ? {
            coupon: {
              code: appliedCoupon.code,
              discount: appliedCoupon.discount,
              type: appliedCoupon.type,
              ...(appliedCoupon.maxDiscount !== undefined && { maxDiscount: appliedCoupon.maxDiscount }),
              appliedDiscount: couponDiscount
            }
          } : {})
        };
        
        // Create FormData for the request
        const formData = new FormData();
        formData.append('order', JSON.stringify(orderData));
        
        // Add prescription files to FormData if any
        prescriptionFiles.forEach((file, index) => {
          formData.append(`prescription_${index}`, file.file);
          formData.append(`prescription_${index}_productId`, file.productId);
        });
        
        // Make the API call to create order with prescription files
        const orderResponse = await axios.post('http://localhost:5000/api/v1/orders', formData, {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'multipart/form-data'
          }
        });
        
        // Handle successful order creation
        if (orderResponse.data.success) {
          // Clear cart and local storage
          clearCart();
          localStorage.removeItem('appliedCoupon');
          setPrescriptionFiles([]);
          
          // Redirect to order confirmation page
          navigate(`/order-confirmation/${orderResponse.data.data._id}`);
        } else {
          throw new Error(orderResponse.data.message || 'Failed to place order');
        }
      } catch (error) {
        console.error('Error placing order:', error);
        const errorMessage = error.response?.data?.message || error.message || 'Failed to place order';
        toast.error(errorMessage);
      } finally {
        setIsLoading(false);
        setIsUploading(false);
      }
    } catch (error) {
      console.error('Unexpected error in handlePlaceOrder:', error);
      toast.error('An unexpected error occurred. Please try again.');
      setIsLoading(false);
      setIsUploading(false);
    }

      // Check if any prescription-required items are missing uploads
      const missingPrescriptions = cartState.items
        .filter(item => item.requiresPrescription && 
          !prescriptionFiles.some(pf => pf.productId === (item.id || item._id)));
      
      if (missingPrescriptions.length > 0) {
        const missingItems = missingPrescriptions.map(item => item.name).join(', ');
        toast.error(`Please upload prescriptions for: ${missingItems}`);
        // Scroll to the prescription upload section
        document.getElementById('prescription-upload-section')?.scrollIntoView({ 
          behavior: 'smooth',
          block: 'center'
        });
        return;
      }

      // Prepare order items with proper types and handle missing fields
      const orderItemPromises = cartState.items.map(async (item) => {
        // Ensure we have a valid item
        if (!item) {
          console.error('Invalid item in cart:', item);
          return null;
        }

        // Ensure we have required fields
        const itemId = item.id || item._id;
        if (!itemId) {
          console.error('Item missing id:', item);
          return null;
        }

        const prescriptionFile = prescriptionFiles.find(pf => pf.productId === itemId);
        
        // Create base item data
        const itemData = {
          id: itemId, // Ensure we have an id
          _id: itemId, // Include _id for backward compatibility
          name: item.name || 'Unnamed Product',
          price: item.price || 0,
          quantity: item.quantity || 1,
          // Include additional fields with defaults
          image: item.image || '',
          originalPrice: item.originalPrice || item.price || 0,
          storeId: item.storeId || 'default-store',
          storeName: item.storeName || 'Default Store',
          store: item.storeId || 'default-store',
          unit: item.unit || 'pcs',
          taxAmount: item.taxAmount || 0,
          discountAmount: item.discountAmount || 0,
          variantId: item.variantId || null,
          brand: item.brand || 'Generic',
          category: item.category || 'General',
          requiresPrescription: item.requiresPrescription || false
        };
        
        // Add prescription data if exists
        if (prescriptionFile) {
          // Convert file to base64
          const base64String = await new Promise<string>((resolve, reject) => {
            const reader = new FileReader();
            reader.onload = () => {
              if (reader.result) {
                resolve(reader.result.toString());
              } else {
                reject(new Error('Failed to read file'));
              }
            };
            reader.onerror = () => reject(reader.error);
            reader.readAsDataURL(prescriptionFile.file);
          });

          return {
            ...itemData,
            prescription: {
              fileName: prescriptionFile.file.name,
              fileType: prescriptionFile.file.type,
              fileSize: prescriptionFile.file.size,
              file: base64String, // This is the base64-encoded image
              productId: itemId,
              productName: item.name || 'Unnamed Product',
              images: [{
                url: base64String,
                fileName: prescriptionFile.file.name,
                fileType: prescriptionFile.file.type,
                uploadedAt: new Date().toISOString()
              }]
            }
          };
        }
        
        return itemData;
      }).filter(Boolean); // Remove any null items from the array

      // Get store information from the first item in the cart
      const firstItem = cartState.items[0];
      const storeInfo = {
        storeId: firstItem?.storeId || 'default-store',
        storeName: firstItem?.storeName || 'Default Store'
      };

      // Prepare order data with required address fields
      // First create the order with a temporary status
      const orderData = {
        items: await Promise.all(cartState.items.map(async (item) => {
          const itemId = item.id || item._id;
          if (!itemId) return null;

          const prescriptionFile = prescriptionFiles.find(pf => pf.productId === itemId);
          
          // Create base item data
          const itemData = {
            id: itemId, // Ensure we have an id
            _id: itemId, // Include _id for backward compatibility
            name: item.name || 'Unnamed Product',
            price: item.price || 0,
            quantity: item.quantity || 1,
            // Include additional fields with defaults
            image: item.image || '',
            originalPrice: item.originalPrice || item.price || 0,
            storeId: item.storeId || 'default-store',
            storeName: item.storeName || 'Default Store',
            store: item.storeId || 'default-store',
            unit: item.unit || 'pcs',
            taxAmount: item.taxAmount || 0,
            discountAmount: item.discountAmount || 0,
            variantId: item.variantId || null,
            brand: item.brand || 'Generic',
            category: item.category || 'General',
            requiresPrescription: item.requiresPrescription || false
          };
          
          // Add prescription data if exists
          if (prescriptionFile) {
            // Convert file to base64
            const base64String = await new Promise<string>((resolve, reject) => {
              const reader = new FileReader();
              reader.onload = () => {
                if (reader.result) {
                  resolve(reader.result.toString());
                } else {
                  reject(new Error('Failed to read file'));
                }
              };
              reader.onerror = () => reject(reader.error);
              reader.readAsDataURL(prescriptionFile.file);
            });

            return {
              ...itemData,
              prescription: {
                fileName: prescriptionFile.file.name,
                fileType: prescriptionFile.file.type,
                fileSize: prescriptionFile.file.size,
                file: base64String, // This is the base64-encoded image
                productId: itemId,
                productName: item.name || 'Unnamed Product',
                images: [{
                  url: base64String,
                  fileName: prescriptionFile.file.name,
                  fileType: prescriptionFile.file.type,
                  uploadedAt: new Date().toISOString()
                }]
              }
            };
          }
          
          return itemData;
        }).filter(Boolean)), // Remove any null items from the array
        store: storeInfo.storeId, // Include store ID at order level
        storeName: storeInfo.storeName, // Include store name at root level
        storeInfo: {  // Include store information object
          storeId: storeInfo.storeId,
          storeName: storeInfo.storeName
        },
        deliveryAddress: {
          _id: selectedAddress._id || selectedAddress.id,
          name: selectedAddress.name,
          address: selectedAddress.address,
          city: selectedAddress.city,
          state: selectedAddress.state,
          country: selectedAddress.country || 'India',
          postcode: selectedAddress.postcode || '',
          contactNumber: selectedAddress.contactNumber,
          landmark: selectedAddress.landmark || '',
          isDefault: selectedAddress.isDefault || selectedAddress.is_default || false,
          type: selectedAddress.type || 'home'
        },
        paymentMethod: active,
        itemsPrice: subtotal,
        taxPrice: 0,
        shippingPrice: 0,
        discountAmount: couponDiscount,
        totalPrice: totalAmount,
        status: 'pending_prescription', // Initial status
        coupon: localCoupon ? {
          code: localCoupon.code,
          discount: localCoupon.discount,
          type: localCoupon.type,
          maxDiscount: localCoupon.maxDiscount,
          appliedDiscount: couponDiscount
        } : undefined
      };

      // Get token from localStorage at the beginning
      const token = localStorage.getItem('token');
      if (!token) {
        toast.error('Please log in to place an order');
        navigate('/login');
        return;
      }

      try {
        // Create the order first to get an order ID
        const orderResponse = await axios.post('http://localhost:5000/api/v1/orders', orderData, {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });

        // Make sure we have a valid order ID from the response
        const orderId = orderResponse.data.data?._id || orderResponse.data.orderId;
        if (!orderId) {
          console.error('Failed to get order ID from response:', orderResponse.data);
          throw new Error('Failed to create order: No order ID returned from server');
        }

        // Upload prescription files if any
        const prescriptionUploads: Array<{
        productId: string;
        fileUrl: string;
        fileName: string;
      }> = [];
      if (prescriptionFiles.length > 0) {
        try {
          setIsUploading(true);
          
          await Promise.all(
            prescriptionFiles.map(async (prescription) => {
              const formData = new FormData();
              // Append the file with the correct field name that the backend expects
              formData.append('file', prescription.file);
              formData.append('productId', prescription.productId);
              formData.append('orderId', orderId);
              formData.append('storeId', storeInfo.storeId);
              
              // Add additional metadata that might be needed
              const product = cartState.items.find(item => (item.id || item._id) === prescription.productId);
              if (product) {
                formData.append('productName', product.name || 'Prescription');
              }
              
              console.log('Uploading prescription file:', {
                fileName: prescription.file.name,
                size: prescription.file.size,
                type: prescription.file.type
              });
              
              try {
                const response = await axios.post('http://localhost:5000/api/v1/prescriptions/upload', formData, {
                  headers: {
                    'Content-Type': 'multipart/form-data',
                    'Authorization': `Bearer ${token}`,
                    'Accept': 'application/json'
                  },
                  // Add timeout and onUploadProgress if needed
                  // timeout: 30000,
                  // onUploadProgress: (progressEvent) => {
                  //   const percentCompleted = Math.round((progressEvent.loaded * 100) / progressEvent.total);
                  //   console.log(`Upload progress: ${percentCompleted}%`);
                  // }
                });
                
                console.log('Upload response:', response.data);
                
                if (response.data.success && response.data.fileUrl) {
                  prescriptionUploads.push({
                    productId: prescription.productId,
                    fileUrl: response.data.fileUrl,
                    fileName: prescription.file.name
                  });
                } else {
                  console.error('Upload failed:', response.data);
                  throw new Error(response.data.message || 'Failed to upload prescription');
                }
              } catch (uploadError) {
                console.error('Error uploading file:', uploadError);
                throw uploadError;
              }
            })
          );

          // Update order status to confirmed after successful uploads
          await axios.put(
            `http://localhost:5000/api/v1/orders/${orderId}/status`,
            { status: 'confirmed' },
            { 
              headers: { 
                'Authorization': `Bearer ${token}`,
                'Content-Type': 'application/json'
              } 
            }
          );

        } catch (error) {
          console.error('Error processing order:', error);
          
          // Update order status to failed if there's an error
          try {
            if (orderId) {
              await axios.put(
                `http://localhost:5000/api/v1/orders/${orderId}/status`,
                { 
                  status: 'failed',
                  statusMessage: 'Failed to process prescription uploads'
                },
                { 
                  headers: { 
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                  } 
                }
              );
            }
          } catch (updateError) {
            console.error('Failed to update order status:', updateError);
          }
          
          throw error; // Re-throw to be caught by the outer catch
        } finally {
          setIsUploading(false);
        }
      }

      // Update the order with the prescription information
      const updatedOrderData = {
        ...orderData,
        status: 'confirmed',
        prescriptions: prescriptionUploads.map(upload => ({
          productId: upload.productId,
          fileUrl: upload.fileUrl,
          fileName: upload.fileName
        }))
      };

      // Update the order with prescription information
      await axios.put(
        `http://localhost:5000/api/v1/orders/${orderId}`,
        updatedOrderData,
        { 
          headers: { 
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          } 
        }
      );

      // Create FormData for the request
      const formData = new FormData();
      
      // Append order data as JSON
      formData.append('order', JSON.stringify(orderData));
      
      // Append prescription files if any
      if (prescriptionFiles && prescriptionFiles.length > 0) {
        prescriptionFiles.forEach((file, index) => {
          formData.append(`prescription_${index}`, file.file);
          formData.append(`prescription_${index}_productId`, file.productId);
        });
      }
      
      // Submit the order with prescription files
      const response = await axios.post('http://localhost:5000/api/v1/orders', formData, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'multipart/form-data'
        }
      });
      
      // Clear cart, coupon, and prescription files on successful order
      clearCart();
      localStorage.removeItem('appliedCoupon');
      setPrescriptionFiles([]);
      
      // Redirect to order confirmation
      if (response?.data?.orderId || response?.data?.data?.order?._id) {
        const orderId = response.data.orderId || response.data.data.order._id;
        navigate(`/order-confirmation/${orderId}`);
      } else {
        throw new Error('Invalid response format from server');
      }
    } catch (error) {
      console.error('Error placing order:', error);
      let errorMessage = 'Failed to place order. Please try again.';
      
      if (axios.isAxiosError(error)) {
        errorMessage = error.response?.data?.message || error.message || errorMessage;
      } else if (error instanceof Error) {
        errorMessage = error.message;
      }
      
      toast.error(errorMessage);
    } finally {
      setIsLoading(false);
      setIsUploading(false);
    }
  };

// Render prescription upload section
  const renderPrescriptionUploads = () => {
    if (!requiresPrescription) return null;
    
    return (
      <div id="prescription-upload-section" className="bg-white rounded-xl border p-6 space-y-4">
        <h3 className="text-lg font-semibold text-gray-800">Prescription Required</h3>
        <p className="text-sm text-gray-600 mb-4">
          The following items require a prescription. Please upload a clear photo or scan of your prescription.
        </p>
        
        <div className="space-y-4">
          {cartState.items
            .filter(item => item.requiresPrescription)
            .map(item => {
              const uploadedFile = prescriptionFiles.find(pf => pf.productId === (item.id || item._id));
              return (
                <div key={item.id || item._id} className="border rounded-lg p-4">
                  <div className="flex justify-between items-center mb-2">
                    <span className="font-medium">{item.name}</span>
                    {uploadedFile ? (
                      <span className="text-green-600 text-sm flex items-center">
                        <Check className="w-4 h-4 mr-1" /> Uploaded
                      </span>
                    ) : (
                      <span className="text-yellow-600 text-sm">Required</span>
                    )}
                  </div>
                  
                  {uploadedFile ? (
                    <div className="mt-2 flex items-center justify-between bg-gray-50 p-2 rounded">
                      <div className="flex items-center">
                        <FileText className="text-blue-500 mr-2" />
                        <span className="text-sm text-gray-700 truncate max-w-xs">
                          {uploadedFile.file.name}
                        </span>
                      </div>
                      <button
                        onClick={() => removePrescription(item.id || item._id)}
                        className="text-red-500 hover:text-red-700"
                        type="button"
                      >
                        <X size={16} />
                      </button>
                    </div>
                  ) : (
                    <label className="mt-2 flex flex-col items-center px-4 py-6 bg-white text-blue-600 rounded-lg border-2 border-dashed border-blue-300 cursor-pointer hover:bg-blue-50">
                      <Upload className="w-8 h-8 mb-2" />
                      <span className="text-sm text-center">
                        Click to upload prescription for {item.name}
                        <span className="block text-xs text-gray-500 mt-1">
                          (JPG, PNG or PDF, max 5MB)
                        </span>
                      </span>
                      <input
                        type="file"
                        className="hidden"
                        accept="image/*,.pdf"
                        onChange={(e) => handleFileChange(e, item.id || item._id, item.name)}
                      />
                    </label>
                  )}
                </div>
              );
            })}
        </div>
      </div>
    );
  };

  return (
    <div className="min-h-screen bg-white flex flex-col">
      <Header />
      
      <main className="flex-1">
        <div className="max-w-7xl mx-auto py-10 px-4 grid lg:grid-cols-3 gap-6">
          {/* Left Section */}
          <div className="lg:col-span-2 space-y-6">
            {/* Prescription Upload Section */}
            {requiresPrescription && renderPrescriptionUploads()}
            
            {/* Payment Methods */}
            <div className="space-y-6">
              <h2 className="text-xl font-semibold text-gray-800">Payment Method</h2>
              {/* UPI */}
              <div className="border rounded-xl">
                <button
                  onClick={() => toggle('upi')}
                  className="w-full flex justify-between items-center px-6 py-4 text-left font-medium"
                >
                  <div className="flex items-center gap-3">
                    <div className="w-8 h-8 rounded-full bg-blue-50 flex items-center justify-center">
                      <CreditCard className="w-4 h-4 text-blue-600" />
                    </div>
                    <span>UPI</span>
                  </div>
                  {active === 'upi' ? <ChevronUp /> : <ChevronDown />}
                </button>
                {active === 'upi' && (
                  <div className="border-t px-6 py-4 space-y-3">
                    <div className="flex justify-between items-center text-sm">
                      <span className="text-gray-600">Scan & Pay</span>
                      <button className="text-blue-600 font-medium">Show QR code</button>
                    </div>
                    <input
                      type="text"
                      placeholder="username@bank"
                      className="w-full border rounded-lg px-4 py-2 text-sm"
                    />
                    <button className="bg-blue-600 text-white px-4 py-2 rounded-lg text-sm w-full">
                      Verify & Pay
                    </button>
                    <label className="flex items-center gap-2 text-xs text-gray-600 mt-2">
                      <input type="checkbox" className="rounded" /> 
                      Save this ID for faster checkout
                    </label>
                  </div>
                )}
              </div>
              {/* Debit/Credit Card */}
              <div className="border rounded-xl">
                <button
                  onClick={() => toggle('card')}
                  className="w-full flex justify-between items-center px-6 py-4 text-left font-medium"
                >
                  <div className="flex items-center gap-3">
                    <div className="w-8 h-8 rounded-full bg-blue-50 flex items-center justify-center">
                      <CreditCard className="w-4 h-4 text-blue-600" />
                    </div>
                    <span>Debit/Credit cards</span>
                  </div>
                  {active === 'card' ? <ChevronUp /> : <ChevronDown />}
                </button>
                {active === 'card' && (
                  <div className="border-t px-6 py-4 space-y-3">
                    <label className="block text-sm text-gray-700">Card Number</label>
                    <input
                      type="text"
                      placeholder="Card Number"
                      className="w-full border rounded-lg px-4 py-2 text-sm"
                    />
                    <label className="block text-sm text-gray-700">Card Holder's Name</label>
                    <input
                      type="text"
                      placeholder="Name on card"
                      className="w-full border rounded-lg px-4 py-2 text-sm"
                    />
                    <div className="grid grid-cols-2 gap-3">
                      <div>
                        <label className="block text-sm text-gray-700 mb-1">Expiry</label>
                        <input
                          type="text"
                          placeholder="MM/YY"
                          className="w-full border rounded-lg px-4 py-2 text-sm"
                        />
                      </div>
                      <div>
                        <label className="block text-sm text-gray-700 mb-1">CVV</label>
                        <input
                          type="password"
                          placeholder="123"
                          className="w-full border rounded-lg px-4 py-2 text-sm"
                        />
                      </div>
                    </div>
                    <label className="flex items-center gap-2 text-xs text-gray-600 mt-2">
                      <input type="checkbox" className="rounded" /> 
                      Save this card for faster checkout
                    </label>
                  </div>
                )}
              </div>
              {/* Cash on Delivery */}
              <div className="border rounded-xl">
                <button
                  onClick={() => toggle('cod')}
                  className="w-full flex justify-between items-center px-6 py-4 text-left font-medium"
                >
                  <div className="flex items-center gap-3">
                    <div className="w-8 h-8 rounded-full bg-blue-50 flex items-center justify-center">
                      <Truck className="w-4 h-4 text-blue-600" />
                    </div>
                    <span>Pay on Delivery</span>
                  </div>
                  {active === 'cod' ? <ChevronUp /> : <ChevronDown />}
                </button>
                {active === 'cod' && (
                  <div className="border-t px-6 py-4 space-y-3">
                    <div className="text-sm text-gray-600 bg-blue-50 border border-blue-100 rounded-lg p-3">
                      Total amount to pay: <b>₹{totalAmount.toFixed(2)}</b>
                    </div>
                    <button 
                      onClick={handlePlaceOrder}
                      disabled={!selectedAddressId || cartState.totalItems === 0 || isLoading}
                      className={`bg-blue-600 text-white px-5 py-2 rounded-lg text-sm w-full hover:bg-blue-700 transition-colors ${
                        !selectedAddressId || cartState.totalItems === 0 || isLoading ? 'opacity-70 cursor-not-allowed' : ''
                      }`}
                    >
                      {isLoading ? (
                        'Processing...'
                      ) : !selectedAddressId ? (
                        'Select Delivery Address'
                      ) : cartState.totalItems > 0 ? (
                        `Place Order (${cartState.totalItems} ${cartState.totalItems === 1 ? 'item' : 'items'})`
                      ) : (
                        'Your cart is empty'
                      )}
                    </button>
                  </div>
                )}
              </div>
            </div>
          </div>
          {/* Right Section - Summary */}
          <div className="lg:col-span-1 space-y-6">
            {/* Delivery Address */}
            <div className="border rounded-xl p-6">
              <h3 className="text-lg font-semibold text-gray-800 mb-4 flex items-center gap-2">
                <MapPin className="w-5 h-5" />
                Delivery Address
              </h3>
              {isLoadingAddresses ? (
                <div className="flex justify-center py-4">
                  <Loader2 className="w-6 h-6 animate-spin text-blue-600" />
                </div>
              ) : savedAddresses.length > 0 ? (
                <div className="space-y-4">
                  <select
                    className="w-full p-3 border rounded-lg text-sm"
                    value={selectedAddressId || ''}
                    onChange={handleAddressChange}
                  >
                    <option value="">Select a delivery address</option>
                    {savedAddresses.map((address) => {
                      const addressId = address._id || address.id;
                      return (
                        <option key={addressId} value={addressId}>
                          {address.name} - {address.address.slice(0, 30)}...
                        </option>
                      );
                    })}
                  </select>
                  {selectedAddressId && (
                    <div className="text-sm text-gray-600 p-3 bg-gray-50 rounded-lg">
                      {savedAddresses.find(a => (a._id || a.id) === selectedAddressId)?.address}
                    </div>
                  )}
                </div>
              ) : (
                <div className="text-center py-4">
                  <p className="text-gray-500 mb-3">No saved addresses found</p>
                  <button
                    onClick={() => navigate('/profile/addresses')}
                    className="text-blue-600 hover:text-blue-800 text-sm font-medium"
                  >
                    Add Address
                  </button>
                </div>
              )}
            </div>

            {/* Order Summary */}
            <div className="border rounded-xl p-6">
              <h3 className="text-lg font-semibold text-gray-800 mb-4">
                Order Summary
              </h3>
              
              {/* Cart Items */}
              <div className="mb-6 space-y-4 max-h-80 overflow-y-auto pr-2">
                {cartState.items.map((item) => (
                  <div key={`${item.id}-${item.variantId || ''}`} className="flex items-start gap-3 pb-3 border-b">
                    <div className="w-16 h-16 bg-gray-100 rounded-lg overflow-hidden flex-shrink-0">
                      <img 
                        src={item.image || '/placeholder-product.png'} 
                        alt={item.name}
                        className="w-full h-full object-cover"
                        onError={(e) => {
                          const target = e.target as HTMLImageElement;
                          target.src = '/placeholder-product.png';
                        }}
                      />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex justify-between items-start">
                        <h4 className="font-medium text-gray-800 truncate">{item.name}</h4>
                        <span className="font-medium ml-2 whitespace-nowrap">₹{(item.price * item.quantity).toFixed(2)}</span>
                      </div>
                      {item.storeName && (
                        <p className="text-xs text-gray-500 mt-1">
                          Sold by: {item.storeName}
                        </p>
                      )}
                      <div className="flex items-center justify-between mt-1 text-sm text-gray-500">
                        <span>Qty: {item.quantity}</span>
                        <span className="text-xs line-through text-gray-400">
                          {item.originalPrice && item.originalPrice > item.price ? 
                            `₹${(item.originalPrice * item.quantity).toFixed(2)}` : ''}
                        </span>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
              
              <div className="space-y-3">
                <div className="flex justify-between text-sm text-gray-600">
                  <span>Subtotal ({cartState.totalItems} {cartState.totalItems === 1 ? 'item' : 'items'})</span>
                  <span>₹{subtotal.toFixed(2)}</span>
                </div>
                
                {appliedCoupon && (
                  <div className="space-y-2">
                    <div className="flex justify-between items-center text-sm">
                      <div className="flex items-center text-green-600">
                        <span>Coupon Applied: {appliedCoupon.code}</span>
                        <button 
                          onClick={() => {
                            localStorage.removeItem('appliedCoupon');
                            setAppliedCoupon(null);
                            toast.success('Coupon removed successfully');
                          }}
                          className="ml-2 text-red-500 hover:text-red-700"
                          title="Remove coupon"
                        >
                          <X size={14} />
                        </button>
                      </div>
                      <span className="text-green-600">-₹{couponDiscount.toFixed(2)}</span>
                    </div>
                    <div className="text-xs text-gray-500">
                      {localCoupon.type === 'percentage' 
                        ? `${localCoupon.discount}% off` 
                        : `₹${localCoupon.discount} off`}
                      {localCoupon.maxDiscount && localCoupon.type === 'percentage' && 
                        ` (max ₹${localCoupon.maxDiscount})`}
                    </div>
                  </div>
                )}

                <div className="border-t pt-3 flex justify-between font-medium">
                  <span>Total Amount</span>
                  <span className="text-lg">₹{totalAmount.toFixed(2)}</span>
                </div>
              </div>
            </div>

            {/* Secure Payment Info */}
            <div className="border rounded-xl p-4 text-center">
              <div className="flex items-center justify-center gap-2 text-gray-500 text-sm">
                <Shield className="w-4 h-4" />
                <span>100% Secure Payment</span>
              </div>
            </div>
          </div>
        </div>
      </main>
      <Footer />
    </div>
  );
}

export default CheckoutProcess;
