import React, { useState, useEffect, useRef, useCallback } from 'react';
import { X, Search, MapPin, User, AlertCircle } from 'lucide-react';
import { useNavigate } from 'react-router-dom';
import axios from 'axios';
import { useAuth } from '../../context/AuthContext';

// Configure axios to use base URL and handle errors globally
const api = axios.create({
  baseURL: import.meta.env.VITE_API_BASE_URL || 'http://localhost:5000/api/v1',
  headers: {
    'Content-Type': 'application/json'
  }
});

// Google Maps API Key - ensure this is set in your .env file
const GOOGLE_MAPS_API_KEY = import.meta.env.VITE_GOOGLE_MAPS_API_KEY || 'AIzaSyCJKwxaSS0glDtxXMX37uHX_KHUEleCMk0';

// Add Google Maps types
declare global {
  interface Window {
    initMap: () => void;
    google: typeof google;
  }
}

// Interceptor for request logging
api.interceptors.request.use(
  (config) => {
    console.log('API Request:', {
      method: config.method,
      url: config.url,
      data: config.data,
      headers: config.headers
    });
    return config;
  },
  (error) => {
    console.error('API Request Error:', error);
    return Promise.reject(error);
  }
);

// Interceptor for response logging
api.interceptors.response.use(
  (response) => {
    console.log('API Response:', {
      status: response.status,
      data: response.data
    });
    return response;
  },
  (error) => {
    console.error('API Response Error:', {
      status: error.response?.status,
      data: error.response?.data,
      message: error.message
    });
    return Promise.reject(error);
  }
);

// Function to fetch pincode data from external API
const fetchPincodeData = async (pincode: string) => {
  try {
    // First try to get from local storage cache
    const cacheKey = `pincode_${pincode}`;
    const cachedData = localStorage.getItem(cacheKey);
    
    if (cachedData) {
      return JSON.parse(cachedData);
    }

    // If not in cache, fetch from API
    const response = await fetch(`https://api.postalpincode.in/pincode/${pincode}`, {
      headers: {
        'Accept': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error('Failed to fetch pincode data');
    }

    const data = await response.json();
    
    if (!data || data[0].Status === 'Error') {
      throw new Error(data[0].Message || 'Invalid pincode');
    }

    const postOffices = data[0].PostOffice || [];
    if (postOffices.length === 0) {
      throw new Error('No address found for this pincode');
    }

    // Get the first post office as primary
    const firstOffice = postOffices[0];
    const pincodeData = {
      area: firstOffice.Name,
      city: firstOffice.District || firstOffice.Region || firstOffice.State,
      state: firstOffice.State,
      country: firstOffice.Country || 'India',
      // Use approximate coordinates for the region (these are fallback values)
      latitude: 0,
      longitude: 0,
      mapDescription: `${firstOffice.Name}, ${firstOffice.District || ''}`.trim()
    };

    // Cache the result
    localStorage.setItem(cacheKey, JSON.stringify(pincodeData));
    
    return pincodeData;
  } catch (error) {
    console.error('Error fetching pincode data:', error);
    throw new Error('Failed to fetch pincode details. Please check the pincode and try again.');
  }
};

interface AddressModalProps {
  isOpen: boolean;
  onClose: () => void;
  onAddressAdded?: (address: any) => void;
}

const AddressModal: React.FC<AddressModalProps> = ({ isOpen, onClose, onAddressAdded }) => {
  const [stage, setStage] = useState<'initial' | 'search' | 'details' | 'map' | 'detailed-address' | 'recipient'>('initial');
  const [isVisible, setIsVisible] = useState(false);
  const [selectedPincode, setSelectedPincode] = useState('');
  const [searchQuery, setSearchQuery] = useState('');
  const [addressDetails, setAddressDetails] = useState({
    houseNumber: '',
    landmark: '',
    area: '',
    city: '',
    state: ''
  });
  const [mapDetails, setMapDetails] = useState({
    latitude: 0,
    longitude: 0,
    description: ''
  });
  const { user } = useAuth();
  const navigate = useNavigate();
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [recipientDetails, setRecipientDetails] = useState({
    name: '',
    phoneNumber: ''
  });
  const [googleMapsLoaded, setGoogleMapsLoaded] = useState(false);
  const [selectedLocation, setSelectedLocation] = useState<{lat: number, lng: number} | null>(null);
  const mapRef = useRef<HTMLDivElement>(null);
  const googleMapRef = useRef<google.maps.Map | null>(null);
  const markerRef = useRef<google.maps.Marker | null>(null);

  useEffect(() => {
    if (isOpen) {
      const timer = setTimeout(() => setIsVisible(true), 10);
      return () => clearTimeout(timer);
    } else {
      setIsVisible(false);
    }
  }, [isOpen]);

  // Handle map click event
  const handleMapClick = useCallback((event: google.maps.MapMouseEvent) => {
    if (event.latLng) {
      const location = {
        lat: event.latLng.lat(),
        lng: event.latLng.lng()
      };

      setSelectedLocation(location);
      setMapDetails(prev => ({
        ...prev,
        latitude: location.lat,
        longitude: location.lng
      }));

      // Reverse geocode to get address
      reverseGeocode(location);
    }
  }, []);

  // Reverse geocode function to get address from coordinates
  const reverseGeocode = useCallback(async (location: {lat: number, lng: number}) => {
    if (!window.google || !window.google.maps) return;

    try {
      const geocoder = new window.google.maps.Geocoder();
      const latlng = { lat: location.lat, lng: location.lng };

      geocoder.geocode({ location: latlng }, (results, status) => {
        if (status === 'OK' && results && results[0]) {
          const address = results[0];
          const addressComponents = address.address_components;

          // Extract address components
          let area = '', city = '', state = '', pincode = '';

          addressComponents.forEach(component => {
            const types = component.types;
            if (types.includes('sublocality') || types.includes('neighborhood')) {
              area = component.long_name;
            }
            if (types.includes('locality') || types.includes('administrative_area_level_3')) {
              city = component.long_name;
            }
            if (types.includes('administrative_area_level_1')) {
              state = component.long_name;
            }
            if (types.includes('postal_code')) {
              pincode = component.long_name;
            }
          });

          // Update address details
          setAddressDetails(prev => ({
            ...prev,
            area: area || prev.area,
            city: city || prev.city,
            state: state || prev.state
          }));

          // Update pincode if found
          if (pincode && pincode.length === 6) {
            setSelectedPincode(pincode);
          }

          // Update map description
          setMapDetails(prev => ({
            ...prev,
            description: address.formatted_address
          }));
        } else {
          console.error('Geocoder failed:', status);
          setError('Failed to get address from selected location');
        }
      });
    } catch (error) {
      console.error('Reverse geocoding error:', error);
      setError('Failed to get address from selected location');
    }
  }, []);

  // Load Google Maps API script
  useEffect(() => {
    if (!GOOGLE_MAPS_API_KEY) {
      console.error('Google Maps API key is not set');
      setError('Google Maps API key is not configured');
      return;
    }

    if (!isOpen) return;

    const loadGoogleMapsScript = () => {
      // Check if script is already added
      if (document.querySelector(`script[src*="maps.googleapis.com/maps/api/js"]`)) {
        if (window.google && window.google.maps) {
          setGoogleMapsLoaded(true);
        }
        return;
      }

      // Check if already loaded
      if (window.google && window.google.maps) {
        setGoogleMapsLoaded(true);
        return;
      }

      // Create script element
      const script = document.createElement('script');
      script.src = `https://maps.googleapis.com/maps/api/js?key=${GOOGLE_MAPS_API_KEY}&libraries=places&callback=initMap`;
      script.async = true;
      script.defer = true;
      
      // Add global init function
      window.initMap = () => {
        console.log('Google Maps script loaded successfully');
        setGoogleMapsLoaded(true);
      };

      script.onerror = (error) => {
        console.error('Error loading Google Maps script:', error);
        setError('Failed to load Google Maps. Please check your internet connection and try again.');
      };

      document.head.appendChild(script);
      console.log('Google Maps script added to document');
    };

    loadGoogleMapsScript();

    return () => {
      // Cleanup
      if (window.google && window.google.maps) {
        // No need to remove the script as it's cached
      }
    };
  }, [isOpen]);

  // Initialize map when Google Maps is loaded and we have coordinates
  useEffect(() => {
    if (!googleMapsLoaded || !mapRef.current) {
      console.log('Maps not ready:', { googleMapsLoaded, mapRef: !!mapRef.current });
      return;
    }

    if (!window.google || !window.google.maps) {
      console.error('Google Maps API not available');
      setError('Google Maps failed to load. Please refresh the page.');
      return;
    }

    console.log('Initializing map...');
    const { google } = window;
    
    try {
      const defaultLocation = {
        lat: mapDetails.latitude || 22.5726, // Default to Kolkata
        lng: mapDetails.longitude || 88.3639
      };

      const mapOptions: google.maps.MapOptions = {
        center: defaultLocation,
        zoom: 15,
        mapTypeId: 'roadmap',
        streetViewControl: false,
        fullscreenControl: false,
        disableDefaultUI: false
      };

      // Initialize the map
      googleMapRef.current = new google.maps.Map(mapRef.current, mapOptions);
      console.log('Map initialized');

      // Add click listener to map
      const clickListener = googleMapRef.current.addListener('click', handleMapClick);

      // Add marker if we have a selected location
      if (selectedLocation) {
        updateMarker(selectedLocation);
      } else if (navigator.geolocation) {
        // Try to get user's current location
        navigator.geolocation.getCurrentPosition(
          (position) => {
            const pos = {
              lat: position.coords.latitude,
              lng: position.coords.longitude
            };
            setSelectedLocation(pos);
            updateMarker(pos);
            googleMapRef.current?.setCenter(pos);
          },
          () => {
            // Use default location if geolocation fails
            updateMarker(defaultLocation);
          }
        );
      } else {
        // Browser doesn't support Geolocation
        updateMarker(defaultLocation);
      }

      return () => {
        // Clean up event listeners
        if (clickListener) {
          google.maps.event.removeListener(clickListener);
        }
        if (markerRef.current) {
          markerRef.current.setMap(null);
          markerRef.current = null;
        }
      };
    } catch (error) {
      console.error('Error initializing map:', error);
      setError('Failed to initialize the map. Please try again.');
    }
  }, [googleMapsLoaded, mapDetails.latitude, mapDetails.longitude, selectedLocation, handleMapClick, reverseGeocode]);

  // Helper function to update marker position
  const updateMarker = (position: { lat: number; lng: number }) => {
    if (!googleMapRef.current || !window.google) return;

    const { google } = window;

    if (markerRef.current) {
      markerRef.current.setPosition(position);
    } else {
      markerRef.current = new google.maps.Marker({
        position,
        map: googleMapRef.current,
        draggable: true,
        title: 'Selected Location'
      });

      // Add drag listener to marker
      markerRef.current.addListener('dragend', (event: google.maps.MapMouseEvent) => {
        if (event.latLng) {
          const newLocation = {
            lat: event.latLng.lat(),
            lng: event.latLng.lng()
          };
          setSelectedLocation(newLocation);
          reverseGeocode(newLocation);
        }
      });
    }
  };

  const handlePincodeSearch = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedPincode) return;
    
    try {
      setIsLoading(true);
      setError(null);
      
      // Fetch pincode data from the API
      const pincodeInfo = await fetchPincodeData(selectedPincode);
      
      if (pincodeInfo) {
        setAddressDetails(prev => ({
          ...prev,
          area: pincodeInfo.area,
          city: pincodeInfo.city,
          state: pincodeInfo.state
        }));
        
        setMapDetails({
          latitude: pincodeInfo.latitude,
          longitude: pincodeInfo.longitude,
          description: pincodeInfo.mapDescription
        });
        
        setStage('details');
      } else {
        setError('Pincode not found. Please try another pincode.');
      }
    } catch (error) {
      console.error('Error handling pincode search:', error);
      const errorMessage = error instanceof Error ? error.message : 'Failed to validate pincode. Please try again.';
      setError(errorMessage);
    } finally {
      setIsLoading(false);
    }
  };
  
  // Handle pincode input change and validation
  const handlePincodeChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const value = e.target.value.replace(/\D/g, '').slice(0, 6);
    setSelectedPincode(value);
    
    // Clear error when user starts typing
    if (value.length === 6) {
      setError(null);
    }
  };

  // Comprehensive validation function
  const validateAddressData = () => {
    const errors: string[] = [];

    // Validate house number
    if (!addressDetails.houseNumber || addressDetails.houseNumber.trim() === '') {
      errors.push('House/Flat Number is required');
    }

    // Validate pincode
    if (!selectedPincode || !/^\d{6}$/.test(selectedPincode)) {
      errors.push('Valid Pincode is required');
    }

      // Validate contact number
    if (!user?.phone) {
      errors.push('Please update your profile with a contact number');
    }

    return errors;
  };

  const handleSaveAddress = async () => {
    // Validate address data
    const validationErrors = validateAddressData();
    
    if (validationErrors.length > 0) {
      setError(validationErrors.join(', '));
      return;
    }

    // Log user info for debugging
    console.log('Current user:', user);
    if (user?.role) {
      console.log('User role:', user.role);
      if (typeof user.role === 'object') {
        console.log('Role details:', {
          name: user.role.name,
          permissions: user.role.permissions
        });
      }
    }

    setIsLoading(true);
    setError(null);

    try {
      // Check for authentication and user role
      const token = localStorage.getItem('token');
      if (!token) {
        throw new Error('No authentication token found. Please log in.');
      }

      // Check if user has permission to add address
      const userRole = user?.role?.name || (user as any)?.role;
      
      // Construct address data with only the most essential fields first
      const addressData: Record<string, any> = {
        // Basic address components
        address: `${addressDetails.houseNumber}, ${addressDetails.landmark}, ${addressDetails.area}`.replace(/,\s*,/g, ',').replace(/^\s*,\s*|\s*,\s*$/g, ''),
        street: addressDetails.houseNumber,
        city: addressDetails.city,
        state: addressDetails.state,
        country: 'India',
        
        // Postal code (optional)
        postcode: '',
        
        // Location data
        latitude: mapDetails.latitude || 0,
        longitude: mapDetails.longitude || 0,
        
        // Contact information
        phone: user?.phone || '',
        name: user?.name || 'My Address',
        
        // Additional details
        type: 'home',
        isDefault: true,
        is_default: true,
        
        // Include all individual components
        houseNumber: addressDetails.houseNumber,
        landmark: addressDetails.landmark,
        area: addressDetails.area,
        
        // Add recipient details if available
        ...(recipientDetails.phoneNumber && { recipientPhone: recipientDetails.phoneNumber }),
        ...(recipientDetails.name && { recipientName: recipientDetails.name })
      };

      // Log the exact payload being sent
      console.log('Sending address data:', JSON.stringify(addressData, null, 2));
      
      if (!['user', 'customer', 'admin'].includes(userRole?.toLowerCase())) {
        throw new Error('You do not have permission to add an address. Please contact support.');
      }

      // Get coordinates from mapDetails or use default values
      let longitude = mapDetails.longitude ? Number(mapDetails.longitude) : 88.3639; // Default to Kolkata
      let latitude = mapDetails.latitude ? Number(mapDetails.latitude) : 22.5726;    // Default to Kolkata

      // Validate coordinates
      if (typeof longitude === 'undefined' || typeof latitude === 'undefined' || 
          isNaN(longitude) || isNaN(latitude) ||
          longitude < -180 || longitude > 180 ||
          latitude < -90 || latitude > 90) {
        throw new Error('Valid coordinates are required to save the address. Please select a valid location on the map.');
      }

      // Use user's contact information
      if (!user?.phone) {
        throw new Error('Please update your profile with a contact number');
      }
      const contactNumber = user.phone;
      const recipientName = user.name || 'My Address';

      // Add GeoJSON location to address data
      const addressDataWithLocation = {
        ...addressData,
        location: {
          type: 'Point',
          coordinates: [longitude, latitude] // GeoJSON uses [longitude, latitude] order
        },
        contactNumber: contactNumber,
        name: recipientName
      };

      console.log('Prepared Address Data:', JSON.stringify(addressDataWithLocation, null, 2));

      // Send address to backend with token
      try {
        const response = await api.post('/users/me/addresses', addressDataWithLocation, {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });
        console.log('Address saved successfully:', response.data);
        return response;
      } catch (error: any) {
        console.error('Detailed error:', {
          message: error.message,
          response: error.response?.data,
          status: error.response?.status,
          headers: error.response?.headers,
          config: {
            url: error.config?.url,
            method: error.config?.method,
            data: error.config?.data
          }
        });
        throw error;
      }

      // Handle successful address creation
      if (response.data.success) {
        // Reset form
        setAddressDetails({
          houseNumber: '',
          landmark: '',
          area: '',
          city: '',
          state: ''
        });
        setRecipientDetails({
          name: '',
          phoneNumber: ''
        });
        setSelectedPincode('');
        setSearchQuery('');

        // Optional: Trigger a refresh of addresses or show success message
        // You might want to pass a callback prop to notify parent component
        
        // Close modal
        handleClose();
        
        // Redirect to cart page
        navigate('/cart');
        
        // Return the response data for potential chaining
        return response.data;
      } else {
        // Server returned success: false
        throw new Error(response.data.msg || 'Failed to save address');
      }
    } catch (error: unknown) {
      console.error('Complete Error Object:', error);
      
      // More detailed error handling
      if (error && typeof error === 'object' && 'response' in error) {
        const err = error as any;
        // The request was made and the server responded with a status code
        // that falls out of the range of 2xx
        const errorMessage = err.response?.data?.msg || 
          err.response?.data?.error || 
          'Server error occurred while saving address';
        
        console.error('Server Response Error:', {
          status: err.response?.status,
          data: err.response?.data,
          message: errorMessage
        });
        
        // If there are validation errors, display them
        if (err.response?.data?.errors) {
          const validationErrors = err.response.data.errors
            .map((error: any) => error.msg || error.message)
            .join(', ');
          setError(validationErrors);
        } else {
          setError(errorMessage);
        }
      } else if (error && typeof error === 'object' && 'request' in error) {
        // The request was made but no response was received
        console.error('No Response Received:', error.request);
        setError('No response received from server. Please check your network connection.');
      } else if (error instanceof Error) {
        // Something happened in setting up the request that triggered an Error
        console.error('Error:', error.message);
        setError(error.message || 'Error saving address');
      } else {
        setError('An unknown error occurred while saving your address.');
      }
    } finally {
      setIsLoading(false);
    }
  };

  if (!isOpen) return null;

  const handleClose = () => {
    setIsVisible(false);
    setTimeout(onClose, 300);
  };

  const renderInitialContent = () => (
    <div className="p-6">
      <button 
        onClick={handleClose}
        className="absolute top-4 right-4 text-gray-600 hover:text-gray-900"
      >
        <X className="w-6 h-6" />
      </button>

      <h2 className="text-xl font-bold text-gray-900 mb-4">Deliver to</h2>
      
      <button 
        onClick={() => setStage('search')}
        className="w-full bg-blue-600 text-white py-3 rounded-md hover:bg-blue-700"
      >
        Add New Address
      </button>
    </div>
  );

  const renderSearchContent = () => (
    <div className="p-6">
      <button 
        onClick={() => setStage('initial')}
        className="absolute top-4 left-4 text-gray-600 hover:text-gray-900"
      >
        <X className="w-6 h-6" />
      </button>

      <h2 className="text-xl font-bold text-gray-900 mb-4">Search for area, locality, pincode</h2>
      
      <form onSubmit={handlePincodeSearch} className="space-y-4">
        <div className="relative">
          <input 
            type="text" 
            placeholder="Enter 6-digit pincode"
            value={selectedPincode}
            onChange={handlePincodeChange}
            maxLength={6}
            pattern="\d{6}"
            className="w-full px-4 py-2 border rounded-md pl-10"
          />
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
        </div>
        {error && (
          <div className="text-red-500 text-sm flex items-center">
            <AlertCircle className="w-4 h-4 mr-1" />
            {error}
          </div>
        )}
        <button 
          type="submit" 
          disabled={!selectedPincode || selectedPincode.length !== 6 || isLoading}
          className="w-full bg-blue-600 text-white py-2 rounded-md hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed"
        >
          {isLoading ? 'Searching...' : 'Search Pincode'}
        </button>
      </form>
      
      <div className="mt-6">
        <h3 className="text-sm font-medium text-gray-500 mb-2">Or search by area/locality</h3>
        <div className="relative">
          <input 
            type="text" 
            placeholder="Search area, locality" 
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="w-full px-4 py-2 border rounded-md pl-10"
          />
          <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
        </div>
      </div>

      <div className="mt-4">
        <p className="text-sm text-gray-500 mb-2">
          Enter a 6-digit pincode to check service availability
        </p>
        <p className="text-sm text-gray-500">
          We currently serve in Kolkata and surrounding areas
        </p>
      </div>
    </div>
  );

  const renderDetailedAddressContent = () => (
    <div className="p-6 h-full overflow-y-auto">
      <button 
        onClick={() => setStage('search')}
        className="absolute top-4 left-4 text-gray-600 hover:text-gray-900"
      >
        <X className="w-6 h-6" />
      </button>

      <div className="grid grid-cols-2 gap-6 h-full">
        {/* Map Section */}
        <div className="relative h-full">
          {!googleMapsLoaded ? (
            <div className="w-full h-full bg-gray-200 rounded-lg flex items-center justify-center">
              <div className="text-center">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2"></div>
                <p className="text-gray-600">Loading map...</p>
              </div>
            </div>
          ) : (
            <div 
              ref={mapRef}
              className="w-full h-full rounded-lg"
              style={{ minHeight: '400px' }}
            />
          )}
          <div className="absolute top-4 left-4 bg-black bg-opacity-50 text-white px-3 py-1 rounded">
            {mapDetails.description || 'Click on the map to select your location'}
          </div>
          {selectedLocation && (
            <div className="absolute bottom-4 left-4 bg-blue-600 text-white px-3 py-1 rounded text-sm">
              Location selected: {selectedLocation.lat.toFixed(6)}, {selectedLocation.lng.toFixed(6)}
            </div>
          )}
        </div>

        {/* Details Section */}
        <div className="overflow-y-auto">
          <div className="mb-4">
            <div className="flex items-center space-x-4 mb-4">
              <div>
                <h2 className="text-xl font-bold text-gray-900 flex items-center">
                  <MapPin className="mr-2 text-blue-600" /> Enter Address Details
                </h2>
                <p className="text-sm text-gray-600">{selectedPincode}, {addressDetails.area}</p>
              </div>
            </div>

            <div className="flex space-x-4 mb-4">
              <button 
                onClick={() => setStage('details')}
                className={`px-4 py-2 rounded-md flex items-center ${
                  stage === 'details' 
                    ? 'bg-blue-600 text-white' 
                    : 'bg-gray-100 text-gray-700'
                }`}
              >
                <MapPin className="mr-2 w-5 h-5" />
                Address Details
              </button>
              <button 
                onClick={() => setStage('recipient')}
                className={`px-4 py-2 rounded-md flex items-center ${
                  stage === 'recipient' 
                    ? 'bg-blue-600 text-white' 
                    : 'bg-gray-100 text-gray-700'
                }`}
              >
                <User className="mr-2 w-5 h-5" />
                Recipient Details
              </button>
            </div>
          </div>

          {stage === 'details' && (
            <div className="grid grid-cols-1 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">House/ Floor/ Flat Number*</label>
                <input 
                  type="text" 
                  placeholder="Type here" 
                  value={addressDetails.houseNumber}
                  onChange={(e) => setAddressDetails(prev => ({
                    ...prev, 
                    houseNumber: e.target.value
                  }))}
                  className="w-full px-3 py-2 border rounded-md"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Area Details</label>
                <input 
                  type="text" 
                  value={addressDetails.area} 
                  readOnly
                  className="w-full px-3 py-2 border rounded-md bg-gray-100 text-gray-700"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Landmark (Optional)</label>
                <input 
                  type="text" 
                  placeholder="Type here" 
                  value={addressDetails.landmark}
                  onChange={(e) => setAddressDetails(prev => ({
                    ...prev, 
                    landmark: e.target.value
                  }))}
                  className="w-full px-3 py-2 border rounded-md"
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Pincode</label>
                  <input 
                    type="text" 
                    value={selectedPincode} 
                    readOnly
                    className="w-full px-3 py-2 border rounded-md bg-gray-100 text-gray-700"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">City</label>
                  <input 
                    type="text" 
                    value={addressDetails.city} 
                    readOnly
                    className="w-full px-3 py-2 border rounded-md bg-gray-100 text-gray-700"
                  />
                </div>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">State</label>
                <input 
                  type="text" 
                  value={addressDetails.state} 
                  readOnly
                  className="w-full px-3 py-2 border rounded-md bg-gray-100 text-gray-700"
                />
              </div>
            </div>
          )}

          {stage === 'recipient' && (
            <div className="grid grid-cols-1 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Recipient Name*</label>
                <input 
                  type="text" 
                  placeholder="Enter recipient name" 
                  value={recipientDetails.name}
                  onChange={(e) => setRecipientDetails(prev => ({
                    ...prev,
                    name: e.target.value
                  }))}
                  className="w-full px-3 py-2 border rounded-md"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Phone Number*</label>
                <div className="flex">
                  <span className="inline-flex items-center px-3 border border-r-0 rounded-l-md bg-gray-50 text-gray-500">
                    +91
                  </span>
                  <input 
                    type="tel" 
                    placeholder="Enter phone number"
                    value={recipientDetails.phoneNumber}
                    onChange={(e) => setRecipientDetails(prev => ({
                      ...prev,
                      phoneNumber: e.target.value
                    }))}
                    className="flex-1 px-3 py-2 border rounded-r-md"
                  />
                </div>
                <p className="text-xs text-gray-500 mt-1">Billing will be done using the name on the prescription</p>
              </div>
            </div>
          )}

          {error && (
            <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative mb-4 flex items-center" role="alert">
              <AlertCircle className="mr-2 text-red-500" />
              <span className="flex-1">{error}</span>
              <button 
                onClick={() => setError(null)}
                className="ml-4 text-red-700 hover:text-red-900"
              >
                <X className="w-4 h-4" />
              </button>
            </div>
          )}

          <button 
            onClick={handleSaveAddress}
            disabled={isLoading}
            className={`w-full bg-blue-600 text-white py-3 rounded-md mt-6 hover:bg-blue-700 ${
              isLoading ? 'opacity-50 cursor-not-allowed' : ''
            }`}
          >
            {isLoading ? 'Saving...' : (stage === 'details' ? 'Save & Next' : 'Save Address')}
          </button>
        </div>
      </div>
    </div>
  );

  return (
    <div 
      className={`fixed inset-0 z-50 bg-black bg-opacity-50 flex items-center justify-center 
        ${isVisible ? 'opacity-100' : 'opacity-0'} 
        transition-opacity duration-300 ease-in-out`}
    >
      <div 
      className={`fixed top-0 right-0 h-full w-full max-w-4xl bg-white shadow-lg transform 
        ${(stage === 'detailed-address' || stage === 'details' || stage === 'recipient') && isVisible ? 'translate-x-0' : 'translate-x-full'} 
        transition-transform duration-300 ease-in-out`}
    >
      {(stage === 'detailed-address' || stage === 'details' || stage === 'recipient') && renderDetailedAddressContent()}
      </div>

      <div 
        className={`fixed top-0 right-0 h-full w-full max-w-md bg-white shadow-lg transform 
          ${(stage === 'initial' || stage === 'search') && isVisible ? 'translate-x-0' : 'translate-x-full'} 
          transition-transform duration-300 ease-in-out`}
      >
        {stage === 'initial' && renderInitialContent()}
        {stage === 'search' && renderSearchContent()}
      </div>
    </div>
  );
};

export default AddressModal;
