import React, { useEffect } from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate, useLocation } from 'react-router-dom';
import { AuthProvider, useAuth } from './context/AuthContext';
import { StoreProvider } from './context/StoreContext';
import { Toaster } from 'react-hot-toast';
import { CartProvider } from './context/CartContext';
import { WebSettingsProvider } from './context/WebSettingsContext';
import MaintenanceWrapper from './components/MaintenanceWrapper';
import LoginForm from './components/shared/LoginForm';
import RegisterForm from './components/shared/RegisterForm';
import ProductDetails from './components/HomepageComponent/ProductDetails';
import CartPage from './components/HomepageComponent/CartPage';
import CheckoutProcess from './components/HomepageComponent/Checkout';
import OrderConfirmation from './components/HomepageComponent/OrderConfirmation';
import Profile from './components/HomepageComponent/Profile';
import UploadPrescription from './components/HomepageComponent/UploadPrescription';
import ProtectedRoute from './components/ProtectedRoute';
import Home from './pages/Home/Home';
import ProductList from './components/HomepageComponent/ProductList';
import PharmacistDashboard from './pages/Pharmacist/Dashboard';
import PrescriptionsPage from './pages/Pharmacist/PrescriptionsPage';
import PickerPackerDashboard from './pages/PickerPacker/Dashboard';
import OrdersToPick from './pages/PickerPacker/OrdersToPick';
import AdminDashboard from './pages/admin/Dashboard';
import AdminOrders from './pages/admin/orders/Orders';
import OrderTracking from './pages/admin/orders/OrderTracking';
import ReturnsRefunds from './pages/admin/orders/ReturnsRefunds';
import OrderInvoices from './pages/admin/orders/OrderInvoices';

// Import Products components
import Products from './pages/admin/Products';
import Brands from './pages/admin/Products/Brands/Brands';
import Categories from './pages/admin/Products/Categories/Categories';

// Import Store components
import StoreDashboard from './pages/store/Dashboard';
import StoreProducts from './pages/store/Products/Products';
import StoreOrders from './pages/store/Orders/Orders';
import StoreInventory from './pages/store/Inventory/Inventory';
import StorePharmacists from './pages/store/Pharmacist/Pharmacist';
import StorePickerPackers from './pages/store/PickerPacker/PickerPacker';
import StoreDelivery from './pages/store/Delivery/Delivery';
import Taxes from './pages/admin/Products/Taxes/Taxes';
import Discounts from './pages/admin/Products/Discounts/Discounts';
import Items from './pages/admin/Products/Items/Items';
import Units from './pages/admin/Products/Units/Units';
import BulkOperations from './pages/admin/Products/BulkOperations/BulkOperations';
import Coupon from './pages/admin/Coupon/Coupon';
import ProductSections from './pages/admin/HomepageSection/ProductSections';
import HomepageSections from './pages/admin/HomepageSection';
import Banners from './pages/admin/HomepageSection/Banners';

// Web Settings
import WebSettingsRoutes from './pages/admin/web-settings';

// Import Delivery Boys components
import DeliveryBoysList from './pages/admin/DeliveryBoys/DeliveryBoysList';
import FundsTransfer from './pages/admin/DeliveryBoys/FundsTransfer';
import CashCollections from './pages/admin/DeliveryBoys/CashCollections';
import Bonuses from './pages/admin/DeliveryBoys/Bonuses';

// Store Manager components
import { CreateStoreManagers, ManageStoreManagers } from './pages/admin/StoreManager';
import Settings from './pages/admin/Settings/Settings';
import SalesReport from './pages/admin/Reports/SalesReport';

// Store Pages - Using the renamed imports with Store prefix
// These components are already imported above with Store prefix

// Delivery Pages
import DeliveryDashboard from './pages/delivery boy/Dashboard';

// Customer Pages
import Customers from './pages/admin/Customers/Customers';

// Helper function to get the appropriate dashboard route based on user role
const getDashboardRoute = (user: any) => {
  // Handle case where user is an object with a role property
  let userRole = user?.role || user;
  
  // If it's still an object with a role property, extract it
  if (userRole && typeof userRole === 'object' && 'role' in userRole) {
    userRole = userRole.role;
  }
  switch (userRole) {
    case 'admin':
    case 'superadmin':
      return '/admin/dashboard';
    case 'store_manager':
      return '/store/dashboard';
    case 'pharmacist':
      return '/pharmacist/dashboard';
    case 'picker_packer':
      return '/picker-packer/dashboard';
    case 'customer':
      return '/';
    case 'delivery_boy':
      return '/delivery/dashboard';
    case 'delivery': // For backward compatibility
      return '/delivery/dashboard';
    default:
      return '/';
  }
};



// Define AppRoutes as a separate component that uses the AuthProvider
const AppRoutesWithAuth: React.FC = () => {
  return (
    <AuthProvider>
      <AppRoutes />
    </AuthProvider>
  );
};

const AppRoutes: React.FC = () => {
  const { user, loading } = useAuth();
  const location = useLocation();

  // Debug logging for route changes
  useEffect(() => {
    console.log('Route changed to:', location.pathname);
    console.log('Current user:', user);
    console.log('Loading state:', loading);
  }, [location.pathname, user, loading]);

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    );
  }

  // If no user is logged in, only allow access to public routes
  if (!user) {
    return (
      <Routes>
        <Route path="/login" element={<LoginForm />} />
        <Route path="/register" element={<RegisterForm />} />
        <Route path="/" element={<Home />} />
        <Route path="/products" element={<ProductList />} />
        <Route path="/product/:id" element={<ProductDetails />} />
        <Route path="*" element={<Navigate to="/login" state={{ from: location.pathname }} replace />} />
      </Routes>
    );
  }

  return (
    <Routes>
      {/* Public routes */}
      <Route path="/" element={<Home />} />
      <Route path="/login" element={<LoginForm />} />
      <Route path="/register" element={<RegisterForm />} />
      <Route path="/products" element={<ProductList />} />
      <Route path="/product/:id" element={<ProductDetails />} />
      <Route path="/cart" element={
        <ProtectedRoute allowedRoles={['customer']}>
          <CartPage />
        </ProtectedRoute>
      } />
      <Route path="/checkout" element={
        <ProtectedRoute allowedRoles={['customer']}>
          <CheckoutProcess />
        </ProtectedRoute>
      } />
      <Route path="/checkout/process" element={
        <ProtectedRoute allowedRoles={['customer']}>
          <CheckoutProcess />
        </ProtectedRoute>
      } />
      <Route path="/order-confirmation" element={
        <ProtectedRoute allowedRoles={['customer']}>
          <OrderConfirmation />
        </ProtectedRoute>
      } />
      
      {/* Admin Routes */}
      <Route path="/admin/*" element={
        <ProtectedRoute allowedRoles={['admin', 'superadmin']}>
          <Routes>
            <Route index element={<Navigate to="dashboard" replace />} />
            <Route path="dashboard" element={<AdminDashboard />} />
            <Route path="customers" element={<Customers />} />
            <Route path="orders">
              <Route index element={<AdminOrders />} />
              <Route path="tracking" element={<OrderTracking />} />
              <Route path="returns" element={<ReturnsRefunds />} />
              <Route path="invoices" element={<OrderInvoices />} />
            </Route>
            <Route path="products" element={<Products />} />
            <Route path="products/brands" element={<Brands />} />
            <Route path="products/categories" element={<Categories />} />
            <Route path="products/taxes" element={<Taxes />} />
            <Route path="products/discounts" element={<Discounts />} />
            <Route path="coupons" element={<Coupon />} />
            <Route path="products/items" element={<Items />} />
            <Route path="products/units" element={<Units />} />
            <Route path="products/bulk-operations" element={<BulkOperations />} />
            <Route path="homepage" element={<HomepageSections />}>
              <Route index element={<Navigate to="banners" replace />} />
              <Route path="banners" element={<Banners />} />
              <Route path="sections" element={<ProductSections />} />
            </Route>
            <Route path="web-settings/*" element={<WebSettingsRoutes />} />
            <Route path="store-managers" element={<ManageStoreManagers />} />
            <Route path="store-managers/create" element={<CreateStoreManagers />} />
            <Route path="bonuses" element={<Bonuses />} />
            
            <Route path="delivery">
              <Route index element={<Navigate to="boys" replace />} />
              <Route path="boys" element={<DeliveryBoysList />} />
              <Route path="funds" element={<FundsTransfer />} />
              <Route path="collections" element={<CashCollections />} />
            </Route>
            
            <Route path="settings" element={<Settings />} />
            <Route path="reports">
              <Route index element={<Navigate to="sales" replace />} />
              <Route path="sales" element={<SalesReport />} />
            </Route>
            <Route path="*" element={<Navigate to="/admin/dashboard" replace />} />
          </Routes>
        </ProtectedRoute>
      } />

      {/* Store Manager Routes */}
      <Route path="/store/*" element={
        <ProtectedRoute allowedRoles={['store_manager']}>
          <Routes>
            <Route path="dashboard" element={<StoreDashboard />} />
            <Route path="orders" element={<StoreOrders />} />
            <Route path="products" element={<StoreProducts />} />
            <Route path="inventory" element={<StoreInventory />} />
            <Route path="pharmacist" element={<StorePharmacists />} />
            <Route path="picker-packer" element={<StorePickerPackers />} />
            <Route path="delivery" element={<StoreDelivery />} />
            <Route index element={<Navigate to="dashboard" replace />} />
          </Routes>
        </ProtectedRoute>
      } />

      {/* Delivery Boy Routes */}
      <Route path="/delivery/*" element={
        <ProtectedRoute allowedRoles={['delivery_boy']}>
          <Routes>
            <Route path="dashboard" element={<DeliveryDashboard />} />
            <Route index element={<Navigate to="dashboard" replace />} />
          </Routes>
        </ProtectedRoute>
      } />

      {/* Pharmacist Routes */}
      <Route path="/pharmacist/*" element={
        <ProtectedRoute allowedRoles={['pharmacist']}>
          <Routes>
            <Route index element={<Navigate to="dashboard" replace />} />
            <Route path="dashboard" element={<PharmacistDashboard />} />
            <Route path="prescriptions" element={<PrescriptionsPage />} />
            <Route path="*" element={<Navigate to="/pharmacist/dashboard" replace />} />
          </Routes>
        </ProtectedRoute>
      } />

      {/* Picker Packer Routes */}
      <Route path="/pickerpacker/*" element={
        <ProtectedRoute allowedRoles={['picker_packer']}>
          <Routes>
            <Route index element={<Navigate to="dashboard" replace />} />
            <Route path="dashboard" element={<PickerPackerDashboard />} />
            <Route path="to-pick" element={<OrdersToPick />} />
            <Route path="in-progress" element={<div className="p-8">In Progress Orders</div>} />
            <Route path="ready" element={<div className="p-8">Ready for Dispatch</div>} />
            <Route path="dispatched" element={<div className="p-8">Dispatched Orders</div>} />
            <Route path="profile" element={<div className="p-8">Profile</div>} />
            <Route path="settings" element={<div className="p-8">Settings</div>} />
            <Route path="*" element={<Navigate to="/pickerpacker/dashboard" replace />} />
          </Routes>
        </ProtectedRoute>
      } />

      {/* Customer Routes */}
      <Route path="/home" element={
        <ProtectedRoute allowedRoles={['customer']}>
          <Home />
        </ProtectedRoute>
      } />
      <Route
        path="/home/*"
        element={
          <ProtectedRoute allowedRoles={['customer']}>
            <Routes>
              <Route index element={<Home />} />
              <Route path="orders" element={<Profile />} />
              <Route path="profile" element={<Profile />} />
              <Route path="addresses" element={<Profile />} />
              <Route path="upload-prescription" element={<UploadPrescription />} />
              <Route path="prescription-history" element={<UploadPrescription />} />
              <Route path="wishlist" element={<div className="p-8">My Wishlist</div>} />
              <Route path="wallet" element={<div className="p-8">My Wallet</div>} />
              <Route path="notifications" element={<div className="p-8">Notifications</div>} />
              <Route path="settings" element={<div className="p-8">Account Settings</div>} />
              <Route path="*" element={<Navigate to="/home" replace />} />
            </Routes>
          </ProtectedRoute>
        }
      />
      <Route
        path="/cart"
        element={
          <ProtectedRoute allowedRoles={['customer']}>
            <CartPage />
          </ProtectedRoute>
        }
      />
      <Route
        path="/orders"
        element={
          <ProtectedRoute allowedRoles={['customer']}>
            <div className="p-8">My Orders Page</div>
          </ProtectedRoute>
        }
      />
      
      {/* Store Routes */}
      <Route path="/store">
        <Route index element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StoreDashboard />
          </ProtectedRoute>
        } />
        <Route path="products" element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StoreProducts />
          </ProtectedRoute>
        } />
        <Route path="orders" element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StoreOrders />
          </ProtectedRoute>
        } />
        <Route path="inventory" element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StoreInventory />
          </ProtectedRoute>
        } />
        <Route path="pharmacist" element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StorePharmacists />
          </ProtectedRoute>
        } />
        <Route path="picker-packer" element={
          <ProtectedRoute allowedRoles={['store_manager']}>
            <StorePickerPackers />
          </ProtectedRoute>
        } />
      </Route>

      {/* Catch all - redirect to dashboard */}
      <Route path="*" element={<Navigate to={getDashboardRoute(user?.role)} replace />} />
    </Routes>
  );
};

function App() {
  return (
    <Router>
      <WebSettingsProvider>
        <AuthProvider>
          <CartProvider>
            <StoreProvider>
              <Toaster position="top-center" reverseOrder={false} />
              <MaintenanceWrapper>
                <AppRoutes />
              </MaintenanceWrapper>
            </StoreProvider>
          </CartProvider>
        </AuthProvider>
      </WebSettingsProvider>
    </Router>
  );
}

export default App;
