import React, { useState, useEffect } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';
import {
  Search,
  Plus,
  Edit,
  Eye,
  Trash2,
  User,
  X
} from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';

interface Pharmacist {
  _id: string;
  name: string;
  email: string;
  phone: string;
  employee_id?: string;
  qualification: string;
  license_number: string;
  experience_years: number;
  shift: 'morning' | 'afternoon' | 'night';
  performance_rating?: number;
  status: 'active' | 'inactive' | 'on_leave';
  image: string;
  joining_date: string | Date;
  prescriptions_verified?: number;
  store: {
    _id: string;
    storeName: string;
  };
  user?: {
    _id: string;
    name: string;
    email: string;
  };
}

const Pharmacist: React.FC = () => {
  const [pharmacists, setPharmacists] = useState<Pharmacist[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [totalPages, setTotalPages] = useState<number>(1);
  const [selectedPharmacist, setSelectedPharmacist] = useState<Pharmacist | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState<boolean>(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState<boolean>(false);
  const [isDeleteModalOpen, setIsDeleteModalOpen] = useState<boolean>(false);
  const [isCreateModalOpen, setIsCreateModalOpen] = useState<boolean>(false);
  
  // Define initial form data as a constant
  const initialFormData = {
    name: '',
    email: '',
    password: '',
    phone: '',
    employee_id: '',
    qualification: '',
    license_number: '',
    experience_years: 0,
    shift: 'morning' as const,
    status: 'active' as const,
    image: null as File | null,
    imagePreview: null as string | null,
    joining_date: new Date().toISOString().split('T')[0],
    store: '',
    performance_rating: 0,
    prescriptions_verified: 0
  };

  const [formData, setFormData] = useState<typeof initialFormData>({ ...initialFormData });
  const [showPassword, setShowPassword] = useState(false);
  const [isUploading, setIsUploading] = useState(false);
  const [stores, setStores] = useState<Array<{ _id: string; storeName: string }>>([]);
  const [selectedFile, setSelectedFile] = useState<File | null>(null);

  // Initialize component with default data
  useEffect(() => {
    // Fetch initial data
    fetchPharmacists();
    
    // Set default store if available in localStorage
    const defaultStore = localStorage.getItem('defaultStore');
    if (defaultStore) {
      try {
        const store = JSON.parse(defaultStore);
        setStores([store]);
        setFormData(prev => ({
          ...prev,
          store: store._id
        }));
      } catch (e) {
        console.error('Error parsing default store:', e);
      }
    }
    
    // You can also initialize with some default data if needed
    // For example, if you have a way to get the current store from context or props
    // const currentStore = getCurrentStoreFromContext();
    // if (currentStore) {
    //   setStores([currentStore]);
    //   setFormData(prev => ({
    //     ...prev,
    //     store: currentStore._id
    //   }));
    // }
  }, []);

  // Fetch pharmacists from API
  const fetchPharmacists = async (page = 1, search = '', status = 'all') => {
    setLoading(true);
    try {
      const params = new URLSearchParams({
        page: page.toString(),
        limit: '10',
        ...(search && { search }),
        ...(status !== 'all' && { status })
      });

      // Remove the leading /api/v1 since it's already included in the baseURL
      const response = await api.get(`/pharmacists?${params.toString()}`);
      
      if (response.data) {
        const { data, pages } = response.data;
        setPharmacists(data);
        setTotalPages(pages || 1);
        setCurrentPage(page);
      } else {
        console.error('Failed to fetch pharmacists:', response.data?.msg || 'Unknown error');
        toast.error('Failed to load pharmacists');
        setPharmacists([]);
      }
    } catch (error: any) {
      console.error('Error fetching pharmacists:', error);
      const errorMessage = error.response?.data?.msg || 'Failed to load pharmacists';
      toast.error(typeof errorMessage === 'string' ? errorMessage : 'An error occurred');
      setPharmacists([]);
    } finally {
      setLoading(false);
    }
  };

  const handleSearch = (v: string) => {
    setSearchTerm(v);
    fetchPharmacists(1, v, statusFilter);
  };

  const handleStatusFilterChange = (v: string) => {
    setStatusFilter(v);
    fetchPharmacists(1, searchTerm, v);
  };

  const handlePageChange = (page: number) => {
    fetchPharmacists(page, searchTerm, statusFilter);
  };

  const handleViewPharmacist = (ph: Pharmacist) => {
    setSelectedPharmacist(ph);
    setIsViewModalOpen(true);
  };

  const handleEditPharmacist = (ph: Pharmacist) => {
    setSelectedPharmacist(ph);
    setFormData({
      name: ph.name,
      email: ph.email,
      password: '',
      phone: ph.phone || '',
      employee_id: ph.employee_id || '',
      qualification: ph.qualification || '',
      license_number: ph.license_number || '',
      experience_years: ph.experience_years || 0,
      shift: ph.shift || 'morning',
      status: ph.status || 'active',
      image: null,
      imagePreview: ph.image || null,
      joining_date: ph.joining_date
        ? new Date(ph.joining_date).toISOString().split('T')[0]
        : new Date().toISOString().split('T')[0],
      store: ph.store?._id || '',
      performance_rating: ph.performance_rating || 0,
      prescriptions_verified: ph.prescriptions_verified || 0
    });
    setIsEditModalOpen(true);
  };

  const handleDeletePharmacist = (ph: Pharmacist) => {
    setSelectedPharmacist(ph);
    setIsDeleteModalOpen(true);
  };

  const handleCreatePharmacist = () => {
    resetFormData();
    setIsCreateModalOpen(true);
  };

  const resetFormData = () => {
    setFormData({
      name: '',
      email: '',
      password: '',
      phone: '',
      employee_id: '',
      qualification: '',
      license_number: '',
      experience_years: 0,
      shift: 'morning',
      status: 'active',
      image: null,
      imagePreview: null,
      joining_date: new Date().toISOString().split('T')[0],
      store: stores[0]?._id || '',
      performance_rating: 0,
      prescriptions_verified: 0
    });
    setSelectedFile(null);
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    
    // Prevent changing store for non-admin users
    if (name === 'store') {
      const userRole = localStorage.getItem('userRole');
      if (userRole !== 'admin') {
        toast.error('You do not have permission to change the store');
        return;
      }
    }
    
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      const file = e.target.files[0];
      setSelectedFile(file);
      setFormData(prev => ({
        ...prev,
        image: file,
        imagePreview: URL.createObjectURL(file)
      }));
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Get the store ID from form data or user data
    let storeId = formData.store;
    
    // If no store ID in form data, try to get it from user data
    if (!storeId) {
      try {
        const userResponse = await api.get('/auth/me');
        const userData = userResponse.data?.data || {};
        storeId = userData.store?._id || userData.storeManager || userData.store;
        
        // If we still don't have a store ID, show an error
        if (!storeId && stores.length > 0) {
          storeId = stores[0]._id; // Use the first available store
        } else if (!storeId) {
          toast.error('No store assigned to your account. Please contact an administrator.');
          return;
        }
      } catch (error) {
        console.error('Error fetching user store info:', error);
        toast.error('Error verifying store information. Please try again.');
        return;
      }
    }
    
    // Ensure store ID is a string and trim any whitespace
    storeId = String(storeId).trim();
    
    // Validate store ID format (MongoDB ObjectId format)
    const objectIdRegex = /^[0-9a-fA-F]{24}$/;
    if (!objectIdRegex.test(storeId)) {
      console.error('Invalid store ID format:', storeId);
      toast.error('Invalid store configuration. Please contact support.');
      return;
    }

    setIsUploading(true);
    try {
      const fd = new FormData();
      
      // Ensure storeId is a string and trim any whitespace
      const storeIdStr = String(storeId).trim();
      
      // Map form data to match backend model
      const pharmacistData = {
        name: formData.name.trim(),
        email: formData.email.trim(),
        phone: formData.phone.trim(),
        employee_id: formData.employee_id.trim(),
        qualification: formData.qualification.trim(),
        license_number: formData.license_number.trim(),
        experience_years: Number(formData.experience_years) || 0,
        shift: formData.shift,
        status: formData.status,
        joining_date: formData.joining_date,
        store: storeIdStr, // Use the string store ID
        // The following fields are managed by the backend
        // - _id (auto-generated)
        // - user (handled during user creation)
        // - image (handled separately)
        // - performance_rating (defaults to 0)
        // - prescriptions_verified (defaults to 0)
      };

      // Add all data to FormData
      Object.entries(pharmacistData).forEach(([key, value]) => {
        if (value !== null && value !== undefined) {
          fd.append(key, String(value));
        }
      });

      // Add image if present
      if (formData.image instanceof File) {
        fd.append('image', formData.image);
      }

      // Add password only for new pharmacist
      if (!selectedPharmacist && formData.password) {
        fd.append('password', formData.password);
      }

      if (selectedPharmacist) {
        await api.put(`/pharmacists/${selectedPharmacist._id}`, fd, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Pharmacist updated successfully');
        setIsEditModalOpen(false);
      } else {
        await api.post('/pharmacists', fd, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });
        toast.success('Pharmacist created successfully');
        setIsCreateModalOpen(false);
        // Reset form data to initial state
        setFormData({ ...initialFormData });
        setSelectedFile(null);
      }
      
      // Refresh the list with current filters
      fetchPharmacists(currentPage, searchTerm, statusFilter);
    } catch (err: any) {
      console.error('Error saving pharmacist:', err);
      const errorMessage = err.response?.data?.msg || err.message || 'Failed to save pharmacist';
      toast.error(errorMessage);
    } finally {
      setIsUploading(false);
    }
  };

  const confirmDelete = async () => {
    if (!selectedPharmacist) return;
    try {
      await api.delete(`/pharmacists/${selectedPharmacist._id}`);
      toast.success('Deleted successfully');
      setIsDeleteModalOpen(false);
      fetchPharmacists(currentPage, searchTerm, statusFilter);
    } catch {
      toast.error('Delete failed');
    }
  };

  const getStatusDisplayName = (s: string) =>
    s === 'active' ? 'Active' : s === 'inactive' ? 'Inactive' : 'On Leave';

  const getStatusColor = (s: string) =>
    s === 'active'
      ? 'bg-green-100 text-green-800'
      : s === 'inactive'
      ? 'bg-red-100 text-red-800'
      : 'bg-yellow-100 text-yellow-800';

  // Create Pharmacist Modal
  const renderCreateModal = () => {
    if (!isCreateModalOpen) return null;

    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div className="bg-white rounded-lg shadow-xl w-full max-w-2xl max-h-[90vh] overflow-y-auto">
          <div className="p-6">
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-xl font-semibold text-gray-800">Add New Pharmacist</h2>
              <button
                onClick={() => setIsCreateModalOpen(false)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <form onSubmit={handleSubmit}>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                {/* Profile Picture Upload */}
                <div className="col-span-2">
                  <div className="flex items-center justify-center">
                    <div className="relative">
                      <div className="w-32 h-32 rounded-full bg-gray-200 flex items-center justify-center overflow-hidden">
                        {formData.imagePreview ? (
                          <img
                            src={formData.imagePreview}
                            alt="Profile Preview"
                            className="w-full h-full object-cover"
                          />
                        ) : (
                          <User className="w-16 h-16 text-gray-400" />
                        )}
                      </div>
                      <label className="absolute bottom-0 right-0 bg-blue-600 text-white p-2 rounded-full cursor-pointer hover:bg-blue-700">
                        <input
                          type="file"
                          className="hidden"
                          accept="image/*"
                          onChange={handleFileChange}
                        />
                        <Edit className="w-4 h-4" />
                      </label>
                    </div>
                  </div>
                </div>

                {/* Personal Information */}
                <div>
                  <h3 className="text-lg font-medium text-gray-700 mb-4">Personal Information</h3>
                  <div className="space-y-4">
                    <div>
                      <label htmlFor="name" className="block text-sm font-medium text-gray-700 mb-1">
                        Full Name <span className="text-red-500">*</span>
                      </label>
                      <input
                        type="text"
                        id="name"
                        name="name"
                        value={formData.name}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>

                    <div>
                      <label htmlFor="email" className="block text-sm font-medium text-gray-700 mb-1">
                        Email <span className="text-red-500">*</span>
                      </label>
                      <input
                        type="email"
                        id="email"
                        name="email"
                        value={formData.email}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        required
                      />
                    </div>

                    <div>
                      <label htmlFor="password" className="block text-sm font-medium text-gray-700 mb-1">
                        Password <span className="text-red-500">*</span>
                      </label>
                      <div className="relative">
                        <input
                          type={showPassword ? 'text' : 'password'}
                          id="password"
                          name="password"
                          value={formData.password}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 pr-10"
                          required
                          minLength={6}
                        />
                        <button
                          type="button"
                          onClick={() => setShowPassword(!showPassword)}
                          className="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-500 hover:text-gray-700"
                        >
                          {showPassword ? (
                            <Eye className="w-5 h-5" />
                          ) : (
                            <Eye className="w-5 h-5" />
                          )}
                        </button>
                      </div>
                    </div>

                    <div>
                      <label htmlFor="phone" className="block text-sm font-medium text-gray-700 mb-1">
                        Phone Number
                      </label>
                      <input
                        type="tel"
                        id="phone"
                        name="phone"
                        value={formData.phone}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>
                  </div>
                </div>

                {/* Professional Information */}
                <div>
                  <h3 className="text-lg font-medium text-gray-700 mb-4">Professional Information</h3>
                  <div className="space-y-4">
                    <div>
                      <label htmlFor="employee_id" className="block text-sm font-medium text-gray-700 mb-1">
                        Employee ID
                      </label>
                      <input
                        type="text"
                        id="employee_id"
                        name="employee_id"
                        value={formData.employee_id}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    <div>
                      <label htmlFor="qualification" className="block text-sm font-medium text-gray-700 mb-1">
                        Qualification
                      </label>
                      <input
                        type="text"
                        id="qualification"
                        name="qualification"
                        value={formData.qualification}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                        placeholder="e.g., PharmD, B.Pharm"
                      />
                    </div>

                    <div>
                      <label htmlFor="license_number" className="block text-sm font-medium text-gray-700 mb-1">
                        License Number
                      </label>
                      <input
                        type="text"
                        id="license_number"
                        name="license_number"
                        value={formData.license_number}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    <div>
                      <label htmlFor="experience_years" className="block text-sm font-medium text-gray-700 mb-1">
                        Experience (years)
                      </label>
                      <input
                        type="number"
                        id="experience_years"
                        name="experience_years"
                        min="0"
                        value={formData.experience_years}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    <div>
                      <label htmlFor="shift" className="block text-sm font-medium text-gray-700 mb-1">
                        Shift
                      </label>
                      <select
                        id="shift"
                        name="shift"
                        value={formData.shift}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      >
                        <option value="morning">Morning</option>
                        <option value="afternoon">Afternoon</option>
                        <option value="night">Night</option>
                      </select>
                    </div>

                    <div>
                      <label htmlFor="status" className="block text-sm font-medium text-gray-700 mb-1">
                        Status
                      </label>
                      <select
                        id="status"
                        name="status"
                        value={formData.status}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      >
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="on_leave">On Leave</option>
                      </select>
                    </div>

                    <div>
                      <label htmlFor="joining_date" className="block text-sm font-medium text-gray-700 mb-1">
                        Joining Date
                      </label>
                      <input
                        type="date"
                        id="joining_date"
                        name="joining_date"
                        value={formData.joining_date}
                        onChange={handleInputChange}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>

                    {stores.length > 0 && (
                      <div>
                        <label htmlFor="store" className="block text-sm font-medium text-gray-700 mb-1">
                          Assigned Store
                        </label>
                        <select
                          id="store"
                          name="store"
                          value={formData.store}
                          onChange={handleInputChange}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                          required={stores.length > 0}
                        >
                          <option value="">Select Store</option>
                          {stores.map((store) => (
                            <option key={store._id} value={store._id}>
                              {store.storeName}
                            </option>
                          ))}
                        </select>
                      </div>
                    )}
                  </div>
                </div>
              </div>

              <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200">
                <button
                  type="button"
                  onClick={() => setIsCreateModalOpen(false)}
                  className="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                  disabled={isUploading}
                >
                  Cancel
                </button>
                <button
                  type="submit"
                  className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
                  disabled={isUploading}
                >
                  {isUploading ? 'Saving...' : 'Save Pharmacist'}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    );
  };
  const renderEditModal = () => isEditModalOpen && <div>Edit Pharmacist Modal</div>;
  const renderViewModal = () => isViewModalOpen && <div>View Pharmacist Modal</div>;
  const renderDeleteModal = () =>
    isDeleteModalOpen && (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
        <div className="bg-white rounded-lg shadow-xl p-6 w-full max-w-md">
          <div className="mb-4">
            <h3 className="text-lg font-medium text-gray-900">Delete Pharmacist</h3>
            <p className="text-sm text-gray-500 mt-1">Are you sure you want to delete this pharmacist? This action cannot be undone.</p>
          </div>
          <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200">
            <button
              type="button"
              onClick={() => setIsDeleteModalOpen(false)}
              className="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              Cancel
            </button>
            <button
              type="button"
              onClick={confirmDelete}
              className="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
            >
              Delete
            </button>
          </div>
        </div>
      </div>
    );

  return (
    <Layout title="Pharmacist Management" sidebar={<Sidebar items={storeSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <h1 className="text-2xl font-bold">Pharmacist Management</h1>
          <button
            onClick={handleCreatePharmacist}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="h-4 w-4 mr-2" />
            Add Pharmacist
          </button>
        </div>

        {/* Filters */}
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="relative flex-grow">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search by name, email, license number..."
              className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>
          <select
            value={statusFilter}
            onChange={(e) => handleStatusFilterChange(e.target.value)}
            className="px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
          >
            <option value="all">All Status</option>
            <option value="active">Active</option>
            <option value="inactive">Inactive</option>
            <option value="on_leave">On Leave</option>
          </select>
          <button
            onClick={() => {
              setSearchTerm('');
              setStatusFilter('all');
              fetchPharmacists(1, '', 'all');
            }}
            className="flex items-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
          >
            <X className="w-4 h-4 mr-1" />
            Reset
          </button>
        </div>

        {/* Table */}
        <div className="bg-white rounded-lg shadow overflow-hidden">
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    Name
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    Contact
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    License
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    Specialization
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {loading ? (
                  <tr>
                    <td colSpan={6} className="px-6 py-4 text-center text-sm text-gray-500">
                      Loading...
                    </td>
                  </tr>
                ) : pharmacists.length === 0 ? (
                  <tr>
                    <td colSpan={6} className="px-6 py-4 text-center text-sm text-gray-500">
                      No pharmacists found
                    </td>
                  </tr>
                ) : (
                  pharmacists.map((p) => (
                    <tr key={p._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <div className="h-10 w-10">
                            {p.image ? (
                              <img
                                className="h-10 w-10 rounded-full object-cover"
                                src={p.image}
                                alt={p.name}
                              />
                            ) : (
                              <div className="h-10 w-10 rounded-full bg-gray-200 flex items-center justify-center">
                                <User className="h-6 w-6 text-gray-500" />
                              </div>
                            )}
                          </div>
                          <div className="ml-4">
                            <div className="text-sm font-medium text-gray-900">{p.name}</div>
                            <div className="text-sm text-gray-500">{p.email}</div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {p.phone || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {p.license_number || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {p.specialization}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span
                          className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusColor(
                            p.status
                          )}`}
                        >
                          {getStatusDisplayName(p.status)}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex space-x-2">
                          <button
                            onClick={() => handleViewPharmacist(p)}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            <Eye className="h-5 w-5" />
                          </button>
                          <button
                            onClick={() => handleEditPharmacist(p)}
                            className="text-indigo-600 hover:text-indigo-900"
                          >
                            <Edit className="h-5 w-5" />
                          </button>
                          <button
                            onClick={() => handleDeletePharmacist(p)}
                            className="text-red-600 hover:text-red-900"
                          >
                            <Trash2 className="h-5 w-5" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </div>
      </div>

      {/* Modals */}
      {renderCreateModal()}
      {renderEditModal()}
      {renderViewModal()}
      {renderDeleteModal()}
    </Layout>
  );
};

export default Pharmacist;
