import React, { useState, Fragment, useEffect } from 'react';
import { Dialog, Transition } from '@headlessui/react';
import { Search, Filter, Download, Eye, Edit, Truck } from 'lucide-react';
import { format } from 'date-fns';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';
import api, { orderService } from '../../../services/api';
import { useAuth } from '../../../context/AuthContext';
import { toast } from 'react-hot-toast';

// Define the Order type based on backend schema
interface Order {
  _id: string;
  orderNumber: string;
  user: {
    _id: string;
    name: string;
    email: string;
    phone?: string;
  };
  deliveryBoy?: {
    _id: string;
    name: string;
    phone?: string;
  };
  pharmacist?: {
    _id: string;
    name: string;
    email?: string;
  };
  prescription?: {
    fileUrl: string;
    originalName: string;
    mimeType: string;
    size: number;
    verified: boolean;
    verifiedBy?: string;
    verifiedAt?: string;
    rejectionReason?: string;
    images?: Array<{
      path: string;
      original_name: string;
      mimetype: string;
      size: number;
      uploaded_at: string;
      is_primary: boolean;
    }>;
  };
  items: Array<{
    item: string; // ObjectId reference to Item
    name: string;
    quantity: number;
    price: number;
    taxAmount: number;
    discountAmount: number;
    totalAmount: number;
    image?: string;
    brand?: string;
    category?: string;
    unit?: string;
    weight?: number;
    variantId?: string;
    storeName?: string;
  }>;
  deliveryAddress: {
    name: string;
    contactNumber?: string;
    address: string;
    landmark?: string;
    city: string;
    state: string;
    postcode?: string;
    country?: string;
    latitude?: number;
    longitude?: number;
  };
  totalAmount: number;
  discountAmount: number;
  taxAmount: number;
  finalAmount: number;
  paymentMethod: string;
  paymentStatus: string;
  status: string;
  statusHistory?: Array<{
    status: string;
    changedAt: string;
    comment?: string;
    updatedBy?: string;
  }>;
  createdAt: string;
  updatedAt: string;
  deliveredAt?: string;
  cancelledAt?: string;
  cancellationReason?: string;
  notes?: string;
}

interface TrackingInfo {
  status: string;
  estimatedDelivery: string;
  trackingNumber: string;
  carrier: string;
  orderNumber: string;
}

const Orders = () => {
  const { user } = useAuth();
  const [orders, setOrders] = useState<Order[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState(false);
  const [isTrackingModalOpen, setIsTrackingModalOpen] = useState(false);
  const [trackingInfo, setTrackingInfo] = useState<TrackingInfo | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalOrders, setTotalOrders] = useState(0);

  // Fetch orders for the current store
  const fetchOrders = async (page: number = 1) => {
    console.log('Current user object:', user); // Debug log
    
    const userId = user?.id || user?._id;
    
    if (!userId) {
      const errorMsg = 'No user ID found. Please log in again.';
      console.error(errorMsg);
      setError(errorMsg);
      setLoading(false);
      return;
    }

    try {
      setLoading(true);
      setError(null);

      // Create a clean copy of the user data
      const userData = JSON.parse(JSON.stringify(user));
      
      // Log the complete user data in a well-formatted way
      console.group('👤 Complete User Data from Database');
      
      // Basic Information
      console.group('🔑 Basic Information');
      console.log('ID:', userData.id || userData._id);
      console.log('Name:', userData.name);
      console.log('Email:', userData.email);
      console.log('Phone:', userData.phone);
      console.log('Role:', userData.role?.name || userData.role);
      console.log('Status:', userData.status ? 'Active' : 'Inactive');
      console.log('Email Verified:', userData.emailVerified ? '✅' : '❌');
      console.log('Phone Verified:', userData.phoneVerified ? '✅' : '❌');
      console.groupEnd();
      
      // Authentication & Status
      console.group('🔐 Authentication & Status');
      console.log('Login Attempts:', userData.loginAttempts || 0);
      console.log('Created At:', new Date(userData.createdAt).toLocaleString());
      console.log('Updated At:', new Date(userData.updatedAt).toLocaleString());
      if (userData.lastLogin) {
        console.log('Last Login:', new Date(userData.lastLogin).toLocaleString());
      }
      console.groupEnd();
      
      // Avatar
      console.group('🖼️ Avatar');
      if (userData.avatar) {
        if (typeof userData.avatar === 'string') {
          console.log('URL:', userData.avatar);
        } else {
          console.log('Public ID:', userData.avatar.public_id || 'Not set');
          console.log('URL:', userData.avatar.url || 'Not set');
        }
      } else {
        console.log('No avatar set');
      }
      console.groupEnd();
      
      // Location
      console.group('📍 Location');
      if (userData.location) {
        const loc = userData.location;
        if (loc.address) console.log('Address:', loc.address);
        if (loc.city) console.log('City:', loc.city);
        if (loc.state) console.log('State:', loc.state);
        if (loc.country) console.log('Country:', loc.country);
        if (loc.pincode) console.log('Postal Code:', loc.pincode);
        
        if (loc.coordinates) {
          console.group('Coordinates:');
          console.log('Type:', loc.coordinates.type || 'Point');
          console.log('Longitude:', loc.coordinates.coordinates ? loc.coordinates.coordinates[0] : 'N/A');
          console.log('Latitude:', loc.coordinates.coordinates ? loc.coordinates.coordinates[1] : 'N/A');
          console.groupEnd();
        }
        
        console.log('Is Default:', loc.isDefault ? '✅' : '❌');
        if (loc.lastUpdated) {
          console.log('Last Updated:', new Date(loc.lastUpdated).toLocaleString());
        }
      } else {
        console.log('No location data available');
      }
      console.groupEnd();
      
      // References and store manager ID handling
      console.log('🔗 References');
      
      // Get store manager ID from user data
      let storeManagerId: string | undefined;
      
      // Debug log the entire user data to understand the structure
      console.log('🔍 User data (raw):', JSON.parse(JSON.stringify(userData)));
      
      // The storeManager field can be in different formats based on how it's populated
      if (userData?.storeManager) {
        // Case 1: Direct ObjectId reference (not populated)
        if (typeof userData.storeManager === 'string') {
          storeManagerId = userData.storeManager;
          console.log('✅ Using direct storeManager string as storeManagerId:', storeManagerId);
        }
        // Case 2: Object with $oid (MongoDB format)
        else if (userData.storeManager.$oid) {
          storeManagerId = userData.storeManager.$oid;
          console.log('✅ Using storeManager.$oid as storeManagerId:', storeManagerId);
        }
        // Case 3: Object with _id (Mongoose populated)
        else if (userData.storeManager._id) {
          storeManagerId = typeof userData.storeManager._id === 'string' 
            ? userData.storeManager._id 
            : userData.storeManager._id.toString();
          console.log('✅ Using storeManager._id as storeManagerId:', storeManagerId);
        }
        // Case 4: If storeManager is an object with id field
        else if (userData.storeManager.id) {
          storeManagerId = userData.storeManager.id;
          console.log('✅ Using storeManager.id as storeManagerId:', storeManagerId);
        }
      }
      
      console.log('Store Manager ID:', storeManagerId || 'Not assigned');
      console.log('Delivery Boy ID:', userData.deliveryBoy ? 
        (userData.deliveryBoy._id || userData.deliveryBoy) : 'None');
      
      // Debug: Log the entire storeManager object
      console.log('🔍 Store Manager Object:', userData.storeManager ? 
        JSON.stringify(userData.storeManager, null, 2) : 'No store manager assigned');
      
      // Addresses
      console.group('🏠 Addresses');
      if (userData.addresses && userData.addresses.length > 0) {
        userData.addresses.forEach((addr: any, index: any) => {
          console.group(`Address #${index + 1}:`);
          if (addr.name) console.log('Name:', addr.name);
          if (addr.contactNumber) console.log('Contact:', addr.contactNumber);
          if (addr.address) console.log('Address:', addr.address);
          if (addr.landmark) console.log('Landmark:', addr.landmark);
          if (addr.city) console.log('City:', addr.city);
          if (addr.state) console.log('State:', addr.state);
          if (addr.country) console.log('Country:', addr.country);
          if (addr.postcode) console.log('Postal Code:', addr.postcode);
          if (addr.isDefault) console.log('Default: ✅');
          console.groupEnd();
        });
      } else {
        console.log('No saved addresses');
      }
      console.groupEnd(); // Addresses
      
      console.groupEnd(); // References
      
      // Raw data (collapsed by default)
      console.groupCollapsed('📋 Raw JSON Data');
      console.log(JSON.stringify(userData, null, 2));
      console.groupEnd();
      
      console.groupEnd(); // Complete User Data
      
      // If user is a store manager and we couldn't find a storeManager reference
      if (!storeManagerId && (userData.role === 'store_manager' || userData.role?.name === 'store_manager')) {
        // Use the user's own ID as the store manager ID
        storeManagerId = userData.id || userData._id;
        console.log('⚠️ User is a store manager but no store reference found. Using user ID as storeManagerId:', storeManagerId);
      }
      
      if (!storeManagerId) {
        throw new Error('No valid store manager ID found. User data: ' + JSON.stringify({
          hasStoreManager: !!userData?.storeManager,
          storeManagerType: typeof userData?.storeManager,
          userRole: userData?.role,
          userId: userData?.id
        }, null, 2));
      }
      
      console.log('🔑 Final storeManagerId being used:', storeManagerId);

      // Now fetch orders for this store manager
      const response = await orderService.getOrders({
        storeManagerId, // Only pass storeManagerId
        search: searchTerm,
        status: statusFilter !== 'all' ? statusFilter : undefined,
        page,
        limit: 10
      });

      console.log('Orders API Response:', response); // Debug log

      if (response.data?.success) {
        // Handle different response structures
        const responseData = response.data as any; // Temporary any to handle API response
        let ordersData: Order[] = [];
        let paginationData = {
          total: 0,
          pages: 1,
          page: 1
        };

        // Check if data is directly in response.data or nested in response.data.data
        if (Array.isArray(responseData.data)) {
          ordersData = responseData.data as Order[];
          paginationData = {
            total: (responseData as any).total || responseData.data.length,
            pages: (responseData as any).pages || 1,
            page: (responseData as any).page || 1
          };
        } else if (responseData.data && Array.isArray(responseData.data.data)) {
          ordersData = responseData.data.data as Order[];
          paginationData = {
            total: responseData.data.total || 0,
            pages: responseData.data.pages || 1,
            page: responseData.data.page || 1
          };
        }

        setOrders(ordersData);
        setTotalPages(paginationData.pages);
        setTotalOrders(paginationData.total);
        setCurrentPage(paginationData.page);
      } else {
        throw new Error(response.data?.message || 'Failed to fetch orders');
      }
    } catch (err: any) {
      console.error('Error fetching orders:', err);
      const errorMessage = err.response?.data?.message || err.message || 'Failed to fetch orders';
      setError(errorMessage);
      toast.error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  // Fetch orders when component mounts or when filters change
  useEffect(() => {
    fetchOrders(1);
  }, [searchTerm, statusFilter]);

  // Handle page changes
  const handlePageChange = (page: number) => {
    fetchOrders(page);
  };

  const handleViewOrder = (order: Order) => {
    setSelectedOrder(order);
    setIsViewModalOpen(true);
  };

  const handleEditOrder = (order: Order) => {
    alert('Edit functionality - Update order details here');
  };

  const openTrackingModal = (order: Order) => {
    setSelectedOrder(order);
    setTrackingInfo({
      status: order.status,
      estimatedDelivery: format(new Date(Date.now() + 3 * 24 * 60 * 60 * 1000), 'MMM dd, yyyy'),
      trackingNumber: order.orderNumber,
      carrier: 'Express Delivery Services',
      orderNumber: order.orderNumber
    });
    setIsTrackingModalOpen(true);
  };

  // Filter orders based on search term and status
  const filteredOrders = orders.filter(order => {
    const matchesSearch = searchTerm === '' ||
      order.orderNumber.toLowerCase().includes(searchTerm.toLowerCase()) ||
      order.user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      order.user.email.toLowerCase().includes(searchTerm.toLowerCase());

    const matchesStatus = statusFilter === 'all' || order.status === statusFilter;

    return matchesSearch && matchesStatus;
  });

  return (
    <Layout title="Store Orders" sidebar={<Sidebar items={storeSidebarItems} />}>
      <div className="space-y-6">
        {/* Header Actions */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search orders..."
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
            
            <select
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
            >
              <option value="all">All Status</option>
              <option value="pending">Pending</option>
              <option value="confirmed">Confirmed</option>
              <option value="packed">Packed</option>
              <option value="out_for_delivery">Out for Delivery</option>
              <option value="delivered">Delivered</option>
              <option value="cancelled">Cancelled</option>
            </select>
          </div>
          
          <div className="flex gap-2">
            <button 
              className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50"
              onClick={() => alert('Filter functionality')}
            >
              <Filter className="w-4 h-4 mr-2" />
              Filter
            </button>
            <button 
              className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50"
              onClick={() => alert('Export functionality')}
            >
              <Download className="w-4 h-4 mr-2" />
              Export
            </button>
          </div>
        </div>

        {/* Orders Table */}
        <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
          <div className="overflow-x-auto">
            {loading ? (
              <div className="flex justify-center items-center py-12">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-500"></div>
                <span className="ml-2 text-gray-600">Loading orders...</span>
              </div>
            ) : error ? (
              <div className="text-center py-12">
                <div className="text-red-500 mb-2">❌ Error loading orders</div>
                <div className="text-gray-600">{error}</div>
                <button
                  onClick={() => fetchOrders(currentPage)}
                  className="mt-4 px-4 py-2 bg-blue-500 text-white rounded hover:bg-blue-600"
                >
                  Retry
                </button>
              </div>
            ) : (
              <>
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Order Details
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Customer
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Amount
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Status
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Delivery Boy
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Pharmacist
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Prescription
                      </th>
                      <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Actions
                      </th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {filteredOrders.length > 0 ? (
                      filteredOrders.map((order: Order) => (
                        <tr key={order._id}>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <div className="text-sm font-medium text-gray-900">
                              {order.orderNumber}
                            </div>
                            <div className="text-sm text-gray-500">
                              {format(new Date(order.createdAt), 'MMM dd, yyyy')}
                            </div>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <div className="text-sm font-medium text-gray-900">
                              {order.user?.name}
                            </div>
                            <div className="text-sm text-gray-500">
                              {order.user?.email}
                            </div>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                            ${order.finalAmount?.toFixed(2) || order.totalAmount?.toFixed(2)}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full
                              ${order.status === 'delivered' ? 'bg-green-100 text-green-800' :
                                order.status === 'cancelled' ? 'bg-red-100 text-red-800' :
                                'bg-yellow-100 text-yellow-800'}`}
                            >
                              {order.status.replace(/_/g, ' ')}
                            </span>
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            {order.deliveryBoy ? (
                              <div>
                                <div className="text-sm font-medium text-gray-900">
                                  {order.deliveryBoy.name}
                                </div>
                                {order.deliveryBoy.phone && (
                                  <div className="text-sm text-gray-500">
                                    {order.deliveryBoy.phone}
                                  </div>
                                )}
                              </div>
                            ) : (
                              <span className="text-sm text-gray-500">Not assigned</span>
                            )}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            {order.pharmacist ? (
                              <div>
                                <div className="text-sm font-medium text-gray-900">
                                  {order.pharmacist.name}
                                </div>
                                {order.pharmacist.email && (
                                  <div className="text-sm text-gray-500">
                                    {order.pharmacist.email}
                                  </div>
                                )}
                              </div>
                            ) : (
                              <span className="text-sm text-gray-500">-</span>
                            )}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap">
                            {order.prescription ? (
                              <div className="flex flex-col space-y-1">
                                {order.prescription.images ? (
                                  order.prescription.images.map((img, idx) => {
                                    // Use the full path if it's already a URL, otherwise construct it
                                    let imageUrl = img.path;
                                    
                                    // If it's not a full URL, construct it
                                    if (!imageUrl.startsWith('http')) {
                                      // Remove any leading slashes or uploads/ prefix if present
                                      const cleanPath = img.path.replace(/^\/|^uploads\//, '');
                                      imageUrl = `${import.meta.env.VITE_API_BASE_URL || 'http://localhost:5000'}/api/uploads/prescriptions/${cleanPath}`;
                                    }
                                    
                                    return (
                                      <a 
                                        key={idx}
                                        href={imageUrl}
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        className="text-blue-500 hover:text-blue-700 text-sm font-medium"
                                      >
                                        {img.original_name || `Prescription ${idx + 1}`}
                                      </a>
                                    );
                                  })
                                ) : order.prescription.fileUrl ? (
                                  <a 
                                    href={order.prescription.fileUrl} 
                                    target="_blank" 
                                    rel="noopener noreferrer"
                                    className="text-blue-500 hover:text-blue-700 text-sm font-medium"
                                  >
                                    {order.prescription.originalName || 'View Prescription'}
                                  </a>
                                ) : (
                                  <span className="text-sm text-gray-500">No prescription available</span>
                                )}
                              </div>
                            ) : (
                              <span className="text-sm text-gray-500">Not required</span>
                            )}
                          </td>
                          <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <button
                              onClick={() => handleViewOrder(order)}
                              className="text-blue-600 hover:text-blue-900 mr-3"
                            >
                              <Eye className="h-4 w-4" />
                            </button>
                          </td>
                        </tr>
                      ))
                    ) : (
                      <tr>
                        <td colSpan={6} className="px-6 py-12 text-center text-gray-500">
                          No orders found
                        </td>
                      </tr>
                    )}
                  </tbody>
                </table>

                {/* Pagination */}
                {totalPages > 1 && (
                  <div className="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
                    <div className="flex-1 flex justify-between sm:hidden">
                      <button
                        onClick={() => handlePageChange(currentPage - 1)}
                        disabled={currentPage === 1}
                        className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        Previous
                      </button>
                      <button
                        onClick={() => handlePageChange(currentPage + 1)}
                        disabled={currentPage === totalPages}
                        className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                      >
                        Next
                      </button>
                    </div>
                    <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                      <div>
                        <p className="text-sm text-gray-700">
                          Showing <span className="font-medium">{((currentPage - 1) * 10) + 1}</span> to{' '}
                          <span className="font-medium">{Math.min(currentPage * 10, totalOrders)}</span> of{' '}
                          <span className="font-medium">{totalOrders}</span> results
                        </p>
                      </div>
                      <div>
                        <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                          <button
                            onClick={() => handlePageChange(currentPage - 1)}
                            disabled={currentPage === 1}
                            className="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                          >
                            <span className="sr-only">Previous</span>
                            <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                              <path fillRule="evenodd" d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z" clipRule="evenodd" />
                            </svg>
                          </button>

                          {/* Page numbers */}
                          {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                            const pageNum = i + 1;
                            return (
                              <button
                                key={pageNum}
                                onClick={() => handlePageChange(pageNum)}
                                className={`relative inline-flex items-center px-4 py-2 border text-sm font-medium ${
                                  currentPage === pageNum
                                    ? 'z-10 bg-blue-50 border-blue-500 text-blue-600'
                                    : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'
                                }`}
                              >
                                {pageNum}
                              </button>
                            );
                          })}

                          <button
                            onClick={() => handlePageChange(currentPage + 1)}
                            disabled={currentPage === totalPages}
                            className="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
                          >
                            <span className="sr-only">Next</span>
                            <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                              <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
                            </svg>
                          </button>
                        </nav>
                      </div>
                    </div>
                  </div>
                )}
              </>
            )}
          </div>
        </div>

        {/* View Order Modal */}
        <Transition appear show={isViewModalOpen} as={Fragment}>
          <Dialog
            as="div"
            className="fixed inset-0 z-10 overflow-y-auto"
            onClose={() => setIsViewModalOpen(false)}
          >
            <div className="min-h-screen px-4 text-center">
              <Transition.Child
                as={Fragment}
                enter="ease-out duration-300"
                enterFrom="opacity-0"
                enterTo="opacity-100"
                leave="ease-in duration-200"
                leaveFrom="opacity-100"
                leaveTo="opacity-0"
              >
                <div className="fixed inset-0 bg-black opacity-30" />
              </Transition.Child>
              <span className="inline-block h-screen align-middle" aria-hidden="true">&#8203;</span>
              <div className="inline-block w-full max-w-3xl p-6 my-8 overflow-hidden text-left align-middle transition-all transform bg-white shadow-xl rounded-2xl">
                {selectedOrder && (
                  <div>
                    <div className="flex justify-between items-start">
                      <div>
                        <Dialog.Title as="h3" className="text-lg font-medium leading-6 text-gray-900">
                          Order #{selectedOrder.orderNumber}
                        </Dialog.Title>
                        <p className="mt-1 text-sm text-gray-500">
                          Placed on {format(new Date(selectedOrder.createdAt), 'MMMM dd, yyyy')}
                        </p>
                      </div>
                      <button
                        className="text-gray-400 hover:text-gray-500"
                        onClick={() => setIsViewModalOpen(false)}
                      >
                        <span className="sr-only">Close</span>
                        <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                        </svg>
                      </button>
                    </div>

                    <div className="mt-4">
                      <div className="border-t border-gray-200 py-4">
                        <h4 className="font-medium text-gray-900">Customer Information</h4>
                        <div className="mt-2 text-sm text-gray-500">
                          <p>{selectedOrder.user?.name}</p>
                          <p>{selectedOrder.user?.email}</p>
                          {selectedOrder.user?.phone && <p>{selectedOrder.user.phone}</p>}
                        </div>
                      </div>

                      <div className="border-t border-gray-200 py-4">
                        <h4 className="font-medium text-gray-900">Delivery Address</h4>
                        <div className="mt-2 text-sm text-gray-500">
                          <p>{selectedOrder.deliveryAddress?.name}</p>
                          <p>{selectedOrder.deliveryAddress?.address}</p>
                          <p>{selectedOrder.deliveryAddress?.city}, {selectedOrder.deliveryAddress?.state} {selectedOrder.deliveryAddress?.postcode}</p>
                          {selectedOrder.deliveryAddress?.contactNumber && <p>Phone: {selectedOrder.deliveryAddress.contactNumber}</p>}
                        </div>
                      </div>

                      <div className="border-t border-gray-200 py-4">
                        <h4 className="font-medium text-gray-900">Order Items</h4>
                        <div className="mt-4">
                          <table className="min-w-full divide-y divide-gray-200">
                            <thead>
                              <tr>
                                <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase">Product</th>
                                <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase">Qty</th>
                                <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase">Price</th>
                                <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase">Total</th>
                              </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-200">
                              {selectedOrder.items.map((item, index) => (
                                <tr key={index}>
                                  <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-900">
                                    {item.name}
                                    {item.brand && <span className="text-xs text-gray-500 ml-1">({item.brand})</span>}
                                  </td>
                                  <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-500 text-right">
                                    {item.quantity}
                                  </td>
                                  <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-500 text-right">
                                    ${(item.price || 0).toFixed(2)}
                                  </td>
                                  <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-900 font-medium text-right">
                                    ${(item.totalAmount || 0).toFixed(2)}
                                  </td>
                                </tr>
                              ))}
                            </tbody>
                            <tfoot className="bg-gray-50">
                              <tr>
                                <td colSpan={3} className="px-4 py-2 text-right text-sm font-medium text-gray-700">
                                  Total Amount:
                                </td>
                                <td className="px-4 py-2 text-right text-sm font-medium text-gray-900">
                                  ${selectedOrder.finalAmount?.toFixed(2) || selectedOrder.totalAmount?.toFixed(2)}
                                </td>
                              </tr>
                            </tfoot>
                          </table>
                        </div>
                      </div>

                      <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                        <div>
                          <h4 className="font-medium text-gray-700">Payment Method</h4>
                          <p className="text-gray-900 capitalize">{selectedOrder.paymentMethod}</p>
                        </div>
                        <div>
                          <h4 className="font-medium text-gray-700">Payment Status</h4>
                          <p className="text-gray-900 capitalize">{selectedOrder.paymentStatus}</p>
                        </div>
                      </div>

                      <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                        <div>
                          <h4 className="font-medium text-gray-700">Delivery Person</h4>
                          <p className="text-gray-900">
                            {selectedOrder.deliveryBoy?.name || 'Not assigned'}
                          </p>
                        </div>
                        <div>
                          <h4 className="font-medium text-gray-700">Order Status</h4>
                          <p className="text-gray-900 capitalize">{selectedOrder.status.replace(/_/g, ' ')}</p>
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </Dialog>
        </Transition>

        {/* Tracking Modal */}
        <Transition appear show={isTrackingModalOpen} as={Fragment}>
          <Dialog
            as="div"
            className="fixed inset-0 z-10 overflow-y-auto"
            onClose={() => setIsTrackingModalOpen(false)}
          >
            <div className="min-h-screen px-4 text-center">
              <Transition.Child
                as={Fragment}
                enter="ease-out duration-300"
                enterFrom="opacity-0"
                enterTo="opacity-100"
                leave="ease-in duration-200"
                leaveFrom="opacity-100"
                leaveTo="opacity-0"
              >
                <div className="fixed inset-0 bg-black opacity-30" />
              </Transition.Child>
              <span className="inline-block h-screen align-middle" aria-hidden="true">&#8203;</span>
              <div className="inline-block w-full max-w-md p-6 my-8 overflow-hidden text-left align-middle transition-all transform bg-white shadow-xl rounded-2xl">
                <div className="flex justify-between items-start">
                  <Dialog.Title as="h3" className="text-lg font-medium leading-6 text-gray-900">
                    Track Order
                  </Dialog.Title>
                  <button
                    className="text-gray-400 hover:text-gray-500"
                    onClick={() => setIsTrackingModalOpen(false)}
                  >
                    <span className="sr-only">Close</span>
                    <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                    </svg>
                  </button>
                </div>

                {trackingInfo && (
                  <div className="mt-4">
                    <div className="bg-gray-50 rounded-lg p-4">
                      <div className="flex justify-between items-center">
                        <div>
                          <p className="text-sm font-medium text-gray-500">Status</p>
                          <p className="text-lg font-semibold text-gray-900 capitalize">{trackingInfo.status.replace(/_/g, ' ')}</p>
                        </div>
                        <div className="text-right">
                          <p className="text-sm font-medium text-gray-500">Estimated Delivery</p>
                          <p className="text-sm font-medium text-gray-900">{trackingInfo.estimatedDelivery}</p>
                        </div>
                      </div>
                    </div>

                    <div className="border-t border-gray-200 pt-4 mt-4">
                      <h4 className="text-sm font-medium text-gray-900 mb-3">Tracking Information</h4>
                      <div className="space-y-3">
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Tracking Number:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.trackingNumber}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Carrier:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.carrier}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Order Number:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.orderNumber}</span>
                        </div>
                      </div>
                    </div>

                    <div className="mt-6 pt-4 border-t border-gray-200">
                      <h4 className="text-sm font-medium text-gray-900 mb-3">Delivery Status</h4>
                      <div className="space-y-4">
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className="w-2.5 h-2.5 rounded-full bg-green-500 mt-1"></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Order Placed</p>
                            <p className="text-xs text-gray-500">Your order has been received</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['confirmed', 'packed', 'out_for_delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Order Confirmed</p>
                            <p className="text-xs text-gray-500">We're preparing your order</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['packed', 'out_for_delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Order Packed</p>
                            <p className="text-xs text-gray-500">Your order is packed and ready</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['out_for_delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Out for Delivery</p>
                            <p className="text-xs text-gray-500">Your order is on the way</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${trackingInfo.status === 'delivered' ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Delivered</p>
                            <p className="text-xs text-gray-500">Order has been delivered</p>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                )}
              </div>
            </div>
          </Dialog>
        </Transition>
      </div>
    </Layout>
  );
};

export default Orders;