import React, { useState, useEffect } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';
import { Search, Plus, Minus, Download, Edit2, Eye, Package, RefreshCw, X, Check } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';
import { useAuth } from '@/context/AuthContext';

// Format currency helper
const formatCurrency = (amount: number) => {
  return new Intl.NumberFormat('en-IN', {
    style: 'currency',
    currency: 'INR',
    minimumFractionDigits: 2,
    maximumFractionDigits: 2
  }).format(amount);
};

interface Brand {
  _id: string;
  name: string;
}

interface Category {
  _id: string;
  name: string;
}

interface Item {
  _id: string;
  name: string;
  sku: string;
  description?: string;
  price: number;
  sale_price?: number;
  quantity: number;
  min_stock_level: number;
  max_stock_level: number;
  category_id: Category | string;
  brand_id: Brand | string | null;
  image?: string;
  images?: string[];
  status: boolean;
  created_at: string;
  updated_at: string;
}

interface Pagination {
  total: number;
  totalPages: number;
  currentPage: number;
  limit: number;
  hasNextPage: boolean;
  hasPreviousPage: boolean;
}

const Inventory: React.FC = () => {
  const { user } = useAuth();
  const [items, setItems] = useState<Item[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [pagination, setPagination] = useState<Pagination>({
    total: 0,
    totalPages: 1,
    currentPage: 1,
    limit: 10,
    hasNextPage: false,
    hasPreviousPage: false
  });
  const [selectedItem, setSelectedItem] = useState<Item | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState<boolean>(false);
  const [isStockUpdateModalOpen, setIsStockUpdateModalOpen] = useState<boolean>(false);
  const [stockUpdateData, setStockUpdateData] = useState({
    quantity: 0,
    type: 'in', // 'in', 'out', 'adjustment'
    reference: '',
    notes: ''
  });

  // Fetch active items with inventory data for the current store
  const fetchInventory = async (page = 1, search = '') => {
    setLoading(true);
    
    try {
      const storeId = user?.store?._id || user?.store_id;

      if (!storeId) {
        console.error('No store ID found for user:', user);
        toast.error('No store associated with your account');
        setLoading(false);
        return;
      }

      const response = await api.get('/items', {
        params: {
          page,
          limit: 10,
          search,
          status: 'active',
          store_id: storeId,
          populate: 'category_id,brand_id,store_id'
        }
      });
      
      if (response.data?.data) {
        setItems(response.data.data);
        
        // Update pagination
        setPagination({
          total: response.data.pagination?.total || 0,
          totalPages: response.data.pagination?.totalPages || 1,
          currentPage: response.data.pagination?.currentPage || 1,
          limit: response.data.pagination?.limit || 10,
          hasNextPage: response.data.pagination?.hasNextPage || false,
          hasPreviousPage: response.data.pagination?.hasPreviousPage || false
        });
        
        setCurrentPage(page);
      }
    } catch (error) {
      console.error('Error fetching inventory:', error);
      toast.error('Failed to load inventory items');
    } finally {
      setLoading(false);
    }
  };

  // Handle search
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    fetchInventory(1, searchTerm);
  };

  // Handle page change
  const handlePageChange = (page: number) => {
    fetchInventory(page, searchTerm);
  };

  // Handle view item details
  const handleViewItem = (item: Item) => {
    setSelectedItem(item);
    setIsViewModalOpen(true);
  };

  // Handle update stock
  const handleUpdateStock = (item: Item) => {
    setSelectedItem(item);
    setStockUpdateData({
      quantity: 0,
      type: 'in',
      reference: '',
      notes: ''
    });
    setIsStockUpdateModalOpen(true);
  };

  // Handle stock update form submit
  const handleStockUpdateSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!selectedItem) return;

    try {
      await api.post(`/inventory/${selectedItem._id}/stock`, stockUpdateData);
      toast.success('Stock updated successfully');
      setIsStockUpdateModalOpen(false);
      fetchInventory(currentPage, searchTerm);
    } catch (error) {
      console.error('Error updating stock:', error);
      toast.error('Failed to update stock');
    }
  };

  // Initial fetch
  useEffect(() => {
    if (user) {
      fetchInventory(1, '');
    }
  }, [user]);

  // Stock management states
  const [editingStock, setEditingStock] = useState<Record<string, boolean>>({});
  const [stockAdjustment, setStockAdjustment] = useState<Record<string, { type: 'add' | 'subtract' | 'set'; quantity: number }>>({});
  
  // Stock level thresholds with state for dynamic updates
  const [outOfStockNumber, setOutOfStockNumber] = useState<number>(5);
  const [inStockNumber, setInStockNumber] = useState<number>(10);

  // Load stock settings from store settings on component mount
  useEffect(() => {
    const loadStockSettings = async () => {
      try {
        const storeId = user?.store?._id || user?.store_id;
        if (!storeId) {
          console.log('No store ID found for user');
          return;
        }

        console.log('Loading stock settings for store:', storeId);
        
        try {
          // Get stock alert settings using the dedicated endpoint
          const response = await api.get(`/stores/${storeId}/stock-alert-settings`);
          console.log('Stock alert settings response:', response.data);
          
          // Check if we have stock alert settings
          if (response.data?.data?.stockAlertSettings) {
            const { outOfStockThreshold, lowStockThreshold } = response.data.data.stockAlertSettings;
            console.log('Loaded stock settings from API:', { outOfStockThreshold, lowStockThreshold });
            
            if (outOfStockThreshold !== undefined) {
              setOutOfStockNumber(outOfStockThreshold);
            }
            if (lowStockThreshold !== undefined) {
              setInStockNumber(lowStockThreshold);
            }
          } else {
            console.log('No stockAlertSettings found in store data, using defaults');
            // Fallback to localStorage if no settings found
            const savedOutOfStock = localStorage.getItem('outOfStockNumber');
            const savedInStock = localStorage.getItem('inStockNumber');
            
            if (savedOutOfStock) setOutOfStockNumber(parseInt(savedOutOfStock, 10));
            if (savedInStock) setInStockNumber(parseInt(savedInStock, 10));
          }
        } catch (apiError) {
          console.error('API Error loading store settings:', apiError);
          // Fallback to localStorage if API call fails
          const savedOutOfStock = localStorage.getItem('outOfStockNumber');
          const savedInStock = localStorage.getItem('inStockNumber');
          
          if (savedOutOfStock) setOutOfStockNumber(parseInt(savedOutOfStock, 10));
          if (savedInStock) setInStockNumber(parseInt(savedInStock, 10));
          
          toast.error('Failed to load store settings. Using local values.');
        }
      } catch (error) {
        console.error('Error loading stock settings:', error);
        // Fallback to localStorage if API call fails
        const savedOutOfStock = localStorage.getItem('outOfStockNumber');
        const savedInStock = localStorage.getItem('inStockNumber');
        
        if (savedOutOfStock) setOutOfStockNumber(parseInt(savedOutOfStock, 10));
        if (savedInStock) setInStockNumber(parseInt(savedInStock, 10));
      }
    };

    loadStockSettings();
  }, [user]);

  // Save stock settings to localStorage when they change
  useEffect(() => {
    localStorage.setItem('outOfStockNumber', outOfStockNumber.toString());
    localStorage.setItem('inStockNumber', inStockNumber.toString());
  }, [outOfStockNumber, inStockNumber]);

  // Save stock settings to store settings
  const saveStockSettings = async () => {
    try {
      const storeId = user?.store?._id || user?.store_id;
      if (!storeId) {
        toast.error('No store associated with your account');
        return;
      }

      if (outOfStockNumber >= inStockNumber) {
        toast.error('Low stock threshold must be greater than out of stock threshold');
        return;
      }

      console.log('Saving stock alert settings:', { outOfStockNumber, inStockNumber });

      // Update stock alert settings using the dedicated endpoint
      const response = await api.put(`/stores/${storeId}/stock-alert-settings`, {
        outOfStockThreshold: outOfStockNumber,
        lowStockThreshold: inStockNumber
      });

      console.log('Save settings response:', response.data);

      if (response.data.success) {
        // Save to localStorage as fallback
        localStorage.setItem('outOfStockNumber', outOfStockNumber.toString());
        localStorage.setItem('inStockNumber', inStockNumber.toString());
        
        toast.success('Stock alert settings saved successfully');
      } else {
        throw new Error(response.data.msg || 'Failed to save settings');
      }
    } catch (error: any) {
      console.error('Error saving stock settings:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to save stock alert settings';
      toast.error(errorMessage);
    }
  };

  // Reset stock settings to defaults
  const resetToDefaults = () => {
    setOutOfStockNumber(5);
    setInStockNumber(10);
    toast.success('Stock settings reset to default');
  };

  // Handle quick stock adjustment
  const handleQuickAdjust = async (itemId: string, amount: number) => {
    try {
      const storeId = user?.store?._id || user?.store_id;
      if (!storeId) {
        toast.error('No store associated with your account');
        return;
      }

      const item = items.find(i => i._id === itemId);
      if (!item) return;

      // Check if we have enough stock to subtract
      if (amount < 0 && (item.quantity || 0) < Math.abs(amount)) {
        toast.error('Insufficient stock');
        return;
      }

      // Optimistic update
      const updatedItems = items.map(i => {
        if (i._id === itemId) {
          return {
            ...i,
            quantity: Math.max(0, (i.quantity || 0) + amount) // Ensure we don't go below 0
          };
        }
        return i;
      });
      setItems(updatedItems);

      // API call
      await api.patch(`/items/${itemId}/stock`, {
        quantity: Math.abs(amount),
        type: amount > 0 ? 'in' : 'out',
        reference: 'adjustment',
        notes: `Quick ${amount > 0 ? 'addition' : 'removal'} of ${Math.abs(amount)} units`,
        store_id: storeId
      });
      
      toast.success('Stock updated successfully');
      // Refresh the inventory to ensure we have the latest data
      fetchInventory(currentPage, searchTerm);
    } catch (error: unknown) {
      console.error('Error updating stock:', error);
      // Revert optimistic update on error
      fetchInventory(currentPage, searchTerm);
      const errorMessage = error instanceof Error ? error.message : 'Failed to update stock';
      toast.error(errorMessage);
    }
  };

  // Handle stock adjustment from the edit modal
  const handleStockAdjustment = async (itemId: string) => {
    const adjustment = stockAdjustment[itemId];
    if (!adjustment) return;

    try {
      const storeId = user?.store?._id || user?.store_id;
      if (!storeId) {
        toast.error('No store associated with your account');
        return;
      }

      const item = items.find(i => i._id === itemId);
      if (!item) {
        toast.error('Item not found');
        return;
      }

      const { type, quantity } = adjustment;
      let amount = quantity;

      // For 'set' type, calculate the difference from current quantity
      if (type === 'set') {
        amount = quantity - (item.quantity || 0);
      }

      // Update local state optimistically
      setItems(prevItems => 
        prevItems.map(i => 
          i._id === itemId 
            ? { ...i, quantity: type === 'set' ? quantity : Math.max(0, (i.quantity || 0) + (type === 'add' ? quantity : -quantity)) }
            : i
        )
      );

      // Make API call to update stock
      await api.patch(`/items/${itemId}/stock`, {
        quantity: Math.abs(amount), // Always send positive quantity
        type: amount >= 0 ? 'in' : 'out',
        reference: 'manual',
        notes: `Manual ${type} of ${Math.abs(amount)} units`,
        store_id: storeId
      });

      // Show success message and reset form
      toast.success(`Stock ${type === 'add' ? 'increased' : type === 'subtract' ? 'decreased' : 'updated'} successfully`);
      
      // Reset the editing state
      setEditingStock(prev => ({ ...prev, [itemId]: false }));
      setStockAdjustment(prev => {
        const newState = { ...prev };
        delete newState[itemId];
        return newState;
      });

      // Refresh the inventory to ensure we have the latest data
      fetchInventory(currentPage, searchTerm);
    } catch (error: unknown) {
      console.error('Error updating stock:', error);
      
      // Revert optimistic update on error
      fetchInventory(currentPage, searchTerm);
      
      // Show error message
      const errorMessage = error instanceof Error ? error.message : 'Failed to update stock';
      toast.error(errorMessage);
    }
  };

  // Get stock status class
  const getStockStatusClass = (quantity: number | null | undefined) => {
    const q = quantity ?? 0;
    if (q <= 0) return 'bg-red-100 text-red-800';
    if (q <= outOfStockNumber) return 'bg-red-50 text-red-800 border border-red-200';
    if (q < inStockNumber) return 'bg-yellow-50 text-yellow-800 border border-yellow-200';
    return 'bg-green-50 text-green-800 border border-green-200';
  };

  // Get stock status text
  const getStockStatusText = (quantity: number | null | undefined) => {
    const q = quantity ?? 0;
    if (q <= 0) return 'Out of Stock';
    if (q <= outOfStockNumber) return 'Very Low Stock';
    if (q < inStockNumber) return 'Low Stock';
    return 'In Stock';
  };

  return (
    <Layout title="Inventory Management" sidebar={<Sidebar items={storeSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col space-y-4">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
            <h1 className="text-2xl font-bold">Inventory Management</h1>
            <div className="flex items-center space-x-2">
              <button className="px-4 py-2 bg-white border border-gray-300 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-50">
                <Download className="w-4 h-4 mr-2 inline" />
                Export
              </button>
            </div>
          </div>

          {/* Stock Settings */}
          <div className="bg-white shadow rounded-lg p-6">
            <div className="mb-6">
              <h3 className="text-lg font-medium text-gray-900">Stock Alert Settings</h3>
              <p className="mt-1 text-sm text-gray-500">
                Configure the stock levels that trigger "Out of Stock" and "In Stock" alerts for products.
              </p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {/* Out of Stock Settings */}
              <div className="bg-red-50 border border-red-200 rounded-lg p-6">
                <div className="flex items-center mb-4">
                  <div className="bg-red-100 p-2 rounded-lg mr-3">
                    <Package className="w-5 h-5 text-red-600" />
                  </div>
                  <div>
                    <h4 className="text-lg font-medium text-gray-900">Out of Stock Alert</h4>
                    <p className="text-sm text-gray-600">Products with stock below this number will show as "Out of Stock"</p>
                  </div>
                </div>
                <div className="flex items-center space-x-4">
                  <label htmlFor="outOfStockNumber" className="text-sm font-medium text-gray-700">
                    Alert Threshold:
                  </label>
                  <input
                    id="outOfStockNumber"
                    type="number"
                    min="0"
                    max="1000"
                    className="block w-24 pl-3 pr-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-red-500 focus:border-red-500 sm:text-sm"
                    value={outOfStockNumber}
                    onChange={(e) => setOutOfStockNumber(parseInt(e.target.value) || 0)}
                  />
                  <span className="text-sm text-gray-600">units</span>
                </div>
                <div className="mt-3 text-sm text-gray-600">
                  Current setting: Products with ≤ {outOfStockNumber} units will show as "Out of Stock"
                </div>
              </div>

              {/* In Stock Settings */}
              <div className="bg-green-50 border border-green-200 rounded-lg p-6">
                <div className="flex items-center mb-4">
                  <div className="bg-green-100 p-2 rounded-lg mr-3">
                    <Check className="w-5 h-5 text-green-600" />
                  </div>
                  <div>
                    <h4 className="text-lg font-medium text-gray-900">In Stock Alert</h4>
                    <p className="text-sm text-gray-600">Products with stock above this number will show as "In Stock"</p>
                  </div>
                </div>
                <div className="flex items-center space-x-4">
                  <label htmlFor="inStockNumber" className="text-sm font-medium text-gray-700">
                    Alert Threshold:
                  </label>
                  <input
                    id="inStockNumber"
                    type="number"
                    min="0"
                    max="1000"
                    className="block w-24 pl-3 pr-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 sm:text-sm"
                    value={inStockNumber}
                    onChange={(e) => setInStockNumber(parseInt(e.target.value) || 0)}
                  />
                  <span className="text-sm text-gray-600">units</span>
                </div>
                <div className="mt-3 text-sm text-gray-600">
                  Current setting: Products with ≥ {inStockNumber} units will show as "In Stock"
                </div>
              </div>
            </div>

            <div className="mt-6 flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
              <div className="text-sm text-gray-600">
                <p className="mb-2">These settings will affect how stock status is displayed across the system:</p>
                <ul className="list-disc list-inside space-y-1 text-sm text-gray-600">
                  <li>• Products with stock ≤ {outOfStockNumber} units: <span className="font-medium text-red-600">Out of Stock</span></li>
                  <li>• Products with stock between {outOfStockNumber + 1} and {inStockNumber - 1} units: <span className="font-medium text-yellow-600">Low Stock</span></li>
                  <li>• Products with stock ≥ {inStockNumber} units: <span className="font-medium text-green-600">In Stock</span></li>
                </ul>
              </div>
              <div className="flex flex-shrink-0 space-x-3">
                <button 
                  onClick={resetToDefaults}
                  className="px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Reset to Default
                </button>
                <button 
                  onClick={saveStockSettings}
                  className="px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Save Settings
                </button>
              </div>
            </div>
          </div>
        </div>

        {/* Search and Filters */}
        <div className="bg-white shadow rounded-lg p-4">
          <form onSubmit={handleSearch} className="flex flex-col sm:flex-row gap-4">
            <div className="relative flex-grow">
              <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                <Search className="h-5 w-5 text-gray-400" />
              </div>
              <input
                type="text"
                className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-lg bg-white shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                placeholder="Search by name or SKU..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
            <button
              type="submit"
              className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              Search
            </button>
            <button
              type="button"
              onClick={() => {
                setSearchTerm('');
                fetchInventory(1, '');
              }}
              className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            >
              Reset
            </button>
          </form>
        </div>

        {/* Inventory Table */}
        <div className="bg-white shadow overflow-hidden sm:rounded-lg">
          {loading ? (
            <div className="flex justify-center items-center p-8">
              <RefreshCw className="animate-spin h-8 w-8 text-blue-500" />
              <span className="ml-2">Loading inventory...</span>
            </div>
          ) : items.length === 0 ? (
            <div className="text-center p-8">
              <Package className="mx-auto h-12 w-12 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">No items found</h3>
              <p className="mt-1 text-sm text-gray-500">
                {searchTerm ? 'Try adjusting your search or filter' : 'Get started by adding a new item'}
              </p>
            </div>
          ) : (
            <>
              <div className="overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                  <thead className="bg-gray-50">
                    <tr>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Item
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        SKU
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Category
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Price
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Stock
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Status
                      </th>
                      <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Actions
                      </th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {items.map((item) => (
                      <tr key={item._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="flex items-center">
                            {item.image ? (
                              <img
                                className="h-10 w-10 rounded-md object-cover"
                                src={item.image}
                                alt={item.name}
                              />
                            ) : (
                              <div className="h-10 w-10 rounded-md bg-gray-200 flex items-center justify-center">
                                <Package className="h-5 w-5 text-gray-400" />
                              </div>
                            )}
                            <div className="ml-4">
                              <div className="text-sm font-medium text-gray-900">{item.name}</div>
                              <div className="text-sm text-gray-500 truncate max-w-xs">{item.description}</div>
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {item.sku || 'N/A'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {typeof item.category_id === 'object' ? item.category_id.name : 'Uncategorized'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {formatCurrency(item.price)}
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <div className="flex flex-col space-y-2">
                            <div className="flex items-center">
                              <div className="w-24 bg-gray-200 rounded-full h-2.5 mr-2">
                                <div 
                                  className={
                                    'h-2.5 rounded-full ' + 
                                    (item.quantity <= outOfStockNumber 
                                      ? 'bg-red-500' 
                                      : item.quantity < inStockNumber
                                        ? 'bg-yellow-500' 
                                        : 'bg-green-500')
                                  }
                                  style={{
                                    width: Math.min(100, (item.quantity / (item.max_stock_level || 100)) * 100) + '%'
                                  }}
                                ></div>
                              </div>
                              <span className={
                                'text-xs font-medium ' +
                                (item.quantity <= outOfStockNumber 
                                  ? 'text-red-700' 
                                  : item.quantity < inStockNumber
                                    ? 'text-yellow-700' 
                                    : 'text-green-700')
                              }>
                                {getStockStatusText(item.quantity)}
                              </span>
                            </div>
                            <div className="text-xs text-gray-500">
                              {item.max_stock_level > 0 && (
                                <span>{'Max: ' + item.max_stock_level + ' | '}</span>
                              )}
                              <span>Updated: {new Date(item.updated_at).toLocaleDateString()}</span>
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <div className="flex flex-col items-center">
                            <span 
                              className={
                                'inline-flex items-center justify-center w-20 px-3 py-1.5 rounded-md text-sm font-semibold ' +
                                getStockStatusClass(item.quantity)
                              }
                              title={'Current stock: ' + (item.quantity || 0) + ' units'}
                            >
                              {item.quantity ?? 'N/A'}
                            </span>
                            <span className="text-xs text-gray-500 mt-1">
                              {(item.quantity ?? 0) === 1 ? 'unit' : 'units'}
                              {item.min_stock_level !== undefined && (
                                <span className="block text-xs text-gray-400">Min: {item.min_stock_level}</span>
                              )}
                            </span>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <div className="flex items-center space-x-2">
                            {editingStock[item._id] ? (
                              <div className="flex items-center space-x-2">
                                <select
                                  className="block w-24 pl-3 pr-8 py-1.5 text-base border border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                  value={stockAdjustment[item._id]?.type || 'add'}
                                  onChange={(e) => 
                                    setStockAdjustment(prev => ({
                                      ...prev,
                                      [item._id]: {
                                        ...(prev[item._id] || { type: 'add', quantity: 0 }),
                                        type: e.target.value as 'add' | 'subtract' | 'set'
                                      }
                                    }))
                                  }
                                >
                                  <option value="add">Add</option>
                                  <option value="subtract">Subtract</option>
                                  <option value="set">Set to</option>
                                </select>
                                <input
                                  type="number"
                                  min="0"
                                  className="block w-20 pl-2 pr-2 py-1.5 border border-gray-300 rounded-md text-sm"
                                  value={stockAdjustment[item._id]?.quantity || ''}
                                  onChange={(e) => 
                                    setStockAdjustment(prev => ({
                                      ...prev,
                                      [item._id]: {
                                        ...(prev[item._id] || { type: 'add', quantity: 0 }),
                                        quantity: parseInt(e.target.value) || 0
                                      }
                                    }))
                                  }
                                  placeholder="Qty"
                                />
                                <button
                                  onClick={() => handleStockAdjustment(item._id)}
                                  className="text-green-600 hover:text-green-800"
                                >
                                  <Check className="w-5 h-5" />
                                </button>
                                <button
                                  onClick={() => {
                                    setEditingStock(prev => ({ ...prev, [item._id]: false }));
                                    setStockAdjustment(prev => {
                                      const newState = { ...prev };
                                      delete newState[item._id];
                                      return newState;
                                    });
                                  }}
                                  className="text-gray-500 hover:text-gray-700"
                                >
                                  <X className="w-5 h-5" />
                                </button>
                              </div>
                            ) : (
                              <>
                                <span className="font-medium mr-2">{item.quantity ?? 'N/A'}</span>
                                <div className="flex space-x-1">
                                  <button
                                    onClick={() => handleQuickAdjust(item._id, 1)}
                                    className="text-green-600 hover:text-green-800 p-1 rounded-full hover:bg-green-50"
                                    title="Add 1"
                                  >
                                    <Plus className="w-4 h-4" />
                                  </button>
                                  <button
                                    onClick={() => handleQuickAdjust(item._id, -1)}
                                    className="text-red-600 hover:text-red-800 p-1 rounded-full hover:bg-red-50"
                                    title="Subtract 1"
                                    disabled={(item.quantity ?? 0) <= 0}
                                  >
                                    <Minus className="w-4 h-4" />
                                  </button>
                                  <button
                                    onClick={() => setEditingStock(prev => ({ ...prev, [item._id]: true }))}
                                    className="text-blue-600 hover:text-blue-800 p-1 rounded-full hover:bg-blue-50"
                                    title="Adjust stock"
                                  >
                                    <Edit2 className="w-4 h-4" />
                                  </button>
                                  <button
                                    onClick={() => handleViewItem(item)}
                                    className="text-indigo-600 hover:text-indigo-800 p-1 rounded-full hover:bg-indigo-50"
                                    title="View details"
                                  >
                                    <Eye className="w-4 h-4" />
                                  </button>
                                </div>
                              </>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {/* Pagination */}
              {!loading && pagination.totalPages > 1 && (
                <div className="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
                  <div className="flex-1 flex justify-between sm:hidden">
                    <button
                      onClick={() => handlePageChange(pagination.currentPage - 1)}
                      disabled={!pagination.hasPreviousPage}
                      className={"relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md " + 
                        (pagination.hasPreviousPage
                          ? 'bg-white text-gray-700 hover:bg-gray-50'
                          : 'bg-gray-100 text-gray-400 cursor-not-allowed')
                        }
                    >
                      Previous
                    </button>
                    <button
                      onClick={() => handlePageChange(pagination.currentPage + 1)}
                      disabled={!pagination.hasNextPage}
                      className={"ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md " +
                        (pagination.hasNextPage
                          ? 'bg-white text-gray-700 hover:bg-gray-50'
                          : 'bg-gray-100 text-gray-400 cursor-not-allowed')
                      }
                    >
                      Next
                    </button>
                  </div>
                  <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                    <div>
                      <p className="text-sm text-gray-700">
                        Showing <span className="font-medium">{(pagination.currentPage - 1) * pagination.limit + 1}</span> to{' '}
                        <span className="font-medium">
                          {Math.min(pagination.currentPage * pagination.limit, pagination.total)}
                        </span>{' '}
                        of <span className="font-medium">{pagination.total}</span> results
                      </p>
                    </div>
                    <div>
                      <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                        <button
                          onClick={() => handlePageChange(1)}
                          disabled={!pagination.hasPreviousPage}
                          className={"relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium " +
                            (pagination.hasPreviousPage
                              ? 'text-gray-500 hover:bg-gray-50'
                              : 'text-gray-300 cursor-not-allowed')
                          }
                        >
                          <span className="sr-only">First</span>
                          <svg
                            className="h-5 w-5"
                            xmlns="http://www.w3.org/2000/svg"
                            viewBox="0 0 20 20"
                            fill="currentColor"
                            aria-hidden="true"
                          >
                            <path
                              fillRule="evenodd"
                              d="M15.707 15.707a1 1 0 01-1.414 0l-5-5a1 1 0 010-1.414l5-5a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 010 1.414zm-6 0a1 1 0 01-1.414 0l-5-5a1 1 0 010-1.414l5-5a1 1 0 011.414 1.414L5.414 10l4.293 4.293a1 1 0 010 1.414z"
                              clipRule="evenodd"
                            />
                          </svg>
                        </button>
                        <button
                          onClick={() => handlePageChange(pagination.currentPage - 1)}
                          disabled={!pagination.hasPreviousPage}
                          className={"relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium " +
                            (pagination.hasPreviousPage
                              ? 'text-gray-500 hover:bg-gray-50'
                              : 'text-gray-300 cursor-not-allowed')
                          }
                        >
                          <span className="sr-only">Previous</span>
                          <svg
                            className="h-5 w-5"
                            xmlns="http://www.w3.org/2000/svg"
                            viewBox="0 0 20 20"
                            fill="currentColor"
                            aria-hidden="true"
                          >
                            <path
                              fillRule="evenodd"
                              d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z"
                              clipRule="evenodd"
                            />
                          </svg>
                        </button>
                        {Array.from({ length: Math.min(5, pagination.totalPages) }, (_, i) => {
                          let pageNum;
                          if (pagination.totalPages <= 5) {
                            pageNum = i + 1;
                          } else if (pagination.currentPage <= 3) {
                            pageNum = i + 1;
                          } else if (pagination.currentPage >= pagination.totalPages - 2) {
                            pageNum = pagination.totalPages - 4 + i;
                          } else {
                            pageNum = pagination.currentPage - 2 + i;
                          }
                          return (
                            <button
                              key={pageNum}
                              onClick={() => handlePageChange(pageNum)}
                              className={"relative inline-flex items-center px-4 py-2 border text-sm font-medium " +
                                (pageNum === pagination.currentPage
                                  ? 'z-10 bg-blue-50 border-blue-500 text-blue-600'
                                  : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50')
                              }
                            >
                              {pageNum}
                            </button>
                          );
                        })}
                        <button
                          onClick={() => handlePageChange(pagination.currentPage + 1)}
                          disabled={!pagination.hasNextPage}
                          className={"relative inline-flex items-center px-2 py-2 border border-gray-300 bg-white text-sm font-medium " +
                            (pagination.hasNextPage
                              ? 'text-gray-500 hover:bg-gray-50'
                              : 'text-gray-300 cursor-not-allowed')
                          }
                        >
                          <span className="sr-only">Next</span>
                          <svg
                            className="h-5 w-5"
                            xmlns="http://www.w3.org/2000/svg"
                            viewBox="0 0 20 20"
                            fill="currentColor"
                            aria-hidden="true"
                          >
                            <path
                              fillRule="evenodd"
                              d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z"
                              clipRule="evenodd"
                            />
                          </svg>
                        </button>
                        <button
                          onClick={() => handlePageChange(pagination.totalPages)}
                          disabled={!pagination.hasNextPage}
                          className={`relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium ${
                            pagination.hasNextPage
                              ? 'text-gray-500 hover:bg-gray-50'
                              : 'text-gray-300 cursor-not-allowed'
                          }`}
                        >
                          <span className="sr-only">Last</span>
                          <svg
                            className="h-5 w-5"
                            xmlns="http://www.w3.org/2000/svg"
                            viewBox="0 0 20 20"
                            fill="currentColor"
                            aria-hidden="true"
                          >
                            <path
                              fillRule="evenodd"
                              d="M10.293 15.707a1 1 0 010-1.414L14.586 10l-4.293-4.293a1 1 0 111.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z"
                              clipRule="evenodd"
                            />
                            <path
                              fillRule="evenodd"
                              d="M4.293 15.707a1 1 0 010-1.414L8.586 10 4.293 5.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z"
                              clipRule="evenodd"
                            />
                          </svg>
                        </button>
                      </nav>
                    </div>
                  </div>
                </div>
              )}
            </>
          )}
        </div>
      </div>

      {/* View Item Modal */}
      {isViewModalOpen && selectedItem && (
        <div className="fixed z-10 inset-0 overflow-y-auto">
          <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div className="fixed inset-0 transition-opacity" aria-hidden="true">
              <div className="absolute inset-0 bg-gray-500 opacity-75"></div>
            </div>
            <span className="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">
              &#8203;
            </span>
            <div className="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-2xl sm:w-full">
              <div className="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                <div className="sm:flex sm:items-start">
                  <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                    <h3 className="text-lg leading-6 font-medium text-gray-900 mb-4">
                      {selectedItem.name}
                    </h3>
                    <div className="mt-2">
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                          <p className="text-sm text-gray-500">SKU</p>
                          <p className="mt-1 text-sm text-gray-900">{selectedItem.sku || 'N/A'}</p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-500">Category</p>
                          <p className="mt-1 text-sm text-gray-900">
                            {typeof selectedItem.category_id === 'object' ? selectedItem.category_id.name : 'Uncategorized'}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-500">Brand</p>
                          <p className="mt-1 text-sm text-gray-900">
                            {selectedItem.brand_id && typeof selectedItem.brand_id === 'object' 
                              ? selectedItem.brand_id.name 
                              : 'No Brand'}
                          </p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-500">Price</p>
                          <p className="mt-1 text-sm text-gray-900">{formatCurrency(selectedItem.price)}</p>
                        </div>
                        <div>
                          <p className="text-sm text-gray-500">Current Stock</p>
                          <div className="flex flex-col items-center">
                            <span 
                              className={`inline-flex items-center justify-center w-20 px-3 py-1.5 rounded-md text-sm font-semibold ${getStockStatusClass(selectedItem.quantity)}`}
                              title={`Current stock: ${selectedItem.quantity ?? 0} units`}
                            >
                              {selectedItem.quantity ?? 'N/A'}
                            </span>
                            <span className="text-xs text-gray-500 mt-1">
                              {(selectedItem.quantity ?? 0) === 1 ? 'unit' : 'units'}
                              {selectedItem.min_stock_level !== undefined && (
                                <span className="block text-xs text-gray-400">Min: {selectedItem.min_stock_level}</span>
                              )}
                            </span>
                          </div>
                        </div>
                        <div className="md:col-span-2">
                          <p className="text-sm text-gray-500">Description</p>
                          <p className="mt-1 text-sm text-gray-900">
                            {selectedItem.description || 'No description available'}
                          </p>
                        </div>
                        <div className="md:col-span-2">
                          <p className="text-sm text-gray-500">Stock Levels</p>
                          <div className="mt-1">
                            <div className="flex items-center justify-between text-sm">
                              <span className="text-gray-600">Minimum:</span>
                              <span className="font-medium">{selectedItem.min_stock_level} units</span>
                            </div>
                            <div className="flex items-center justify-between text-sm mt-1">
                              <span className="text-gray-600">Maximum:</span>
                              <span className="font-medium">{selectedItem.max_stock_level} units</span>
                            </div>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
              <div className="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                <button
                  type="button"
                  className="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-blue-600 text-base font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 sm:ml-3 sm:w-auto sm:text-sm"
                  onClick={() => setIsViewModalOpen(false)}
                >
                  Close
                </button>
                <button
                  type="button"
                  className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                  onClick={() => {
                    setIsViewModalOpen(false);
                    handleUpdateStock(selectedItem);
                  }}
                >
                  Update Stock
                </button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Stock Update Modal */}
      {isStockUpdateModalOpen && selectedItem && (
        <div className="fixed z-10 inset-0 overflow-y-auto">
          <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div className="fixed inset-0 transition-opacity" aria-hidden="true">
              <div className="absolute inset-0 bg-gray-500 opacity-75"></div>
            </div>
            <span className="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">
              &#8203;
            </span>
            <div className="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
              <form onSubmit={handleStockUpdateSubmit}>
                <div className="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                  <div className="sm:flex sm:items-start">
                    <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                      <h3 className="text-lg leading-6 font-medium text-gray-900 mb-4">
                        Update Stock - {selectedItem.name}
                      </h3>
                      <div className="mt-2 space-y-4">
                        <div>
                          <label htmlFor="type" className="block text-sm font-medium text-gray-700">
                            Transaction Type
                          </label>
                          <select
                            id="type"
                            name="type"
                            value={stockUpdateData.type}
                            onChange={(e) =>
                              setStockUpdateData({ ...stockUpdateData, type: e.target.value })
                            }
                            className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                          >
                            <option value="in">Stock In</option>
                            <option value="out">Stock Out</option>
                            <option value="adjustment">Adjustment</option>
                          </select>
                        </div>
                        <div>
                          <label htmlFor="quantity" className="block text-sm font-medium text-gray-700">
                            Quantity
                          </label>
                          <input
                            type="number"
                            name="quantity"
                            id="quantity"
                            min="1"
                            value={stockUpdateData.quantity}
                            onChange={(e) =>
                              setStockUpdateData({
                                ...stockUpdateData,
                                quantity: parseInt(e.target.value) || 0,
                              })
                            }
                            className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                            required
                          />
                        </div>
                        <div>
                          <label htmlFor="reference" className="block text-sm font-medium text-gray-700">
                            Reference (Optional)
                          </label>
                          <input
                            type="text"
                            name="reference"
                            id="reference"
                            value={stockUpdateData.reference}
                            onChange={(e) =>
                              setStockUpdateData({ ...stockUpdateData, reference: e.target.value })
                            }
                            className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                            placeholder="e.g., PO #12345"
                          />
                        </div>
                        <div>
                          <label htmlFor="notes" className="block text-sm font-medium text-gray-700">
                            Notes (Optional)
                          </label>
                          <textarea
                            id="notes"
                            name="notes"
                            rows={3}
                            value={stockUpdateData.notes}
                            onChange={(e) =>
                              setStockUpdateData({ ...stockUpdateData, notes: e.target.value })
                            }
                            className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                            placeholder="Add any additional notes about this stock update"
                          />
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
                <div className="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                  <button
                    type="submit"
                    className="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-blue-600 text-base font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 sm:ml-3 sm:w-auto sm:text-sm"
                  >
                    Update Stock
                  </button>
                  <button
                    type="button"
                    className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                    onClick={() => setIsStockUpdateModalOpen(false)}
                  >
                    Cancel
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
};

export default Inventory;