import React, { useState, Fragment, useEffect } from 'react';
import { Dialog, Transition } from '@headlessui/react';
import {
  Search,
  Filter,
  Download,
  Eye,
  Truck,
  Package,
  Clock,
  CheckCircle,
  XCircle
} from 'lucide-react';
import { format } from 'date-fns';
import { useNavigate } from 'react-router-dom';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { storeSidebarItems } from '../StoreLayout';
import api from '../../../services/api';
import { useAuth } from '../../../context/AuthContext';
import { toast } from 'react-hot-toast';

// Local small UI primitives (so this file is self-contained and won't fail if your design system Button/StatsCard are missing)
// Button component definition
const Button: React.FC<React.ButtonHTMLAttributes<HTMLButtonElement> & { variant?: 'default' | 'outline'; size?: 'sm' | 'md' }> = ({ children, variant = 'default', size = 'md', className = '', ...props }) => {
  const base = 'inline-flex items-center justify-center rounded-md font-medium focus:outline-none focus:ring-2 focus:ring-offset-2';
  const variants: Record<string, string> = {
    default: 'bg-blue-600 text-white hover:bg-blue-700 focus:ring-blue-500',
    outline: 'bg-white border border-gray-300 text-gray-700 hover:bg-gray-50 focus:ring-blue-500'
  };
  const sizes: Record<string, string> = { sm: 'px-3 py-1.5 text-sm', md: 'px-4 py-2 text-sm' };
  return (
    <button {...props} className={`${base} ${variants[variant]} ${sizes[size]} ${className}`.trim()}>
      {children}
    </button>
  );
};

// StatsCard component definition
const StatsCard: React.FC<{ title: string; value: number | string; icon?: any; color?: 'blue' | 'green' | 'yellow' }> = ({ title, value, icon: Icon, color = 'blue' }) => {
  const colorMap: Record<string, string> = {
    blue: 'bg-blue-100 text-blue-800',
    green: 'bg-green-100 text-green-800',
    yellow: 'bg-yellow-100 text-yellow-800'
  };
  return (
    <div className="bg-white rounded-lg shadow p-4 flex items-center justify-between">
      <div>
        <p className="text-sm text-gray-500">{title}</p>
        <p className="text-2xl font-semibold text-gray-900">{value}</p>
      </div>
      {Icon && (
        <div className={`p-3 rounded-full ${colorMap[color]} ml-4`}>
          <Icon className="h-6 w-6" />
        </div>
      )}
    </div>
  );
};

// Define the Order type based on backend schema
interface Order {
  _id: string;
  orderNumber: string;
  user?: {
    _id?: string;
    name?: string;
    email?: string;
    phone?: string;
  };
  deliveryBoy?: {
    _id?: string;
    name?: string;
    phone?: string;
  };
  items: Array<{
    item?: string;
    name: string;
    quantity: number;
    price?: number;
    taxAmount?: number;
    discountAmount?: number;
    totalAmount: number;
    image?: string;
  }>;
  deliveryAddress?: {
    name?: string;
    contactNumber?: string;
    address?: string;
    landmark?: string;
    city?: string;
    state?: string;
    postcode?: string;
  };
  totalAmount?: number;
  discountAmount?: number;
  taxAmount?: number;
  finalAmount?: number;
  paymentMethod?: string;
  paymentStatus?: string;
  status?: string;
  statusHistory?: Array<{
    status: string;
    changedAt: string;
    comment?: string;
    updatedBy?: string;
  }>;
  createdAt?: string;
  updatedAt?: string;
  deliveredAt?: string;
  estimatedDelivery?: string;
}

interface TrackingInfo {
  status?: string;
  estimatedDelivery?: string;
  trackingNumber?: string;
  carrier?: string;
  orderNumber?: string;
}

const PAGE_SIZE = 10;

const DeliveryPage: React.FC = () => {
  const { user } = useAuth();
  const navigate = useNavigate();

  const [orders, setOrders] = useState<Order[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState(false);
  const [isTrackingModalOpen, setIsTrackingModalOpen] = useState(false);
  const [trackingInfo, setTrackingInfo] = useState<TrackingInfo | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [totalOrders, setTotalOrders] = useState(0);

  // Fetch delivery orders
  const fetchDeliveryOrders = async (page: number = 1) => {
    try {
      setLoading(true);
      setError(null);

      const params: any = {
        page,
        limit: PAGE_SIZE
      };
      if (statusFilter && statusFilter !== 'all') params.status = statusFilter;
      if (searchTerm) params.search = searchTerm;
      if (user?.storeId) params.storeId = user.storeId;

      // Use store-specific endpoint for store managers
      const endpoint = user?.storeId ? `orders/store/${user.storeId}` : 'orders';
      const response = await api.get(endpoint, { params });

      // defensive checks for response structure
      const data = response?.data || {};
      setOrders(Array.isArray(data.orders) ? data.orders : []);
      setTotalOrders(typeof data.total === 'number' ? data.total : (data.orders?.length ?? 0));
      setTotalPages(typeof data.pages === 'number' ? data.pages : Math.max(1, Math.ceil((data.total ?? data.orders?.length ?? 0) / PAGE_SIZE)));
      setCurrentPage(page);
    } catch (err) {
      console.error('Error fetching delivery orders:', err);
      setError('Failed to load delivery orders. Please try again.');
      toast.error('Failed to load delivery orders');
    } finally {
      setLoading(false);
    }
  };

  // Handle search
  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    setCurrentPage(1);
    fetchDeliveryOrders(1);
  };

  // Handle filter change
  const handleFilterChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setStatusFilter(e.target.value);
    setCurrentPage(1);
    fetchDeliveryOrders(1);
  };

  // View order details
  const viewOrderDetails = (order: Order) => {
    setSelectedOrder(order);
    setIsViewModalOpen(true);
  };

  // Track order
  const trackOrder = async (order: Order) => {
    try {
      setTrackingInfo(null);
      setIsTrackingModalOpen(true);
      const response = await api.get(`/api/orders/${order._id}/tracking`);
      const data = response?.data || null;
      setTrackingInfo(data);
    } catch (err) {
      console.error('Error fetching tracking info:', err);
      toast.error('Failed to load tracking information');
      setTrackingInfo(null);
    }
  };

  // Update order status
  const updateOrderStatus = async (orderId: string, status: string) => {
    try {
      await api.patch(`/api/orders/${orderId}/status`, { status });
      toast.success('Order status updated successfully');
      // refresh current page
      fetchDeliveryOrders(currentPage);
    } catch (err) {
      console.error('Error updating order status:', err);
      toast.error('Failed to update order status');
    }
  };

  // Export CSV
  const exportCSV = () => {
    if (!orders || orders.length === 0) {
      toast('No orders to export');
      return;
    }

    const header = ['OrderNumber', 'Customer', 'Email', 'Phone', 'Address', 'City', 'Status', 'Total', 'CreatedAt'];
    const rows = orders.map((o) => [
      o.orderNumber,
      o.user?.name ?? '',
      o.user?.email ?? '',
      o.user?.phone ?? '',
      o.deliveryAddress?.address ?? '',
      o.deliveryAddress?.city ?? '',
      o.status ?? '',
      typeof o.finalAmount === 'number' ? o.finalAmount.toFixed(2) : (o.finalAmount ?? ''),
      o.createdAt ?? ''
    ]);

    const csv = [header.join(','), ...rows.map(r => r.map((c) => `"${String(c).replace(/"/g, '""')}"`).join(','))].join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `deliveries_page_${currentPage}.csv`;
    a.click();
    URL.revokeObjectURL(url);
  };

  // Initial data fetch
  useEffect(() => {
    fetchDeliveryOrders(1);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Get status badge
  const getStatusBadge = (status?: string) => {
    const s = (status ?? '').toLowerCase();
    switch (s) {
      case 'in_transit':
      case 'in-transit':
      case 'in transit':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
            <Truck className="h-3 w-3 mr-1" /> In Transit
          </span>
        );
      case 'pending':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
            <Clock className="h-3 w-3 mr-1" /> Pending
          </span>
        );
      case 'delivered':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
            <CheckCircle className="h-3 w-3 mr-1" /> Delivered
          </span>
        );
      default:
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
            <XCircle className="h-3 w-3 mr-1" /> {status}
          </span>
        );
    }
  };

  // Format date safely
  const formatDate = (dateString?: string) => {
    if (!dateString) return '-';
    try {
      return format(new Date(dateString), 'MMM d, yyyy h:mm a');
    } catch (e) {
      return dateString;
    }
  };

  // Calculate delivery stats
  const deliveryStats = {
    total: totalOrders,
    inTransit: orders.filter(o => (o.status ?? '').toLowerCase().includes('in')).length,
    pending: orders.filter(o => (o.status ?? '').toLowerCase() === 'pending').length,
    delivered: orders.filter(o => (o.status ?? '').toLowerCase() === 'delivered').length,
    today: orders.filter(o => {
      const orderDate = o.createdAt ? new Date(o.createdAt) : null;
      if (!orderDate) return false;
      const today = new Date();
      return orderDate.toDateString() === today.toDateString();
    }).length
  };

  return (
    <div className="h-full">
      <Layout title="Delivery Management" sidebar={<Sidebar items={storeSidebarItems} />}>
        <div className="container mx-auto px-4 py-6">
          {/* Header */}
          <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Delivery Management</h1>
              <p className="text-sm text-gray-500">Track and manage all deliveries</p>
            </div>
            <div className="mt-4 md:mt-0 flex space-x-2">
              <Button variant="outline" onClick={() => navigate('/store/delivery/schedule')} className="whitespace-nowrap">Schedule Delivery</Button>
              <Button onClick={() => navigate('/store/delivery/assign')} className="whitespace-nowrap">Assign Delivery</Button>
            </div>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
            <StatsCard title="Total Deliveries" value={deliveryStats.total} icon={Truck} color="blue" />
            <StatsCard title="In Transit" value={deliveryStats.inTransit} icon={Truck} color="blue" />
            <StatsCard title="Pending" value={deliveryStats.pending} icon={Clock} color="yellow" />
            <StatsCard title="Delivered" value={deliveryStats.delivered} icon={CheckCircle} color="green" />
          </div>

          {/* Filters and Search */}
          <div className="bg-white rounded-lg shadow-sm border mb-6">
            <div className="p-4 border-b">
              <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0">
                <form onSubmit={handleSearch} className="flex-1 max-w-md">
                  <div className="relative">
                    <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                      <Search className="h-5 w-5 text-gray-400" />
                    </div>
                    <input
                      type="text"
                      className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                      placeholder="Search deliveries..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                    />
                  </div>
                </form>

                <div className="flex items-center space-x-2">
                  <div className="flex items-center">
                    <Filter className="h-5 w-5 text-gray-400 mr-2" />
                    <select value={statusFilter} onChange={handleFilterChange} className="block w-full pl-3 pr-10 py-2 text-base border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md">
                      <option value="all">All Status</option>
                      <option value="pending">Pending</option>
                      <option value="in_transit">In Transit</option>
                      <option value="delivered">Delivered</option>
                      <option value="cancelled">Cancelled</option>
                    </select>
                  </div>
                  <Button variant="outline" size="sm" onClick={exportCSV}>
                    <Download className="h-4 w-4 mr-2" />
                    Export
                  </Button>
                </div>
              </div>
            </div>

            {/* Orders Table */}
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Order ID</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Customer</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Delivery Address</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {loading ? (
                    <tr>
                      <td colSpan={6} className="px-6 py-4 text-center text-sm text-gray-500">Loading...</td>
                    </tr>
                  ) : error ? (
                    <tr>
                      <td colSpan={6} className="px-6 py-4 text-center text-sm text-red-500">{error}</td>
                    </tr>
                  ) : orders.length === 0 ? (
                    <tr>
                      <td colSpan={6} className="px-6 py-4 text-center text-sm text-gray-500">No delivery orders found</td>
                    </tr>
                  ) : (
                    orders.map((order) => (
                      <tr key={order._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">{order.orderNumber}</td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-gray-900">{order.user?.name}</div>
                          <div className="text-sm text-gray-500">{order.user?.email}</div>
                        </td>
                        <td className="px-6 py-4">
                          <div className="text-sm text-gray-900">{order.deliveryAddress?.address}, {order.deliveryAddress?.city}</div>
                          <div className="text-sm text-gray-500">{order.deliveryAddress?.contactNumber}</div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">{getStatusBadge(order.status)}</td>
                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">{typeof order.finalAmount === 'number' ? `₹${order.finalAmount.toFixed(2)}` : (order.finalAmount ?? '-')}</td>
                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-2">
                          <button onClick={() => viewOrderDetails(order)} className="text-blue-600 hover:text-blue-900" aria-label="View">
                            <Eye className="h-5 w-5" />
                          </button>
                          <button onClick={() => trackOrder(order)} className="text-indigo-600 hover:text-indigo-900 ml-2" aria-label="Track">
                            <Truck className="h-5 w-5" />
                          </button>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>

            {/* Pagination */}
            <div className="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
              <div className="flex-1 flex justify-between sm:hidden">
                <button onClick={() => fetchDeliveryOrders(Math.max(1, currentPage - 1))} disabled={currentPage === 1} className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                  Previous
                </button>
                <button onClick={() => fetchDeliveryOrders(Math.min(totalPages, currentPage + 1))} disabled={currentPage >= totalPages} className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                  Next
                </button>
              </div>
              <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                <div>
                  <p className="text-sm text-gray-700">Showing <span className="font-medium">{(currentPage - 1) * PAGE_SIZE + 1}</span> to <span className="font-medium">{Math.min(currentPage * PAGE_SIZE, totalOrders)}</span> of <span className="font-medium">{totalOrders}</span> results</p>
                </div>
                <div>
                  <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                    <button onClick={() => fetchDeliveryOrders(Math.max(1, currentPage - 1))} disabled={currentPage === 1} className="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                      <span className="sr-only">Previous</span>
                      <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                        <path fillRule="evenodd" d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z" clipRule="evenodd" />
                      </svg>
                    </button>
                    {Array.from({ length: totalPages }, (_, i) => i + 1).map((page) => (
                      <button key={page} onClick={() => fetchDeliveryOrders(page)} className={`relative inline-flex items-center px-4 py-2 border text-sm font-medium ${currentPage === page ? 'z-10 bg-blue-50 border-blue-500 text-blue-600' : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'}`}>
                        {page}
                      </button>
                    ))}
                    <button onClick={() => fetchDeliveryOrders(Math.min(totalPages, currentPage + 1))} disabled={currentPage >= totalPages} className="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed">
                      <span className="sr-only">Next</span>
                      <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                        <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
                      </svg>
                    </button>
                  </nav>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* View Order Modal */}
        <Transition.Root show={isViewModalOpen} as={Fragment}>
          <Dialog as="div" className="fixed z-10 inset-0 overflow-y-auto" onClose={setIsViewModalOpen}>
            <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
              <Transition.Child as={Fragment} enter="ease-out duration-300" enterFrom="opacity-0" enterTo="opacity-100" leave="ease-in duration-200" leaveFrom="opacity-100" leaveTo="opacity-0">
                <Dialog.Overlay className="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" />
              </Transition.Child>

              <span className="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>

              <Transition.Child as={Fragment} enter="ease-out duration-300" enterFrom="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95" enterTo="opacity-100 translate-y-0 sm:scale-100" leave="ease-in duration-200" leaveFrom="opacity-100 translate-y-0 sm:scale-100" leaveTo="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95">
                <div className="inline-block align-bottom bg-white rounded-lg px-4 pt-5 pb-4 text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-4xl sm:w-full sm:p-6">
                  {selectedOrder && (
                    <div>
                      <div className="flex justify-between items-start">
                        <div>
                          <Dialog.Title as="h3" className="text-lg leading-6 font-medium text-gray-900">Order #{selectedOrder.orderNumber}</Dialog.Title>
                          <p className="text-sm text-gray-500 mt-1">Placed on {formatDate(selectedOrder.createdAt)}</p>
                        </div>
                        <div className="ml-3 h-7 flex items-center">
                          <button type="button" className="bg-white rounded-md text-gray-400 hover:text-gray-500 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500" onClick={() => setIsViewModalOpen(false)}>
                            <span className="sr-only">Close</span>
                            <svg className="h-6 w-6" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" aria-hidden="true">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </div>
                      </div>

                      <div className="mt-6 grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div>
                          <h4 className="text-lg font-medium text-gray-900 mb-4">Order Items</h4>
                          <div className="space-y-4">
                            {selectedOrder.items.map((item, index) => (
                              <div key={index} className="flex items-start">
                                <div className="flex-shrink-0 h-16 w-16 rounded-md overflow-hidden">
                                  {item.image ? (
                                    <img src={item.image} alt={item.name} className="h-full w-full object-cover object-center" />
                                  ) : (
                                    <div className="h-full w-full bg-gray-200 flex items-center justify-center text-gray-400">
                                      <Package className="h-6 w-6" />
                                    </div>
                                  )}
                                </div>
                                <div className="ml-4 flex-1">
                                  <h5 className="text-sm font-medium text-gray-900">{item.name}</h5>
                                  <p className="text-sm text-gray-500">Qty: {item.quantity}</p>
                                  <p className="text-sm font-medium text-gray-900">{typeof item.totalAmount === 'number' ? `₹${item.totalAmount.toFixed(2)}` : `₹${item.totalAmount}`}</p>
                                </div>
                              </div>
                            ))}
                          </div>

                          <div className="mt-6 border-t border-gray-200 pt-4">
                            <div className="flex justify-between text-base font-medium text-gray-900">
                              <p>Subtotal</p>
                              <p>{typeof selectedOrder.totalAmount === 'number' ? `₹${selectedOrder.totalAmount.toFixed(2)}` : (selectedOrder.totalAmount ?? '-')}</p>
                            </div>
                            <div className="flex justify-between text-sm text-gray-500 mt-1">
                              <p>Shipping</p>
                              <p>₹0.00</p>
                            </div>
                            <div className="flex justify-between text-sm text-gray-500 mt-1">
                              <p>Tax</p>
                              <p>{typeof selectedOrder.taxAmount === 'number' ? `₹${selectedOrder.taxAmount.toFixed(2)}` : (selectedOrder.taxAmount ?? '-')}</p>
                            </div>
                            <div className="flex justify-between text-sm text-gray-500 mt-1">
                              <p>Discount</p>
                              <p>{typeof selectedOrder.discountAmount === 'number' ? `₹${selectedOrder.discountAmount.toFixed(2)}` : (selectedOrder.discountAmount ?? '-')}</p>
                            </div>
                            <div className="flex justify-between text-lg font-medium text-gray-900 mt-2 pt-2 border-t border-gray-200">
                              <p>Total</p>
                              <p>{typeof selectedOrder.finalAmount === 'number' ? `₹${selectedOrder.finalAmount.toFixed(2)}` : (selectedOrder.finalAmount ?? '-')}</p>
                            </div>
                          </div>
                        </div>

                        <div>
                          <div className="mb-6">
                            <h4 className="text-lg font-medium text-gray-900 mb-2">Delivery Information</h4>
                            <div className="bg-gray-50 p-4 rounded-md">
                              <p className="text-sm font-medium text-gray-900">{selectedOrder.deliveryAddress?.name}</p>
                              <p className="text-sm text-gray-500">{selectedOrder.deliveryAddress?.address}</p>
                              <p className="text-sm text-gray-500">{selectedOrder.deliveryAddress?.city}, {selectedOrder.deliveryAddress?.state} {selectedOrder.deliveryAddress?.postcode}</p>
                              {selectedOrder.deliveryAddress?.contactNumber && (
                                <p className="text-sm text-gray-500 mt-1">Phone: {selectedOrder.deliveryAddress.contactNumber}</p>
                              )}
                            </div>
                          </div>

                          <div>
                            <h4 className="text-lg font-medium text-gray-900 mb-2">Delivery Status</h4>
                            <div className="space-y-4">
                              <div>
                                <div className="flex items-center justify-between">
                                  <span className="text-sm font-medium text-gray-900">Current Status</span>
                                  {getStatusBadge(selectedOrder.status)}
                                </div>
                                {selectedOrder.estimatedDelivery && (
                                  <p className="text-sm text-gray-500 mt-1">Estimated delivery: {formatDate(selectedOrder.estimatedDelivery)}</p>
                                )}
                                {selectedOrder.deliveredAt && (
                                  <p className="text-sm text-gray-500 mt-1">Delivered on: {formatDate(selectedOrder.deliveredAt)}</p>
                                )}
                              </div>

                              <div className="space-y-2">
                                <Button variant="outline" size="sm" onClick={() => trackOrder(selectedOrder)} className="w-full flex items-center justify-center">
                                  <Truck className="h-4 w-4 mr-2" />
                                  Track Order
                                </Button>
                                {selectedOrder.status === 'pending' && (
                                  <Button variant="default" size="sm" onClick={() => { updateOrderStatus(selectedOrder._id, 'in_transit'); setIsViewModalOpen(false); }} className="w-full">Mark as In Transit</Button>
                                )}
                                {selectedOrder.status === 'in_transit' && (
                                  <Button variant="default" size="sm" onClick={() => { updateOrderStatus(selectedOrder._id, 'delivered'); setIsViewModalOpen(false); }} className="w-full">Mark as Delivered</Button>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>

                      <div className="mt-5 sm:mt-6">
                        <button type="button" className="w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 sm:text-sm" onClick={() => setIsViewModalOpen(false)}>Close</button>
                      </div>
                    </div>
                  )}
                </div>
              </Transition.Child>
            </div>
          </Dialog>
        </Transition.Root>

        {/* Tracking Modal */}
        <Transition.Root show={isTrackingModalOpen} as={Fragment}>
          <Dialog as="div" className="fixed z-10 inset-0 overflow-y-auto" onClose={setIsTrackingModalOpen}>
            <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
              <Transition.Child as={Fragment} enter="ease-out duration-300" enterFrom="opacity-0" enterTo="opacity-100" leave="ease-in duration-200" leaveFrom="opacity-100" leaveTo="opacity-0">
                <Dialog.Overlay className="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" />
              </Transition.Child>

              <span className="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>

              <Transition.Child as={Fragment} enter="ease-out duration-300" enterFrom="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95" enterTo="opacity-100 translate-y-0 sm:scale-100" leave="ease-in duration-200" leaveFrom="opacity-100 translate-y-0 sm:scale-100" leaveTo="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95">
                <div className="inline-block align-bottom bg-white rounded-lg px-4 pt-5 pb-4 text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full sm:p-6">
                  {trackingInfo ? (
                    <div>
                      <div className="flex justify-between items-start">
                        <div>
                          <Dialog.Title as="h3" className="text-lg leading-6 font-medium text-gray-900">Order Tracking</Dialog.Title>
                          <p className="text-sm text-gray-500 mt-1">Order #{trackingInfo.orderNumber ?? '-'}</p>
                        </div>
                        <div className="ml-3 h-7 flex items-center">
                          <button type="button" className="bg-white rounded-md text-gray-400 hover:text-gray-500 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500" onClick={() => setIsTrackingModalOpen(false)}>
                            <span className="sr-only">Close</span>
                            <svg className="h-6 w-6" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke="currentColor" aria-hidden="true">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </div>
                      </div>

                      <div className="mt-6">
                        <div className="flow-root">
                          <ul className="-mb-8">
                            <li>
                              <div className="relative pb-8">
                                <span className="absolute top-4 left-4 -ml-px h-full w-0.5 bg-gray-200" aria-hidden="true"></span>
                                <div className="relative flex space-x-3">
                                  <div>
                                    <span className="h-8 w-8 rounded-full bg-blue-500 flex items-center justify-center ring-8 ring-white">
                                      <CheckCircle className="h-5 w-5 text-white" aria-hidden="true" />
                                    </span>
                                  </div>
                                  <div className="min-w-0 flex-1 pt-1.5 flex justify-between space-x-4">
                                    <div>
                                      <p className="text-sm text-gray-500">Order placed on {formatDate(new Date().toISOString())}</p>
                                    </div>
                                  </div>
                                </div>
                              </div>
                            </li>

                            <li>
                              <div className="relative pb-8">
                                <span className="absolute top-4 left-4 -ml-px h-full w-0.5 bg-gray-200" aria-hidden="true"></span>
                                <div className="relative flex space-x-3">
                                  <div>
                                    <span className="h-8 w-8 rounded-full bg-green-500 flex items-center justify-center ring-8 ring-white">
                                      <Truck className="h-5 w-5 text-white" aria-hidden="true" />
                                    </span>
                                  </div>
                                  <div className="min-w-0 flex-1 pt-1.5 flex justify-between space-x-4">
                                    <div>
                                      <p className="text-sm text-gray-500">Your order is out for delivery</p>
                                      <p className="text-sm text-gray-500">Estimated delivery: {formatDate(trackingInfo.estimatedDelivery)}</p>
                                    </div>
                                  </div>
                                </div>
                              </div>
                            </li>

                            <li>
                              <div className="relative pb-8">
                                <div className="relative flex space-x-3">
                                  <div>
                                    <span className="h-8 w-8 rounded-full bg-gray-300 flex items-center justify-center ring-8 ring-white">
                                      <CheckCircle className="h-5 w-5 text-white" aria-hidden="true" />
                                    </span>
                                  </div>
                                  <div className="min-w-0 flex-1 pt-1.5 flex justify-between space-x-4">
                                    <div>
                                      <p className="text-sm text-gray-500">Order delivered on {formatDate(trackingInfo.estimatedDelivery)}</p>
                                    </div>
                                  </div>
                                </div>
                              </div>
                            </li>
                          </ul>
                        </div>
                      </div>
                    </div>
                  ) : (
                    <div className="text-center">
                      <div className="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-gray-100">
                        <Truck className="h-6 w-6 text-gray-400" />
                      </div>
                      <h3 className="mt-2 text-lg font-medium text-gray-900">No tracking information available</h3>
                      <p className="mt-1 text-sm text-gray-500">We couldn't find any tracking information for this order.</p>
                    </div>
                  )}

                  <div className="mt-5 sm:mt-6">
                    <Button type="button" variant="outline" className="w-full" onClick={() => setIsTrackingModalOpen(false)}>Close</Button>
                  </div>
                </div>
              </Transition.Child>
            </div>
          </Dialog>
        </Transition.Root>
      </Layout>
    </div>
  );
};

export default DeliveryPage;
