import React from 'react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import StatsCard from '../../components/shared/StatsCard';
import { storeSidebarItems } from './StoreLayout';
import { 
  DollarSign, 
  ShoppingCart, 
  Package, 
  Truck,
  Clock,
  CheckCircle
} from 'lucide-react';

const StoreDashboard: React.FC = () => {
  const stats = [
    {
      title: 'Store Revenue',
      value: '$12,450',
      icon: DollarSign,
      trend: { value: 8, isPositive: true },
      color: 'green' as const
    },
    {
      title: 'Store Orders',
      value: '145',
      icon: ShoppingCart,
      trend: { value: 12, isPositive: true },
      color: 'blue' as const
    },
    {
      title: 'Products in Stock',
      value: '1,247',
      icon: Package,
      trend: { value: -3, isPositive: false },
      color: 'purple' as const
    },
    {
      title: 'Pending Deliveries',
      value: '23',
      icon: Truck,
      trend: { value: 5, isPositive: false },
      color: 'orange' as const
    }
  ];

  const recentOrders = [
    { id: '#ORD-101', customer: 'Alice Johnson', amount: '$89.50', status: 'packed', time: '2 hours ago' },
    { id: '#ORD-102', customer: 'Bob Smith', amount: '$156.25', status: 'confirmed', time: '3 hours ago' },
    { id: '#ORD-103', customer: 'Carol Williams', amount: '$67.80', status: 'delivered', time: '5 hours ago' },
    { id: '#ORD-104', customer: 'David Brown', amount: '$124.00', status: 'out_for_delivery', time: '6 hours ago' },
  ];

  const lowStockProducts = [
    { name: 'Aspirin 100mg', current: 5, minimum: 20, category: 'Pain Relief' },
    { name: 'Vitamin D3', current: 8, minimum: 15, category: 'Supplements' },
    { name: 'Blood Pressure Monitor', current: 2, minimum: 10, category: 'Medical Devices' },
    { name: 'Thermometer Digital', current: 3, minimum: 12, category: 'Medical Devices' }
  ];

  const getStatusBadge = (status: string) => {
    const statusColors = {
      delivered: 'bg-green-100 text-green-800',
      out_for_delivery: 'bg-blue-100 text-blue-800',
      packed: 'bg-yellow-100 text-yellow-800',
      confirmed: 'bg-purple-100 text-purple-800',
      pending: 'bg-gray-100 text-gray-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status as keyof typeof statusColors] || statusColors.pending}`;
  };

  return (
    <Layout title="Store Dashboard" sidebar={<Sidebar items={storeSidebarItems} />}>
      <div className="space-y-6">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <StatsCard
              key={index}
              title={stat.title}
              value={stat.value}
              icon={stat.icon}
              trend={stat.trend}
              color={stat.color}
            />
          ))}
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Orders */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Recent Store Orders</h3>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Time
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {recentOrders.map((order) => (
                    <tr key={order.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">{order.id}</div>
                          <div className="text-sm text-gray-500">{order.amount}</div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.customer}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getStatusBadge(order.status)}>
                          {order.status.replace('_', ' ')}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {order.time}
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {/* Low Stock Alert */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Low Stock Alert</h3>
            </div>
            <div className="p-6">
              <div className="space-y-4">
                {lowStockProducts.map((product, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-red-50 border border-red-200 rounded-lg">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-gray-900">{product.name}</p>
                      <p className="text-xs text-gray-500">{product.category}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-red-600">
                        {product.current} left
                      </p>
                      <p className="text-xs text-gray-500">
                        Min: {product.minimum}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
              <button className="w-full mt-4 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                Reorder Products
              </button>
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Quick Actions</h3>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {[
                { label: 'Process Orders', icon: Clock, count: '5 pending' },
                { label: 'Update Inventory', icon: Package, count: '12 items' },
                { label: 'Assign Delivery', icon: Truck, count: '3 ready' },
                { label: 'Mark Delivered', icon: CheckCircle, count: '8 completed' }
              ].map((action, index) => {
                const Icon = action.icon;
                return (
                  <button
                    key={index}
                    className="p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-all text-left"
                  >
                    <div className="flex items-center mb-2">
                      <Icon className="w-5 h-5 text-blue-600 mr-2" />
                      <span className="text-sm font-medium text-gray-900">{action.label}</span>
                    </div>
                    <p className="text-xs text-gray-500">{action.count}</p>
                  </button>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </Layout>
  );
};

export default StoreDashboard;