import React, { useState, useEffect, useCallback } from 'react';
import { Save } from 'lucide-react';
import api from '@/services/api';
import Layout from '@/components/shared/Layout';
import Sidebar from '@/components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { toast } from 'react-hot-toast';

// Custom Input component
const Input = React.forwardRef<HTMLInputElement, React.InputHTMLAttributes<HTMLInputElement>>(
  ({ className = '', ...props }, ref) => (
    <input
      ref={ref}
      className={`flex h-10 w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 ${className}`}
      {...props}
    />
  )
);
Input.displayName = 'Input';

// Custom Select component
const Select = React.forwardRef<HTMLSelectElement, React.SelectHTMLAttributes<HTMLSelectElement>>(
  ({ className = '', ...props }, ref) => (
    <select
      ref={ref}
      className={`flex h-10 w-full rounded-md border border-gray-300 bg-white px-3 py-2 text-sm shadow-sm focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 ${className}`}
      {...props}
    >
      {props.children}
    </select>
  )
);
Select.displayName = 'Select';

// Simple reusable button component
const Button = ({
  onClick,
  children,
  className = '',
  variant = 'default',
  disabled = false,
  type = 'button',
  ...props
}: {
  onClick?: () => void;
  children: React.ReactNode;
  className?: string;
  variant?: 'default' | 'outline' | 'destructive';
  disabled?: boolean;
  type?: 'button' | 'submit' | 'reset';
} & React.ButtonHTMLAttributes<HTMLButtonElement>) => {
  const base =
    'inline-flex items-center justify-center rounded-md px-4 py-2 text-sm font-medium focus:outline-none focus:ring-2 focus:ring-offset-2 disabled:opacity-50 disabled:pointer-events-none';
  const variants = {
    default: 'bg-blue-600 text-white hover:bg-blue-700 focus:ring-blue-500',
    outline: 'border border-gray-300 bg-white text-gray-700 hover:bg-gray-50 focus:ring-blue-500',
    destructive: 'bg-red-600 text-white hover:bg-red-700 focus:ring-red-500',
  };
  return (
    <button
      type={type}
      onClick={onClick}
      disabled={disabled}
      className={`${base} ${variants[variant]} ${className}`}
      {...props}
    >
      {children}
    </button>
  );
};

interface WebSettingsData {
  // General
  siteName: string;
  siteUrl: string;
  siteEmail: string;
  sitePhone: string;
  siteAddress: string;
  siteDescription: string;
  maintenanceMode: boolean;
  allowRegistration: boolean;
  enableEmailVerification: boolean;
  defaultUserRole: string;
  
  // Localization
  timezone: string;
  dateFormat: string;
  timeFormat: string;
  currency: string;
  currencySymbol: string;
  currencyPosition: 'left' | 'right';
  thousandSeparator: string;
  decimalSeparator: string;
  numberOfDecimals: number;
  
  // Store Information
  siteStoreName: string;
  siteStoreEmail: string;
  siteStorePhone: string;
  siteStoreAddress: string;
  siteStoreCountry: string;
  siteStoreState: string;
  siteStoreCity: string;
  siteStoreZipCode: string;
  siteStoreCurrency: string;
  siteStoreCurrencySymbol: string;
  siteStoreTimezone: string;
  siteStoreLanguage: string;
  siteStoreLogo: string;
  siteStoreIcon: string;
  siteStoreMetaTitle: string;
  siteStoreMetaDescription: string;
  siteStoreMetaKeywords: string;
}

const TABS = [
  { key: 'general', label: 'General' },
  { key: 'store', label: 'Site' },
  { key: 'localization', label: 'Localization' },
  { key: 'email', label: 'Email' },
  { key: 'seo', label: 'SEO' },
  { key: 'social', label: 'Social Media' },
  { key: 'payments', label: 'Payments' },
] as const;

type TabKey = typeof TABS[number]['key'];

const WebSettings: React.FC = () => {
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [activeTab, setActiveTab] = useState<TabKey>('general');
  
  // Initialize active tab from URL
  useEffect(() => {
    const pathParts = window.location.pathname.split('/');
    const tabFromUrl = pathParts[pathParts.length - 1] as TabKey;
    if (TABS.some(tab => tab.key === tabFromUrl)) {
      setActiveTab(tabFromUrl);
    }
  }, []);

  // Update URL when tab changes
  const handleTabChange = useCallback((tabKey: TabKey) => {
    setActiveTab(tabKey);
    window.history.pushState({}, '', `/admin/web-settings/${tabKey}`);
  }, []);
  const [formData, setFormData] = useState<WebSettingsData>({
    // General
    siteName: 'Helthstra',
    siteUrl: 'http://localhost:3000',
    siteEmail: 'contact@helthstra.com',
    sitePhone: '',
    siteAddress: '',
    siteDescription: '',
    maintenanceMode: false,
    allowRegistration: true,
    enableEmailVerification: true,
    defaultUserRole: 'customer',
    
    // Localization
    timezone: 'Asia/Kolkata',
    dateFormat: 'DD/MM/YYYY',
    timeFormat: '12h',
    currency: 'INR',
    currencySymbol: '₹',
    currencyPosition: 'left',
    thousandSeparator: ',',
    decimalSeparator: '.',
    numberOfDecimals: 2,
    
    // Store Information
    siteStoreName: 'Helthstra Store',
    siteStoreEmail: 'store@helthstra.com',
    siteStorePhone: '',
    siteStoreAddress: '',
    siteStoreCountry: 'India',
    siteStoreState: '',
    siteStoreCity: '',
    siteStoreZipCode: '',
    siteStoreCurrency: 'INR',
    siteStoreCurrencySymbol: '₹',
    siteStoreTimezone: 'Asia/Kolkata',
    siteStoreLanguage: 'en',
    siteStoreLogo: '',
    siteStoreIcon: '',
    siteStoreMetaTitle: '',
    siteStoreMetaDescription: '',
    siteStoreMetaKeywords: '',
  });

  const fetchSettings = useCallback(async () => {
    try {
      setIsLoading(true);
      const response = await api.get('/web-settings');
      if (response.data.success) {
        setFormData((prev) => ({
          ...prev,
          ...response.data.data,
        }));
      }
    } catch (error) {
      toast.error('Failed to load settings');
    } finally {
      setIsLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchSettings();
  }, [fetchSettings]);

  const handleInputChange = useCallback(
    async (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
      const { name, type } = e.target;
      const value = type === 'checkbox' 
        ? (e.target as HTMLInputElement).checked 
        : e.target.value;
      
      // Update the form data immediately for a responsive UI
      setFormData(prev => ({
        ...prev,
        [name]: value
      }));
      
      // For maintenance mode, submit immediately
      if (name === 'maintenanceMode') {
        try {
          setIsSaving(true);
          const updatedData = { ...formData, [name]: value };
          const response = await api.put('/web-settings', updatedData);
          
          if (response.data.success) {
            toast.success('Maintenance mode updated successfully');
          } else {
            // Revert the toggle if the update fails
            setFormData(prev => ({
              ...prev,
              [name]: !value
            }));
            toast.error('Failed to update maintenance mode');
          }
        } catch (error: any) {
          // Revert the toggle on error
          setFormData(prev => ({
            ...prev,
            [name]: !value
          }));
          toast.error('Error updating maintenance mode: ' + (error.response?.data?.message || error.message));
        } finally {
          setIsSaving(false);
        }
      }
    },
    [formData]
  );

  const handleDescriptionChange = useCallback(
    (e: React.ChangeEvent<HTMLTextAreaElement>) => {
      setFormData((prev) => ({ ...prev, siteDescription: e.target.value }));
    },
    []
  );

  const handleRadioChange = useCallback(
    (name: keyof WebSettingsData, value: any) => {
      setFormData((prev) => ({ ...prev, [name]: value }));
    },
    []
  );

  const handleSubmit = async (e?: React.FormEvent) => {
    if (e) e.preventDefault();
    try {
      setIsSaving(true);
      console.log('Submitting form data:', JSON.stringify(formData, null, 2));
      
      const response = await api.put('/web-settings', formData);
      
      console.log('API Response:', JSON.stringify(response.data, null, 2));
      
      if (response.data.success) {
        console.log('Settings saved successfully');
        toast.success('Settings saved successfully');
      } else {
        console.error('API returned success:false', response.data);
        toast.error('Failed to save settings: ' + (response.data.message || 'Unknown error'));
      }
    } catch (error: any) {
      console.error('Error saving settings:', error);
      console.error('Error response data:', error.response?.data);
      toast.error('Failed to save settings: ' + (error.response?.data?.message || error.message));
    } finally {
      setIsSaving(false);
    }
  };

  if (isLoading) {
    return (
      <Layout title="Web Settings" sidebar={<Sidebar items={adminSidebarItems} />}>
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
        </div>
      </Layout>
    );
  }

  return (
    <Layout title="Web Settings" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6 max-w-4xl mx-auto pb-12">
        {/* Tab navigation */}
        <div className="flex border-b mb-6">
          {TABS.map((tab) => (
            <button
              key={tab.key}
              type="button"
              className={`py-2 px-6 -mb-px border-b-2 text-sm font-medium transition-colors ${
                activeTab === tab.key
                  ? 'border-blue-600 text-blue-600 bg-blue-50'
                  : 'border-transparent text-gray-500'
              }`}
              onClick={() => handleTabChange(tab.key as TabKey)}
              aria-current={activeTab === tab.key ? 'page' : undefined}
            >
              {tab.label}
            </button>
          ))}
        </div>

        <form onSubmit={handleSubmit} className="space-y-6">
          {/* General Tab */}
          {activeTab === 'general' && (
            <div className="bg-white p-6 rounded-lg shadow space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <label htmlFor="siteName" className="block text-sm font-medium text-gray-700">
                    Site Name
                  </label>
                  <Input
                    id="siteName"
                    name="siteName"
                    value={formData.siteName}
                    onChange={handleInputChange}
                    placeholder="Enter site name"
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="siteUrl" className="block text-sm font-medium text-gray-700">
                    Site URL
                  </label>
                  <Input
                    id="siteUrl"
                    name="siteUrl"
                    type="url"
                    value={formData.siteUrl}
                    onChange={handleInputChange}
                    placeholder="https://example.com"
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="siteEmail" className="block text-sm font-medium text-gray-700">
                    Contact Email
                  </label>
                  <Input
                    id="siteEmail"
                    name="siteEmail"
                    type="email"
                    value={formData.siteEmail}
                    onChange={handleInputChange}
                    placeholder="contact@example.com"
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="sitePhone" className="block text-sm font-medium text-gray-700">
                    Contact Phone
                  </label>
                  <Input
                    id="sitePhone"
                    name="sitePhone"
                    type="tel"
                    value={formData.sitePhone}
                    onChange={handleInputChange}
                    placeholder="+1234567890"
                  />
                </div>
              </div>
              <div>
                <label htmlFor="siteAddress" className="block text-sm font-medium text-gray-700">
                  Business Address
                </label>
                <Input
                  id="siteAddress"
                  name="siteAddress"
                  value={formData.siteAddress}
                  onChange={handleInputChange}
                />
              </div>
              <div>
                <label htmlFor="siteDescription" className="block text-sm font-medium text-gray-700">
                  Site Description
                </label>
                <textarea
                  id="siteDescription"
                  name="siteDescription"
                  rows={3}
                  className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border border-gray-300 rounded-md"
                  value={formData.siteDescription}
                  onChange={handleDescriptionChange}
                />
              </div>
              <div className="flex items-center space-x-6 pt-2">
                <div className="flex items-center space-x-2">
                  <label htmlFor="maintenanceMode" className="text-sm font-medium text-gray-700">
                    Maintenance Mode:
                  </label>
                  <select
                    id="maintenanceMode"
                    name="maintenanceMode"
                    value={formData.maintenanceMode ? 'true' : 'false'}
                    onChange={handleInputChange}
                    className="h-8 pl-2 pr-8 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                  >
                    <option value="false">Off</option>
                    <option value="true">On</option>
                  </select>
                </div>
                <label className="flex items-center space-x-2">
                  <input
                    type="checkbox"
                    name="allowRegistration"
                    checked={formData.allowRegistration}
                    onChange={handleInputChange}
                    className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span>Allow User Registration</span>
                </label>
                <label className="flex items-center space-x-2">
                  <input
                    type="checkbox"
                    name="enableEmailVerification"
                    checked={formData.enableEmailVerification}
                    onChange={handleInputChange}
                    className="h-4 w-4 text-blue-600 border-gray-300 rounded focus:ring-blue-500"
                  />
                  <span>Enable Email Verification</span>
                </label>
              </div>
            </div>
          )}

          {/* Site Tab */}
          {activeTab === 'store' && (
            <div className="bg-white p-6 rounded-lg shadow space-y-6">
              <div className="space-y-1">
                <h2 className="text-lg font-medium text-gray-900">Site Information</h2>
                <p className="text-sm text-gray-500">Set your site details and contact information.</p>
              </div>
              
              <div className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                  <div className="space-y-2">
                    <label htmlFor="siteStoreName" className="block text-sm font-medium text-gray-700">
                      Site Name *
                    </label>
                    <Input
                      id="siteStoreName"
                      name="siteStoreName"
                      value={formData.siteStoreName}
                      onChange={handleInputChange}
                      placeholder="Enter site name"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreEmail" className="block text-sm font-medium text-gray-700">
                      Email *
                    </label>
                    <Input
                      id="siteStoreEmail"
                      name="siteStoreEmail"
                      type="email"
                      value={formData.siteStoreEmail}
                      onChange={handleInputChange}
                      placeholder="site@example.com"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStorePhone" className="block text-sm font-medium text-gray-700">
                      Phone
                    </label>
                    <Input
                      id="siteStorePhone"
                      name="siteStorePhone"
                      type="tel"
                      value={formData.siteStorePhone}
                      onChange={handleInputChange}
                      placeholder="+91 1234567890"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreCountry" className="block text-sm font-medium text-gray-700">
                      Country/Region *
                    </label>
                    <Input
                      id="siteStoreCountry"
                      name="siteStoreCountry"
                      value={formData.siteStoreCountry}
                      onChange={handleInputChange}
                      placeholder="e.g. India"
                      required
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreState" className="block text-sm font-medium text-gray-700">
                      State/Province
                    </label>
                    <Input
                      id="siteStoreState"
                      name="siteStoreState"
                      value={formData.siteStoreState}
                      onChange={handleInputChange}
                      placeholder="e.g. Maharashtra"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreCity" className="block text-sm font-medium text-gray-700">
                      City
                    </label>
                    <Input
                      id="siteStoreCity"
                      name="siteStoreCity"
                      value={formData.siteStoreCity}
                      onChange={handleInputChange}
                      placeholder="e.g. Mumbai"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreZipCode" className="block text-sm font-medium text-gray-700">
                      ZIP/Postal Code
                    </label>
                    <Input
                      id="siteStoreZipCode"
                      name="siteStoreZipCode"
                      value={formData.siteStoreZipCode}
                      onChange={handleInputChange}
                      placeholder="e.g. 400001"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreAddress" className="block text-sm font-medium text-gray-700">
                      Address Line 1
                    </label>
                    <Input
                      id="siteStoreAddress"
                      name="siteStoreAddress"
                      value={formData.siteStoreAddress}
                      onChange={handleInputChange}
                      placeholder="123 Main St"
                    />
                  </div>
                </div>

                <div className="pt-4 border-t border-gray-200">
                  <h3 className="text-md font-medium text-gray-900 mb-4">Site Display</h3>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div className="space-y-2">
                      <label htmlFor="siteStoreCurrency" className="block text-sm font-medium text-gray-700">
                        Currency
                      </label>
                      <Select
                        id="siteStoreCurrency"
                        name="siteStoreCurrency"
                        value={formData.siteStoreCurrency}
                        onChange={handleInputChange}
                      >
                        <option value="INR">Indian Rupee (₹)</option>
                        <option value="USD">US Dollar ($)</option>
                        <option value="EUR">Euro (€)</option>
                        <option value="GBP">British Pound (£)</option>
                      </Select>
                    </div>
                    <div className="space-y-2">
                      <label htmlFor="siteStoreLanguage" className="block text-sm font-medium text-gray-700">
                        Language
                      </label>
                      <Select
                        id="siteStoreLanguage"
                        name="siteStoreLanguage"
                        value={formData.siteStoreLanguage}
                        onChange={handleInputChange}
                      >
                        <option value="en">English</option>
                        <option value="hi">हिंदी</option>
                        <option value="mr">मराठी</option>
                        <option value="ta">தமிழ்</option>
                      </Select>
                    </div>
                  </div>
                </div>

                <div className="pt-4 border-t border-gray-200">
                  <h3 className="text-md font-medium text-gray-900 mb-4">Site Logo</h3>
                  <div className="flex items-center space-x-4">
                    <div className="flex-shrink-0">
                      {formData.siteStoreLogo ? (
                        <img
                          className="h-16 w-16 object-cover rounded-md"
                          src={formData.siteStoreLogo}
                          alt="Site Logo"
                        />
                      ) : (
                        <div className="h-16 w-16 rounded-md bg-gray-200 flex items-center justify-center">
                          <span className="text-gray-400 text-xs">Logo</span>
                        </div>
                      )}
                    </div>
                    <div className="flex-1">
                      <label className="block">
                        <span className="sr-only">Choose logo</span>
                        <input
                          type="file"
                          className="block w-full text-sm text-gray-500
                            file:mr-4 file:py-2 file:px-4
                            file:rounded-md file:border-0
                            file:text-sm file:font-semibold
                            file:bg-blue-50 file:text-blue-700
                            hover:file:bg-blue-100"
                          onChange={(e) => {
                            const file = e.target.files?.[0];
                            if (file) {
                              const reader = new FileReader();
                              reader.onloadend = () => {
                                setFormData(prev => ({
                                  ...prev,
                                  siteStoreLogo: reader.result as string
                                }));
                              };
                              reader.readAsDataURL(file);
                            }
                          }}
                        />
                      </label>
                      <p className="mt-1 text-xs text-gray-500">
                        Recommended size: 200x200px. Max file size: 2MB
                      </p>
                    </div>
                  </div>
                </div>

                <div className="pt-4 border-t border-gray-200">
                  <h3 className="text-md font-medium text-gray-900 mb-4">Site Icon</h3>
                  <div className="flex items-center space-x-4">
                    <div className="flex-shrink-0">
                      {formData.siteStoreIcon ? (
                        <img
                          className="h-10 w-10 object-cover rounded-md"
                          src={formData.siteStoreIcon}
                          alt="Site Icon"
                        />
                      ) : (
                        <div className="h-10 w-10 rounded-md bg-gray-200 flex items-center justify-center">
                          <span className="text-gray-400 text-xs">Icon</span>
                        </div>
                      )}
                    </div>
                    <div className="flex-1">
                      <label className="block">
                        <span className="sr-only">Choose icon</span>
                        <input
                          type="file"
                          className="block w-full text-sm text-gray-500
                            file:mr-4 file:py-2 file:px-4
                            file:rounded-md file:border-0
                            file:text-sm file:font-semibold
                            file:bg-blue-50 file:text-blue-700
                            hover:file:bg-blue-100"
                          onChange={(e) => {
                            const file = e.target.files?.[0];
                            if (file) {
                              const reader = new FileReader();
                              reader.onloadend = () => {
                                setFormData(prev => ({
                                  ...prev,
                                  siteStoreIcon: reader.result as string
                                }));
                              };
                              reader.readAsDataURL(file);
                            }
                          }}
                        />
                      </label>
                      <p className="mt-1 text-xs text-gray-500">
                        Recommended size: 100x100px. Max file size: 1MB
                      </p>
                    </div>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-6 pt-4">
                  <div className="space-y-2">
                    <label htmlFor="siteStoreMetaKeywords" className="block text-sm font-medium text-gray-700">
                      Meta Keywords
                    </label>
                    <Input
                      id="siteStoreMetaKeywords"
                      name="siteStoreMetaKeywords"
                      value={formData.siteStoreMetaKeywords}
                      onChange={handleInputChange}
                      placeholder="Enter meta keywords (comma separated)"
                    />
                  </div>
                  <div className="space-y-2">
                    <label htmlFor="siteStoreMetaDescription" className="block text-sm font-medium text-gray-700">
                      Meta Description
                    </label>
                    <textarea
                      id="siteStoreMetaDescription"
                      name="siteStoreMetaDescription"
                      rows={2}
                      className="shadow-sm focus:ring-blue-500 focus:border-blue-500 block w-full sm:text-sm border border-gray-300 rounded-md"
                      value={formData.siteStoreMetaDescription}
                      onChange={(e) => setFormData({...formData, siteStoreMetaDescription: e.target.value})}
                    />
                  </div>
                </div>
              </div>
            </div>
          )}

          {/* Localization Tab */}
          {activeTab === 'localization' && (
            <div className="bg-white p-6 rounded-lg shadow space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <label htmlFor="timezone" className="block text-sm font-medium text-gray-700">
                    Timezone
                  </label>
                  <Select
                    id="timezone"
                    name="timezone"
                    value={formData.timezone}
                    onChange={handleInputChange}
                  >
                    <option value="UTC">UTC</option>
                    <option value="America/New_York">Eastern Time (ET)</option>
                    <option value="America/Chicago">Central Time (CT)</option>
                    <option value="America/Denver">Mountain Time (MT)</option>
                    <option value="America/Los_Angeles">Pacific Time (PT)</option>
                    <option value="Asia/Kolkata">India Standard Time (IST)</option>
                  </Select>
                </div>
                <div className="space-y-2">
                  <label htmlFor="dateFormat" className="block text-sm font-medium text-gray-700">
                    Date Format
                  </label>
                  <Select
                    id="dateFormat"
                    name="dateFormat"
                    value={formData.dateFormat}
                    onChange={handleInputChange}
                  >
                    <option value="DD/MM/YYYY">DD/MM/YYYY</option>
                    <option value="MM/DD/YYYY">MM/DD/YYYY</option>
                    <option value="YYYY-MM-DD">YYYY-MM-DD</option>
                  </Select>
                </div>
                <div className="space-y-2">
                  <label htmlFor="timeFormat" className="block text-sm font-medium text-gray-700">
                    Time Format
                  </label>
                  <Select
                    id="timeFormat"
                    name="timeFormat"
                    value={formData.timeFormat}
                    onChange={handleInputChange}
                  >
                    <option value="12h">12-hour</option>
                    <option value="24h">24-hour</option>
                  </Select>
                </div>
                <div className="space-y-2">
                  <label htmlFor="currency" className="block text-sm font-medium text-gray-700">
                    Currency
                  </label>
                  <Select
                    id="currency"
                    name="currency"
                    value={formData.currency}
                    onChange={handleInputChange}
                  >
                    <option value="USD">US Dollar ($)</option>
                    <option value="EUR">Euro (€)</option>
                    <option value="GBP">British Pound (£)</option>
                    <option value="INR">Indian Rupee (₹)</option>
                    <option value="JPY">Japanese Yen (¥)</option>
                  </Select>
                </div>
              </div>
              <div className="grid gap-6 md:grid-cols-3">
                <div className="space-y-2">
                  <label htmlFor="currencySymbol" className="block text-sm font-medium text-gray-700">
                    Currency Symbol
                  </label>
                  <Input
                    id="currencySymbol"
                    name="currencySymbol"
                    value={formData.currencySymbol}
                    onChange={handleInputChange}
                    maxLength={3}
                  />
                </div>
                <div className="space-y-2">
                  <label className="block text-sm font-medium text-gray-700">
                    Currency Position
                  </label>
                  <div className="flex items-center space-x-4 mt-2">
                    <label className="flex items-center space-x-2">
                      <input
                        type="radio"
                        name="currencyPosition"
                        value="left"
                        checked={formData.currencyPosition === 'left'}
                        onChange={() => handleRadioChange('currencyPosition', 'left')}
                        className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
                      />
                      <span>Left ($99.99)</span>
                    </label>
                    <label className="flex items-center space-x-2">
                      <input
                        type="radio"
                        name="currencyPosition"
                        value="right"
                        checked={formData.currencyPosition === 'right'}
                        onChange={() => handleRadioChange('currencyPosition', 'right')}
                        className="h-4 w-4 text-blue-600 border-gray-300 focus:ring-blue-500"
                      />
                      <span>Right (99.99$)</span>
                    </label>
                  </div>
                </div>
                <div className="space-y-2">
                  <label htmlFor="thousandSeparator" className="block text-sm font-medium text-gray-700">
                    Thousand Separator
                  </label>
                  <Input
                    id="thousandSeparator"
                    name="thousandSeparator"
                    value={formData.thousandSeparator}
                    onChange={handleInputChange}
                    maxLength={1}
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="decimalSeparator" className="block text-sm font-medium text-gray-700">
                    Decimal Separator
                  </label>
                  <Input
                    id="decimalSeparator"
                    name="decimalSeparator"
                    value={formData.decimalSeparator}
                    onChange={handleInputChange}
                    maxLength={1}
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="numberOfDecimals" className="block text-sm font-medium text-gray-700">
                    Number of Decimals
                  </label>
                  <Input
                    id="numberOfDecimals"
                    name="numberOfDecimals"
                    type="number"
                    min={0}
                    max={4}
                    value={formData.numberOfDecimals}
                    onChange={handleInputChange}
                  />
                </div>
              </div>
            </div>
          )}

          {/* Placeholder notice for other tabs */}
          {(activeTab === 'email' ||
            activeTab === 'payments' ||
            activeTab === 'social' ||
            activeTab === 'seo') && (
            <div className="bg-yellow-50 border-l-4 border-yellow-400 p-8 rounded">
              <div className="flex">
                <div className="flex-shrink-0">
                  <svg
                    className="h-5 w-5 text-yellow-400"
                    xmlns="http://www.w3.org/2000/svg"
                    viewBox="0 0 20 20"
                    fill="currentColor"
                  >
                    <path
                      fillRule="evenodd"
                      d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z"
                      clipRule="evenodd"
                    />
                  </svg>
                </div>
                <div className="ml-3">
                  <p className="text-sm text-yellow-700">
                    {activeTab.charAt(0).toUpperCase() + activeTab.slice(1)} settings will be available in the next update.
                  </p>
                </div>
              </div>
            </div>
          )}

          <div className="flex justify-end pt-6">
            <Button
              type="submit"
              disabled={isSaving}
              className="inline-flex justify-center"
            >
              <Save className="w-4 h-4 mr-2" />
              {isSaving ? 'Saving...' : 'Save Changes'}
            </Button>
          </div>
        </form>
      </div>
    </Layout>
  );
};

export default WebSettings;
