import React, { useState } from 'react';
import { useQuery, useMutation, useQueryClient } from 'react-query';
import { format } from 'date-fns';
import { useNavigate } from 'react-router-dom';
import { toast } from 'react-toastify';
import api from '../../../services/api';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, Filter, Eye, CheckCircle, XCircle, Clock, RefreshCw, Trash2, Edit, X } from 'lucide-react';

// Type definitions
interface ReturnItem {
  _id: string;
  returnId: string;
  order: {
    _id: string;
    orderNumber: string;
  } | string; // Can be string if not populated
  user: {
    _id: string;
    name: string;
    email: string;
    phone?: string;
  } | string; // Can be string if not populated
  reason: string;
  notes?: string;
  status: 'pending' | 'approved' | 'rejected' | 'processing' | 'completed';
  refundStatus?: 'pending' | 'processing' | 'completed' | 'failed';
  amount: number;
  refundAmount?: number;
  refundMethod?: 'original_payment' | 'store_credit' | 'bank_transfer';
  items: Array<{
    product: string | {
      _id: string;
      name: string;
      image?: string;
    };
    name: string;
    quantity: number;
    price: number;
    returnReason?: string;
    status?: string;
  }>;
  returnDate?: string;
  type?: string;
  processedBy?: string | {
    _id: string;
    name: string;
    email: string;
  };
  createdAt: string;
  updatedAt: string;
  orderNumber?: string; // Virtual field from backend
  userDetails?: {
    name: string;
    email: string;
    phone?: string;
  };
}

interface RefundItem {
  _id: string;
  refundId: string;
  order: {
    _id: string;
    orderNumber: string;
  };
  user: {
    _id: string;
    name: string;
    email: string;
  };
  amount: number;
  status: 'pending' | 'processing' | 'completed' | 'failed';
  method: string;
  transactionId?: string;
  createdAt: string;
  updatedAt: string;
}

// Type definitions
interface Return {
  _id: string;
  returnId: string;
  order: { orderNumber: string };
  customer: { name: string };
  reason: string;
  status: string;
  createdAt: string;
  amount: number;
  items: { name: string }[];
}

interface Refund {
  _id: string;
  refundId: string;
  order: { orderNumber: string };
  customer: { name: string };
  amount: number;
  status: string;
  createdAt: string;
  method: string;
  transactionId: string;
}

// Fetch returns data with proper typing and error handling
const fetchReturns = async (status: string): Promise<ReturnItem[]> => {
  try {
    // Convert array to comma-separated string for the backend
    const populate = 'order,user,items.product,processedBy';
    
    const { data } = await api.get('/returns', {
      params: { 
        status: status === 'all' ? '' : status,
        populate,
        sort: '-createdAt' // Sort by newest first
      },
      paramsSerializer: (params) => {
        // Handle array parameters correctly
        const query = new URLSearchParams();
        Object.entries(params).forEach(([key, value]) => {
          if (value !== undefined && value !== null && value !== '') {
            query.append(key, value);
          }
        });
        return query.toString();
      }
    });
    
    // Transform the API response to match the expected frontend structure
    return (data.data || []).map((item: any) => {
      // If items array is missing or empty, create a default item from the return data
      let items = [];
      if (!item.items || item.items.length === 0) {
        items = [{
          _id: item._id,
          name: 'Return Item',
          quantity: 1,
          price: item.amount || 0
        }];
      } else {
        items = item.items.map((i: any) => ({
          _id: i._id || Math.random().toString(36).substr(2, 9),
          name: i.name || 'Unknown Product',
          quantity: i.quantity || 1,
          price: i.price || 0
        }));
      }

      return {
        _id: item._id,
        returnId: item.returnId || `RTN-${item._id.toString().substr(-6).toUpperCase()}`,
        order: {
          _id: item.order?._id || '',
          orderNumber: item.order?.orderNumber || 'N/A'
        },
        user: {
          _id: item.user?._id || '',
          name: item.user?.name || 'Unknown',
          email: item.user?.email || ''
        },
        reason: item.reason || item.notes || 'No reason provided',
        status: item.status || 'pending',
        amount: item.amount || 0,
        items: items,
        createdAt: item.createdAt || new Date().toISOString(),
        updatedAt: item.updatedAt || new Date().toISOString()
      };
    });
  } catch (error) {
    console.error('Error fetching returns:', error);
    throw error;
  }
};

// Fetch refunds data with proper typing and error handling
const fetchRefunds = async (status: string): Promise<RefundItem[]> => {
  try {
    const { data } = await api.get('/refunds', {
      params: { 
        status: status === 'all' ? '' : status,
        $populate: ['user', 'order', 'return']
      },
    });
    return data.data || [];
  } catch (error) {
    console.error('Error fetching refunds:', error);
    throw error;
  }
};

const ReturnsRefunds: React.FC = () => {
  const queryClient = useQueryClient();
  const navigate = useNavigate();
  const [activeTab, setActiveTab] = useState<'returns' | 'refunds'>('returns');
  const [statusFilter, setStatusFilter] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');
  
  // State for modals and confirmation dialogs
  const [selectedItem, setSelectedItem] = useState<(ReturnItem & { type: 'return' }) | (RefundItem & { type: 'refund' }) | null>(null);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [isProcessing, setIsProcessing] = useState(false);
  
  // Mutation for updating return/refund status
  const updateStatusMutation = useMutation(
    async ({ type, id, status, data = {} }: { type: 'return' | 'refund', id: string, status: string, data?: any }) => {
      const endpoint = type === 'return' ? `/returns/${id}/status` : `/refunds/${id}/status`;
      return api.patch(endpoint, { status, ...data });
    },
    {
      onSuccess: (_, { type }) => {
        const message = type === 'return' ? 'Return' : 'Refund';
        toast.success(`${message} status updated successfully`);
        queryClient.invalidateQueries([type === 'return' ? 'returns' : 'refunds']);
      },
      onError: (error: any) => {
        toast.error(error.response?.data?.message || 'Failed to update status');
      },
    }
  );

  // Mutation for deleting return/refund
  const deleteMutation = useMutation(
    async ({ type, id }: { type: 'return' | 'refund', id: string }) => {
      const endpoint = type === 'return' ? `/returns/${id}` : `/refunds/${id}`;
      return api.delete(endpoint);
    },
    {
      onSuccess: (_, { type }) => {
        const message = type === 'return' ? 'Return' : 'Refund';
        toast.success(`${message} deleted successfully`);
        queryClient.invalidateQueries([type === 'return' ? 'returns' : 'refunds']);
      },
      onError: (error: any) => {
        toast.error(error.response?.data?.message || 'Failed to delete');
      },
    }
  );

  // Handle view details
  const handleViewDetails = (item: ReturnItem | RefundItem) => {
    const type = 'returnId' in item ? 'returns' : 'refunds';
    navigate(`/admin/orders/${type}/${item._id}`);
  };

  // Handle edit
  const handleEdit = (item: ReturnItem | RefundItem) => {
    const type = 'returnId' in item ? 'returns' : 'refunds';
    navigate(`/admin/orders/${type}/edit/${item._id}`);
  };

  // Handle status update
  const handleStatusUpdate = (type: 'return' | 'refund', id: string, status: string, data: any = {}) => {
    updateStatusMutation.mutate({ type, id, status, data });
  };

  // Handle delete confirmation
  const handleDeleteClick = (item: ReturnItem | RefundItem) => {
    const type = 'returnId' in item ? 'return' : 'refund';
    setSelectedItem({ ...item, type });
    setShowDeleteModal(true);
  };

  // Confirm delete
  const confirmDelete = () => {
    if (!selectedItem) return;
    
    const type = 'returnId' in selectedItem ? 'return' : 'refund';
    deleteMutation.mutate({ type, id: selectedItem._id });
    setShowDeleteModal(false);
  };

  // Cancel delete
  const cancelDelete = () => {
    setSelectedItem(null);
    setShowDeleteModal(false);
  };

  // Query options for better error handling and caching
  const queryOptions = {
    retry: 2,
    refetchOnWindowFocus: false,
    staleTime: 5 * 60 * 1000, // 5 minutes
  };

  // Fetch returns data with error boundary and loading states
  const {
    data: returnsData = [],
    isLoading: isLoadingReturns,
    isError: isErrorReturns,
    error: returnsError,
    refetch: refetchReturns,
    isRefetching: isRefreshingReturns
  } = useQuery<ReturnItem[], Error>(
    ['returns', statusFilter],
    () => fetchReturns(statusFilter),
    {
      ...queryOptions,
      enabled: activeTab === 'returns',
      onError: (error) => {
        console.error('Error fetching returns:', error);
      },
      select: (data) => data.map(returnItem => ({
        ...returnItem,
        // Ensure all required fields have default values
        order: returnItem.order || { _id: '', orderNumber: 'N/A' },
        user: returnItem.user || { _id: '', name: 'Unknown', email: '' },
        items: returnItem.items || [],
        status: returnItem.status || 'pending',
        amount: returnItem.amount || 0,
        reason: returnItem.reason || 'No reason provided'
      }))
    }
  );

  // Fetch refunds data with error boundary and loading states
  const {
    data: refundsData = [],
    isLoading: isLoadingRefunds,
    isError: isErrorRefunds,
    error: refundsError,
    refetch: refetchRefunds,
    isRefetching: isRefreshingRefunds
  } = useQuery<RefundItem[], Error>(
    ['refunds', statusFilter],
    () => fetchRefunds(statusFilter),
    {
      ...queryOptions,
      enabled: activeTab === 'refunds',
      onError: (error) => {
        console.error('Error fetching refunds:', error);
      },
      select: (data) => data.map(refund => ({
        ...refund,
        // Ensure all required fields have default values
        order: refund.order || { _id: '', orderNumber: 'N/A' },
        user: refund.user || { _id: '', name: 'Unknown', email: '' },
        status: refund.status || 'pending',
        amount: refund.amount || 0,
        method: refund.method || 'credit_card'
      }))
    }
  );

  // Format date
  const formatDate = (dateString: string) => {
    try {
      return format(new Date(dateString), 'MMM d, yyyy h:mm a');
    } catch (error) {
      return dateString;
    }
  };

  // Handle refresh data
  const handleRefresh = () => {
    if (activeTab === 'returns') {
      refetchReturns();
    } else {
      refetchRefunds();
    }
  };

  // Filter data based on search term
  const filterData = <T extends { _id: string; order: { orderNumber: string }; user: { name: string; email: string } }>(
    data: T[], 
    search: string
  ): T[] => {
    if (!search) return data;
    const searchLower = search.toLowerCase();
    return data.filter(item => 
      item.order?.orderNumber?.toLowerCase().includes(searchLower) ||
      item.user?.name?.toLowerCase().includes(searchLower) ||
      item.user?.email?.toLowerCase().includes(searchLower) ||
      ('returnId' in item ? (item as any).returnId.toLowerCase().includes(searchLower) : false) ||
      ('refundId' in item ? (item as any).refundId.toLowerCase().includes(searchLower) : false)
    );
  };

  const filteredReturns = filterData(returnsData, searchTerm);
  const filteredRefunds = filterData(refundsData, searchTerm);
  const isLoading = activeTab === 'returns' ? isLoadingReturns : isLoadingRefunds;
  const isRefreshing = activeTab === 'returns' ? isRefreshingReturns : isRefreshingRefunds;
  const noResults = activeTab === 'returns' 
    ? filteredReturns.length === 0 
    : filteredRefunds.length === 0;

  const getStatusBadge = (status: string) => {
    const statusColors: { [key: string]: string } = {
      pending: 'bg-yellow-100 text-yellow-800',
      approved: 'bg-green-100 text-green-800',
      rejected: 'bg-red-100 text-red-800',
      completed: 'bg-green-100 text-green-800',
      processing: 'bg-blue-100 text-blue-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status] || 'bg-gray-100 text-gray-800'}`;
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'approved':
      case 'completed':
        return <CheckCircle className="w-4 h-4 text-green-500" />;
      case 'rejected':
        return <XCircle className="w-4 h-4 text-red-500" />;
      case 'pending':
      case 'processing':
        return <Clock className="w-4 h-4 text-yellow-500" />;
      default:
        return <Clock className="w-4 h-4 text-gray-400" />;
    }
  };

  return (
    <Layout title="Returns & Refunds" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Tabs and Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          {/* Tabs */}
          <div className="border-b border-gray-200 flex-1">
            <nav className="-mb-px flex space-x-8">
              <button
                onClick={() => setActiveTab('returns')}
                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                  activeTab === 'returns'
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                Returns
              </button>
              <button
                onClick={() => setActiveTab('refunds')}
                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                  activeTab === 'refunds'
                    ? 'border-blue-500 text-blue-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                Refunds
              </button>
            </nav>
          </div>

          {/* Refresh Button */}
          <button
            onClick={handleRefresh}
            disabled={isRefreshing}
            className="flex items-center px-3 py-2 text-sm text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
            title="Refresh"
          >
            <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
            {isRefreshing ? 'Refreshing...' : 'Refresh'}
          </button>
        </div>

        {/* Filters */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex flex-col sm:flex-row gap-4 flex-1">
            <div className="relative flex-1 max-w-md">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder={`Search ${activeTab} by ID, name, or email...`}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
            
            <select
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 min-w-[180px]"
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
            >
              <option value="all">All Status</option>
              {activeTab === 'returns' ? (
                <>
                  <option value="pending">Pending</option>
                  <option value="approved">Approved</option>
                  <option value="rejected">Rejected</option>
                  <option value="processing">Processing</option>
                  <option value="completed">Completed</option>
                </>
              ) : (
                <>
                  <option value="pending">Pending</option>
                  <option value="processing">Processing</option>
                  <option value="completed">Completed</option>
                  <option value="failed">Failed</option>
                </>
              )}
            </select>
          </div>
          
          <button 
            className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 whitespace-nowrap"
            onClick={() => {
              // TODO: Implement more filters
              console.log('More filters clicked');
            }}
          >
            <Filter className="w-4 h-4 mr-2" />
            More Filters
          </button>
        </div>

        {/* Content Area */}
        {isLoading ? (
          <div className="text-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto"></div>
            <p className="mt-2 text-gray-600">Loading {activeTab}...</p>
          </div>
        ) : noResults ? (
          <div className="text-center py-12 bg-white rounded-lg shadow-sm border">
            <div className="text-gray-400 mb-2">
              <svg
                className="mx-auto h-12 w-12"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={1}
                  d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                />
              </svg>
            </div>
            <h3 className="text-lg font-medium text-gray-900">No {activeTab} found</h3>
            <p className="mt-1 text-sm text-gray-500">
              {searchTerm || statusFilter !== 'all'
                ? 'Try adjusting your search or filter criteria.'
                : `There are no ${activeTab} at the moment.`}
            </p>
          </div>
        ) : (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      {activeTab === 'returns' ? 'Return' : 'Refund'} Details
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Amount
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Date
                    </th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {activeTab === 'returns' ? (
                    // Returns Table Rows
                    filteredReturns.map((item) => (
                      <tr key={item._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-gray-900">
                            {item.returnId}
                          </div>
                          <div className="text-xs text-gray-500">
                            {formatDate(item.createdAt)}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-blue-600">
                            #{item.order?.orderNumber || 'N/A'}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-gray-900">
                            {item.user?.name || 'N/A'}
                          </div>
                          <div className="text-xs text-gray-500">
                            {item.user?.email || ''}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          ${item.amount?.toFixed(2) || '0.00'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={getStatusBadge(item.status)}>
                            {item.status.charAt(0).toUpperCase() + item.status.slice(1)}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 text-right">
                          {new Date(item.updatedAt).toLocaleDateString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                          <div className="flex items-center justify-end space-x-2">
                            <button 
                              className="text-blue-600 hover:text-blue-900"
                              onClick={() => handleViewDetails(item)}
                              title="View details"
                            >
                              <Eye className="w-4 h-4" />
                            </button>
                            <button 
                              className="text-blue-600 hover:text-blue-900"
                              onClick={() => handleEdit(item)}
                              title="Edit return"
                            >
                              <Edit className="w-4 h-4" />
                            </button>
                            {item.status === 'pending' && (
                              <>
                                <button 
                                  className="text-green-600 hover:text-green-900"
                                  onClick={() => handleStatusUpdate('return', item._id, 'approved')}
                                  disabled={isProcessing}
                                  title="Approve return"
                                >
                                  <CheckCircle className="w-4 h-4" />
                                </button>
                                <button 
                                  className="text-red-600 hover:text-red-900"
                                  onClick={() => handleStatusUpdate('return', item._id, 'rejected')}
                                  disabled={isProcessing}
                                  title="Reject return"
                                >
                                  <XCircle className="w-4 h-4" />
                                </button>
                                <button 
                                  className="text-red-600 hover:text-red-900"
                                  onClick={() => handleDeleteClick(item)}
                                  disabled={isProcessing}
                                  title="Delete return"
                                >
                                  <Trash2 className="w-4 h-4" />
                                </button>
                              </>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))
                  ) : (
                    // Refunds Table Rows
                    filteredRefunds.map((item) => (
                      <tr key={item._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-gray-900">
                            {item.refundId}
                          </div>
                          <div className="text-xs text-gray-500">
                            {formatDate(item.createdAt)}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-blue-600">
                            #{item.order?.orderNumber || 'N/A'}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <div className="text-sm font-medium text-gray-900">
                            {item.user?.name || 'N/A'}
                          </div>
                          <div className="text-xs text-gray-500">
                            {item.user?.email || ''}
                          </div>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          ${item.amount?.toFixed(2) || '0.00'}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <span className={getStatusBadge(item.status)}>
                            {item.status.charAt(0).toUpperCase() + item.status.slice(1)}
                          </span>
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 text-right">
                          {new Date(item.updatedAt).toLocaleDateString()}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                          <div className="flex items-center justify-end space-x-2">
                            <button 
                              className="text-blue-600 hover:text-blue-900"
                              onClick={() => handleViewDetails(item)}
                              title="View details"
                            >
                              <Eye className="w-4 h-4" />
                            </button>
                            <button 
                              className="text-blue-600 hover:text-blue-900"
                              onClick={() => handleEdit(item)}
                              title="Edit refund"
                            >
                              <Edit className="w-4 h-4" />
                            </button>
                            {item.status === 'pending' || item.status === 'processing' ? (
                              <>
                                <button 
                                  className="text-green-600 hover:text-green-900"
                                  onClick={() => handleStatusUpdate('refund', item._id, 'processing')}
                                  disabled={isProcessing}
                                  title="Process refund"
                                >
                                  <CheckCircle className="w-4 h-4" />
                                </button>
                                <button 
                                  className="text-red-600 hover:text-red-900"
                                  onClick={() => handleDeleteClick(item)}
                                  disabled={isProcessing}
                                  title="Delete refund"
                                >
                                  <Trash2 className="w-4 h-4" />
                                </button>
                              </>
                            ) : null}
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>

            {/* Pagination */}
            <div className="bg-white px-4 py-3 flex items-center justify-between border-t border-gray-200 sm:px-6">
              <div className="flex-1 flex justify-between sm:hidden">
                <button className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                  Previous
                </button>
                <button className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">
                  Next
                </button>
              </div>
              <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                <div>
                  <p className="text-sm text-gray-700">
                    Showing <span className="font-medium">1</span> to{' '}
                    <span className="font-medium">
                      {activeTab === 'returns' ? filteredReturns.length : filteredRefunds.length}
                    </span>{' '}
                    of{' '}
                    <span className="font-medium">
                      {activeTab === 'returns' ? returnsData.length : refundsData.length}
                    </span>{' '}
                    {activeTab}
                  </p>
                </div>
                <div>
                  <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px" aria-label="Pagination">
                    <button className="relative inline-flex items-center px-2 py-2 rounded-l-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                      <span className="sr-only">Previous</span>
                      <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                        <path fillRule="evenodd" d="M12.707 5.293a1 1 0 010 1.414L9.414 10l3.293 3.293a1 1 0 01-1.414 1.414l-4-4a1 1 0 010-1.414l4-4a1 1 0 011.414 0z" clipRule="evenodd" />
                      </svg>
                    </button>
                    <button className="bg-blue-50 border-blue-500 text-blue-600 relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                      1
                    </button>
                    <button className="border-gray-300 text-gray-500 hover:bg-gray-50 relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                      2
                    </button>
                    <button className="border-gray-300 text-gray-500 hover:bg-gray-50 relative inline-flex items-center px-4 py-2 border text-sm font-medium">
                      3
                    </button>
                    <button className="relative inline-flex items-center px-2 py-2 rounded-r-md border border-gray-300 bg-white text-sm font-medium text-gray-500 hover:bg-gray-50">
                      <span className="sr-only">Next</span>
                      <svg className="h-5 w-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                        <path fillRule="evenodd" d="M7.293 14.707a1 1 0 010-1.414L10.586 10 7.293 6.707a1 1 0 011.414-1.414l4 4a1 1 0 010 1.414l-4 4a1 1 0 01-1.414 0z" clipRule="evenodd" />
                      </svg>
                    </button>
                  </nav>
                </div>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Inline Delete Confirmation Modal */}
      {showDeleteModal && (
        <div className="fixed inset-0 z-50 overflow-y-auto">
          <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            {/* Background overlay */}
            <div className="fixed inset-0 transition-opacity" aria-hidden="true" onClick={cancelDelete}>
              <div className="absolute inset-0 bg-gray-500 opacity-75"></div>
            </div>

            {/* Modal panel */}
            <span className="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
            
            <div className="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
              <div className="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                <div className="sm:flex sm:items-start">
                  <div className="mx-auto flex-shrink-0 flex items-center justify-center h-12 w-12 rounded-full bg-red-100 sm:mx-0 sm:h-10 sm:w-10">
                    <Trash2 className="h-6 w-6 text-red-600" />
                  </div>
                  <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left">
                    <h3 className="text-lg leading-6 font-medium text-gray-900">
                      {`Delete ${selectedItem?.type === 'return' ? 'Return' : 'Refund'}?`}
                    </h3>
                    <div className="mt-2">
                      <p className="text-sm text-gray-500">
                        Are you sure you want to delete this {selectedItem?.type || 'item'}? This action cannot be undone.
                      </p>
                    </div>
                  </div>
                </div>
              </div>
              <div className="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                <button
                  type="button"
                  className={`w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-red-600 text-base font-medium text-white hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 sm:ml-3 sm:w-auto sm:text-sm ${
                    deleteMutation.isLoading ? 'opacity-70 cursor-not-allowed' : ''
                  }`}
                  onClick={confirmDelete}
                  disabled={deleteMutation.isLoading}
                >
                  {deleteMutation.isLoading ? (
                    <>
                      <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                      </svg>
                      Deleting...
                    </>
                  ) : 'Delete'}
                </button>
                <button
                  type="button"
                  className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                  onClick={cancelDelete}
                  disabled={deleteMutation.isLoading}
                >
                  Cancel
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
};

export default ReturnsRefunds;