import React, { useState } from 'react';
import { Dialog, Transition } from '@headlessui/react';
import { Fragment } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, Filter, Download, Eye, Edit, Truck } from 'lucide-react';
import { format } from 'date-fns';

import api from '../../../services/api';
import { useQuery, useQueryClient } from 'react-query';
import { toast } from 'react-hot-toast';

// Define the Order type based on your backend model
interface Order {
  _id: string;
  orderNumber: string;
  user: string | {
    _id: string;
    name: string;
    email: string;
  };
  items: Array<{
    _id?: string;
    product: string | { _id: string; name: string };
    item?: {
      _id: string;
      name: string;
      price: number;
      image?: string;
      brand?: string | { _id: string; name: string };
      category?: string | { _id: string; name: string };
      unit?: string | { _id: string; name: string; shortCode?: string };
    };
    quantity: number;
    price: number;
    name?: string;
  }>;
  totalAmount: number;
  originalAmount?: number; // Original amount before discount
  coupon?: {
    code: string;
    discount: number;
    type: 'percentage' | 'fixed';
    maxDiscount?: number;
  };
  status: string;
  store?: string | { _id: string; name: string; storeUrl?: string };
  deliveryInfo?: {
    _id?: string;
    deliveryBoy?: string | { _id: string; name: string; phone?: string };
    // Add other delivery info fields as needed
  };
  deliveryBoy?: string | { _id: string; name: string; phone?: string };
  paymentStatus?: string;
  paymentMethod?: string;
  deliveryAddress?: any;
  createdAt: string | Date;
  updatedAt: string | Date;
}

const fetchOrders = async (searchTerm: string, statusFilter: string) => {
  try {
    const params: Record<string, any> = {
      sort: '-createdAt',
      limit: 50, // Increased limit to show more items by default
    };

    // Add search term if provided
    if (searchTerm) {
      params.search = searchTerm;
    }

    // Add status filter if not 'all'
    if (statusFilter && statusFilter !== 'all') {
      params.status = statusFilter;
    }

    const { data } = await api.get('/orders', { params });
    
    // Handle both response formats for backward compatibility
    if (data.data) {
      return data.data; // Standard response format with pagination
    }
    
    return Array.isArray(data) ? data : []; // Fallback for array response
  } catch (error) {
    console.error('Error fetching orders:', error);
    // Return empty array instead of throwing to prevent UI crash
    return [];
  }
};

const AdminOrders: React.FC = () => {
  const queryClient = useQueryClient();
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
  const [isViewModalOpen, setIsViewModalOpen] = useState(false);
  const [isEditModalOpen, setIsEditModalOpen] = useState(false);
  const [editingOrder, setEditingOrder] = useState<Order | null>(null);
  const [orderStatus, setOrderStatus] = useState('');
  const [deliveryBoy, setDeliveryBoy] = useState('');
  const [isUpdating, setIsUpdating] = useState(false);
  const [trackingModalOpen, setTrackingModalOpen] = useState(false);
  const [trackingInfo, setTrackingInfo] = useState({
    orderNumber: '',
    status: '',
    estimatedDelivery: '',
    trackingNumber: '',
    carrier: ''
  });

  const { data: orders = [], isLoading, isError, error } = useQuery<Order[]>(
    ['orders', searchTerm, statusFilter],
    () => fetchOrders(searchTerm, statusFilter),
    {
      retry: 2, // Retry failed requests twice
      refetchOnWindowFocus: false, // Don't refetch when window regains focus
      keepPreviousData: true, // Keep previous data while fetching new data
      staleTime: 5 * 60 * 1000, // Data is considered fresh for 5 minutes
    }
  );

  const getStatusBadge = (status: string) => {
    const statusColors: Record<string, string> = {
      pending: 'bg-gray-100 text-gray-800',
      confirmed: 'bg-blue-100 text-blue-800',
      packed: 'bg-yellow-100 text-yellow-800',
      out_for_delivery: 'bg-purple-100 text-purple-800',
      delivered: 'bg-green-100 text-green-800',
      cancelled: 'bg-red-100 text-red-800',
      returned: 'bg-orange-100 text-orange-800',
      processing: 'bg-blue-100 text-blue-800',
      shipped: 'bg-indigo-100 text-indigo-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status] || 'bg-gray-100 text-gray-800'}`;
  };

  // Open view order modal
  const handleViewOrder = (order: Order) => {
    setSelectedOrder(order);
    setIsViewModalOpen(true);
  };

  // Open edit order modal
  const handleEditOrder = (order: Order) => {
    setEditingOrder(order);
    setOrderStatus(order.status);
    setDeliveryBoy(order.deliveryBoy?.name || '');
    setIsEditModalOpen(true);
  };

  // Handle order update
  const handleUpdateOrder = async () => {
    if (!editingOrder) return;
    
    try {
      setIsUpdating(true);
      const updateData = {
        status: orderStatus,
        deliveryBoy: deliveryBoy || undefined
      };
      
      await api.put(`/orders/${editingOrder._id}`, updateData);
      
      // Refresh orders data
      queryClient.invalidateQueries('orders');
      
      // Close modal and show success message
      setIsEditModalOpen(false);
      toast.success('Order updated successfully');
    } catch (error) {
      console.error('Error updating order:', error);
      toast.error('Failed to update order');
    } finally {
      setIsUpdating(false);
    }
  };

  // Close view order modal
  const closeViewModal = () => {
    setIsViewModalOpen(false);
    // Small delay to allow the modal to close before clearing the order
    setTimeout(() => setSelectedOrder(null), 300);
  };
  
  // Close edit order modal
  const closeEditModal = () => {
    setIsEditModalOpen(false);
    setEditingOrder(null);
    setOrderStatus('');
    setDeliveryBoy('');
  };

  const openTrackingModal = (order: Order) => {
    // In a real app, you would fetch tracking info from your API
    setTrackingInfo({
      orderNumber: order.orderNumber,
      status: order.status || 'Processing',
      estimatedDelivery: '2-3 business days',
      trackingNumber: `TRK${Math.floor(1000000000 + Math.random() * 9000000000)}`,
      carrier: 'Standard Shipping'
    });
    setTrackingModalOpen(true);
  };

  const closeTrackingModal = () => {
    setTrackingModalOpen(false);
  };

  // Format date
  const formatDate = (dateString: string) => {
    try {
      return format(new Date(dateString), 'MMM d, yyyy h:mm a');
    } catch (error) {
      return dateString;
    }
  };

  // Calculate total items in order
  const getTotalItems = (items: any[]) => {
    return items.reduce((total, item) => total + (item.quantity || 0), 0);
  };

  // Filter orders based on search term and status
  const filteredOrders = orders.filter(order => {
    const searchLower = searchTerm.toLowerCase();
    const matchesSearch = 
      order.orderNumber?.toLowerCase().includes(searchLower) ||
      order.user?.name?.toLowerCase().includes(searchLower) ||
      order.user?.email?.toLowerCase().includes(searchLower);
      
    const matchesStatus = statusFilter === 'all' || order.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  return (
    <Layout title="Orders Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header Actions */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="flex flex-col sm:flex-row gap-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search orders..."
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
            
            <select
              className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value)}
            >
              <option value="all">All Status</option>
              <option value="pending">Pending</option>
              <option value="confirmed">Confirmed</option>
              <option value="packed">Packed</option>
              <option value="out_for_delivery">Out for Delivery</option>
              <option value="delivered">Delivered</option>
              <option value="cancelled">Cancelled</option>
            </select>
          </div>
          
          <div className="flex gap-2">
            <button className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50">
              <Filter className="w-4 h-4 mr-2" />
              Filter
            </button>
            <button className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50">
              <Download className="w-4 h-4 mr-2" />
              Export
            </button>
          </div>
        </div>

        {/* Orders Table */}
        {isLoading ? (
          <div className="text-center py-12">Loading orders...</div>
        ) : isError ? (
          <div className="text-center py-12 text-red-500">
            Failed to load orders. {error instanceof Error ? error.message : ''}
          </div>
        ) : (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Order Details
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Customer
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Amount
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Status
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Store
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Delivery Boy
                  </th>
                  <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Actions
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {filteredOrders.length > 0 ? (
                  filteredOrders.map((order) => (
                    <tr key={order._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">#{order.orderNumber}</div>
                          <div className="text-xs text-gray-500">{formatDate(order.createdAt)}</div>
                          <div className="text-xs text-gray-400">
                            {getTotalItems(order.items || [])} items
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {order.userId?.name || 'N/A'}
                          </div>
                          <div className="text-xs text-gray-500">
                            {order.userId?.email || 'N/A'}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                        ${order.totalAmount?.toFixed(2) || '0.00'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getStatusBadge(order.status)}>
                          {order.status.replace(/_/g, ' ')}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {typeof order.store === 'object' ? order.store?.name || 'N/A' : order.store || 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.deliveryBoy?.name || (
                          <span className="text-gray-400">Not assigned</span>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          <button 
                            onClick={() => handleViewOrder(order)}
                            className="text-blue-600 hover:text-blue-900"
                            title="View Order"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button 
                            onClick={() => handleEditOrder(order)}
                            className="text-gray-600 hover:text-gray-900"
                            title="Edit Order"
                          >
                            <Edit className="w-4 h-4" />
                          </button>
                          <button 
                            className="text-green-600 hover:text-green-900"
                            title="Track Order"
                            onClick={() => openTrackingModal(order)}
                          >
                            <Truck className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))
                ) : (
                  <tr>
                    <td colSpan={7} className="px-6 py-12 text-center text-gray-500">
                      No orders found
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
          </div>
        )}

        {/* Pagination */}
        <div className="flex items-center justify-between">
          <div className="text-sm text-gray-700">
            Showing <span className="font-medium">1</span> to <span className="font-medium">10</span> of{' '}
            <span className="font-medium">97</span> results
          </div>
          <div className="flex space-x-2">
            <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">
              Previous
            </button>
            <button className="px-3 py-1 text-sm bg-blue-600 text-white rounded">1</button>
            <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">2</button>
            <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">3</button>
            <button className="px-3 py-1 text-sm border border-gray-300 rounded hover:bg-gray-50">
              Next
            </button>
          </div>
        </div>
      </div>

      {/* Tracking Order Modal */}
      <Transition appear show={trackingModalOpen} as={Fragment}>
        <Dialog as="div" className="relative z-10" onClose={closeTrackingModal}>
          <Transition.Child
            as={Fragment}
            enter="ease-out duration-300"
            enterFrom="opacity-0"
            enterTo="opacity-100"
            leave="ease-in duration-200"
            leaveFrom="opacity-100"
            leaveTo="opacity-0"
          >
            <div className="fixed inset-0 bg-black bg-opacity-25" />
          </Transition.Child>

          <div className="fixed inset-0 overflow-y-auto">
            <div className="flex min-h-full items-center justify-center p-4 text-center">
              <Transition.Child
                as={Fragment}
                enter="ease-out duration-300"
                enterFrom="opacity-0 scale-95"
                enterTo="opacity-100 scale-100"
                leave="ease-in duration-200"
                leaveFrom="opacity-100 scale-100"
                leaveTo="opacity-0 scale-95"
              >
                <Dialog.Panel className="w-full max-w-md transform overflow-hidden rounded-2xl bg-white p-6 text-left align-middle shadow-xl transition-all">
                  <Dialog.Title
                    as="h3"
                    className="text-lg font-medium leading-6 text-gray-900 flex items-center"
                  >
                    <Truck className="w-5 h-5 mr-2 text-green-600" />
                    Track Order #{trackingInfo.orderNumber}
                  </Dialog.Title>
                  
                  <div className="mt-4 space-y-4">
                    <div className="bg-gray-50 p-4 rounded-lg">
                      <div className="flex justify-between items-center">
                        <div>
                          <p className="text-sm font-medium text-gray-500">Status</p>
                          <p className="text-lg font-semibold text-gray-900 capitalize">{trackingInfo.status}</p>
                        </div>
                        <div className="text-right">
                          <p className="text-sm font-medium text-gray-500">Estimated Delivery</p>
                          <p className="text-sm font-medium text-gray-900">{trackingInfo.estimatedDelivery}</p>
                        </div>
                      </div>
                    </div>

                    <div className="border-t border-gray-200 pt-4">
                      <h4 className="text-sm font-medium text-gray-900 mb-3">Tracking Information</h4>
                      <div className="space-y-3">
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Tracking Number:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.trackingNumber}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Carrier:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.carrier}</span>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-500">Order Number:</span>
                          <span className="text-sm font-medium text-gray-900">{trackingInfo.orderNumber}</span>
                        </div>
                      </div>
                    </div>

                    <div className="mt-6 pt-4 border-t border-gray-200">
                      <h4 className="text-sm font-medium text-gray-900 mb-3">Delivery Status</h4>
                      <div className="space-y-4">
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className="w-2.5 h-2.5 rounded-full bg-green-500 mt-1"></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Order Placed</p>
                            <p className="text-xs text-gray-500">Your order has been received</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['shipped', 'out-for-delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Order Processed</p>
                            <p className="text-xs text-gray-500">We're preparing your order</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['shipped', 'out-for-delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Shipped</p>
                            <p className="text-xs text-gray-500">Your order is on the way</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${['out-for-delivery', 'delivered'].includes(trackingInfo.status) ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                            <div className="w-px h-8 bg-gray-300 my-1"></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Out for Delivery</p>
                            <p className="text-xs text-gray-500">Your order is with the delivery person</p>
                          </div>
                        </div>
                        <div className="flex items-start">
                          <div className="flex flex-col items-center mr-4">
                            <div className={`w-2.5 h-2.5 rounded-full ${trackingInfo.status === 'delivered' ? 'bg-green-500' : 'bg-gray-300'} mt-1`}></div>
                          </div>
                          <div>
                            <p className="text-sm font-medium text-gray-900">Delivered</p>
                            <p className="text-xs text-gray-500">Your order has been delivered</p>
                          </div>
                        </div>
                      </div>
                    </div>
                  </div>

                  <div className="mt-6 flex justify-end">
                    <button
                      type="button"
                      className="inline-flex justify-center rounded-md border border-transparent bg-blue-100 px-4 py-2 text-sm font-medium text-blue-900 hover:bg-blue-200 focus:outline-none focus-visible:ring-2 focus-visible:ring-blue-500 focus-visible:ring-offset-2"
                      onClick={closeTrackingModal}
                    >
                      Close
                    </button>
                  </div>
                </Dialog.Panel>
              </Transition.Child>
            </div>
          </div>
        </Dialog>
      </Transition>

      {/* View Order Modal */}
      <Transition appear show={isViewModalOpen} as={Fragment}>
        <Dialog as="div" className="relative z-10" onClose={closeViewModal}>
          <Transition.Child
            as={Fragment}
            enter="ease-out duration-300"
            enterFrom="opacity-0"
            enterTo="opacity-100"
            leave="ease-in duration-200"
            leaveFrom="opacity-100"
            leaveTo="opacity-0"
          >
            <div className="fixed inset-0 bg-black bg-opacity-25" />
          </Transition.Child>

          <div className="fixed inset-0 overflow-y-auto">
            <div className="flex min-h-full items-center justify-center p-4 text-center">
              <Transition.Child
                as={Fragment}
                enter="ease-out duration-300"
                enterFrom="opacity-0 scale-95"
                enterTo="opacity-100 scale-100"
                leave="ease-in duration-200"
                leaveFrom="opacity-100 scale-100"
                leaveTo="opacity-0 scale-95"
              >
                <Dialog.Panel className="w-full max-w-2xl transform overflow-hidden rounded-2xl bg-white p-6 text-left align-middle shadow-xl transition-all">
                  {selectedOrder && (
                    <>
                      <Dialog.Title
                        as="h3"
                        className="text-lg font-medium leading-6 text-gray-900 flex justify-between items-center"
                      >
                        <span>Order #{selectedOrder.orderNumber}</span>
                        <span className={getStatusBadge(selectedOrder.status)}>
                          {selectedOrder.status ? String(selectedOrder.status).replace(/_/g, ' ') : 'N/A'}
                        </span>
                      </Dialog.Title>
                      
                      <div className="mt-4 space-y-4">
                        <div className="grid grid-cols-2 gap-4">
                          <div>
                            <h4 className="font-medium text-gray-700">Customer</h4>
                            <p className="text-gray-900">
                              {selectedOrder.userId?.name || 'N/A'}
                            </p>
                            <p className="text-sm text-gray-500">
                              {selectedOrder.userId?.email || 'N/A'}
                            </p>
                          </div>
                          <div>
                            <h4 className="font-medium text-gray-700">Order Date</h4>
                            <p className="text-gray-900">
                              {selectedOrder.createdAt ? formatDate(selectedOrder.createdAt) : 'N/A'}
                            </p>
                          </div>
                        </div>

                        <div>
                          <h4 className="font-medium text-gray-700 mb-2">Order Items</h4>
                          <div className="border rounded-lg overflow-hidden">
                            <table className="min-w-full divide-y divide-gray-200">
                              <thead className="bg-gray-50">
                                <tr>
                                  <th className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Item</th>
                                  <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Qty</th>
                                  <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Price</th>
                                  <th className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Total</th>
                                </tr>
                              </thead>
                              <tbody className="bg-white divide-y divide-gray-200">
                                {selectedOrder.items?.map((item, index) => (
                                  <tr key={index}>
                                    <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-900">
                                      {typeof item.product === 'string' ? item.product : item.product?.name || 'Product'}
                                    </td>
                                    <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-500 text-right">
                                      {item.quantity}
                                    </td>
                                    <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-500 text-right">
                                      ${(item.price || 0).toFixed(2)}
                                    </td>
                                    <td className="px-4 py-2 whitespace-nowrap text-sm text-gray-900 font-medium text-right">
                                      ${((item.price || 0) * (item.quantity || 0)).toFixed(2)}
                                    </td>
                                  </tr>
                                ))}
                              </tbody>
                              <tfoot className="bg-gray-50">
                                <tr>
                                  <td colSpan={3} className="px-4 py-2 text-right text-sm font-medium text-gray-700">
                                    Subtotal:
                                  </td>
                                  <td className="px-4 py-2 text-right text-sm font-medium text-gray-900">
                                    ${(selectedOrder.originalAmount || selectedOrder.totalAmount)?.toFixed(2) || '0.00'}
                                  </td>
                                </tr>
                                {selectedOrder.coupon && (
                                  <tr>
                                    <td colSpan={3} className="px-4 py-2 text-right text-sm font-medium text-gray-700">
                                      Coupon ({selectedOrder.coupon.code}):
                                    </td>
                                    <td className="px-4 py-2 text-right text-sm font-medium text-red-600">
                                      -${(selectedOrder.originalAmount ? (selectedOrder.originalAmount - selectedOrder.totalAmount).toFixed(2) : '0.00')}
                                    </td>
                                  </tr>
                                )}
                                <tr>
                                  <td colSpan={3} className="px-4 py-2 text-right text-sm font-medium text-gray-700">
                                    Total:
                                  </td>
                                  <td className="px-4 py-2 text-right text-lg font-bold text-blue-600">
                                    ${selectedOrder.totalAmount?.toFixed(2) || '0.00'}
                                    {selectedOrder.originalAmount && selectedOrder.originalAmount > selectedOrder.totalAmount && (
                                      <span className="ml-2 text-sm text-gray-500 line-through">
                                        ${selectedOrder.originalAmount.toFixed(2)}
                                      </span>
                                    )}
                                  </td>
                                </tr>
                              </tfoot>
                            </table>
                          </div>
                        </div>

                        <div className="grid grid-cols-2 gap-4 pt-4 border-t">
                          <div>
                            <h4 className="font-medium text-gray-700">Store</h4>
                            <p className="text-gray-900">
                              {selectedOrder.store && typeof selectedOrder.store === 'object' 
                                ? selectedOrder.store.name || 'N/A'
                                : selectedOrder.store || 'N/A'}
                            </p>
                          </div>
                          <div>
                            <h4 className="font-medium text-gray-700">Delivery Person</h4>
                            <p className="text-gray-900">
                              {selectedOrder.deliveryBoy 
                                ? typeof selectedOrder.deliveryBoy === 'object' 
                                  ? selectedOrder.deliveryBoy.name || 'Not assigned'
                                  : selectedOrder.deliveryBoy
                                : 'Not assigned'}
                            </p>
                          </div>
                        </div>
                      </div>

                      <div className="mt-6 flex justify-end space-x-3">
                        <button
                          type="button"
                          className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                          onClick={closeViewModal}
                        >
                          Close
                        </button>
                        <button
                          type="button"
                          className="px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                          onClick={() => {
                            // Handle print or other action
                            window.print();
                          }}
                        >
                          Print Invoice
                        </button>
                      </div>
                    </>
                  )}
                </Dialog.Panel>
              </Transition.Child>
            </div>
          </div>
        </Dialog>
      </Transition>

      {/* Edit Order Modal */}
      <Transition appear show={isEditModalOpen} as={Fragment}>
        <Dialog as="div" className="relative z-10" onClose={closeEditModal}>
          <Transition.Child
            as={Fragment}
            enter="ease-out duration-300"
            enterFrom="opacity-0"
            enterTo="opacity-100"
            leave="ease-in duration-200"
            leaveFrom="opacity-100"
            leaveTo="opacity-0"
          >
            <div className="fixed inset-0 bg-black bg-opacity-25" />
          </Transition.Child>

          <div className="fixed inset-0 overflow-y-auto">
            <div className="flex min-h-full items-center justify-center p-4 text-center">
              <Transition.Child
                as={Fragment}
                enter="ease-out duration-300"
                enterFrom="opacity-0 scale-95"
                enterTo="opacity-100 scale-100"
                leave="ease-in duration-200"
                leaveFrom="opacity-100 scale-100"
                leaveTo="opacity-0 scale-95"
              >
                <Dialog.Panel className="w-full max-w-md transform overflow-hidden rounded-2xl bg-white p-6 text-left align-middle shadow-xl transition-all">
                  {editingOrder && (
                    <>
                      <Dialog.Title
                        as="h3"
                        className="text-lg font-medium leading-6 text-gray-900"
                      >
                        Edit Order #{editingOrder.orderNumber}
                      </Dialog.Title>
                      
                      <div className="mt-4 space-y-4">
                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-1">
                            Status
                          </label>
                          <select
                            value={orderStatus}
                            onChange={(e) => setOrderStatus(e.target.value)}
                            className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                          >
                            <option value="pending">Pending</option>
                            <option value="confirmed">Confirmed</option>
                            <option value="processing">Processing</option>
                            <option value="packed">Packed</option>
                            <option value="shipped">Shipped</option>
                            <option value="out_for_delivery">Out for Delivery</option>
                            <option value="delivered">Delivered</option>
                            <option value="cancelled">Cancelled</option>
                            <option value="returned">Returned</option>
                          </select>
                        </div>

                        <div>
                          <label className="block text-sm font-medium text-gray-700 mb-1">
                            Delivery Person
                          </label>
                          <input
                            type="text"
                            value={deliveryBoy}
                            onChange={(e) => setDeliveryBoy(e.target.value)}
                            placeholder="Enter delivery person name"
                            className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500"
                          />
                        </div>

                        <div className="mt-6 flex justify-end space-x-3">
                          <button
                            type="button"
                            className="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                            onClick={closeEditModal}
                            disabled={isUpdating}
                          >
                            Cancel
                          </button>
                          <button
                            type="button"
                            className="px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 disabled:opacity-50"
                            onClick={handleUpdateOrder}
                            disabled={isUpdating}
                          >
                            {isUpdating ? 'Updating...' : 'Update Order'}
                          </button>
                        </div>
                      </div>
                    </>
                  )}
                </Dialog.Panel>
              </Transition.Child>
            </div>
          </div>
        </Dialog>
      </Transition>
    </Layout>
  );
};

export default AdminOrders;