import React, { useState } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, MapPin, Clock, CheckCircle, Truck, Package } from 'lucide-react';

import api from '../../../services/api';
import { useQuery } from 'react-query';

// Type definitions
interface User {
  name: string;
}

interface DeliveryBoy {
  name: string;
  phone: string;
}

interface StatusHistory {
  status: string;
  changedAt: string;
}

interface Order {
  _id: string;
  orderNumber: string;
  user: User;
  status: string;
  deliveryBoy?: DeliveryBoy;
  statusHistory: StatusHistory[];
  estimatedDelivery?: string; // This might not exist, handle it gracefully
  currentLocation?: string; // This might not exist, handle it gracefully
}

const fetchOrders = async (searchTerm: string) => {
  try {
    const { data } = await api.get('/orders', {
      params: {
        search: searchTerm,
        sort: '-createdAt',
        limit: 50, // Limit the number of orders to fetch
      },
    });
    // Ensure we always return an array, even if the response structure is different
    return Array.isArray(data?.data) ? data.data : [];
  } catch (error) {
    console.error('Error fetching orders:', error);
    throw error;
  }
};

const OrderTracking: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');

    const { data: orders = [], isLoading, isError, error } = useQuery<Order[], Error>(
    ['orders', searchTerm],
    () => fetchOrders(searchTerm),
    {
      retry: 2,
      refetchOnWindowFocus: false,
      onError: (err) => {
        console.error('Error in useQuery:', err);
      }
    }
  );

  const filteredOrders = React.useMemo(() => {
    if (!Array.isArray(orders)) return [];
    return orders.filter(
      (order) =>
        order?.orderNumber?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        order?.user?.name?.toLowerCase().includes(searchTerm.toLowerCase())
    );
  }, [orders, searchTerm]);

  const getStatusIcon = (status: string, completed: boolean) => {
    if (!completed) return <Clock className="w-5 h-5 text-gray-400" />;
    
    switch (status) {
      case 'pending':
        return <CheckCircle className="w-5 h-5 text-gray-500" />;
      case 'confirmed':
        return <CheckCircle className="w-5 h-5 text-blue-500" />;
      case 'processing':
      case 'packed':
      case 'ready_for_delivery':
        return <Package className="w-5 h-5 text-yellow-500" />;
      case 'out_for_delivery':
        return <Truck className="w-5 h-5 text-purple-500" />;
      case 'delivered':
        return <CheckCircle className="w-5 h-5 text-green-500" />;
      default:
        return <Clock className="w-5 h-5 text-gray-400" />;
    }
  };

  const getStatusBadge = (status: string) => {
    const statusColors = {
      pending: 'bg-gray-100 text-gray-800',
      confirmed: 'bg-blue-100 text-blue-800',
      packed: 'bg-yellow-100 text-yellow-800',
      out_for_delivery: 'bg-purple-100 text-purple-800',
      delivered: 'bg-green-100 text-green-800'
    };
    
    return `px-3 py-1 text-sm font-medium rounded-full ${statusColors[status as keyof typeof statusColors]}`;
  };

  return (
    <Layout title="Order Tracking" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Search */}
        <div className="flex items-center space-x-4">
          <div className="relative flex-1 max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search by Order ID or Customer..."
              className="pl-10 pr-4 py-2 w-full border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
          </div>
        </div>

        {/* Tracking Cards */}
        <div className="space-y-6">
          {isLoading ? (
            <div className="text-center py-12">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto"></div>
              <p className="mt-4 text-gray-600">Loading tracking data...</p>
            </div>
          ) : isError ? (
            <div className="text-center py-12">
              <div className="mx-auto flex items-center justify-center h-12 w-12 rounded-full bg-red-100">
                <svg className="h-6 w-6 text-red-600" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                </svg>
              </div>
              <h3 className="mt-2 text-sm font-medium text-gray-900">Failed to load tracking data</h3>
              <p className="mt-1 text-sm text-gray-500">{error?.message || 'An error occurred while fetching orders'}</p>
              <div className="mt-6">
                <button
                  onClick={() => window.location.reload()}
                  className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                >
                  Retry
                </button>
              </div>
            </div>
          ) : !filteredOrders || filteredOrders.length === 0 ? (
            <div className="text-center py-12">
              <svg
                className="mx-auto h-12 w-12 text-gray-400"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
                aria-hidden="true"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"
                />
              </svg>
              <h3 className="mt-2 text-sm font-medium text-gray-900">No orders found</h3>
              <p className="mt-1 text-sm text-gray-500">
                {searchTerm ? 'Try adjusting your search or filter to find what you\'re looking for.' : 'No orders have been placed yet.'}
              </p>
            </div>
          ) : (
            filteredOrders.map((order: Order) => (
            <div key={order.orderNumber} className="bg-white rounded-lg shadow-sm border">
              {/* Order Header */}
              <div className="px-6 py-4 border-b border-gray-200">
                <div className="flex items-center justify-between">
                  <div>
                    <h3 className="text-lg font-medium text-gray-900">{order.orderNumber}</h3>
                    <p className="text-sm text-gray-500">Customer: {order.user.name}</p>
                  </div>
                  <div className="text-right">
                    <span className={getStatusBadge(order.status)}>
                      {order.status.replace(/_/g, ' ')}
                    </span>
                    <p className="text-sm text-gray-500 mt-1">ETA: {order.estimatedDelivery || 'Not available'}</p>
                  </div>
                </div>
              </div>

              <div className="p-6">
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                  {/* Current Status */}
                  <div>
                    <h4 className="text-sm font-medium text-gray-900 mb-4">Current Status</h4>
                    <div className="bg-blue-50 rounded-lg p-4">
                      <div className="flex items-center">
                        <MapPin className="w-5 h-5 text-blue-600 mr-3" />
                        <div>
                          <p className="text-sm font-medium text-gray-900">{order.currentLocation || order.status.replace(/_/g, ' ')}</p>
                          <p className="text-xs text-gray-500">Delivery Boy: {order.deliveryBoy?.name || 'N/A'}</p>
                          <p className="text-xs text-gray-500">Contact: {order.deliveryBoy?.phone || 'N/A'}</p>
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Timeline */}
                  <div>
                    <h4 className="text-sm font-medium text-gray-900 mb-4">Order Timeline</h4>
                    <div className="space-y-4">
                      {(order.statusHistory || []).map((step: StatusHistory, index: number) => (
                        <div key={index} className="flex items-start">
                          <div className="flex-shrink-0 mr-4">
                            {getStatusIcon(step.status, true)}
                          </div>
                          <div className="flex-1">
                            <p className={`text-sm font-medium ${
                              true ? 'text-gray-900' : 'text-gray-500'
                            }`}>
                              {step.status}
                            </p>
                            <p className="text-xs text-gray-500">{new Date(step.changedAt).toLocaleString()}</p>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>

                {/* Action Buttons */}
                <div className="mt-6 flex flex-wrap gap-3">
                  <button 
                    className="px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors"
                    onClick={() => {
                      // Add view details functionality
                      console.log('View details for order:', order._id);
                    }}
                  >
                    View Details
                  </button>
                  <button 
                    className="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-200 transition-colors"
                    onClick={() => {
                      // Add contact customer functionality
                      console.log('Contact customer for order:', order._id);
                    }}
                  >
                    Contact Customer
                  </button>
                  {order.deliveryBoy && (
                    <button 
                      className="px-4 py-2 bg-gray-100 text-gray-700 text-sm font-medium rounded-lg hover:bg-gray-200 transition-colors"
                      onClick={() => {
                        // Add contact delivery boy functionality
                        console.log('Contact delivery boy for order:', order._id);
                      }}
                    >
                      Contact Delivery Boy
                    </button>
                  )}
                </div>
              </div>
            </div>
          )))}
        </div>
      </div>
    </Layout>
  );
};

export default OrderTracking;