import React, { useState, useEffect, ReactNode } from "react";
import { useNavigate } from "react-router-dom";
import api from "@/services/api";
import { toast } from "react-hot-toast";
import Layout from "../../../components/shared/Layout";

// Format currency helper function
const formatCurrency = (amount: number) => {
  return new Intl.NumberFormat('en-IN', {
    style: 'currency',
    currency: 'INR',
    minimumFractionDigits: 2,
    maximumFractionDigits: 2
  }).format(amount);
};
import Sidebar from "../../../components/shared/Sidebar";
import { adminSidebarItems } from "../AdminLayout";
import {
  Plus,
  Minus,
  Search,
  Download,
  Edit,
  Edit2,
  X,
  Check,
  Upload,
  RefreshCw,
  PackagePlus,
  History,
} from "lucide-react";
import BulkUpload from "./BulkOperations/BulkUpload";
import BulkUpdate from "./BulkOperations/BulkUpdate";


type TabType = "list" | "bulk-upload" | "bulk-update" | "stock-history" | "stock-settings";

// Define TypeScript interfaces

interface StockMovement {
  _id: string;
  product_name?: string;
  sku?: string;
  product_id: string;
  type: 'in' | 'out' | 'adjustment' | 'return';
  quantity: number;
  previous_quantity: number;
  new_quantity: number;
  reference: string;
  notes: string;
  created_at: string;
  user: {
    name: string;
    _id: string;
  };
}

interface Brand {
  _id: string;
  name: string;
  brandName?: string; // Some APIs might use brandName instead of name
}

interface Category {
  _id: string;
  name: string;
}

interface Product {
  _id: string;
  name: string;
  sku: string;
  category_id: Category | string;
  brand_id: Brand | string | null;
  brand_type?: string;
  brand_has_name?: boolean;
  price: number;
  quantity: number;
  status: boolean;
  expire_date: string;
  min_stock_level: number;
  max_stock_level: number;
  last_stock_update: string;
  stock_movements?: StockMovement[];
  updatedAt?: string;
  // Add any additional fields that might be present in the API response
  [key: string]: any;
}

interface Pagination {
  total: number;
  totalPages: number;
  currentPage: number;
  limit: number;
  hasNextPage: boolean;
  hasPreviousPage: boolean;
}

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: "default" | "outline" | "ghost";
  size?: "sm" | "md" | "lg";
  children: ReactNode;
  className?: string;
}

const Button: React.FC<ButtonProps> = ({
  children,
  className = "",
  variant = "default",
  size = "md",
  ...props
}) => {
  const baseStyles =
    "inline-flex items-center justify-center rounded-md font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 disabled:opacity-50 disabled:pointer-events-none";
  const variants = {
    default: "bg-blue-600 text-white hover:bg-blue-700",
    outline:
      "border border-gray-300 bg-white text-gray-700 hover:bg-gray-50",
    ghost: "hover:bg-gray-100",
  } as const;

  const sizes = {
    sm: "h-9 px-3 text-sm",
    md: "h-10 py-2 px-4",
    lg: "h-11 px-8",
  } as const;

  return (
    <button
      className={`${baseStyles} ${variants[variant]} ${sizes[size]} ${className}`}
      {...props}
    >
      {children}
    </button>
  );
};

// Tab content component with conditional rendering
interface TabContentProps {
  isActive: boolean;
  children: React.ReactNode;
  className?: string;
}

const TabContent: React.FC<TabContentProps> = ({ isActive, children, className = "" }) => {
  if (!isActive) return null;
  return <div className={className}>{children}</div>;
};

const Products: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState("");
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [stockLevelFilter, setStockLevelFilter] = useState<string>("all");
  const [activeTab, setActiveTab] = useState<TabType>("list");
  const navigate = useNavigate();
  const [products, setProducts] = useState<Product[]>([]);
  const [loading, setLoading] = useState(true);
  const [brandsMap, setBrandsMap] = useState<Record<string, string>>({});
  const [pagination, setPagination] = useState<{
    total: number;
    totalPages: number;
    currentPage: number;
    hasNextPage: boolean;
    hasPreviousPage: boolean;
    limit: number;
    totalProducts?: number;
  } | null>(null);
  const [stockHistory, setStockHistory] = useState<StockMovement[]>([]);
  const [editingStock, setEditingStock] = useState<{ [key: string]: boolean }>({});
  const [stockAdjustment, setStockAdjustment] = useState<{ [key: string]: { quantity: number; type: 'add' | 'subtract' | 'set' } }>({});
  const [selectedProduct, setSelectedProduct] = useState<Product | null>(null);
  // Initialize stockMovements as an empty array to ensure it's always an array
  const [stockMovements, setStockMovements] = useState<StockMovement[]>([]);
  const [loadingMovements, setLoadingMovements] = useState(false);
  const [outOfStockNumber, setOutOfStockNumber] = useState<number>(5);
  const [inStockNumber, setInStockNumber] = useState<number>(10);

  // Fetch all brands when component mounts or when products change
  useEffect(() => {
    fetchBrands();
  }, [products]);

  // Fetch all brands
  const fetchBrands = async () => {
    try {
      const response = await api.get('/brands?limit=1000');
      const brandsData: Record<string, string> = {};
      response.data.data.forEach((brand: any) => {
        if (brand._id && (brand.brandName || brand.name)) {
          brandsData[brand._id] = brand.brandName || brand.name;
        }
      });
      setBrandsMap(brandsData);
    } catch (error) {
      console.error('Error fetching brands:', error);
    }
  };

  const fetchProducts = async (page = 1) => {
    try {
      setLoading(true);
      const response = await api.get('/items', {
        params: {
          page,
          limit: 10,
          search: searchTerm,
          status: statusFilter === 'all' ? undefined : statusFilter,
          stockLevel: stockLevelFilter === 'all' ? undefined : stockLevelFilter,
          tab: activeTab,
          populate: 'brand_id,category_id' // Ensure brand and category are populated
        }
      });
      
      if (response.data && response.data.success) {
        console.log('API Response:', {
          data: response.data.data,
          pagination: response.data.pagination
        });
        
        // Log the first few products with their brand data
        response.data.data.slice(0, 3).forEach((product: any, index: number) => {
          console.log(`Product ${index + 1}:`, {
            id: product._id,
            name: product.name,
            brand_id: product.brand_id,
            brand_type: typeof product.brand_id,
            brand_has_name: product.brand_id ? 'name' in product.brand_id : 'no brand_id'
          });
        });
        
        setProducts(response.data.data || []);
        setPagination({
          total: response.data.pagination.total,
          totalPages: response.data.pagination.totalPages,
          currentPage: response.data.pagination.currentPage,
          limit: response.data.pagination.limit,
          hasNextPage: response.data.pagination.hasNextPage,
          hasPreviousPage: response.data.pagination.hasPreviousPage,
          totalProducts: response.data.pagination.totalProducts
        });
      } else {
        setProducts([]);
        toast.error('Failed to load products: Invalid response from server');
      }
    } catch (error) {
      console.error('Error fetching products:', error);
      toast.error('Failed to load products');
    } finally {
      setLoading(false);
    }
  };

  const fetchStockMovements = async (productId: string) => {
    try {
      setLoadingMovements(true);
      const response = await api.get(`/items/${productId}/movements`);
      
      // Log the response for debugging
      console.log('Stock movements response:', response);
      
      // Extract movements from the nested response structure
      let movements = [];
      
      if (response.data?.data?.movements && Array.isArray(response.data.data.movements)) {
        // Handle the case where movements are in response.data.data.movements
        movements = response.data.data.movements;
      } else if (response.data?.data && Array.isArray(response.data.data)) {
        // Fallback to response.data.data if it's an array
        movements = response.data.data;
      } else if (Array.isArray(response.data)) {
        // Fallback to direct array response
        movements = response.data;
      }
      
      console.log('Extracted movements:', movements);
      
      // Ensure we always set an array, even if empty
      setStockMovements(movements || []);
      
      // Find and set the selected product
      const product = products.find(p => p._id === productId);
      if (product) {
        setSelectedProduct(product);
      }
      
      // If no movements were found, show a message
      if (!movements || movements.length === 0) {
        console.log('No stock movements found for this product');
      }
      
    } catch (error) {
      console.error('Error fetching stock movements:', error);
      toast.error('Failed to load stock movements');
      setStockMovements([]);
    } finally {
      setLoadingMovements(false);
    }
  };

  useEffect(() => {
    fetchProducts();
  }, [searchTerm, statusFilter, stockLevelFilter, activeTab]);

  const handleStockAdjustment = async (productId: string) => {
    const adjustment = stockAdjustment[productId];
    if (!adjustment || adjustment.quantity === undefined || isNaN(adjustment.quantity) || adjustment.quantity < 0) {
      toast.error('Please enter a valid quantity');
      return;
    }
    
    try {
      const product = products.find(p => p._id === productId);
      if (!product) {
        throw new Error('Product not found');
      }
      
      let newQuantity = 0;
      let action = 'set'; // Default action
      
      switch (adjustment.type) {
        case 'add':
          newQuantity = product.quantity + adjustment.quantity;
          action = 'add';
          break;
        case 'subtract':
          newQuantity = Math.max(0, product.quantity - adjustment.quantity);
          action = 'subtract';
          break;
        case 'set':
          newQuantity = adjustment.quantity;
          action = 'set';
          break;
      }
      
      // Ensure quantity is a valid number
      const parsedQuantity = Number(newQuantity);
      if (isNaN(parsedQuantity) || parsedQuantity < 0) {
        throw new Error('Invalid quantity value');
      }
      
      // Prepare the request payload
      const payload = {
        data: {
          quantity: adjustment.type === 'set' ? parsedQuantity : adjustment.quantity,
          action: adjustment.type,
          note: `Manual ${adjustment.type} by admin`,
          reference: 'adjustment' // Using valid enum value from StockMovement model
        }
      };
      
      console.log('Sending stock update request:', payload);
      
      const response = await api.patch(`/items/${productId}/stock`, payload);
      
      // Update the local state with the new quantity
      setProducts(prevProducts => 
        prevProducts.map(p => 
          p._id === productId 
            ? { 
                ...p, 
                quantity: response.data.data.newQuantity,
                last_stock_update: new Date().toISOString()
              } 
            : p
        )
      );
      
      setEditingStock(prev => ({ ...prev, [productId]: false }));
      setStockAdjustment(prev => {
        const newState = { ...prev };
        delete newState[productId];
        return newState;
      });
      
      toast.success("Stock updated successfully");
    } catch (error) {
      console.error("Error updating stock:", error);
      toast.error("Failed to update stock");
    }
  };
  
  const handleQuickAdjust = async (productId: string, amount: number) => {
    try {
      const product = products.find(p => p._id === productId);
      if (!product) {
        throw new Error('Product not found');
      }
      
      const newQuantity = Math.max(0, product.quantity + amount);
      
      // Optimistic UI update
      const updatedProducts = products.map(p => 
        p._id === productId ? { ...p, quantity: newQuantity } : p
      );
      setProducts(updatedProducts);
      
      // Prepare the request payload
      const payload = {
        data: {
          quantity: Math.abs(amount),
          action: amount > 0 ? 'add' : 'subtract',
          note: `Quick ${amount > 0 ? 'addition' : 'removal'} by admin`,
          reference: 'adjustment' // Using valid enum value from StockMovement model
        }
      };
      
      console.log('Sending quick adjust request:', payload);
      
      const response = await api.patch(`/items/${productId}/stock`, payload);
      
      // Update the local state with the new quantity
      setProducts(prevProducts => 
        prevProducts.map(p => 
          p._id === productId 
            ? { 
                ...p, 
                quantity: response.data.data.newQuantity,
                last_stock_update: new Date().toISOString()
              } 
            : p
        )
      );
      
      // If we're viewing the stock history for this product, refresh it
      if (selectedProduct && selectedProduct._id === productId) {
        fetchStockMovements(productId);
      }
      
      toast.success(`Stock ${amount > 0 ? 'increased' : 'decreased'} successfully`);
    } catch (error) {
      console.error("Error updating stock:", error);
      toast.error("Failed to update stock");
      // Revert optimistic update on error
      fetchProducts(pagination?.currentPage || 1);
    }
  };
  
  const handleViewStockHistory = (product: Product) => {
    setSelectedProduct(product);
    setActiveTab('stock-history');
    fetchStockMovements(product._id);
  };

  // Update the stock movements when the selected product changes
  useEffect(() => {
    if (activeTab === 'stock-history' && selectedProduct) {
      fetchStockMovements(selectedProduct._id);
    }
  }, [activeTab, selectedProduct]);

  useEffect(() => {
    fetchProducts();
  }, [searchTerm, statusFilter, stockLevelFilter, activeTab]);

  const getStatusBadgeClass = (stock: number, status: boolean, stockStatus?: string) => {
    if (!status) return "px-3 py-1 text-xs font-medium rounded-full bg-gray-100 text-gray-800";
    if (stockStatus === 'out_of_stock' || stock <= outOfStockNumber) return "px-3 py-1 text-xs font-medium rounded-full bg-red-100 text-red-800";
    if (stockStatus === 'low_stock' || (stock > outOfStockNumber && stock < inStockNumber)) return "px-3 py-1 text-xs font-medium rounded-full bg-yellow-100 text-yellow-800";
    return "px-3 py-1 text-xs font-medium rounded-full bg-green-100 text-green-800";
  };

  const getStockStatus = (stock: number, status: boolean, stockStatus?: string) => {
    if (!status) return 'Inactive';
    if (stockStatus === 'out_of_stock' || stock <= outOfStockNumber) return 'Out of Stock';
    if (stockStatus === 'low_stock' || (stock > outOfStockNumber && stock < inStockNumber)) return 'Low Stock';
    return 'In Stock';
  };
  
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };
  
  const getMovementTypeClass = (type: string) => {
    switch (type) {
      case 'in': return 'bg-green-100 text-green-800';
      case 'out': return 'bg-red-100 text-red-800';
      case 'adjustment': return 'bg-blue-100 text-blue-800';
      case 'return': return 'bg-purple-100 text-purple-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };
  
  const getMovementTypeLabel = (type: string) => {
    return type.charAt(0).toUpperCase() + type.slice(1).replace('_', ' ');
  };

  // Load stock settings from backend
  const loadStockSettings = async () => {
    try {
      const response = await api.get('/settings');
      if (response.data && response.data.success) {
        const settings = response.data.data;
        setOutOfStockNumber(settings.outOfStockThreshold || 5);
        setInStockNumber(settings.inStockThreshold || 10);
      }
    } catch (error) {
      console.error('Error loading stock settings:', error);
      // Use default values if loading fails
      setOutOfStockNumber(5);
      setInStockNumber(10);
    }
  };

  // Save stock settings to backend
  const saveStockSettings = async () => {
    try {
      const response = await api.post('/settings', {
        outOfStockThreshold: outOfStockNumber,
        inStockThreshold: inStockNumber
      });

      if (response.data && response.data.success) {
        toast.success('Stock settings saved successfully');
      } else {
        throw new Error('Failed to save settings');
      }
    } catch (error) {
      console.error('Error saving stock settings:', error);
      toast.error('Failed to save stock settings');
    }
  };

  // Reset to default values
  const resetToDefaults = () => {
    setOutOfStockNumber(5);
    setInStockNumber(10);
    toast.success('Settings reset to defaults');
  };

  // Load stock settings when component mounts or when active tab changes to stock-settings
  useEffect(() => {
    if (activeTab === 'stock-settings') {
      loadStockSettings();
    }
  }, [activeTab]);

  // Load stock settings when component mounts for stock overview
  useEffect(() => {
    loadStockSettings();
  }, []);

  return (
    <Layout
      title="Product Management"
      sidebar={<Sidebar items={adminSidebarItems} />}
    >
      <div className="p-6 space-y-6">
        {/* Header with Title and Actions */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">
              Stock Management
            </h1>
            <p className="mt-1 text-sm text-gray-500">
              Manage your pharmacy inventory and products
            </p>
            {pagination && products && (
              <div className="mt-3 text-sm text-gray-600">
                Showing <span className="font-medium">{products.length}</span> of <span className="font-medium">{pagination.total}</span> products
                {pagination.totalProducts !== undefined && (
                  <span className="ml-2">(Total in system: {pagination.totalProducts})</span>
                )}
              </div>
            )}
          </div>
          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              size="sm"
              onClick={() => setActiveTab("bulk-upload")}
            >
              <Upload className="w-4 h-4 mr-2" />
              Bulk Upload
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setActiveTab("bulk-update")}
            >
              <RefreshCw className="w-4 h-4 mr-2" />
              Bulk Update
            </Button>
            <Button
              size="sm"
              onClick={() => navigate("/admin/products/new")}
            >
              <Plus className="w-4 h-4 mr-2" />
              Add Product
            </Button>
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 overflow-x-auto">
            {[
              { id: "list", label: "Stock Overview" },
              { id: "stock-settings", label: "Stock Settings" },
              { id: "stock-history", label: "Stock History" },
              { id: "bulk-upload", label: "Bulk Upload" },
              { id: "bulk-update", label: "Bulk Update" }
            ].map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id as TabType)}
                className={`whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm ${
                  activeTab === tab.id
                    ? "border-blue-500 text-blue-600"
                    : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
                }`}
              >
                {tab.label}
              </button>
            ))}
          </nav>
        </div>

        {/* Tab Content */}
        <div className="mt-6">
          {/* Product List Tab */}
          <TabContent isActive={activeTab === "list"}>
            {/* Search and Filter */}
            <div className="px-4 py-4 border-b border-gray-200 bg-white rounded-t-lg">
              {pagination && products && (
                <div className="mb-3 text-sm text-gray-600">
                  Showing <span className="font-medium">{products.length}</span> of <span className="font-medium">{pagination.total}</span> products
                  {pagination.totalProducts !== undefined && (
                    <span className="ml-2">(Total in system: {pagination.totalProducts})</span>
                  )}
                </div>
              )}
              <div className="flex flex-col space-y-4 sm:space-y-0 sm:flex-row sm:items-center sm:justify-between">
                <div className="flex-1 max-w-2xl">
                  <div className="flex flex-col space-y-3 sm:flex-row sm:space-y-0 sm:space-x-3">
                    <div className="relative flex-1">
                      <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <Search className="h-5 w-5 text-gray-400" />
                      </div>
                      <input
                        type="text"
                        placeholder="Search products..."
                        className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                        value={searchTerm}
                        onChange={(e) => setSearchTerm(e.target.value)}
                      />
                    </div>
                    
                    <div className="flex space-x-2">
                      <select
                        className="block w-full pl-3 pr-10 py-2 text-base border border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                        value={statusFilter}
                        onChange={(e) => setStatusFilter(e.target.value)}
                      >
                        <option value="all">All Status</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                      </select>
                      
                      <select
                        className="block w-full pl-3 pr-10 py-2 text-base border border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                        value={stockLevelFilter}
                        onChange={(e) => setStockLevelFilter(e.target.value)}
                      >
                        <option value="all">All Stock Levels</option>
                        <option value="in_stock">In Stock (≥ {inStockNumber})</option>
                        <option value="low_stock">Low Stock ({outOfStockNumber + 1}-{inStockNumber - 1})</option>
                        <option value="out_of_stock">Out of Stock (≤ {outOfStockNumber})</option>
                        <option value="below_min">Below Minimum</option>
                        <option value="above_max">Above Maximum</option>
                      </select>
                    </div>
                  </div>
                </div>
                
                <div className="flex items-center space-x-2">
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={() => {
                      setSearchTerm('');
                      setStatusFilter('all');
                      setStockLevelFilter('all');
                    }}
                  >
                    <RefreshCw className="w-4 h-4 mr-2" />
                    Reset
                  </Button>
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={() => toast('Export functionality coming soon')}
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Export
                  </Button>
                </div>
              </div>
            </div>

            {/* Products Table */}
            <div className="bg-white shadow-sm border rounded-lg overflow-hidden">
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      {[
                        "#",
                        "Product Details",
                        "SKU",
                        "Category",
                        "Brand",
                        "Price",
                        "Stock Level",
                        "Quantity"
                      ].map((header) => (
                        <th
                          key={header}
                          className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider bg-gray-50"
                        >
                          {header}
                        </th>
                      ))}
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-200">
                    {loading ? (
                      <tr>
                        <td colSpan={7} className="text-center py-12">Loading products...</td>
                      </tr>
                    ) : !products || products.length === 0 ? (
                      <tr>
                        <td colSpan={7} className="text-center py-12">No products found.</td>
                      </tr>
                    ) : products.map((product) => (
                      <tr key={product._id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 text-sm text-gray-500 text-center">
                          {products.findIndex(p => p._id === product._id) + 1}
                        </td>
                        <td className="px-6 py-4">
                          <div>
                            <div className="text-sm font-medium text-gray-900">
                              {product.name}
                            </div>
                            <div className="text-sm text-gray-500">
                              ID: {product._id}
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm">{product.sku}</td>
                        <td className="px-6 py-4 text-sm">
                          {typeof product.category_id === 'string' ? 'Loading...' : product.category_id?.name || 'N/A'}
                        </td>
                        <td className="px-6 py-4 text-sm">
                          {(() => {
                            // If no brand_id, return N/A
                            if (!product.brand_id) {
                              return 'N/A';
                            }
                            
                            // If brand_id is a string (just the ID), we don't have the name yet
                            if (typeof product.brand_id === 'string') {
                              const brand = brandsMap[product.brand_id];
                              return brand || 'Loading...';
                            }
                            
                            // If brand_id is an object, try to get the name from the brandsMap first
                            if (typeof product.brand_id === 'object' && product.brand_id !== null) {
                              // If we have the brand in our map, use that
                              if (product.brand_id._id && brandsMap[product.brand_id._id]) {
                                return brandsMap[product.brand_id._id];
                              }
                              // Otherwise try to get name from the object directly
                              return product.brand_id.name || product.brand_id.brandName || 'N/A';
                            }
                            
                            // Fallback
                            return 'N/A';
                          })()}
                        </td>
                        <td className="px-6 py-4 text-sm text-center">
                          {formatCurrency(product.price || 0)}
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <span className={getStatusBadgeClass(
                            product.quantity || 0,
                            Boolean(product.status),
                            (product as any).stockStatus
                          )}>
                            {getStockStatus(
                              product.quantity || 0,
                              Boolean(product.status),
                              (product as any).stockStatus
                            )}
                          </span>
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <div className="flex flex-col space-y-2">
                            <div className="flex items-center">
                              <div className="w-24 bg-gray-200 rounded-full h-2.5 mr-2">
                                <div 
                                  className={`h-2.5 rounded-full ${
                                    !product.quantity || product.quantity <= outOfStockNumber 
                                      ? 'bg-red-500' 
                                      : product.quantity < inStockNumber
                                        ? 'bg-yellow-500' 
                                        : 'bg-green-500'
                                  }`}
                                  style={{
                                    width: `${Math.min(100, (product.quantity / (product.max_stock_level || 100)) * 100)}%`
                                  }}
                                ></div>
                              </div>
                              <span className={`text-xs font-medium ${
                                !product.quantity || product.quantity <= outOfStockNumber 
                                  ? 'text-red-700' 
                                  : product.quantity < inStockNumber
                                    ? 'text-yellow-700' 
                                    : 'text-green-700'
                              }`}>
                                {!product.quantity || product.quantity <= outOfStockNumber 
                                  ? 'Out of Stock' 
                                  : product.quantity < inStockNumber
                                    ? 'Low Stock' 
                                    : 'In Stock'}
                              </span>
                            </div>
                            <div className="text-xs text-gray-500">
                              {product.max_stock_level > 0 && (
                                <span>Max: {product.max_stock_level} | </span>
                              )}
                              <span>Updated: {new Date(product.updatedAt).toLocaleDateString()}</span>
                            </div>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm text-center">
                          <div className="flex flex-col items-center">
                            <span 
                              className={`inline-flex items-center justify-center w-20 px-3 py-1.5 rounded-md text-sm font-semibold ${
                                !product.quantity && product.quantity !== 0 
                                  ? 'bg-gray-100 text-gray-800 border border-gray-200' 
                                  : product.quantity <= outOfStockNumber 
                                    ? 'bg-red-50 text-red-800 border border-red-200' 
                                    : product.quantity < inStockNumber
                                      ? 'bg-yellow-50 text-yellow-800 border border-yellow-200' 
                                      : 'bg-green-50 text-green-800 border border-green-200'
                              }`}
                              title={`Current stock: ${product.quantity || 0} units`}
                            >
                              {product.quantity !== undefined && product.quantity !== null ? product.quantity : 'N/A'}
                            </span>
                            <span className="text-xs text-gray-500 mt-1">
                              {product.quantity === 1 ? 'unit' : 'units'}
                              {product.min_stock_level !== undefined && (
                                <span className="block text-xs text-gray-400">Min: {product.min_stock_level}</span>
                              )}
                            </span>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <div className="flex items-center space-x-2">
                            {editingStock[product._id] ? (
                              <div className="flex items-center space-x-2">
                                <select
                                  className="block w-24 pl-3 pr-8 py-1.5 text-base border border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                  value={stockAdjustment[product._id]?.type || 'add'}
                                  onChange={(e) => 
                                    setStockAdjustment(prev => ({
                                      ...prev,
                                      [product._id]: {
                                        ...prev[product._id],
                                        type: e.target.value as 'add' | 'subtract' | 'set'
                                      }
                                    }))
                                  }
                                >
                                  <option value="add">Add</option>
                                  <option value="subtract">Subtract</option>
                                  <option value="set">Set to</option>
                                </select>
                                <input
                                  type="number"
                                  min="0"
                                  className="block w-20 pl-2 pr-2 py-1.5 border border-gray-300 rounded-md text-sm"
                                  value={stockAdjustment[product._id]?.quantity || ''}
                                  onChange={(e) => 
                                    setStockAdjustment(prev => ({
                                      ...prev,
                                      [product._id]: {
                                        ...prev[product._id],
                                        quantity: parseInt(e.target.value) || 0
                                      }
                                    }))
                                  }
                                  placeholder="Qty"
                                />
                                <button
                                  onClick={() => handleStockAdjustment(product._id)}
                                  className="text-green-600 hover:text-green-800"
                                >
                                  <Check className="w-5 h-5" />
                                </button>
                                <button
                                  onClick={() => {
                                    setEditingStock(prev => ({ ...prev, [product._id]: false }));
                                    setStockAdjustment(prev => {
                                      const newState = { ...prev };
                                      delete newState[product._id];
                                      return newState;
                                    });
                                  }}
                                  className="text-gray-500 hover:text-gray-700"
                                >
                                  <X className="w-5 h-5" />
                                </button>
                              </div>
                            ) : (
                              <>
                                <span className="font-medium">{product.quantity || 0}</span>
                                <div className="flex space-x-1">
                                  <button
                                    onClick={() => handleQuickAdjust(product._id, 1)}
                                    className="text-green-600 hover:text-green-800 p-1 rounded-full hover:bg-green-50"
                                    title="Add 1"
                                  >
                                    <Plus className="w-4 h-4" />
                                  </button>
                                  <button
                                    onClick={() => handleQuickAdjust(product._id, -1)}
                                    className="text-red-600 hover:text-red-800 p-1 rounded-full hover:bg-red-50"
                                    title="Subtract 1"
                                    disabled={product.quantity <= 0}
                                  >
                                    <Minus className="w-4 h-4" />
                                  </button>
                                  <button
                                    onClick={() => setEditingStock(prev => ({ ...prev, [product._id]: true }))}
                                    className="text-blue-600 hover:text-blue-800 p-1 rounded-full hover:bg-blue-50"
                                    title="Adjust stock"
                                  >
                                    <Edit2 className="w-4 h-4" />
                                  </button>
                                </div>
                              </>
                            )}
                          </div>
                          {product.min_stock_level > 0 && (
                            <div className="text-xs text-gray-500 mt-1">
                              Min: {product.min_stock_level}
                              {product.max_stock_level > 0 && ` | Max: ${product.max_stock_level}`}
                            </div>
                          )}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>

              {/* Pagination */}
              <div className="px-6 py-3 flex items-center justify-between border-t border-gray-200">
                <p className="text-sm text-gray-700">
                  Showing{" "}
                  <span className="font-medium">{pagination ? (pagination.currentPage - 1) * pagination.limit + 1 : 0}</span> to{" "}
                  <span className="font-medium">{pagination ? Math.min(pagination.currentPage * pagination.limit, pagination.total) : 0}</span> of{" "}
                  <span className="font-medium">{pagination ? pagination.total : 0}</span> results
                </p>
                {pagination && pagination.totalPages > 1 && (
                  <div className="flex items-center space-x-2">
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => fetchProducts(pagination.currentPage - 1)}
                      disabled={!pagination.hasPreviousPage}
                    >
                      Previous
                    </Button>
                    <span className="text-sm">
                      Page {pagination.currentPage} of {pagination.totalPages}
                    </span>
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => fetchProducts(pagination.currentPage + 1)}
                      disabled={!pagination.hasNextPage}
                    >
                      Next
                    </Button>
                  </div>
                )}
              </div>
            </div>
          </TabContent>

          {/* Stock Settings Tab */}
          <TabContent isActive={activeTab === "stock-settings"}>
            <div className="bg-white shadow rounded-lg p-6">
              <div className="mb-6">
                <h3 className="text-lg font-medium text-gray-900">Stock Alert Settings</h3>
                <p className="mt-1 text-sm text-gray-500">
                  Configure the stock levels that trigger "Out of Stock" and "In Stock" alerts for products.
                </p>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {/* Out of Stock Settings */}
                <div className="bg-red-50 border border-red-200 rounded-lg p-6">
                  <div className="flex items-center mb-4">
                    <div className="bg-red-100 p-2 rounded-lg mr-3">
                      <PackagePlus className="w-5 h-5 text-red-600" />
                    </div>
                    <div>
                      <h4 className="text-lg font-medium text-gray-900">Out of Stock Alert</h4>
                      <p className="text-sm text-gray-600">Products with stock below this number will show as "Out of Stock"</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-4">
                    <label htmlFor="outOfStockNumber" className="text-sm font-medium text-gray-700">
                      Alert Threshold:
                    </label>
                    <input
                      id="outOfStockNumber"
                      type="number"
                      min="0"
                      max="1000"
                      className="block w-24 pl-3 pr-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-red-500 focus:border-red-500 sm:text-sm"
                      value={outOfStockNumber}
                      onChange={(e) => setOutOfStockNumber(parseInt(e.target.value) || 0)}
                    />
                    <span className="text-sm text-gray-600">units</span>
                  </div>
                  <div className="mt-3 text-sm text-gray-600">
                    Current setting: Products with ≤ {outOfStockNumber} units will show as "Out of Stock"
                  </div>
                </div>

                {/* In Stock Settings */}
                <div className="bg-green-50 border border-green-200 rounded-lg p-6">
                  <div className="flex items-center mb-4">
                    <div className="bg-green-100 p-2 rounded-lg mr-3">
                      <Check className="w-5 h-5 text-green-600" />
                    </div>
                    <div>
                      <h4 className="text-lg font-medium text-gray-900">In Stock Alert</h4>
                      <p className="text-sm text-gray-600">Products with stock above this number will show as "In Stock"</p>
                    </div>
                  </div>
                  <div className="flex items-center space-x-4">
                    <label htmlFor="inStockNumber" className="text-sm font-medium text-gray-700">
                      Alert Threshold:
                    </label>
                    <input
                      id="inStockNumber"
                      type="number"
                      min="0"
                      max="1000"
                      className="block w-24 pl-3 pr-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-green-500 focus:border-green-500 sm:text-sm"
                      value={inStockNumber}
                      onChange={(e) => setInStockNumber(parseInt(e.target.value) || 0)}
                    />
                    <span className="text-sm text-gray-600">units</span>
                  </div>
                  <div className="mt-3 text-sm text-gray-600">
                    Current setting: Products with ≥ {inStockNumber} units will show as "In Stock"
                  </div>
                </div>
              </div>

              <div className="mt-6 flex items-center justify-between">
                <div className="text-sm text-gray-600">
                  <p className="mb-2">These settings will affect how stock status is displayed across the system:</p>
                  <ul className="list-disc list-inside space-y-1 text-sm text-gray-600">
                    <li>• Products with stock ≤ {outOfStockNumber} units: <span className="font-medium text-red-600">Out of Stock</span></li>
                    <li>• Products with stock between {outOfStockNumber + 1} and {inStockNumber - 1} units: <span className="font-medium text-yellow-600">Low Stock</span></li>
                    <li>• Products with stock ≥ {inStockNumber} units: <span className="font-medium text-green-600">In Stock</span></li>
                  </ul>
                </div>
                <div className="flex space-x-3">
                  <Button variant="outline" size="sm" onClick={resetToDefaults}>
                    Reset to Default
                  </Button>
                  <Button size="sm" onClick={saveStockSettings}>
                    Save Settings
                  </Button>
                </div>
              </div>
            </div>
          </TabContent>

          {/* Stock History Tab */}
          <TabContent isActive={activeTab === "stock-history"}>
            <div className="bg-white shadow rounded-lg overflow-hidden">
              <div className="px-6 py-5 border-b border-gray-200 bg-white flex justify-between items-center">
                <div>
                  <h3 className="text-lg font-medium text-gray-900">
                    {selectedProduct ? `Stock History - ${selectedProduct.name}` : 'Stock Movement History'}
                  </h3>
                  <p className="mt-1 text-sm text-gray-500">
                    {selectedProduct 
                      ? `Track all stock movements for ${selectedProduct.name}` 
                      : 'Select a product to view its stock movement history'}
                    {selectedProduct?.sku && ` (SKU: ${selectedProduct.sku})`}
                  </p>
                </div>
                {selectedProduct && (
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={() => {
                      setSelectedProduct(null);
                      setStockMovements([]);
                    }}
                  >
                    <X className="w-4 h-4 mr-2" />
                    Clear Selection
                  </Button>
                )}
              </div>
              
              {!selectedProduct ? (
                <div className="p-6 text-center text-gray-500">
                  <p className="mb-4">Please select a product to view its stock movement history.</p>
                  <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 mt-4">
                    {Array.isArray(products) && products.length > 0 ? (
                      products.slice(0, 8).map(product => (
                        <div 
                          key={product._id} 
                          className="border rounded-lg p-4 hover:bg-gray-50 cursor-pointer flex items-center"
                          onClick={() => handleViewStockHistory(product)}
                        >
                          <div className="bg-blue-100 p-2 rounded-lg mr-3">
                            <PackagePlus className="w-5 h-5 text-blue-600" />
                          </div>
                          <div className="text-left">
                            <div className="font-medium text-gray-900 truncate max-w-xs">{product.name}</div>
                            <div className="text-sm text-gray-500">SKU: {product.sku || 'N/A'}</div>
                            <div className="text-sm">
                              Stock: <span className="font-medium">{product.quantity || 0}</span>
                            </div>
                          </div>
                        </div>
                      ))
                    ) : (
                      <div className="col-span-full text-center py-4 text-gray-500">
                        No products available to show history.
                      </div>
                    )}
                  </div>
                </div>
              ) : loadingMovements ? (
                <div className="p-6 text-center">
                  <p className="mb-4">Loading stock history for {selectedProduct?.name || 'selected product'}...</p>
                  <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-500 mx-auto"></div>
                  <p className="mt-2 text-gray-600">Loading stock movements...</p>
                </div>
              ) : (
                <div className="overflow-x-auto">
                  <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                      <tr>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Date & Time
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Type
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Quantity
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Previous
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          New
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Reference
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          User
                        </th>
                        <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                          Notes
                        </th>
                      </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                      {Array.isArray(stockMovements) ? (
                        stockMovements.length > 0 ? (
                          stockMovements.map((movement) => (
                            <tr key={movement._id} className="hover:bg-gray-50">
                              <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {formatDate(movement.created_at)}
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap">
                                <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${getMovementTypeClass(movement.type)}`}>
                                  {getMovementTypeLabel(movement.type)}
                                </span>
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                {movement.quantity}
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {movement.previous_quantity}
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                {movement.new_quantity}
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {movement.reference || 'N/A'}
                              </td>
                              <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                {movement.user?.name || 'System'}
                              </td>
                              <td className="px-6 py-4 text-sm text-gray-500 max-w-xs truncate" title={movement.notes}>
                                {movement.notes || 'N/A'}
                              </td>
                            </tr>
                          ))
                        ) : (
                          <tr>
                            <td colSpan={8} className="px-6 py-8 text-center text-gray-500">
                              No stock movement history found for this product.
                            </td>
                          </tr>
                        )
                      ) : (
                        <tr>
                          <td colSpan={8} className="px-6 py-8 text-center text-gray-500">
                            No stock movement history found for this product.
                          </td>
                        </tr>
                      )}
                    </tbody>
                  </table>
                </div>
              )}
            </div>
          </TabContent>

          {/* Bulk Upload Tab */}
          <TabContent isActive={activeTab === "bulk-upload"}>
            <div className="bg-white shadow rounded-lg p-6">
              <BulkUpload onBackToList={() => setActiveTab("list")} />
            </div>
          </TabContent>

          {/* Bulk Update Tab */}
          <TabContent isActive={activeTab === "bulk-update"}>
            <div className="bg-white shadow rounded-lg p-6">
              <BulkUpdate onBackToList={() => setActiveTab("list")} />
            </div>
          </TabContent>
        </div>
      </div>
    </Layout>
  );
};

export default Products;
