import React, { useEffect, useMemo, useState, Suspense, lazy } from 'react';
import Layout from '../../../../components/shared/Layout';
import Sidebar from '../../../../components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';
import { Search, Plus, Edit, Trash2, Eye, Package } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';
import 'react-quill/dist/quill.snow.css';

// Lazy load ReactQuill for better performance
const ReactQuill = lazy(() => import('react-quill'));

// Loading component for the editor
const EditorLoading = () => (
  <div className="h-32 bg-gray-100 rounded-lg animate-pulse"></div>
);

// Quill modules and formats
const modules = {
  toolbar: [
    [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
    ['bold', 'italic', 'underline', 'strike', 'blockquote'],
    [{'list': 'ordered'}, {'list': 'bullet'}, {'indent': '-1'}, {'indent': '+1'}],
    ['link', 'image'],
    ['clean']
  ],
};

const formats = [
  'header',
  'bold', 'italic', 'underline', 'strike', 'blockquote',
  'list', 'bullet', 'indent',
  'link', 'image'
];

interface BrandOption {
  _id: string;
  name?: string;
  brandName?: string;
}

interface CategoryOption {
  _id: string;
  name: string;
}

interface UnitOption {
  _id: string;
  name: string;
  shortCode?: string;
}

interface TaxOption {
  _id: string;
  name: string;
  percentage: number;
}

interface DiscountOption {
  _id: string;
  name: string;
  type: 'percentage' | 'fixed';
  value: number;
}

interface StoreOption {
  _id: string;
  name: string;
  storeName?: string;
}

interface ItemImage {
  _id?: string;
  url: string;
  is_primary: boolean;
  position: number;
}

interface Item {
  _id: string;
  name: string;
  sku: string;
  hsn?: string;
  barcode?: string;
  description?: string;
  images?: ItemImage[];
  price: number;
  sales_price?: number;
  quantity: number;
  expire_date?: string | null;
  status: boolean;
  brand_id?: { _id: string; name: string } | string;
  category_id?: { _id: string; name: string } | string;
  unit_id?: { _id: string; name: string; shortCode?: string } | string;
  tax_id?: { _id: string; name: string; percentage: number } | string;
  tax_type?: 'inclusive' | 'exclusive';
  discount_id?: { _id: string; name: string } | string;
  store_id?: { _id: string; name: string } | string;
  createdAt: string;
  updatedAt: string;
  // For backward compatibility
  image?: string;
}

interface PaginationMeta {
  total: number;
  totalPages: number;
  currentPage: number;
  hasNextPage: boolean;
  hasPreviousPage: boolean;
  limit: number;
}

interface ItemFormData {
  name: string;
  sku: string;
  price: string;
  brand_id: string;
  category_id: string;
  unit_id: string;
  description: string;
  hsn: string;
  barcode: string;
  images?: File[] | string[] | ItemImage[];
  quantity: string;
  expire_date: string;
  sales_price: string;
  tax_id: string;
  tax_type: 'inclusive' | 'exclusive';
  discount_id: string;
  store_id: string;
  status: boolean;
  requiresPrescription: boolean;
  // For backward compatibility
  image?: string;
}

const Items: React.FC = () => {
  const [items, setItems] = useState<Item[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [searchTerm, setSearchTerm] = useState<string>('');
  const [currentPage, setCurrentPage] = useState<number>(1);
  const [totalPages, setTotalPages] = useState<number>(1);
  const [pagination, setPagination] = useState<PaginationMeta | null>(null); // eslint-disable-line @typescript-eslint/no-unused-vars
  const [selectedImages, setSelectedImages] = useState<File[]>([]);
  const [imagePreviews, setImagePreviews] = useState<string[]>([]);

  const [showEditModal, setShowEditModal] = useState<boolean>(false);
  const [showViewModal, setShowViewModal] = useState<boolean>(false);
  const [selectedItem, setSelectedItem] = useState<Item | null>(null);

  const [brands, setBrands] = useState<BrandOption[]>([]);
  const [categories, setCategories] = useState<CategoryOption[]>([]);
  const [units, setUnits] = useState<UnitOption[]>([]);
  const [taxes, setTaxes] = useState<TaxOption[]>([]);
  const [discounts, setDiscounts] = useState<DiscountOption[]>([]);
  const [stores, setStores] = useState<StoreOption[]>([]);

  // Local preview states (optional)
  const [formData, setFormData] = useState<ItemFormData>({
    name: '',
    sku: '',
    price: '',
    brand_id: '',
    category_id: '',
    unit_id: '',
    description: '',
    hsn: '',
    barcode: '',
    quantity: '0',
    expire_date: '',
    sales_price: '',
    tax_id: '',
    tax_type: 'exclusive',
    discount_id: '',
    store_id: '',
    status: true,
    requiresPrescription: false,
    images: [],
    image: ''
  });

  const [selectedImageFile, setSelectedImageFile] = useState<File | null>(null); // eslint-disable-line @typescript-eslint/no-unused-vars
  const [imagePreview, setImagePreview] = useState<string | null>(null); // eslint-disable-line @typescript-eslint/no-unused-vars
  const [selectedDiscount, setSelectedDiscount] = useState<DiscountOption | null>(null);

  // Calculate final price based on base price and selected discount
  const calculateFinalPrice = (price: string, discount: DiscountOption | null) => {
    const basePrice = parseFloat(price) || 0;
    
    // If no discount is selected or price is not a valid number, return the base price
    if (!discount || isNaN(basePrice)) {
      return basePrice;
    }

    // Calculate discount based on type
    if (discount.type === 'percentage') {
      const discountAmount = (basePrice * discount.value) / 100;
      return Math.max(0, basePrice - discountAmount);
    } else if (discount.type === 'fixed') {
      return Math.max(0, basePrice - discount.value);
    }
    
    // Default return base price if something unexpected happens
    return basePrice;
  };

  // Calculate final price whenever price or discount changes
  const finalPrice = useMemo(() => {
    return calculateFinalPrice(formData?.price || '', selectedDiscount);
  }, [formData?.price, selectedDiscount]);

  const [errors, setErrors] = useState<Record<keyof ItemFormData, string>>({
    name: '',
    sku: '',
    price: '',
    brand_id: '',
    category_id: '',
    unit_id: '',
    description: '',
    hsn: '',
    barcode: '',
    image: '',
    quantity: '',
    expire_date: '',
    sales_price: '',
    tax_id: '',
    tax_type: '',
    discount_id: '',
    store_id: '',
    status: ''
  });

  const validateForm = (): boolean => {
    console.log('Validating form with data:', formData);
    const newErrors: Record<string, string> = {};
    let isValid = true;

    // Validate store_id
    if (!formData.store_id) {
      newErrors.store_id = 'Store is required';
      isValid = false;
    }

    // Validate image for new items (not required for updates)
    if (!selectedItem && selectedImages.length === 0) {
      newErrors.image = 'At least one image is required';
      isValid = false;
      console.log('Image validation failed - no images selected');
    }

    if (!formData.name.trim()) {
      newErrors.name = 'Item name is required';
      isValid = false;
    }

    if (!formData.sku.trim()) {
      newErrors.sku = 'SKU is required';
      isValid = false;
    }

    if (!formData.brand_id) {
      newErrors.brand_id = 'Brand is required';
      isValid = false;
    }

    if (!formData.category_id) {
      newErrors.category_id = 'Category is required';
      isValid = false;
    }

    if (!formData.unit_id) {
      newErrors.unit_id = 'Unit is required';
      isValid = false;
    }

    // Price validation
    if (!formData.price) {
      newErrors.price = 'Price is required';
      isValid = false;
    } else if (isNaN(Number(formData.price)) || Number(formData.price) <= 0) {
      newErrors.price = 'Please enter a valid price';
      isValid = false;
    }

    // Sales price validation if provided
    if (formData.sales_price && (isNaN(Number(formData.sales_price)) || Number(formData.sales_price) < 0)) {
      newErrors.sales_price = 'Please enter a valid sales price';
      isValid = false;
    }

    // Quantity validation
    if (isNaN(Number(formData.quantity)) || Number(formData.quantity) < 0) {
      newErrors.quantity = 'Please enter a valid quantity';
      isValid = false;
    }

    setErrors(prev => ({ ...prev, ...newErrors }));
    return isValid;
  };

  const handleFieldChange = (field: keyof ItemFormData, value: any) => {
    const newFormData = {
      ...formData,
      [field]: value,
      // Ensure images array is always present
      images: formData.images || [],
      // Ensure image is always a string
      image: formData.image || ''
    };
    
    // Calculate new sales price if either price or discount changes
    const shouldUpdateSalesPrice = field === 'price' || field === 'discount_id';
    
    if (shouldUpdateSalesPrice) {
      const currentPrice = field === 'price' ? parseFloat(value) || 0 : parseFloat(formData.price) || 0;
      const currentDiscountId = field === 'discount_id' ? value : formData.discount_id;
      
      // Find the current discount (if any)
      const currentDiscount = currentDiscountId ? discounts.find(d => d._id === currentDiscountId) : null;
      
      // Update selected discount in state
      if (field === 'discount_id') {
        setSelectedDiscount(currentDiscount || null);
      }
      
      // Calculate new sales price
      if (currentDiscount) {
        if (currentDiscount.type === 'percentage') {
          const discountAmount = (currentPrice * currentDiscount.value) / 100;
          newFormData.sales_price = (currentPrice - discountAmount).toFixed(2);
        } else if (currentDiscount.type === 'fixed') {
          newFormData.sales_price = Math.max(0, currentPrice - currentDiscount.value).toFixed(2);
        }
      } else {
        // If no discount or discount removed, set sales_price to the original price
        newFormData.sales_price = currentPrice.toFixed(2);
      }
    }
    
    setFormData(newFormData);
    
    // Clear error when user starts typing
    if (errors[field as string]) {
      setErrors(prev => ({
        ...prev,
        [field]: ''
      }));
    }
  };

  const hasItems = useMemo(() => items.length > 0, [items]);

  const fetchReferenceData = async () => {
    try {
      const [brandsRes, categoriesRes, unitsRes, taxesRes, discountsRes, storesRes] = await Promise.all([
        api.get('/brands'),
        api.get('/categories'),
        api.get('/units'),
        api.get('/taxes'),
        api.get('/discounts'),
        api.get('/stores')
      ]);
      const normalize = (res: any) =>
        Array.isArray(res?.data?.data) ? res.data.data : (Array.isArray(res?.data) ? res.data : (Array.isArray(res) ? res : []));
      const rawBrands = normalize(brandsRes);
      // Map backend brand shape (brandName) to { _id, name }
      const mappedBrands: BrandOption[] = rawBrands.map((b: any) => ({
        _id: b._id || b.id,
        name: b.name || b.brandName,
        brandName: b.brandName
      })).filter((b: BrandOption) => !!b._id);
      setBrands(mappedBrands);
      setCategories(normalize(categoriesRes));
      setUnits(normalize(unitsRes));
      setTaxes(normalize(taxesRes));
      setDiscounts(normalize(discountsRes));
      const rawStores = normalize(storesRes);
      const mappedStores: StoreOption[] = rawStores.map((s: any) => ({
        _id: s._id || s.id,
        name: s.name || s.store?.storeName || s.storeName || '',
        storeName: s.store?.storeName || s.storeName
      })).filter((s: StoreOption) => !!s._id);
      setStores(mappedStores);
    } catch (err: any) {
      console.error('Error fetching reference data', err);
      toast.error(err.response?.data?.msg || 'Failed to load reference data');
    }
  };

  const fetchItems = async (page = 1, search = '') => {
    try {
      setLoading(true);
      const response = await api.get('/items', {
        params: {
          page,
          limit: 100, // Increased limit to show more items per page
          search,
          status: 'all' // Request all items (both active and inactive)
        }
      });

      const { data, pagination: meta } = response.data;
      setItems(data || []);
      setPagination(meta || null);
      setTotalPages(meta?.totalPages || 1);
      setCurrentPage(page);
    } catch (error: any) {
      console.error('Error fetching items:', error);
      toast.error(error.response?.data?.msg || 'Failed to fetch items');
    } finally {
      setLoading(false);
    }
  };

  const createItem = async (payload: ItemFormData) => {
    console.log('createItem called with payload:', payload);
    try {
      const form = new FormData();
      console.log('Created new FormData instance');
      
      // Add basic fields with proper formatting
      form.append('name', payload.name.trim());
      form.append('sku', payload.sku.trim());
      form.append('price', String(parseFloat(payload.price) || 0));
      form.append('brand_id', payload.brand_id);
      form.append('category_id', payload.category_id);
      form.append('unit_id', payload.unit_id);
      form.append('status', payload.status === false ? 'false' : 'true');
      form.append('requiresPrescription', String(payload.requiresPrescription));
      
      // Add optional fields if they exist
      if (payload.description) form.append('description', payload.description);
      if (payload.hsn) form.append('hsn', payload.hsn);
      if (payload.barcode) form.append('barcode', payload.barcode);
      if (payload.minimum_qty) form.append('minimum_qty', payload.minimum_qty);
      if (payload.expire_date) form.append('expire_date', payload.expire_date);
      if (payload.sales_price) form.append('sales_price', payload.sales_price);
      if (payload.tax_id) form.append('tax_id', payload.tax_id);
      if (payload.tax_type) form.append('tax_type', payload.tax_type);
      if (payload.discount_id) form.append('discount_id', payload.discount_id);
      if (payload.store_id) form.append('store_id', payload.store_id);
      
      // Ensure all required fields are present
      if (!payload.name || !payload.sku || !payload.brand_id || !payload.category_id || !payload.unit_id) {
        throw new Error('Missing required fields');
      }
      
      // Add multiple images - use 'images[]' for array format expected by multer
      if (selectedImages.length > 0) {
        console.log(`Appending ${selectedImages.length} images to form data`);
        selectedImages.forEach((file, index) => {
          console.log(`Appending image ${index + 1}:`, file.name, 'size:', file.size, 'type:', file.type);
          form.append('images', file); // Keep as 'images' (not 'images[]') for multer.array()
        });
      } else {
        console.log('No images selected');
        // If no images, ensure we still send an empty array
        form.append('images', new Blob(), 'empty');
      }
      
      // Log form data keys for debugging
      console.log('Form data keys:', Array.from(form.keys()));
      console.log('Sending request to /items with form data');
      
      const response = await api.post('/items', form, {
        headers: {
          'Content-Type': 'multipart/form-data'
        }

      });
      
      return response.data;
    } catch (error: any) {
      console.error('Error creating item:', error);
      throw error;
    }
  };

  const updateItem = async (id: string, payload: ItemFormData) => {
    try {
      const form = new FormData();
      
      // Helper function to handle empty strings for ObjectId fields
      const handleObjectIdField = (value: string | undefined) => {
        return value && value.trim() !== '' ? value : null;
      };
      
      // Always append the basic fields
      const fields = {
        name: payload.name,
        sku: payload.sku,
        price: String(parseFloat(payload.price) || 0),
        brand_id: handleObjectIdField(payload.brand_id),
        category_id: handleObjectIdField(payload.category_id),
        unit_id: handleObjectIdField(payload.unit_id),
        status: Boolean(payload.status),
        description: payload.description || '',
        hsn: payload.hsn || '',
        barcode: payload.barcode || '',
        minimum_qty: payload.minimum_qty || '0',
        sales_price: payload.sales_price || '0',
        tax_id: handleObjectIdField(payload.tax_id),
        tax_type: payload.tax_type || '',
        discount_id: handleObjectIdField(payload.discount_id),
        store_id: handleObjectIdField(payload.store_id),
        requiresPrescription: payload.requiresPrescription
      };

      // Append all fields to form data
      Object.entries(fields).forEach(([key, value]) => {
        if (value !== undefined && value !== null) {
          form.append(key, value);
        }
      });
      
      // Handle image uploads and removals
      const validImages = Array.isArray(selectedImages) ? 
        selectedImages.filter(file => file instanceof File) : [];
      
      // Add new images to form data with the correct field name 'images' that matches the backend
      validImages.forEach((file) => {
        form.append('images', file);  // Must match the field name in the backend's multer config
      });
      
      // Handle image removals and primary image updates
      if (selectedItem?.images?.length > 0) {
        const currentImageUrls = new Set(selectedItem.images.map(img => img.url));
        const remainingImageUrls = new Set(
          (imagePreviews || [])
            .filter((preview): preview is string => typeof preview === 'string')
            .filter(url => url.startsWith('http') || url.startsWith('/'))
        );
        
        // Find images that were removed
        const removedImages = Array.from(currentImageUrls).filter(
          url => !remainingImageUrls.has(url)
        );
        
        if (removedImages.length > 0) {
          form.append('removedImages', JSON.stringify(removedImages));
        }

        // Create a clean array of existing images with their current primary status
        const existingImages = selectedItem.images
          .filter(img => remainingImageUrls.has(img.url) && !removedImages.includes(img.url))
          .map(img => ({
            url: img.url,
            is_primary: img.is_primary || false,
            position: img.position || 0,
            _id: img._id
          }));

        // If we have new images, they're already being added to the form data
        // Now ensure we have exactly one primary image in the existing images
        const hasPrimary = existingImages.some(img => img.is_primary);
        if (existingImages.length > 0 && !hasPrimary) {
          // If no primary is set, make the first image primary
          existingImages[0].is_primary = true;
        }

        // Send the updated image data to the backend
        if (existingImages.length > 0) {
          form.append('existingImages', JSON.stringify(existingImages));
        }
      }
      
      // Set the correct content type for file upload
      const config = {
        headers: {
          'Content-Type': 'multipart/form-data',
          'Accept': 'application/json'
        }
      };
      
      const response = await api.put(`/items/${id}`, form, config);
      return response.data;
    } catch (error: any) {
      console.error('Error updating item:', error);
      throw error;
    }
  };

  const toggleItemStatus = async (item: Item) => {
    try {
      const newStatus = !item.status;
      const response = await api.put(`/items/${item._id}`, { status: newStatus });
      
      // Update the local state to reflect the change immediately
      setItems(prevItems => 
        prevItems.map(i => 
          i._id === item._id ? { ...i, status: newStatus } : i
        )
      );
      
      toast.success(`Item marked as ${newStatus ? 'Active' : 'Inactive'}`);
    } catch (error: any) {
      console.error('Error toggling item status:', error);
      toast.error(error.response?.data?.msg || 'Failed to update item status');
    }
  };

  const deleteItem = async (id: string) => {
    if (window.confirm('Are you sure you want to delete this item?')) {
      try {
        await api.delete(`/items/${id}`);
        toast.success('Item deleted successfully');
        fetchItems(currentPage, searchTerm);
      } catch (error: any) {
        console.error('Error deleting item:', error);
        toast.error(error.response?.data?.msg || 'Failed to delete item');
      }
    }
  };

  const resetForm = () => {
    setFormData({
      name: '',
      sku: '',
      price: '',
      brand_id: '',
      category_id: '',
      unit_id: '',
      description: '',
      hsn: '',
      barcode: '',
      quantity: '0',
      expire_date: '',
      sales_price: '',
      tax_id: '',
      tax_type: 'exclusive',
      discount_id: '',
      store_id: '',
      status: true,
      images: []
    });
    setSelectedImages([]);
    setImagePreviews([]);
    // Clear all errors when resetting the form
    setErrors({
      name: '',
      sku: '',
      price: '',
      brand_id: '',
      category_id: '',
      unit_id: '',
      description: '',
      hsn: '',
      barcode: '',
      quantity: '',
      expire_date: '',
      sales_price: '',
      tax_id: '',
      tax_type: '',
      discount_id: '',
      store_id: '',
      status: ''
    });
    setSelectedItem(null);
    setSelectedImageFile(null);
    setImagePreview(null);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    console.log('Form submission started');
    e.preventDefault();
    
    // Validate form
    console.log('Validating form...');
    if (!validateForm()) {
      console.log('Form validation failed');
      return;
    }
    console.log('Form validation passed');
    
    // File validation for multiple images
    if (selectedImages && selectedImages.length > 0) {
      for (const file of selectedImages) {
        if (!file) continue; // Skip if file is null or undefined
        
        // Check if file has size property (it's a File object)
        if (file instanceof File || (file as File).size !== undefined) {
          const fileSize = (file as File).size;
          if (fileSize > 5 * 1024 * 1024) { // 5MB limit per file
            const fileName = file.name || 'A file';
            toast.error(`Image '${fileName}' exceeds the 5MB size limit`);
            return;
          }
        }
      }
    }
    
    try {
      if (selectedItem) {
        await updateItem(selectedItem._id, formData);
        toast.success('Item updated successfully');
      } else {
        await createItem(formData);
        toast.success('Item created successfully');
      }
      setShowEditModal(false);
      resetForm();
      fetchItems(currentPage, searchTerm);
    } catch (error: any) {
      console.error('Error in handleSubmit:', error);
      
      // Handle file size limit error
      if (error.code === 'ERR_NETWORK' && error.message.includes('413')) {
        toast.error('File size is too large. Maximum allowed size is 5MB');
      } 
      // Handle Multer file size limit error
      else if (error.response?.data?.code === 'LIMIT_FILE_SIZE') {
        toast.error('File size is too large. Maximum allowed size is 5MB');
      }
      // Handle file type errors
      else if (error.response?.data?.code === 'INVALID_FILE_TYPE') {
        toast.error('Invalid file type. Please upload an image (JPG, PNG, JPEG)');
      }
      // Handle other errors
      else {
        const msg = error.response?.data?.msg || error.message || 'An error occurred while saving the item';
        toast.error(msg);
      }
    }
  };

  const handleImageUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files) {
      const files = Array.from(e.target.files);
      const newImages = [...selectedImages];
      const newPreviews = [...imagePreviews];
      
      files.forEach(file => {
        // Check file type
        if (!file.type.match('image.*')) {
          toast.error('Only image files are allowed');
          return;
        }
        
        // Check file size (5MB max)
        if (file.size > 5 * 1024 * 1024) {
          toast.error('Image size should be less than 5MB');
          return;
        }
        
        newImages.push(file);
        
        // Create preview
        const reader = new FileReader();
        reader.onload = (e) => {
          if (e.target?.result) {
            newPreviews.push(e.target.result as string);
            setImagePreviews([...newPreviews]);
          }
        };
        reader.readAsDataURL(file);
      });
      
      setSelectedImages(newImages);
    }
  };

  const removeImage = (index: number) => {
    const newImages = [...selectedImages];
    const newPreviews = [...imagePreviews];
    
    // Remove the image at the specified index
    newImages.splice(index, 1);
    newPreviews.splice(index, 1);
    
    setSelectedImages(newImages);
    setImagePreviews(newPreviews);
    
    // Update form data
    setFormData(prev => ({
      ...prev,
      images: newImages
    }));
  };

  const setPrimaryImage = (index: number, isExistingImage = false) => {
    if (isExistingImage) {
      // Handle existing images (from selectedItem.images)
      if (!selectedItem?.images?.length) return;
      
      const updatedImages = [...selectedItem.images];
      
      // Mark all images as not primary first
      updatedImages.forEach(img => {
        img.is_primary = false;
      });
      
      // Set the selected image as primary
      updatedImages[index].is_primary = true;
      
      // Move the primary image to the first position
      if (index > 0) {
        const [movedImage] = updatedImages.splice(index, 1);
        updatedImages.unshift(movedImage);
      }
      
      // Create a clean array of images with proper structure for the API
      const cleanImages = updatedImages.map((img, idx) => ({
        url: img.url,
        is_primary: img.is_primary,
        position: idx,
        _id: img._id || new mongoose.Types.ObjectId().toString()
      }));
      
      // Update the selectedItem state
      setSelectedItem(prev => ({
        ...prev!,
        images: cleanImages
      }));
      
      // Update formData to include the updated images with primary status
      setFormData(prev => ({
        ...prev,
        images: cleanImages,
        // Also set the first image as the main image for backward compatibility
        image: cleanImages[0]?.url || ''
      }));
      
      // Also update the imagePreviews to maintain consistency
      setImagePreviews(cleanImages.map(img => img.url));
    } else {
      // Handle new uploads (imagePreviews and selectedImages)
      if (imagePreviews.length === 0) return;
      
      const newPreviews = [...imagePreviews];
      const newImages = [...selectedImages];

      // Move the selected image to the first position
      if (index >= 0) {
        const [movedPreview] = newPreviews.splice(index, 1);
        const [movedImage] = newImages.splice(index, 1);

        newPreviews.unshift(movedPreview);
        newImages.unshift(movedImage);

        setImagePreviews(newPreviews);
        setSelectedImages(newImages);

        // Create a clean array of images with proper structure for the API
        const updatedFormImages = newImages.map((img, idx) => {
          // Handle different possible image formats safely
          let url: string;
          let id: string;
          
          if (img instanceof File) {
            // For new file uploads
            url = URL.createObjectURL(img);
            id = new mongoose.Types.ObjectId().toString();
          } else if (typeof img === 'string') {
            // For string URLs
            url = img;
            id = new mongoose.Types.ObjectId().toString();
          } else if (img && typeof img === 'object') {
            // For existing image objects
            url = img.url || '';
            id = img._id || new mongoose.Types.ObjectId().toString();
          } else {
            // Fallback for any other case
            console.warn('Unexpected image format:', img);
            url = '';
            id = new mongoose.Types.ObjectId().toString();
          }
          
          return {
            url: url,
            is_primary: idx === 0, // First image is primary
            position: idx,
            _id: id
          };
        });
        
        // Update form data with the new images
        setFormData(prev => ({
          ...prev,
          images: updatedFormImages,
          // Also set the first image as the main image for backward compatibility
          image: updatedFormImages[0]?.url || ''
        }));
      }
    }
  };

  const handleEdit = (item: Item) => {
    setSelectedItem(item);
    
    // Set form data with proper type safety
    const formData: ItemFormData = {
      name: item.name || '',
      sku: item.sku || '',
      price: String(item.price ?? ''),
      brand_id: typeof item.brand_id === 'object' ? item.brand_id?._id : (item.brand_id || ''),
      category_id: typeof item.category_id === 'string' ? item.category_id : (item.category_id?._id || ''),
      unit_id: typeof item.unit_id === 'string' ? item.unit_id : (item.unit_id?._id || ''),
      description: item.description || '',
      hsn: item.hsn || '',
      barcode: item.barcode || '',
      quantity: item.quantity !== undefined ? String(item.quantity) : '0',
      expire_date: item.expire_date ? String(item.expire_date).slice(0, 10) : '',
      sales_price: item.sales_price !== undefined ? String(item.sales_price) : '',
      tax_id: (item as any).tax_id?._id || '',
      tax_type: (item as any).tax_type || 'exclusive',
      discount_id: (item as any).discount_id?._id || '',
      store_id: (item as any).store_id?._id || '',
      status: item.status,
      // Handle both new images array and legacy single image
      images: item.images || (item.image ? [{ url: item.image, is_primary: true, position: 0 }] : []),
      // Keep legacy image for backward compatibility
      image: item.image || ''
    };
    
    setFormData(formData);
    
    // Clear any previously selected images when editing
    setSelectedImages([]);
    
    // Set image previews for existing images
    if (item.images && item.images.length > 0) {
      // Sort images: primary first, then by position
      const previews = [...item.images]
        .sort((a, b) => {
          if (a.is_primary) return -1;
          if (b.is_primary) return 1;
          return (a.position || 0) - (b.position || 0);
        })
        .map(img => img.url);
      setImagePreviews(previews);
    } else if (item.image) {
      // For backward compatibility with single image
      setImagePreviews([item.image]);
    } else {
      setImagePreviews([]);
    }
    // Clear any previous errors when opening the edit modal
    setErrors({
      name: '',
      sku: '',
      price: '',
      brand_id: '',
      category_id: '',
      unit_id: '',
      description: '',
      hsn: '',
      barcode: '',
      minimum_qty: '',
      expire_date: '',
      sales_price: '',
      tax_id: '',
      tax_type: '',
      discount_id: '',
      store_id: '',
      status: ''
    });
    setShowEditModal(true);
  };

  const handleSearch = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
    fetchItems(1, value);
  };

  useEffect(() => {
    fetchReferenceData();
    fetchItems();
  }, []);

  const getStatusBadge = (status: boolean) => {
    return status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800';
  };

  // Helper function to get brand name consistently
  const getBrandName = (brandId: string | { _id: string; name?: string } | undefined): string => {
    if (!brandId) {
      return 'N/A';
    }
    
    try {
      // If brand_id is a string, find the brand in the brands array
      if (typeof brandId === 'string') {
        const brand = brands.find(b => b._id === brandId);
        return brand?.name || brand?.brandName || 'N/A';
      }
      
      // If brand_id is an object with _id and name properties
      if (brandId && typeof brandId === 'object') {
        // Check if it's a populated brand object
        if (brandId._id) {
          const name = brandId.name || brandId.brandName;
          if (name) return name;
          
          // If no name found, try to find in brands array
          const brand = brands.find(b => b._id === brandId._id);
          return brand?.name || brand?.brandName || 'N/A';
        }
        
        // Handle case where brand object might have different structure
        const possibleKeys = ['name', 'brandName', 'title', 'brand_name'];
        for (const key of possibleKeys) {
          if (brandId[key]) {
            return brandId[key];
          }
        }
      }
      
      return 'N/A';
    } catch (error) {
      return 'N/A';
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  return (
    <Layout title="Items Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="text"
              placeholder="Search items..."
              className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>

          <button
            onClick={() => {
              resetForm();
              setShowEditModal(true);
            }}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Item
          </button>
        </div>

        {loading && (
          <div className="flex justify-center items-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        )}

        {!loading && hasItems && (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Item</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Brand</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Store</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Category</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Unit</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Price (₹)</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Sales Price (₹)</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {items.map((item) => (
                    <tr 
                      key={item._id} 
                      className={`${item.status ? 'hover:bg-gray-50' : 'opacity-70 hover:opacity-90'} transition-opacity`}
                    >
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          {item.images && item.images.length > 0 ? (
                            <div className="relative">
                              <img 
                                src={item.images[0].url.startsWith('http') ? item.images[0].url : `${import.meta.env.VITE_API_URL || 'http://localhost:5000'}${item.images[0].url}`} 
                                alt={item.name} 
                                className={`w-10 h-10 rounded mr-3 object-cover ${!item.status ? 'opacity-60' : ''}`} 
                              />
                              {!item.status && (
                                <div className="absolute inset-0 flex items-center justify-center">
                                  <span className="text-xs font-medium text-red-600 bg-white/90 px-1 rounded">Inactive</span>
                                </div>
                              )}
                            </div>
                          ) : item.image ? (
                            <div className="relative">
                              <img 
                                src={item.image.startsWith('http') ? item.image : `${import.meta.env.VITE_API_URL || 'http://localhost:5000'}${item.image}`} 
                                alt={item.name} 
                                className={`w-10 h-10 rounded mr-3 object-cover ${!item.status ? 'opacity-60' : ''}`} 
                              />
                              {!item.status && (
                                <div className="absolute inset-0 flex items-center justify-center">
                                  <span className="text-xs font-medium text-red-600 bg-white/90 px-1 rounded">Inactive</span>
                                </div>
                              )}
                            </div>
                          ) : (
                            <div className="relative">
                              <Package className={`w-10 h-10 ${item.status ? 'text-blue-500' : 'text-gray-400'} mr-3`} />
                              {!item.status && (
                                <div className="absolute inset-0 flex items-center justify-center -mt-1">
                                  <span className="text-xs font-medium text-red-600 bg-white/90 px-1 rounded">Inactive</span>
                                </div>
                              )}
                            </div>
                          )}
                          <div>
                            <div className={`text-sm font-medium ${item.status ? 'text-gray-900' : 'text-gray-500'}`}>
                              {item.name}
                              {!item.status && <span className="ml-2 text-xs text-red-500">(Inactive)</span>}
                            </div>
                            <div className="text-xs text-gray-400">Created {formatDate(item.createdAt)}</div>
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {(() => {
                            // If brand_id is an object with just _id, try to find the brand in the brands list
                            if (item.brand_id && typeof item.brand_id === 'object' && item.brand_id._id) {
                              const brand = brands.find(b => b._id === item.brand_id._id);
                              return brand ? (brand.name || brand.brandName || 'N/A') : 'N/A';
                            }
                            // If brand_id is a string ID
                            if (typeof item.brand_id === 'string') {
                              const brand = brands.find(b => b._id === item.brand_id);
                              return brand ? (brand.name || brand.brandName || 'N/A') : 'N/A';
                            }
                            return 'N/A';
                          })()}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">
                          {(() => {
                            // If store_id is an object with name
                            if (item.store_id && typeof item.store_id === 'object' && item.store_id.name) {
                              return item.store_id.name;
                            }
                            // If store_id is a string ID, find the store in the stores list
                            if (typeof item.store_id === 'string') {
                              const store = stores.find(s => s._id === item.store_id);
                              return store ? (store.name || store.storeName || 'N/A') : 'N/A';
                            }
                            return 'N/A';
                          })()}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">{typeof item.category_id === 'object' ? item.category_id?.name : ''}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">{typeof item.unit_id === 'object' ? (item.unit_id?.shortCode || item.unit_id?.name) : ''}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right">
                        <div className="text-sm text-gray-900 font-medium">
                          {item.price ? `₹${parseFloat(item.price).toFixed(2)}` : '-'}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right">
                        <div className="text-sm font-medium">
                          {item.sales_price ? (
                            <span className={item.sales_price < item.price ? 'text-green-600' : 'text-gray-900'}>
                              ₹{parseFloat(item.sales_price).toFixed(2)}
                            </span>
                          ) : item.price ? (
                            <span className="text-gray-500">₹{parseFloat(item.price).toFixed(2)}</span>
                          ) : (
                            '-'
                          )}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="flex items-center">
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              toggleItemStatus(item);
                            }}
                            className={`relative inline-flex h-6 w-11 items-center rounded-full transition-colors focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 ${
                              item.status ? 'bg-green-600' : 'bg-gray-200'
                            }`}
                          >
                            <span
                              className={`${
                                item.status ? 'translate-x-6' : 'translate-x-1'
                              } inline-block h-4 w-4 transform rounded-full bg-white transition-transform`}
                            />
                          </button>
                          <span className={`ml-2 text-sm font-medium ${item.status ? 'text-green-800' : 'text-red-600 font-semibold'}`}>
                            {item.status ? 'Active' : 'Inactive'}
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          <button
                            onClick={() => {
                              setSelectedItem(item);
                              setShowViewModal(true);
                            }}
                            className="text-blue-600 hover:text-blue-900"
                          >
                            <Eye className="w-4 h-4" />
                          </button>
                          <button onClick={() => handleEdit(item)} className="text-green-600 hover:text-green-900">
                            <Edit className="w-4 h-4" />
                          </button>
                          <button onClick={() => deleteItem(item._id)} className="text-red-600 hover:text-red-900">
                            <Trash2 className="w-4 h-4" />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {!loading && !hasItems && (
          <div className="text-center py-12">
            <Package className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No items found</h3>
            <p className="text-gray-500 mb-4">{searchTerm ? 'Try adjusting your search terms' : 'Get started by adding your first item'}</p>
            {!searchTerm && (
              <button 
                onClick={() => {
                  resetForm();
                  setShowEditModal(true);
                }} 
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Item
              </button>
            )}
          </div>
        )}

        {!loading && hasItems && totalPages > 1 && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">Page {currentPage} of {totalPages}</div>
            <div className="flex space-x-2">
              <button
                onClick={() => fetchItems(currentPage - 1, searchTerm)}
                disabled={currentPage === 1}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <button
                onClick={() => fetchItems(currentPage + 1, searchTerm)}
                disabled={currentPage === totalPages}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Next
              </button>
            </div>
          </div>
        )}

        {showEditModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
              <h3 className="text-lg font-medium text-gray-900 mb-4">{selectedItem ? 'Edit Item' : 'Add New Item'}</h3>

              <form onSubmit={(e) => {
                e.preventDefault();
                handleSubmit(e);
              }} className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Item Name *</label>
                    <input
                      type="text"
                      value={formData.name}
                      onChange={(e) => handleFieldChange('name', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.name ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="Enter item name"
                      required
                    />
                    {errors.name && <p className="mt-1 text-sm text-red-600">{errors.name}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">SKU *</label>
                    <input
                      type="text"
                      value={formData.sku}
                      onChange={(e) => handleFieldChange('sku', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.sku ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="Enter SKU"
                      required
                    />
                    {errors.sku && <p className="mt-1 text-sm text-red-600">{errors.sku}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Price *</label>
                    <input
                      type="number"
                      step="0.01"
                      value={formData.price}
                      onChange={(e) => setFormData({ ...formData, price: e.target.value })}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="Enter price"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Brand *</label>
                    <select
                      value={formData.brand_id}
                      onChange={(e) => handleFieldChange('brand_id', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.brand_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      required
                    >
                      <option value="">Select brand</option>
                      {brands.map((b) => (
                        <option key={b._id} value={b._id}>{b.name || b.brandName}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Category *</label>
                    <select
                      value={formData.category_id}
                      onChange={(e) => handleFieldChange('category_id', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.category_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      required
                    >
                      <option value="">Select category</option>
                      {categories.map((c) => (
                        <option key={c._id} value={c._id}>{c.name}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Unit *</label>
                    <select
                      value={formData.unit_id}
                      onChange={(e) => handleFieldChange('unit_id', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.unit_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      required
                    >
                      <option value="">Select unit</option>
                      {units.map((u) => (
                        <option key={u._id} value={u._id}>{u.shortCode || u.name}</option>
                      ))}
                    </select>
                  </div>
                  <div className="md:col-span-2">
                    <label className="block text-sm font-medium text-gray-700 mb-1">Description</label>
                    <div className={`${errors.description ? 'border border-red-500 rounded-lg' : ''}`}>
                      <Suspense fallback={<EditorLoading />}>
                        <ReactQuill
                          theme="snow"
                          value={formData.description}
                          onChange={(value) => handleFieldChange('description', value)}
                          modules={modules}
                          formats={formats}
                          placeholder="Enter item description..."
                          className="bg-white"
                          style={{ minHeight: '150px' }}
                        />
                      </Suspense>
                    </div>
                    {errors.description && <p className="mt-1 text-sm text-red-600">{errors.description}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">HSN Code</label>
                    <input
                      type="text"
                      value={formData.hsn}
                      onChange={(e) => handleFieldChange('hsn', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.hsn ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="Enter HSN"
                    />
                    {errors.hsn && <p className="mt-1 text-sm text-red-600">{errors.hsn}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Minimum Quantity</label>
                    <input
                      type="number"
                      value={formData.minimum_qty}
                      onChange={(e) => handleFieldChange('minimum_qty', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.minimum_qty ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="0"
                      min={0}
                    />
                    {errors.minimum_qty && <p className="mt-1 text-sm text-red-600">{errors.minimum_qty}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Expiry Date</label>
                    <input
                      type="date"
                      value={formData.expire_date}
                      onChange={(e) => setFormData({ ...formData, expire_date: e.target.value })}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Barcode</label>
                    <input
                      type="text"
                      value={formData.barcode}
                      onChange={(e) => handleFieldChange('barcode', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.barcode ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="Enter barcode"
                    />
                    {errors.barcode && <p className="mt-1 text-sm text-red-600">{errors.barcode}</p>}
                  </div>
                  <div className="md:col-span-2">
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      Images {!selectedItem && <span className="text-red-500">*</span>}
                    </label>
                    
                    {/* Image upload area */}
                    <div 
                      className={`mt-1 flex justify-center px-6 pt-5 pb-6 border-2 border-dashed rounded-lg ${errors.images ? 'border-red-500' : 'border-gray-300'} hover:border-blue-500 transition-colors`}
                      onDragOver={(e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        e.currentTarget.classList.add('border-blue-500', 'bg-blue-50');
                      }}
                      onDragLeave={(e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        e.currentTarget.classList.remove('border-blue-500', 'bg-blue-50');
                      }}
                      onDrop={(e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        e.currentTarget.classList.remove('border-blue-500', 'bg-blue-50');
                        
                        const files = Array.from(e.dataTransfer.files).filter(file => 
                          file.type.match('image.*') && file.size <= 5 * 1024 * 1024
                        );
                        
                        if (files.length > 0) {
                          handleImageUpload({ target: { files } } as any);
                        } else {
                          toast.error('Please upload valid image files (max 5MB each)');
                        }
                      }}
                    >
                      <div className="space-y-1 text-center">
                        <svg
                          className="mx-auto h-12 w-12 text-gray-400"
                          stroke="currentColor"
                          fill="none"
                          viewBox="0 0 48 48"
                          aria-hidden="true"
                        >
                          <path
                            d="M28 8H12a4 4 0 00-4 4v20m32-12v8m0 0v8a4 4 0 01-4 4H12a4 4 0 01-4-4v-4m32-4l-3.172-3.172a4 4 0 00-5.656 0L28 28M8 32l9.172-9.172a4 4 0 015.656 0L28 28m0 0l4 4m4-24h8m-4-4v8m-12 4h.02"
                            strokeWidth={2}
                            strokeLinecap="round"
                            strokeLinejoin="round"
                          />
                        </svg>
                        <div className="flex text-sm text-gray-600">
                          <label
                            htmlFor="file-upload"
                            className="relative cursor-pointer bg-white rounded-md font-medium text-blue-600 hover:text-blue-500 focus-within:outline-none"
                          >
                            <span>Upload files</span>
                            <input
                              id="file-upload"
                              name="file-upload"
                              type="file"
                              className="sr-only"
                              multiple
                              accept="image/*"
                              onChange={handleImageUpload}
                            />
                          </label>
                          <p className="pl-1">or drag and drop</p>
                        </div>
                        <p className="text-xs text-gray-500">
                          PNG, JPG, GIF up to 5MB
                        </p>
                      </div>
                    </div>
                    
                    {/* Image previews */}
                    <div className="mt-4">
                      <p className="text-sm font-medium text-gray-700 mb-2">Selected Images</p>
                      <div className="grid grid-cols-2 sm:grid-cols-3 md:grid-cols-4 lg:grid-cols-5 gap-3">
                        {/* Show either imagePreviews or selectedItem.images, but not both */}
                        {imagePreviews.length > 0 ? (
                          imagePreviews.map((preview, index) => (
                            <div key={`preview-${index}`} className="relative group">
                              <img
                                src={preview}
                                alt={`Preview ${index + 1}`}
                                className="h-24 w-full object-cover rounded-lg border border-gray-200"
                              />
                              <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-40 rounded-lg transition-all duration-200 flex items-center justify-center space-x-2 opacity-0 group-hover:opacity-100">
                                <button
                                  type="button"
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    setPrimaryImage(index, false);
                                  }}
                                  className="p-1.5 bg-white bg-opacity-80 rounded-full text-gray-700 hover:bg-opacity-100"
                                  title="Set as primary"
                                >
                                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                    <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                  </svg>
                                </button>
                                <button
                                  type="button"
                                  onClick={() => removeImage(index)}
                                  className="p-1.5 bg-white bg-opacity-80 rounded-full text-red-600 hover:bg-opacity-100"
                                  title="Remove image"
                                >
                                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                    <path fillRule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clipRule="evenodd" />
                                  </svg>
                                </button>
                              </div>
                              {index === 0 && (
                                <div className="absolute top-1 right-1 bg-blue-500 text-white text-xs px-1.5 py-0.5 rounded-full">
                                  Primary
                                </div>
                              )}
                            </div>
                          ))
                        ) : selectedItem?.images?.length > 0 ? (
                          selectedItem.images.map((img, index) => (
                            <div key={`existing-${index}`} className="relative group">
                              <img
                                src={img.url}
                                alt={`Image ${index + 1}`}
                                className="h-24 w-full object-cover rounded-lg border border-gray-200"
                              />
                              <div className="absolute inset-0 bg-black bg-opacity-0 group-hover:bg-opacity-40 rounded-lg transition-all duration-200 flex items-center justify-center space-x-2 opacity-0 group-hover:opacity-100">
                                <button
                                  type="button"
                                  onClick={(e) => {
                                    e.stopPropagation();
                                    setPrimaryImage(index, true);
                                  }}
                                  className="p-1.5 bg-white bg-opacity-80 rounded-full text-gray-700 hover:bg-opacity-100"
                                  title="Set as primary"
                                >
                                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                    <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z" />
                                  </svg>
                                </button>
                                <button
                                  type="button"
                                  onClick={() => {
                                    // Remove image from the list
                                    const updatedImages = [...(selectedItem?.images || [])];
                                    updatedImages.splice(index, 1);

                                    // If we removed the primary image, make the first remaining image primary
                                    if (updatedImages.length > 0) {
                                      const wasPrimary = selectedItem?.images?.[index]?.is_primary;
                                      if (wasPrimary) {
                                        updatedImages[0].is_primary = true;
                                        // Remove primary flag from other images
                                        for (let i = 1; i < updatedImages.length; i++) {
                                          updatedImages[i].is_primary = false;
                                        }
                                      }
                                    }

                                    // Update selectedItem to reflect changes
                                    setSelectedItem(prev => prev ? {
                                      ...prev,
                                      images: updatedImages
                                    } : null);

                                    // Update image previews to match the new list
                                    setImagePreviews(updatedImages.map(image => image.url));
                                  }}
                                  className="p-1.5 bg-white bg-opacity-80 rounded-full text-red-600 hover:bg-opacity-100"
                                  title="Remove image"
                                >
                                  <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                    <path fillRule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clipRule="evenodd" />
                                  </svg>
                                </button>
                              </div>
                              {img.is_primary && (
                                <div className="absolute top-1 right-1 bg-blue-500 text-white text-xs px-1.5 py-0.5 rounded-full">
                                  Primary
                                </div>
                              )}
                            </div>
                          ))
                        ) : null}
                      </div>
                    </div>
                    
                    {errors.images && <p className="mt-1 text-sm text-red-600">{errors.images}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Price *</label>
                    <input
                      type="number"
                      step="0.01"
                      value={formData.price}
                      onChange={(e) => setFormData({ ...formData, price: e.target.value })}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                      placeholder="Enter price"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Tax</label>
                    <select
                      value={formData.tax_id}
                      onChange={(e) => handleFieldChange('tax_id', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.tax_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                    >
                      <option value="">Select Tax</option>
                      {taxes.map((t) => (
                        <option key={t._id} value={t._id}>{t.name}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Tax Type</label>
                    <select
                      value={formData.tax_type}
                      onChange={(e) => handleFieldChange('tax_type', e.target.value as 'inclusive' | 'exclusive')}
                      className={`w-full px-3 py-2 border ${errors.tax_type ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                    >
                      <option value="exclusive">Exclusive</option>
                      <option value="inclusive">Inclusive</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Sales Price</label>
                    <input
                      type="number"
                      step="0.01"
                      value={formData.sales_price}
                      onChange={(e) => handleFieldChange('sales_price', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.sales_price ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      placeholder="Enter sales price"
                    />
                    {errors.sales_price && <p className="mt-1 text-sm text-red-600">{errors.sales_price}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Discount</label>
                    <select
                      value={formData.discount_id}
                      onChange={(e) => handleFieldChange('discount_id', e.target.value)}
                      className={`w-full px-3 py-2 border ${errors.discount_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                    >
                      <option value="">Select Discount</option>
                      {discounts.map((d) => (
                        <option key={d._id} value={d._id}>{d.name} ({d.type === 'percentage' ? `${d.value}%` : `₹${d.value}`})</option>
                      ))}
                    </select>
                    {errors.discount_id && <p className="mt-1 text-sm text-red-600">{errors.discount_id}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Final Price</label>
                    <div className="relative">
                      <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <span className="text-gray-500 sm:text-sm">₹</span>
                      </div>
                      <input
                        type="text"
                        readOnly
                        value={finalPrice.toFixed(2)}
                        className="mt-1 block w-full pl-7 pr-12 py-2 border border-gray-300 bg-gray-50 rounded-lg text-gray-900 sm:text-sm"
                        placeholder="0.00"
                      />
                    </div>
                    {selectedDiscount && (
                      <p className="mt-1 text-xs text-green-600">
                        {selectedDiscount.type === 'percentage' 
                          ? `${selectedDiscount.value}% off applied` 
                          : `₹${selectedDiscount.value} off applied`}
                      </p>
                    )}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Store *</label>
                    <select
                      value={formData.store_id}
                      onChange={(e) => {
                        setFormData({ ...formData, store_id: e.target.value });
                        // Clear the error when user selects a store
                        if (e.target.value && errors.store_id) {
                          setErrors(prev => {
                            const newErrors = { ...prev };
                            delete newErrors.store_id;
                            return newErrors;
                          });
                        }
                      }}
                      className={`w-full px-3 py-2 border ${errors.store_id ? 'border-red-500' : 'border-gray-300'} rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500`}
                      required
                    >
                      <option value="">Select Store</option>
                      {stores.map((s) => (
                        <option key={s._id} value={s._id}>{s.name || s.storeName}</option>
                      ))}
                    </select>
                    {errors.store_id && <p className="mt-1 text-sm text-red-600">{errors.store_id}</p>}
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select
                      value={formData.status.toString()}
                      onChange={(e) => setFormData({ ...formData, status: e.target.value === 'true' })}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    >
                      <option value="true">Active</option>
                      <option value="false">Inactive</option>
                    </select>
                  </div>
                  
                  <div className="flex items-center space-x-2">
                    <input
                      type="checkbox"
                      id="requiresPrescription"
                      checked={formData.requiresPrescription}
                      onChange={(e) => handleFieldChange('requiresPrescription', e.target.checked)}
                      className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    />
                    <label htmlFor="requiresPrescription" className="block text-sm font-medium text-gray-700">
                      Prescription Required
                    </label>
                  </div>
                </div>
                <div className="flex justify-end space-x-3 mt-6">
                  <button
                    type="button"
                    onClick={() => {
                      setShowEditModal(false);
                      resetForm();
                    }}
                    className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                  >
                    Cancel
                  </button>
                  <button 
                    type="submit" 
                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                    disabled={loading}
                  >
                    {selectedItem ? 'Update Item' : 'Add Item'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        )}

        {showViewModal && selectedItem && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-2xl">
              <h3 className="text-lg font-medium text-gray-900 mb-4">Item Details</h3>
              <div className="space-y-4">
                <div className="flex items-center">
                  {selectedItem.images && selectedItem.images.length > 0 ? (
                    <div className="flex items-center">
                      <img 
                        src={selectedItem.images[0].url.startsWith('http') ? selectedItem.images[0].url : `${import.meta.env.VITE_API_URL || 'http://localhost:5000'}${selectedItem.images[0].url}`} 
                        alt={selectedItem.name} 
                        className="w-16 h-16 rounded-lg object-cover mr-4" 
                      />
                      {selectedItem.images.length > 1 && (
                        <span className="text-xs bg-blue-500 text-white rounded-full w-5 h-5 flex items-center justify-center">
                          +{selectedItem.images.length - 1}
                        </span>
                      )}
                    </div>
                  ) : selectedItem.image ? (
                    <img 
                      src={selectedItem.image.startsWith('http') ? selectedItem.image : `${import.meta.env.VITE_API_URL || 'http://localhost:5000'}${selectedItem.image}`} 
                      alt={selectedItem.name} 
                      className="w-16 h-16 rounded-lg object-cover mr-4" 
                    />
                  ) : (
                    <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center mr-4">
                      <Package className="w-8 h-8 text-gray-400" />
                    </div>
                  )}
                  <div>
                    <h4 className="text-xl font-semibold text-gray-900">{selectedItem.name}</h4>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(selectedItem.status)}`}>
                      {selectedItem.status ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700">SKU</label>
                    <p className="text-sm text-gray-900">{selectedItem.sku}</p>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Price</label>
                    <p className="text-sm text-gray-900">{selectedItem.price?.toFixed ? selectedItem.price.toFixed(2) : selectedItem.price}</p>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Brand</label>
                    <p className="text-sm text-gray-900">{getBrandName(selectedItem.brand_id)}</p>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Category</label>
                    <p className="text-sm text-gray-900">{typeof selectedItem.category_id === 'object' ? selectedItem.category_id?.name : ''}</p>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Unit</label>
                    <p className="text-sm text-gray-900">{typeof selectedItem.unit_id === 'object' ? (selectedItem.unit_id?.shortCode || selectedItem.unit_id?.name) : ''}</p>
                  </div>
                  {selectedItem.hsn && (
                    <div>
                      <label className="block text-sm font-medium text-gray-700">HSN</label>
                      <p className="text-sm text-gray-900">{selectedItem.hsn}</p>
                    </div>
                  )}
                  {selectedItem.barcode && (
                    <div>
                      <label className="block text-sm font-medium text-gray-700">Barcode</label>
                      <p className="text-sm text-gray-900">{selectedItem.barcode}</p>
                    </div>
                  )}
                </div>

                {selectedItem.description && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Description</label>
                    <p className="text-sm text-gray-900">{selectedItem.description}</p>
                  </div>
                )}

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Created</label>
                    <p className="text-sm text-gray-900">{formatDate(selectedItem.createdAt)}</p>
                  </div>
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Last Updated</label>
                    <p className="text-sm text-gray-900">{formatDate(selectedItem.updatedAt)}</p>
                  </div>
                </div>
              </div>

              <div className="flex justify-end space-x-3 mt-6">
                <button onClick={() => setShowViewModal(false)} className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200">
                  Close
                </button>
                <button
                  onClick={() => {
                    setShowViewModal(false);
                    if (selectedItem) handleEdit(selectedItem);
                  }}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Edit Item
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Items;


