import React, { useState, useEffect, useMemo } from 'react';
import { DragDropContext, Droppable, Draggable } from 'react-beautiful-dnd';
import Layout from '../../../../components/shared/Layout';
import Sidebar from '../../../../components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';
import { toast } from "react-hot-toast";
import api from "@/services/api";
import { useAuth } from '../../../../context/AuthContext';
import { Plus, GripVertical, FolderOpen, ArrowUp, ArrowDown, Eye, Edit, Trash2 } from 'lucide-react';

interface Category {
  _id: string;
  name: string;
  description?: string;
  code?: string;
  image?: string;
  subtitle?: string;
  status: boolean;
  position?: number;
  itemCount?: number;
  createdAt: string;
  updatedAt: string;
}

interface CategoryFormData {
  name: string;
  description: string;
  code: string;
  subtitle: string;
  status: boolean;
}

const Categories: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [showAddModal, setShowAddModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [showViewModal, setShowViewModal] = useState(false);
  const [categories, setCategories] = useState<Category[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedCategory, setSelectedCategory] = useState<Category | null>(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [selectedImage, setSelectedImage] = useState<File | null>(null);
  const [imagePreview, setImagePreview] = useState<string | null>(null);
  const [formData, setFormData] = useState<CategoryFormData>({
    name: '',
    description: '',
    code: '',
    subtitle: '',
    status: true
  });
  const [isDragging, setIsDragging] = useState(false);
  const [isMoving, setIsMoving] = useState(false);

  // Fetch categories from API
  const fetchCategories = async (page = 1, search = '') => {
    try {
      setLoading(true);
      const response = await api.get('/categories', {
        params: {
          page,
          limit: 10,
          search
        }
      });
      
      const { data, pagination } = response.data;
      setCategories(data);
      setTotalPages(pagination?.totalPages || 1);
      setCurrentPage(page);
    } catch (error: any) {
      console.error('Error fetching categories:', error);
      toast.error(error.response?.data?.msg || 'Failed to fetch categories');
    } finally {
      setLoading(false);
    }
  };

  // Create category
  const createCategory = async (categoryData: CategoryFormData, imageFile?: File | null) => {
    try {
      const formDataToSend = new FormData();
      formDataToSend.append('name', categoryData.name);
      formDataToSend.append('description', categoryData.description);
      formDataToSend.append('code', categoryData.code);
      formDataToSend.append('subtitle', categoryData.subtitle);
      formDataToSend.append('status', categoryData.status.toString());

      if (imageFile) {
        formDataToSend.append('image', imageFile);
      }

      const response = await api.post('/categories', formDataToSend);
      return response.data;
    } catch (error: any) {
      console.error('Error creating category:', error);
      throw error;
    }
  };

  // Update category
  const updateCategory = async (id: string, categoryData: CategoryFormData, imageFile?: File | null) => {
    try {
      const formDataToSend = new FormData();
      formDataToSend.append('name', categoryData.name);
      formDataToSend.append('description', categoryData.description);
      formDataToSend.append('code', categoryData.code);
      formDataToSend.append('subtitle', categoryData.subtitle);
      formDataToSend.append('status', categoryData.status.toString());

      if (imageFile) {
        formDataToSend.append('image', imageFile);
      }

      const response = await api.put(`/categories/${id}`, formDataToSend);
      return response.data;
    } catch (error: any) {
      console.error('Error updating category:', error);
      throw error;
    }
  };

  // Delete category
  const deleteCategory = async (id: string) => {
    if (!confirm('Are you sure you want to permanently delete this category? This action cannot be undone.')) {
      return;
    }

    try {
      await api.delete(`/categories/${id}`);
      toast.success('Category permanently deleted successfully');
      fetchCategories(currentPage, searchTerm);
    } catch (error: any) {
      console.error('Error deleting category:', error);
      const errorMessage = error.response?.data?.msg || error.message || 'Failed to delete category';
      toast.error(errorMessage);
    }
  };

  const getStatusBadge = (status: boolean) => {
    return status 
      ? 'bg-green-100 text-green-800' 
      : 'bg-red-100 text-red-800';
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Handle image selection
  const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      setSelectedImage(file);
      const reader = new FileReader();
      reader.onloadend = () => {
        setImagePreview(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  };

  // Handle drag end for reordering categories
  const handleDragEnd = async (result: any) => {
    if (!result.destination) {
      return;
    }

    const sourceIndex = result.source.index;
    const destinationIndex = result.destination.index;

    if (sourceIndex === destinationIndex) {
      return;
    }

    console.log('Drag ended:', { sourceIndex, destinationIndex, draggableId: result.draggableId });

    try {
      setIsDragging(true);

      // Create new categories array with reordered items
      const newCategories = Array.from(categories);
      const [reorderedCategory] = newCategories.splice(sourceIndex, 1);
      newCategories.splice(destinationIndex, 0, reorderedCategory);

      // Update positions based on new order
      const updatedCategories = newCategories.map((category, index) => ({
        ...category,
        position: index
      }));

      // Update the UI immediately for better UX
      setCategories(updatedCategories);

      // Update positions in backend
      const positionUpdates = updatedCategories.map((category, index) =>
        api.put(`/categories/${category._id}`, { position: index })
      );

      await Promise.all(positionUpdates);

      toast.success('Category positions updated successfully');
    } catch (error: any) {
      console.error('Error updating category positions:', error);
      // Revert to original order on error
      fetchCategories(currentPage, searchTerm);
      toast.error('Failed to update category positions');
    } finally {
      setIsDragging(false);
    }
  };

  // Move category up or down
  const moveCategory = async (categoryId: string, direction: 'up' | 'down') => {
    try {
      setIsMoving(true);

      // Get current category index
      const currentIndex = memoizedCategories.findIndex(cat => cat._id === categoryId);
      if (currentIndex === -1) {
        toast.error('Category not found');
        return;
      }

      // Calculate new position
      const newIndex = direction === 'up' ? currentIndex - 1 : currentIndex + 1;

      // Check if the new position is valid
      if (newIndex < 0 || newIndex >= memoizedCategories.length) {
        toast.error(`Cannot move category ${direction === 'up' ? 'up' : 'down'}`);
        return;
      }

      // Create a new array with updated positions
      const updatedCategories = [...memoizedCategories];

      // Get the categories to swap
      const currentCategory = { ...updatedCategories[currentIndex] };
      const targetCategory = { ...updatedCategories[newIndex] };

      // Ensure positions are numbers, default to array index if not set
      const currentPos = typeof currentCategory.position === 'number' ? currentCategory.position : currentIndex;
      const targetPos = typeof targetCategory.position === 'number' ? targetCategory.position : newIndex;

      // Update positions
      currentCategory.position = targetPos;
      targetCategory.position = currentPos;

      // Update the categories in the array
      updatedCategories[currentIndex] = targetCategory;
      updatedCategories[newIndex] = currentCategory;

      // Sort the array by position to ensure correct order
      updatedCategories.sort((a, b) => (a.position || 0) - (b.position || 0));

      // Update the backend with new positions
      try {
        // Update both categories in parallel
        await Promise.all([
          api.put(`/categories/${currentCategory._id}`, {
            position: Number(currentCategory.position)
          }),
          api.put(`/categories/${targetCategory._id}`, {
            position: Number(targetCategory.position)
          })
        ]);

        // Update the UI with the new order
        setCategories(updatedCategories);
        toast.success(`Category moved ${direction}`);
      } catch (error) {
        // If there's an error, revert the UI
        fetchCategories(currentPage, searchTerm);
        throw error;
      }
    } catch (error: any) {
      console.error(`Error moving category ${direction}:`, error);
      const errorMessage = error.response?.data?.msg || error.message || `Failed to move category ${direction}`;
      toast.error(errorMessage);
    } finally {
      setIsMoving(false);
    }
  };

  // Reset form
  const resetForm = () => {
    setFormData({
      name: '',
      description: '',
      code: '',
      subtitle: '',
      status: true
    });
    setSelectedCategory(null);
    setSelectedImage(null);
    setImagePreview(null);
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name.trim()) {
      toast.error('Category name is required');
      return;
    }

    try {
      if (selectedCategory) {
        await updateCategory(selectedCategory._id, formData, selectedImage);
        toast.success('Category updated successfully');
        setShowEditModal(false);
      } else {
        await createCategory(formData, selectedImage);
        toast.success('Category created successfully');
        setShowAddModal(false);
      }
      resetForm();
      fetchCategories(currentPage, searchTerm);
    } catch (error: any) {
      const errorMessage = error.response?.data?.msg || error.message || 'An error occurred';
      toast.error(errorMessage);
    }
  };

  // Handle edit
  const handleEdit = (category: Category) => {
    setSelectedCategory(category);
    setFormData({
      name: category.name,
      description: category.description || '',
      code: category.code || '',
      subtitle: category.subtitle || '',
      status: category.status
    });
    if (category.image) {
      setImagePreview(category.image);
    } else {
      setImagePreview(null);
    }
    setSelectedImage(null);
    setShowEditModal(true);
  };

  // Handle search
  const handleSearch = (value: string) => {
    setSearchTerm(value);
    setCurrentPage(1);
    fetchCategories(1, value);
  };

  // Load categories on component mount
  useEffect(() => {
    fetchCategories();
  }, []);

  // Memoize categories to ensure stable reference for drag and drop
  const memoizedCategories = useMemo(() => categories, [categories]);

  return (
    <Layout title="Categories Management" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
          <div className="relative">
            <svg className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
            </svg>
            <input
              type="text"
              placeholder="Search categories..."
              className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              value={searchTerm}
              onChange={(e) => handleSearch(e.target.value)}
            />
          </div>
          
          <button 
            onClick={() => setShowAddModal(true)}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Category
          </button>
        </div>

        {/* Loading State */}
        {loading && (
          <div className="flex justify-center items-center py-12">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        )}

        {/* Categories Table */}
        {!loading && (
          <div className="bg-white rounded-lg shadow-sm border overflow-hidden">
            <DragDropContext onDragEnd={handleDragEnd}>
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Drag
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Category
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Description
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Code
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Products
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Status
                      </th>
                      <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Position
                      </th>
                      <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Actions
                      </th>
                    </tr>
                  </thead>
                  <Droppable droppableId="categories">
                    {(provided, snapshot) => (
                      <tbody
                        {...provided.droppableProps}
                        ref={provided.innerRef}
                        className={`bg-white divide-y divide-gray-200 ${snapshot.isDraggingOver ? 'bg-blue-50' : ''}`}
                      >
                        {memoizedCategories.map((category, index) => (
                          <Draggable key={`category-${index}`} draggableId={`category-${index}`} index={index}>
                            {(provided, snapshot) => (
                              <tr
                                ref={provided.innerRef}
                                {...provided.draggableProps}
                                className={`hover:bg-gray-50 ${snapshot.isDragging ? 'shadow-lg bg-white' : ''}`}
                              >
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <div
                                    {...provided.dragHandleProps}
                                    className="flex items-center justify-center text-gray-400 hover:text-gray-600 cursor-grab active:cursor-grabbing p-2 rounded hover:bg-gray-100"
                                    onDragStart={(e) => e.preventDefault()}
                                  >
                                    <GripVertical className="w-5 h-5" />
                                  </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <div className="flex items-center">
                                    {category.image ? (
                                      <img
                                        src={category.image}
                                        alt={category.name}
                                        className="w-5 h-5 rounded mr-3 object-cover"
                                      />
                                    ) : (
                                      <FolderOpen className="w-5 h-5 text-blue-500 mr-3" />
                                    )}
                                    <div>
                                      <div className="text-sm font-medium text-gray-900">{category.name}</div>
                                      <div className="text-xs text-gray-500">Created {formatDate(category.createdAt)}</div>
                                    </div>
                                  </div>
                                </td>
                                <td className="px-6 py-4">
                                  <div className="text-sm text-gray-900">
                                    {category.description || 'No description available'}
                                  </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <div className="text-sm text-gray-900">
                                    {category.code || 'N/A'}
                                  </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <div className="text-sm font-medium text-gray-900">{category.itemCount || 0}</div>
                                  <div className="text-xs text-gray-500">products</div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(category.status)}`}>
                                    {category.status ? 'active' : 'inactive'}
                                  </span>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                  <div className="flex items-center space-x-1">
                                    <button
                                      onClick={() => moveCategory(category._id, 'up')}
                                      disabled={isMoving || index === 0}
                                      className="text-gray-400 hover:text-gray-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                      title="Move up"
                                    >
                                      <ArrowUp className="w-4 h-4" />
                                    </button>
                                    <span className="text-sm text-gray-900 w-4 text-center">
                                      {index + 1}
                                    </span>
                                    <button
                                      onClick={() => moveCategory(category._id, 'down')}
                                      disabled={isMoving || index === memoizedCategories.length - 1}
                                      className="text-gray-400 hover:text-gray-600 disabled:opacity-30 disabled:cursor-not-allowed"
                                      title="Move down"
                                    >
                                      <ArrowDown className="w-4 h-4" />
                                    </button>
                                  </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                  <div className="flex items-center justify-end space-x-2">
                                    <button
                                      onClick={() => {
                                        setSelectedCategory(category);
                                        setShowViewModal(true);
                                      }}
                                      className="text-blue-600 hover:text-blue-900"
                                    >
                                      <Eye className="w-4 h-4" />
                                    </button>
                                    <button
                                      onClick={() => handleEdit(category)}
                                      className="text-green-600 hover:text-green-900"
                                    >
                                      <Edit className="w-4 h-4" />
                                    </button>
                                    <button
                                      onClick={() => deleteCategory(category._id)}
                                      className="text-red-600 hover:text-red-900"
                                    >
                                      <Trash2 className="w-4 h-4" />
                                    </button>
                                  </div>
                                </td>
                              </tr>
                            )}
                          </Draggable>
                        ))}
                        {provided.placeholder}
                      </tbody>
                    )}
                  </Droppable>
                </table>
              </div>
            </DragDropContext>
          </div>
        )}

        {/* Empty State */}
        {!loading && memoizedCategories.length === 0 && (
          <div className="text-center py-12">
            <FolderOpen className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No categories found</h3>
            <p className="text-gray-500 mb-4">
              {searchTerm ? 'Try adjusting your search terms' : 'Get started by adding your first category'}
            </p>
            {!searchTerm && (
              <button 
                onClick={() => setShowAddModal(true)}
                className="inline-flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
              >
                <Plus className="w-4 h-4 mr-2" />
                Add Category
              </button>
            )}
          </div>
        )}

        {/* Pagination */}
        {!loading && memoizedCategories.length > 0 && totalPages > 1 && (
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Page {currentPage} of {totalPages}
            </div>
            <div className="flex space-x-2">
              <button
                onClick={() => fetchCategories(currentPage - 1, searchTerm)}
                disabled={currentPage === 1}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Previous
              </button>
              <button
                onClick={() => fetchCategories(currentPage + 1, searchTerm)}
                disabled={currentPage === totalPages}
                className="px-3 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                Next
              </button>
            </div>
          </div>
        )}

        {/* Add/Edit Category Modal */}
        {(showAddModal || showEditModal) && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md max-h-[90vh] overflow-y-auto">
              <h3 className="text-lg font-medium text-gray-900 mb-4">
                {selectedCategory ? 'Edit Category' : 'Add New Category'}
              </h3>
              
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Category Name *
                  </label>
                  <input
                    type="text"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter category name"
                    required
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Description
                  </label>
                  <textarea
                    rows={3}
                    value={formData.description}
                    onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter category description"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Category Code
                  </label>
                  <input
                    type="text"
                    value={formData.code}
                    onChange={(e) => setFormData({ ...formData, code: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter category code (e.g., CAT001)"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Subtitle
                  </label>
                  <input
                    type="text"
                    value={formData.subtitle}
                    onChange={(e) => setFormData({ ...formData, subtitle: e.target.value })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                    placeholder="Enter category subtitle"
                  />
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Category Image
                  </label>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={handleImageChange}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  />
                  {imagePreview && (
                    <div className="mt-2">
                      <img
                        src={imagePreview}
                        alt="Preview"
                        className="w-20 h-20 rounded-lg object-cover border"
                      />
                    </div>
                  )}
                </div>
                
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">
                    Status
                  </label>
                  <select 
                    value={formData.status.toString()}
                    onChange={(e) => setFormData({ ...formData, status: e.target.value === 'true' })}
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                  >
                    <option value="true">Active</option>
                    <option value="false">Inactive</option>
                  </select>
                </div>
              </form>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => {
                    setShowAddModal(false);
                    setShowEditModal(false);
                    resetForm();
                  }}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Cancel
                </button>
                <button 
                  onClick={handleSubmit}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  {selectedCategory ? 'Update Category' : 'Add Category'}
                </button>
              </div>
            </div>
          </div>
        )}

        {/* View Category Modal */}
        {showViewModal && selectedCategory && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-md">
              <h3 className="text-lg font-medium text-gray-900 mb-4">Category Details</h3>
              
              <div className="space-y-4">
                <div className="flex items-center">
                  {selectedCategory.image ? (
                    <img
                      src={selectedCategory.image}
                      alt={selectedCategory.name}
                      className="w-16 h-16 rounded-lg object-cover mr-4"
                    />
                  ) : (
                    <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center mr-4">
                      <FolderOpen className="w-8 h-8 text-gray-400" />
                    </div>
                  )}
                  <div>
                    <h4 className="text-xl font-semibold text-gray-900">{selectedCategory.name}</h4>
                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(selectedCategory.status)}`}>
                      {selectedCategory.status ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                </div>

                {selectedCategory.code && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Code</label>
                    <p className="text-sm text-gray-900">{selectedCategory.code}</p>
                  </div>
                )}

                {selectedCategory.description && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Description</label>
                    <p className="text-sm text-gray-900">{selectedCategory.description}</p>
                  </div>
                )}

                {selectedCategory.subtitle && (
                  <div>
                    <label className="block text-sm font-medium text-gray-700">Subtitle</label>
                    <p className="text-sm text-gray-900">{selectedCategory.subtitle}</p>
                  </div>
                )}

                <div>
                  <label className="block text-sm font-medium text-gray-700">Products</label>
                  <p className="text-sm text-gray-900">{selectedCategory.itemCount || 0} products</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Created</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedCategory.createdAt)}</p>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700">Last Updated</label>
                  <p className="text-sm text-gray-900">{formatDate(selectedCategory.updatedAt)}</p>
                </div>
              </div>
              
              <div className="flex justify-end space-x-3 mt-6">
                <button
                  onClick={() => setShowViewModal(false)}
                  className="px-4 py-2 text-gray-700 bg-gray-100 rounded-lg hover:bg-gray-200"
                >
                  Close
                </button>
                <button 
                  onClick={() => {
                    setShowViewModal(false);
                    handleEdit(selectedCategory);
                  }}
                  className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
                >
                  Edit Category
                </button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Categories;