import React, { useState, useRef } from 'react';
import { Upload, FileText, Download, X, Check, AlertCircle } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';

interface UploadError {
  row: number;
  sku: string;
  error: string;
  rawData?: any;
}

interface UploadResult {
  success: boolean;
  insertedCount?: number;
  errorCount: number;
  errors?: UploadError[];
  totalCount?: number;
  msg?: string;
}

interface BulkUploadProps {
  onBackToList: () => void;
}

const BulkUpload: React.FC<BulkUploadProps> = ({ onBackToList }) => {
  // Use the onBackToList prop or provide a default empty function
  const handleBackToList = onBackToList || (() => window.history.back());
  const [file, setFile] = useState<File | null>(null);
  const [isUploading, setIsUploading] = useState(false);
  const [progress, setProgress] = useState(0);
  const [uploadResult, setUploadResult] = useState<UploadResult | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files.length > 0) {
      setFile(e.target.files[0]);
      setUploadResult(null);
    }
  };

  const handleUpload = async () => {
    if (!file) {
      toast.error('Please select a file to upload');
      return;
    }

    // Log file info for debugging
    console.log('File to upload:', {
      name: file.name,
      type: file.type,
      size: file.size,
      lastModified: file.lastModified,
      isFile: file instanceof File
    });

    const formData = new FormData();
    // Use the same field name as expected by multer ('file')
    formData.append('file', file, file.name);

    // Log FormData contents (for debugging)
    for (let [key, value] of (formData as any).entries()) {
      console.log(`FormData - ${key}:`, value);
    }

    setIsUploading(true);
    setProgress(30);
    setUploadResult(null);

    try {
      console.log('Sending file upload request...');
      const response = await api.post('/items/bulk-upload', formData, {
        headers: {
          // Let the browser set the Content-Type with the correct boundary
          'Content-Type': 'multipart/form-data',
        },
        // Add timeout and withCredentials if needed
        timeout: 30000, // 30 seconds timeout
        withCredentials: true,
      });
      
      console.log('Upload response:', response);

      setProgress(100);
      
      // Handle successful upload with results
      const result = {
        success: true,
        insertedCount: response.data.insertedCount || 0,
        errorCount: response.data.errorCount || 0,
        errors: response.data.errors || [],
      };
      
      setUploadResult(result);

      if (result.errorCount > 0) {
        toast.error(`Processed ${result.insertedCount || 0} items. ${result.errorCount} items had errors.`);
      } else {
        toast.success(`Successfully processed ${result.insertedCount || 0} items.`);
      }
    } catch (error: any) {
      console.error('Error uploading file:', error);
      
      // Handle validation errors from the backend
      const errorResponse = error.response?.data;
      const errorMessage = errorResponse?.msg || 'Error uploading file';
      const errors = Array.isArray(errorResponse?.errors) 
        ? errorResponse.errors 
        : [{ row: 0, sku: 'N/A', error: errorMessage }];
      
      const result = {
        success: false,
        errorCount: errors.length,
        errors: errors,
      };
      
      setUploadResult(result);
      
      toast.error(errorMessage);
    } finally {
      setIsUploading(false);
    }
  };

  const handleDownloadTemplate = async () => {
    try {
      const response = await api.get('/items/bulk-upload/template', {
        responseType: 'blob',
      });

      const url = window.URL.createObjectURL(new Blob([response.data]));
      const link = document.createElement('a');
      link.href = url;
      link.setAttribute('download', 'bulk-upload-template.xlsx');
      document.body.appendChild(link);
      link.click();
      link.remove();
    } catch (error) {
      console.error('Error downloading template:', error);
      toast.error('Failed to download template');
    }
  };

  const handleRemoveFile = () => {
    setFile(null);
    setUploadResult(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold tracking-tight">Bulk Upload Products</h2>
          <p className="text-muted-foreground">
            Upload a CSV or Excel file to add multiple products at once
          </p>
        </div>
        <Button onClick={handleDownloadTemplate} variant="outline">
          <Download className="mr-2 h-4 w-4" />
          Download Template
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Upload File</CardTitle>
          <CardDescription>
            Upload a CSV or Excel file with product data. Make sure to use the provided template.
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-6">
            <div
              className={`flex flex-col items-center justify-center border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
                file ? 'border-primary/50' : 'border-border hover:border-primary/50'
              }`}
              onDragOver={(e) => {
                e.preventDefault();
                e.stopPropagation();
              }}
              onDrop={(e) => {
                e.preventDefault();
                e.stopPropagation();
                if (e.dataTransfer.files && e.dataTransfer.files.length > 0) {
                  setFile(e.dataTransfer.files[0]);
                  setUploadResult(null);
                }
              }}
            >
              <input
                type="file"
                ref={fileInputRef}
                onChange={handleFileChange}
                accept=".xlsx,.xls,.csv"
                className="hidden"
                disabled={isUploading}
              />
              
              {!file ? (
                <div className="flex flex-col items-center space-y-2">
                  <Upload className="h-10 w-10 text-muted-foreground" />
                  <div className="text-sm text-muted-foreground">
                    <span className="font-medium text-primary">Click to upload</span> or drag and drop
                  </div>
                  <p className="text-xs text-muted-foreground">CSV or Excel files only (max. 10MB)</p>
                </div>
              ) : (
                <div className="flex flex-col items-center space-y-2 w-full">
                  <FileText className="h-10 w-10 text-primary" />
                  <div className="flex items-center space-x-2">
                    <span className="font-medium truncate max-w-xs">{file.name}</span>
                    <button
                      type="button"
                      onClick={handleRemoveFile}
                      className="text-muted-foreground hover:text-destructive"
                      disabled={isUploading}
                    >
                      <X className="h-4 w-4" />
                    </button>
                  </div>
                  <span className="text-xs text-muted-foreground">
                    {(file.size / 1024 / 1024).toFixed(2)} MB
                  </span>
                </div>
              )}
            </div>

            {isUploading && (
              <div className="space-y-2">
                <div className="flex justify-between text-sm text-muted-foreground">
                  <span>Uploading...</span>
                  <span>{progress}%</span>
                </div>
                <Progress value={progress} className="h-2" />
              </div>
            )}

            <div className="flex justify-end">
              <Button
                onClick={() => fileInputRef.current?.click()}
                variant="outline"
                className="mr-2"
                disabled={isUploading}
              >
                {file ? 'Change File' : 'Select File'}
              </Button>
              <Button
                onClick={handleUpload}
                disabled={!file || isUploading}
                className="min-w-[120px]"
              >
                {isUploading ? 'Uploading...' : 'Upload'}
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {uploadResult && (
        <Card className={uploadResult.success ? 'border-green-200 dark:border-green-900' : 'border-destructive/50'}>
          <CardHeader className="pb-3">
            <div className="flex items-center space-x-2">
              {uploadResult.success ? (
                <>
                  <Check className="h-5 w-5 text-green-600" />
                  <CardTitle className="text-green-600">
                    {uploadResult.errorCount && uploadResult.errorCount > 0 ? 'Upload Completed with Issues' : 'Upload Complete'}
                  </CardTitle>
                </>
              ) : (
                <>
                  <AlertCircle className="h-5 w-5 text-destructive" />
                  <CardTitle className="text-destructive">
                    {uploadResult.errorCount && uploadResult.errorCount > 1 ? 'Validation Errors' : 'Upload Failed'}
                  </CardTitle>
                </>
              )}
            </div>
            <CardDescription>
              {uploadResult.success ? (
                <>
                  Successfully processed {uploadResult.insertedCount || 0} items.
                  {uploadResult.errorCount !== undefined && uploadResult.errorCount > 0 && (
                    <span className="text-yellow-600 dark:text-yellow-400">
                      {' '}{uploadResult.errorCount} item{uploadResult.errorCount !== 1 ? 's had' : ' had'} errors.
                    </span>
                  )}
                </>
              ) : (
                <>
                  {uploadResult.errorCount !== undefined && uploadResult.errorCount > 1 
                    ? `Found ${uploadResult.errorCount} validation errors in your file.`
                    : 'There was an error processing your file.'
                  }
                </>
              )}
            </CardDescription>
          </CardHeader>
          
          {uploadResult.errors && uploadResult.errors.length > 0 && (
            <CardContent>
              <div className="rounded-md border border-border overflow-hidden">
                <div className="bg-muted/50 p-2 flex text-sm font-medium">
                  <div className="w-16">Row</div>
                  <div className="w-32">SKU</div>
                  <div>Error</div>
                </div>
                <div className="max-h-96 overflow-y-auto">
                  {uploadResult.errors.map((error, index) => (
                    <div 
                      key={index} 
                      className="border-t border-border hover:bg-muted/10"
                    >
                      <div className="p-2 flex text-sm items-start">
                        <div className="w-16 font-medium">{error.row}</div>
                        <div className="w-32 font-mono text-sm break-words">
                          {error.sku || 'N/A'}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div className="text-destructive font-medium">{error.error}</div>
                          {error.rawData && (
                            <details className="mt-1 text-xs text-muted-foreground">
                              <summary className="cursor-pointer hover:underline">View row data</summary>
                              <pre className="mt-1 p-2 bg-muted/30 rounded overflow-x-auto">
                                {typeof error.rawData === 'string' 
                                  ? error.rawData 
                                  : JSON.stringify(error.rawData, null, 2)}
                              </pre>
                            </details>
                          )}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </CardContent>
          )}
        </Card>
      )}

      <Card>
        <CardHeader>
          <CardTitle>Instructions</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div>
            <div className="flex justify-between items-center mb-2">
              <h3 className="font-medium">Preparing Your File</h3>
              <Button 
                variant="outline" 
                size="sm" 
                onClick={handleBackToList}
                className="text-sm"
              >
                Back to List
              </Button>
            </div>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>Download and use the provided template to ensure proper formatting</li>
              <li>Only fill in the columns you need to update</li>
              <li>Required fields are marked with an asterisk (*) in the template</li>
              <li>Make sure all IDs (brand_id, category_id, etc.) exist in the system</li>
            </ul>
          </div>
          
          <div>
            <h3 className="font-medium mb-2">File Requirements</h3>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>File size: Maximum 10MB</li>
              <li>File formats: .xlsx, .xls, .csv</li>
              <li>Maximum rows: 10,000 per file</li>
              <li>Date format: YYYY-MM-DD</li>
            </ul>
          </div>
          
          <div>
            <h3 className="font-medium mb-2">Troubleshooting</h3>
            <ul className="list-disc pl-5 space-y-1 text-sm text-muted-foreground">
              <li>Check that all required fields are filled in</li>
              <li>Verify that all IDs reference existing records</li>
              <li>Ensure numeric fields contain only numbers</li>
              <li>Check for extra spaces or special characters</li>
            </ul>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

// Button component to match the design system
interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'default' | 'outline' | 'ghost' | 'link';
  size?: 'sm' | 'default' | 'lg';
  children: React.ReactNode;
  className?: string;
}

const Button: React.FC<ButtonProps> = ({
  variant = 'default',
  size = 'default',
  className = '',
  onClick,
  children,
  disabled = false,
  ...props
}) => {
  const baseStyles = 'inline-flex items-center justify-center rounded-md font-medium transition-colors focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-offset-2 disabled:opacity-50 disabled:pointer-events-none';
  
  const variants = {
    default: 'bg-blue-600 text-white hover:bg-blue-700',
    outline: 'border border-gray-300 bg-white text-gray-700 hover:bg-gray-50',
    ghost: 'hover:bg-gray-100',
    link: 'text-blue-600 hover:underline',
  };

  const sizes = {
    sm: 'h-8 px-3 text-sm',
    default: 'h-10 py-2 px-4',
    lg: 'h-11 px-8',
  };

  return (
    <button
      className={`${baseStyles} ${variants[variant]} ${sizes[size]} ${className}`}
      onClick={onClick}
      disabled={disabled}
      {...props}
    >
      {children}
    </button>
  );
};

// Card components to match the design system
const Card: React.FC<{ className?: string; children: React.ReactNode }> = ({ className = '', children }) => (
  <div className={`bg-white rounded-lg shadow-sm border ${className}`}>
    {children}
  </div>
);

const CardHeader: React.FC<{ children: React.ReactNode }> = ({ children }) => (
  <div className="px-6 py-4 border-b">
    {children}
  </div>
);

const CardTitle: React.FC<{ children: React.ReactNode }> = ({ children }) => (
  <h3 className="text-lg font-medium">{children}</h3>
);

const CardDescription: React.FC<{ children: React.ReactNode }> = ({ children }) => (
  <p className="text-sm text-gray-500">{children}</p>
);

const CardContent: React.FC<{ className?: string; children: React.ReactNode }> = ({ className = '', children }) => (
  <div className={`p-6 ${className}`}>
    {children}
  </div>
);

// Progress component
const Progress: React.FC<{ value: number; className?: string }> = ({ value, className = '' }) => (
  <div className={`w-full bg-gray-200 rounded-full h-2.5 ${className}`}>
    <div 
      className="bg-blue-600 h-2.5 rounded-full transition-all duration-300" 
      style={{ width: `${value}%` }}
    />
  </div>
);

// Tabs components
const Tabs: React.FC<{ 
  value: string; 
  onValueChange: (value: string) => void;
  className?: string;
  children: React.ReactNode;
}> = ({ value, onValueChange, className = '', children }) => {
  return (
    <div className={className}>
      {React.Children.map(children, (child) => {
        if (React.isValidElement(child) && child.type === TabsList) {
          return React.cloneElement(child as React.ReactElement, { value, onValueChange });
        }
        return child;
      })}
    </div>
  );
};

const TabsList: React.FC<{ 
  value: string; 
  onValueChange: (value: string) => void;
  className?: string;
  children: React.ReactNode;
}> = ({ value, onValueChange, className = '', children }) => {
  return (
    <div className={`flex border-b ${className}`}>
      {React.Children.map(children, (child) => {
        if (React.isValidElement(child) && child.type === TabsTrigger) {
          return React.cloneElement(child as React.ReactElement, { 
            isActive: value === child.props.value,
            onClick: () => onValueChange(child.props.value)
          });
        }
        return child;
      })}
    </div>
  );
};

const TabsTrigger: React.FC<{ 
  value: string;
  isActive?: boolean;
  onClick?: () => void;
  children: React.ReactNode;
  className?: string;
}> = ({ isActive = false, onClick, children, className = '' }) => (
  <button
    className={`px-4 py-2 text-sm font-medium ${isActive 
      ? 'text-blue-600 border-b-2 border-blue-600' 
      : 'text-gray-500 hover:text-gray-700'} ${className}`}
    onClick={onClick}
  >
    {children}
  </button>
);

const TabsContent: React.FC<{ 
  value: string; 
  children: React.ReactNode;
  className?: string;
}> = ({ value, children, className = '' }) => {
  return (
    <div className={`mt-4 ${className}`}>
      {React.Children.map(children, (child) => {
        if (React.isValidElement(child) && child.props['data-value'] === value) {
          return child;
        }
        return null;
      })}
    </div>
  );
};

export default BulkUpload;
