import React, { useState, useEffect } from 'react';
import { useSearchParams } from 'react-router-dom';
import { FileUp, RefreshCw, Search } from 'lucide-react';
import Layout from '../../../../components/shared/Layout';
import Sidebar from '../../../../components/shared/Sidebar';
import { adminSidebarItems } from '../../AdminLayout';
import BulkUpload from './BulkUpload';
import BulkUpdate from './BulkUpdate';

// Local tab components
const Tabs = ({ value, onValueChange, className, children }: { 
  value: string; 
  onValueChange: (value: string) => void; 
  className?: string; 
  children: React.ReactNode;
}) => {
  return (
    <div className={className}>
      {React.Children.map(children, (child) => {
        if (React.isValidElement(child) && child.type === TabsList) {
          return React.cloneElement(child as React.ReactElement, { 
            activeTab: value,
            onTabChange: onValueChange 
          });
        }
        return child;
      })}
    </div>
  );
};

const TabsList = ({ 
  className, 
  children, 
  activeTab,
  onTabChange 
}: { 
  className?: string; 
  children: React.ReactNode;
  activeTab?: string;
  onTabChange?: (value: string) => void;
}) => (
  <div className={`inline-flex h-10 items-center justify-center rounded-md bg-muted p-1 text-muted-foreground ${className}`}>
    {React.Children.map(children, (child) => {
      if (React.isValidElement(child) && child.type === TabsTrigger) {
        return React.cloneElement(child as React.ReactElement, { 
          isActive: child.props.value === activeTab,
          onClick: () => onTabChange?.(child.props.value)
        });
      }
      return child;
    })}
  </div>
);

const TabsTrigger = ({ 
  value, 
  children, 
  className = '',
  isActive = false,
  onClick
}: { 
  value: string; 
  children: React.ReactNode; 
  className?: string;
  isActive?: boolean;
  onClick?: () => void;
}) => {
  return (
    <button
      className={`inline-flex items-center justify-center whitespace-nowrap rounded-sm px-3 py-1.5 text-sm font-medium ring-offset-background transition-all focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 disabled:pointer-events-none disabled:opacity-50 data-[state=active]:bg-background data-[state=active]:text-foreground data-[state=active]:shadow-sm ${className}`}
      onClick={(e) => {
        e.preventDefault();
        const url = new URL(window.location.href);
        url.searchParams.set('tab', value);
        window.history.pushState({}, '', url);
        onClick?.();
      }}
      data-state={isActive ? 'active' : 'inactive'}
    >
      {children}
    </button>
  );
};

const TabsContent = ({ 
  value, 
  children, 
  className = '',
  activeTab
}: { 
  value: string; 
  children: React.ReactNode; 
  className?: string;
  activeTab: string;
}) => {
  return value === activeTab ? (
    <div className={`mt-4 ${className}`}>{children}</div>
  ) : null;
};

type TabValue = 'upload' | 'update';

const BulkOperations: React.FC = () => {
  const [searchParams, setSearchParams] = useSearchParams();
  const [activeTab, setActiveTab] = useState<TabValue>('upload');
  const [searchTerm, setSearchTerm] = useState('');

  // Update active tab when URL changes
  useEffect(() => {
    const tabParam = searchParams.get('tab');
    if (tabParam === 'upload' || tabParam === 'update') {
      setActiveTab(tabParam);
    } else {
      // Set default tab if no valid tab in URL
      setSearchParams({ tab: 'upload' }, { replace: true });
    }
  }, [searchParams, setSearchParams]);

  // Handle tab change
  const handleTabChange = (value: TabValue) => {
    setActiveTab(value);
    setSearchParams({ tab: value }, { replace: true });
  };

  return (
    <Layout title="Bulk Operations" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div>
            <h1 className="text-2xl font-bold tracking-tight">Bulk Operations</h1>
            <p className="text-muted-foreground">
              Manage your products in bulk using the tools below
            </p>
          </div>
          
          <div className="flex flex-col sm:flex-row gap-3">
            <div className="relative">
              <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
              <input
                type="search"
                placeholder="Search operations..."
                className="pl-8 w-full rounded-md border border-input bg-background px-3 py-2 text-sm ring-offset-background placeholder:text-muted-foreground focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 md:w-[200px] lg:w-[300px]"
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
          </div>
        </div>

        <div className="rounded-md border bg-card text-card-foreground shadow">
          <Tabs 
            value={activeTab} 
            onValueChange={handleTabChange}
            className="w-full p-4"
          >
            <TabsList activeTab={activeTab} onTabChange={handleTabChange}>
              <TabsTrigger value="upload">
                <FileUp className="mr-2 h-4 w-4" />
                Bulk Upload
              </TabsTrigger>
              <TabsTrigger value="update">
                <RefreshCw className="mr-2 h-4 w-4" />
                Bulk Update
              </TabsTrigger>
            </TabsList>

            <div className="p-4 pt-6">
              <TabsContent value="upload" className="m-0" activeTab={activeTab}>
                <BulkUpload />
              </TabsContent>
              
              <TabsContent value="update" className="m-0" activeTab={activeTab}>
                <BulkUpdate />
              </TabsContent>
            </div>
          </Tabs>
        </div>
      </div>
    </Layout>
  );
};

export default BulkOperations;
