import React, { useState, useEffect } from 'react';
import { 
  Plus, 
  Edit, 
  Trash2, 
  ArrowUp, 
  ArrowDown 
} from 'lucide-react';
import { toast } from 'react-hot-toast';
import { homepageSectionService } from '@/services/api';
import { itemService } from '@/services/api';
import { categoryService } from '@/services/api';

// Define the structure for a homepage section
interface Product {
  _id: string;
  name: string;
  price: number;
  images?: string[];
  status: number;
}

interface HomepageSection {
  _id?: string;
  title: string;
  description?: string;
  categoryIds: string[];
  productType: 'all_products' | 'new_added_products' | 'products_on_sale' | 'most_selling_products' | 'custom_products';
  position: 'top' | 'below_slider' | 'below_category' | 'below_shop_by_seller' | 'below_shop_by_country' | 'below_shop_by_brands';
  backgroundColorLight: string;
  backgroundColorDark: string;
  appSectionStyle: 'grid' | 'horizontal_scroll' | 'vertical_scroll' | 'carousel';
  webSectionStyle: 'grid' | 'horizontal_scroll' | 'vertical_scroll' | 'carousel';
  products: string[] | Product[];
  isActive: boolean;
  sortOrder: number;
}

const ProductSections: React.FC = () => {
  const [sections, setSections] = useState<HomepageSection[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingSection, setEditingSection] = useState<HomepageSection | null>(null);
  const [availableProducts, setAvailableProducts] = useState<Product[]>([]);
  const [availableCategories, setAvailableCategories] = useState<any[]>([]);

  // Fetch existing homepage sections
  const fetchHomepageSections = async () => {
    try {
      const response = await homepageSectionService.getDisplaySections();
      setSections(response.data);
    } catch (error) {
      toast.error('Failed to fetch homepage sections');
      console.error(error);
    } finally {
      setLoading(false);
    }
  };

  // Fetch available products and categories
  const fetchAvailableData = async () => {
    try {
      const [productsResponse, categoriesResponse] = await Promise.all([
        itemService.getItems(1, 100),
        categoryService.getActiveCategories(100)
      ]);

      setAvailableProducts(productsResponse.data || []);
      setAvailableCategories(categoriesResponse.data || []);
    } catch (error) {
      toast.error('Failed to fetch products or categories');
      console.error(error);
    }
  };

  // Fetch and update available products
  const updateAvailableProducts = async () => {
    try {
      const response = await itemService.getItems(1, 1000);
      setAvailableProducts(response.data || []);
      
      // If editing a section with 'all_products' type, update its products
      if (editingSection?.productType === 'all_products') {
        const allActiveProducts = response.data || [];
        setEditingSection({
          ...editingSection,
          products: allActiveProducts.map((p: Product) => p._id)
        });
      }
    } catch (error) {
      console.error('Failed to update products:', error);
    }
  };

  // Set up polling for product updates
  useEffect(() => {
    fetchHomepageSections();
    fetchAvailableData();
    
    // Poll for product updates every 30 seconds
    const productPollingInterval = setInterval(updateAvailableProducts, 30000);
    
    // Clean up interval on component unmount
    return () => clearInterval(productPollingInterval);
  }, [editingSection]); // Add editingSection to dependency array to avoid stale closures

  // Create or update a homepage section
  const handleSaveSection = async () => {
    if (!editingSection) return;

    try {
      // Prepare the section data
      let sectionData = { ...editingSection };

      // If product type is 'all_products', fetch all active products
      if (editingSection.productType === 'all_products') {
        try {
          const response = await itemService.getItems(1, 1000); // Fetch all active products
          const allActiveProducts = response.data || [];
          sectionData.products = allActiveProducts.map((p: any) => p._id);
          toast.success('Automatically included all active products in this section');
        } catch (error) {
          console.error('Failed to fetch active products:', error);
          toast.error('Failed to fetch active products');
          return;
        }
      } else {
        // For other product types, ensure products is an array of IDs
        sectionData.products = Array.isArray(editingSection.products) 
          ? editingSection.products.map(p => typeof p === 'string' ? p : p._id)
          : [];
      }

      if (editingSection._id) {
        // Update existing section
        await homepageSectionService.updateSection(editingSection._id, sectionData);
        toast.success('Section updated successfully');
      } else {
        // Create new section
        await homepageSectionService.createSection(sectionData);
        toast.success('Section created successfully');
      }
      
      // Refresh sections
      fetchHomepageSections();
      setEditingSection(null);
    } catch (error) {
      toast.error('Failed to save section');
      console.error(error);
    }
  };

  // Delete a homepage section
  const handleDeleteSection = async (sectionId: string) => {
    try {
      await homepageSectionService.deleteSection(sectionId);
      toast.success('Section deleted successfully');
      fetchHomepageSections();
    } catch (error) {
      toast.error('Failed to delete section');
      console.error(error);
    }
  };

  // Change section order
  const moveSectionOrder = (index: number, direction: 'up' | 'down') => {
    const newSections = [...sections];
    const [movedSection] = newSections.splice(index, 1);
    const newIndex = direction === 'up' ? index - 1 : index + 1;
    newSections.splice(newIndex, 0, movedSection);

    // Update sort orders
    const updatedSections = newSections.map((section, idx) => ({
      ...section,
      sortOrder: idx + 1
    }));

    setSections(updatedSections);
    
    // Call backend to update order
    homepageSectionService.reorderSections(updatedSections)
      .catch(error => {
        toast.error('Failed to reorder sections');
        console.error(error);
        // Revert local changes
        setSections(sections);
      });
  };

  // Render section edit/create modal
  const renderSectionModal = () => {
    if (!editingSection) return null;

    return (
      <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 overflow-y-auto">
        <div className="bg-white rounded-lg p-6 w-full max-w-4xl max-h-[90vh] overflow-y-auto">
          <h2 className="text-xl font-bold mb-4">
            {editingSection._id ? 'Edit' : 'Create'} Homepage Section
          </h2>

          <div className="grid grid-cols-2 gap-4">
            {/* Section Details Column */}
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Section Title *
                </label>
                <input
                  type="text"
                  value={editingSection.title}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    title: e.target.value
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                  placeholder="Enter section title"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Short Description
                </label>
                <textarea
                  value={editingSection.description || ''}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    description: e.target.value
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                  placeholder="Enter section description"
                  rows={3}
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Categories
                </label>
                <select
                  multiple
                  value={editingSection.categoryIds}
                  onChange={(e) => {
                    const selectedCategories = Array.from(e.target.selectedOptions, 
                      option => option.value);
                    setEditingSection({
                      ...editingSection,
                      categoryIds: selectedCategories
                    });
                  }}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 h-40"
                >
                  {availableCategories.map((category) => (
                    <option key={category._id} value={category._id}>
                      {category.name}
                    </option>
                  ))}
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Product Type *
                </label>
                <select
                  value={editingSection.productType}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    productType: e.target.value as HomepageSection['productType']
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                >
                  <option value="all_products">All Products</option>
                  <option value="new_added_products">New Added Products</option>
                  <option value="products_on_sale">Products On Sale</option>
                  <option value="most_selling_products">Most Selling Products</option>
                  <option value="custom_products">Custom Products</option>
                </select>
              </div>

              {editingSection.productType === 'custom_products' && (
                <div>
                  <label className="block text-sm font-medium text-gray-700">
                    Select Custom Products
                  </label>
                  <select
                    multiple
                    value={Array.isArray(editingSection.products) 
                      ? editingSection.products.map(p => typeof p === 'string' ? p : p._id) 
                      : []}
                    onChange={(e) => {
                      const selectedProducts = Array.from(e.target.selectedOptions, 
                        option => option.value);
                      setEditingSection({
                        ...editingSection,
                        products: selectedProducts
                      });
                    }}
                    className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 h-40"
                  >
                    {availableProducts.map((product) => (
                      <option key={product._id} value={product._id}>
                        {product.name} - ₹{product.price}
                      </option>
                    ))}
                  </select>
                </div>
              )}
            </div>

            {/* Section Configuration Column */}
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Position *
                </label>
                <select
                  value={editingSection.position}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    position: e.target.value as HomepageSection['position']
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                >
                  <option value="top">Top</option>
                  <option value="below_slider">Below Slider</option>
                  <option value="below_category">Below Category</option>
                  <option value="below_shop_by_seller">Below Shop by Seller</option>
                  <option value="below_shop_by_country">Below Shop by Country of Origin</option>
                  <option value="below_shop_by_brands">Below Shop by Brands</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Background Color (Light Theme) *
                </label>
                <input
                  type="color"
                  value={editingSection.backgroundColorLight}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    backgroundColorLight: e.target.value
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Background Color (Dark Theme) *
                </label>
                <input
                  type="color"
                  value={editingSection.backgroundColorDark}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    backgroundColorDark: e.target.value
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  App Section Style *
                </label>
                <select
                  value={editingSection.appSectionStyle}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    appSectionStyle: e.target.value as HomepageSection['appSectionStyle']
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                >
                  <option value="grid">Grid</option>
                  <option value="horizontal_scroll">Horizontal Scroll</option>
                  <option value="vertical_scroll">Vertical Scroll</option>
                  <option value="carousel">Carousel</option>
                </select>
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700">
                  Web Section Style *
                </label>
                <select
                  value={editingSection.webSectionStyle}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    webSectionStyle: e.target.value as HomepageSection['webSectionStyle']
                  })}
                  className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3"
                >
                  <option value="grid">Grid</option>
                  <option value="horizontal_scroll">Horizontal Scroll</option>
                  <option value="vertical_scroll">Vertical Scroll</option>
                  <option value="carousel">Carousel</option>
                </select>
              </div>

              <div className="flex items-center">
                <input
                  type="checkbox"
                  checked={editingSection.isActive}
                  onChange={(e) => setEditingSection({
                    ...editingSection,
                    isActive: e.target.checked
                  })}
                  className="mr-2"
                />
                <label>Active Section</label>
              </div>
            </div>
          </div>

          <div className="mt-6 flex justify-end space-x-3">
            <button
              onClick={() => setEditingSection(null)}
              className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
            >
              Cancel
            </button>
            <button
              onClick={handleSaveSection}
              className="px-4 py-2 bg-blue-600 text-white rounded-md text-sm font-medium hover:bg-blue-700"
            >
              Save Section
            </button>
          </div>
        </div>
      </div>
    );
  };

  return (<div className="p-6 space-y-6">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold text-gray-900">
            Homepage Product Sections
          </h1>
          <button
            onClick={() => setEditingSection({
              title: '',
              description: '',
              categoryIds: [],
              productType: 'all_products',
              position: 'top',
              backgroundColorLight: '#FFFFFF',
              backgroundColorDark: '#000000',
              appSectionStyle: 'grid',
              webSectionStyle: 'grid',
              products: [],
              isActive: true,
              sortOrder: sections.length + 1
            })}
            className="flex items-center bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Section
          </button>
        </div>

        {loading ? (
          <div className="text-center py-12">Loading sections...</div>
        ) : (
          <div className="bg-white shadow rounded-lg overflow-hidden">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  {['Section Title', 'Product Type', 'Position', 'Products', 'Status', 'Actions', 'Order'].map((header) => (
                    <th
                      key={header}
                      className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider"
                    >
                      {header}
                    </th>
                  ))}
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {sections.map((section, index) => (
                  <tr key={section._id} className="hover:bg-gray-50">
                    <td className="px-6 py-4 text-sm font-medium text-gray-900">
                      {section.title}
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-500">
                      {section.productType === 'all_products' ? 'All Products' :
                       section.productType === 'new_added_products' ? 'New Added Products' :
                       section.productType === 'products_on_sale' ? 'Products On Sale' :
                       section.productType === 'most_selling_products' ? 'Most Selling Products' :
                       'Custom Products'}
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-500">
                      {section.position === 'top' ? 'Top' :
                       section.position === 'below_slider' ? 'Below Slider' :
                       section.position === 'below_category' ? 'Below Category' :
                       section.position === 'below_shop_by_seller' ? 'Below Shop by Seller' :
                       section.position === 'below_shop_by_country' ? 'Below Shop by Country' :
                       'Below Shop by Brands'}
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-500">
                      {section.products.length} products
                    </td>
                    <td className="px-6 py-4">
                      <span className={`px-2 py-1 rounded-full text-xs ${
                        section.isActive ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                      }`}>
                        {section.isActive ? 'Active' : 'Inactive'}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm">
                      <div className="flex items-center space-x-2">
                        <button
                          className="text-blue-600 hover:text-blue-900"
                          onClick={async () => {
                            try {
                              setLoading(true);
                              const response = await homepageSectionService.getSectionById(section._id!);
                              setEditingSection({
                                ...response.data,
                                // Ensure products is always an array
                                products: Array.isArray(response.data.products) 
                                  ? response.data.products 
                                  : []
                              });
                            } catch (error) {
                              toast.error('Failed to load section details');
                              console.error(error);
                            } finally {
                              setLoading(false);
                            }
                          }}
                        >
                          <Edit className="w-4 h-4" />
                        </button>
                        <button
                          className="text-red-600 hover:text-red-900"
                          onClick={() => handleDeleteSection(section._id!)}
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex items-center space-x-2">
                        {index > 0 && (
                          <button
                            onClick={() => moveSectionOrder(index, 'up')}
                            className="text-gray-600 hover:text-gray-900"
                          >
                            <ArrowUp className="w-4 h-4" />
                          </button>
                        )}
                        {index < sections.length - 1 && (
                          <button
                            onClick={() => moveSectionOrder(index, 'down')}
                            className="text-gray-600 hover:text-gray-900"
                          >
                            <ArrowDown className="w-4 h-4" />
                          </button>
                        )}
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {/* Section Edit/Create Modal */}
        {renderSectionModal()}
      </div>
    
  );
};

export { ProductSections as default };

                        


