import React, { useEffect, useMemo, useState } from 'react';
import { Search, Plus, Download } from 'lucide-react';
import { toast } from 'react-hot-toast';
import { format } from 'date-fns';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import api from '../../../services/api';

// Helper function to safely get delivery boy name
const getDeliveryBoyName = (deliveryBoy: any): string => {
  if (!deliveryBoy) return 'N/A';
  if (typeof deliveryBoy === 'string') {
    // This case should ideally be handled by populating data on the backend
    // or fetching details on the frontend. For now, we'll show the ID.
    return deliveryBoy; 
  }
  return deliveryBoy.name || 'Unknown';
};

interface DeliveryBoy {
  _id?: string;
  id?: string;
  name: string;
  phone?: string;
}

interface CashCollection {
  _id?: string;
  deliveryBoy: string | { _id: string; name: string };
  orderId: string;
  amount: number;
  collectionDate: string;
  status: 'pending' | 'collected' | 'handed_over';
  notes?: string;
}

const CashCollections: React.FC = () => {
  const [collections, setCollections] = useState<CashCollection[]>([]);
  const [deliveryBoys, setDeliveryBoys] = useState<DeliveryBoy[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [currentCollection, setCurrentCollection] = useState<Partial<CashCollection> | null>(null);
  
  const hasRows = useMemo(() => collections.length > 0, [collections]);

  const fetchDeliveryBoys = async () => {
    try {
      const res = await api.get('/delivery-boys?limit=1000');
      const boys = Array.isArray(res.data?.data) ? res.data.data :
                  Array.isArray(res.data) ? res.data : [];
      
      const validBoys = boys.filter((boy: DeliveryBoy) => 
        boy && (boy._id || boy.id) && boy.name
      ).map((boy: DeliveryBoy) => ({
        _id: boy._id || boy.id,
        id: boy.id || boy._id,
        name: boy.name || 'Unknown',
        phone: boy.phone || ''
      }));
      
      setDeliveryBoys(validBoys);
    } catch (e: any) {
      console.error('Failed to fetch delivery boys:', e);
      toast.error('Failed to load delivery boys. Please try again.');
      setDeliveryBoys([]);
    }
  };

  const fetchCollections = async () => {
    try {
      setLoading(true);
      const { data } = await api.get('/cash-collections', { 
        params: { 
          page: 1, 
          limit: 100, 
          search: searchTerm 
        } 
      });
      setCollections(Array.isArray(data.data) ? data.data : []);
    } catch (e: any) {
      toast.error(e.response?.data?.message || 'Failed to load cash collections');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCollections();
    fetchDeliveryBoys();
  }, [searchTerm]);
  
  const handleDelete = async (id: string) => {
    if (window.confirm('Are you sure you want to delete this cash collection?')) {
      try {
        await api.delete(`/cash-collections/${id}`);
        toast.success('Cash collection deleted successfully');
        fetchCollections();
      } catch (e: any) {
        toast.error(e.response?.data?.message || 'Failed to delete cash collection');
      }
    }
  };

  const handleEdit = (collection: CashCollection) => {
    // Ensure we have the delivery boy ID in the correct format
    const deliveryBoyId = typeof collection.deliveryBoy === 'object' && collection.deliveryBoy
      ? collection.deliveryBoy._id 
      : collection.deliveryBoy;
    
    // Create a new collection object with the processed delivery boy ID
    const updatedCollection = {
      ...collection,
      deliveryBoy: deliveryBoyId,
      // Ensure collectionDate is in the correct format for the date input
      collectionDate: collection.collectionDate 
        ? new Date(collection.collectionDate).toISOString().split('T')[0]
        : new Date().toISOString().split('T')[0]
    };
    
    console.log('Editing collection:', updatedCollection);
    setCurrentCollection(updatedCollection);
    setIsModalOpen(true);
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!currentCollection) return;

    const saveCollection = async () => {
      try {
        // Prepare the data to be sent
        const collectionData = {
          ...currentCollection,
          // Ensure amount is a number
          amount: typeof currentCollection.amount === 'string' 
            ? parseFloat(currentCollection.amount) 
            : currentCollection.amount,
          // Ensure deliveryBoy is just the ID
          deliveryBoy: typeof currentCollection.deliveryBoy === 'object' && currentCollection.deliveryBoy
            ? (currentCollection.deliveryBoy as any)._id
            : currentCollection.deliveryBoy,
          // Format the date correctly
          collectionDate: currentCollection.collectionDate
            ? new Date(currentCollection.collectionDate).toISOString()
            : new Date().toISOString()
        };

        if (currentCollection._id) {
          // Update existing collection
          await api.put(`/cash-collections/${currentCollection._id}`, collectionData);
          toast.success('Cash collection updated successfully');
        } else {
          // Create new collection
          await api.post('/cash-collections', {
            ...collectionData,
            status: 'pending'
          });
          toast.success('Cash collection created successfully');
        }
        
        // Reset and refresh
        setIsModalOpen(false);
        setCurrentCollection(null);
        fetchCollections();
      } catch (e: any) {
        console.error('Error saving cash collection:', e);
        const errorMessage = e.response?.data?.message || 'Failed to save cash collection';
        toast.error(errorMessage);
      }
    };

    saveCollection();
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value, type } = e.target as HTMLInputElement;
    
    setCurrentCollection(prev => {
      if (!prev) return null as any;
      
      // Special handling for different input types
      let processedValue: any = value;
      
      if (type === 'number') {
        processedValue = parseFloat(value) || 0;
      } else if (name === 'deliveryBoy') {
        // Ensure we store just the ID for deliveryBoy
        processedValue = value;
      }
      
      return {
        ...prev,
        [name]: processedValue
      };
    });
  };

  const handleStatusUpdate = async (id: string, status: 'collected' | 'handed_over') => {
    try {
      await api.put(`/cash-collections/${id}`, { status });
      toast.success(`Cash collection marked as ${status.replace('_', ' ')}`);
      fetchCollections();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to update status');
    }
  };

  return (
    <>
      <Layout title="Cash Collections" sidebar={<Sidebar items={adminSidebarItems} />}>
        <div className="relative">
          <div className="container mx-auto p-4">
            <div className="flex justify-between items-center mb-6">
              <h1 className="text-2xl font-bold">Cash Collections</h1>
              <div className="flex space-x-4">
                <button
                  onClick={() => {
                    // Export functionality can be implemented here
                    toast.success('Export functionality will be implemented soon');
                  }}
                  className="flex items-center px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"
                >
                  <Download className="w-4 h-4 mr-2" />
                  Export
                </button>
                <button
                  onClick={() => {
                    setCurrentCollection({
                      deliveryBoy: '',
                      orderId: '',
                      amount: 0,
                      collectionDate: new Date().toISOString().split('T')[0],
                      status: 'pending',
                      notes: ''
                    });
                    setIsModalOpen(true);
                  }}
                  className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700"
                >
                  <Plus className="w-4 h-4 mr-2" />
                  Add Collection
                </button>
              </div>
            </div>
            
            <div className="space-y-6">
              {/* Search */}
              <div className="relative w-full max-w-sm">
                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                  <Search className="h-5 w-5 text-gray-400" />
                </div>
                <input
                  type="search"
                  placeholder="Search collections..."
                  className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500 w-full"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>

              {/* Table / Empty / Loading */}
              {loading ? (
                <div className="py-12 text-center">Loading...</div>
              ) : !hasRows ? (
                <div className="text-center py-12">
                  <Download className="mx-auto h-12 w-12 text-gray-400" />
                  <div className="text-gray-600">No cash collections found</div>
                </div>
              ) : (
                <div className="bg-white border rounded-lg overflow-hidden">
                  <div className="overflow-x-auto">
                    <table className="min-w-full divide-y divide-gray-200">
                      <thead className="bg-gray-50">
                        <tr>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Order ID</th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Delivery Boy</th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Collection Date</th>
                          <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                          <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase">Actions</th>
                        </tr>
                      </thead>
                      <tbody className="bg-white divide-y divide-gray-200">
                        {collections.map((collection) => (
                          <tr key={collection._id || collection.id} className="hover:bg-gray-50">
                            <td className="px-6 py-4 whitespace-nowrap">
                              <div className="text-sm font-medium text-gray-900">{collection.orderId}</div>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <div className="text-sm text-gray-900">
                                {getDeliveryBoyName(collection.deliveryBoy)}
                              </div>
                              <div className="text-xs text-gray-500">
                                {collection.deliveryBoy && typeof collection.deliveryBoy === 'object' 
                                  ? (collection.deliveryBoy as any).phone || 'No phone'
                                  : ''}
                              </div>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <div className="text-sm text-gray-900">${collection.amount.toFixed(2)}</div>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <div className="text-sm text-gray-900">
                                {format(new Date(collection.collectionDate), 'MMM d, yyyy')}
                              </div>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap">
                              <span className={`px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                collection.status === 'handed_over' 
                                  ? 'bg-green-100 text-green-800' 
                                  : collection.status === 'collected'
                                    ? 'bg-blue-100 text-blue-800'
                                    : 'bg-yellow-100 text-yellow-800'
                              }`}>
                                {collection.status.split('_').map(word => 
                                  word.charAt(0).toUpperCase() + word.slice(1)
                                ).join(' ')}
                              </span>
                            </td>
                            <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-2">
                              {collection.status === 'collected' && (
                                <button
                                  onClick={() => handleStatusUpdate(collection._id || collection.id || '', 'handed_over')}
                                  className="text-blue-600 hover:text-blue-800 mr-2"
                                  title="Mark as Handed Over"
                                >
                                  Hand Over
                                </button>
                              )}
                              {collection.status === 'pending' && (
                                <button
                                  onClick={() => handleStatusUpdate(collection._id || collection.id || '', 'collected')}
                                  className="text-blue-600 hover:text-blue-800 mr-2"
                                  title="Mark as Collected"
                                >
                                  Mark Collected
                                </button>
                              )}
                              <button
                                onClick={() => handleEdit(collection)} 
                                className="text-gray-600 hover:text-gray-800"
                                title="View Details"
                              >
                                View
                              </button>
                              <button
                                onClick={() => handleDelete(collection._id!)}
                                className="text-red-600 hover:text-red-800"
                                title="Delete"
                              >
                                Delete
                              </button>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </Layout>

      {/* Add/Edit Collection Modal */}
      {isModalOpen && currentCollection && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
        <div className="bg-white rounded-lg w-full max-w-md">
          <div className="p-6">
            <h2 className="text-xl font-semibold mb-4">
              {currentCollection._id ? 'Edit Cash Collection' : 'Add New Cash Collection'}
            </h2>
            
            <form onSubmit={handleSubmit}>
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Delivery Boy *</label>
                  <select
                    name="deliveryBoy"
                    value={(() => {
                      // Handle different possible formats of deliveryBoy
                      if (!currentCollection.deliveryBoy) return '';
                      if (typeof currentCollection.deliveryBoy === 'string') {
                        return currentCollection.deliveryBoy;
                      }
                      if (typeof currentCollection.deliveryBoy === 'object' && currentCollection.deliveryBoy !== null) {
                        return (currentCollection.deliveryBoy as any)._id || '';
                      }
                      return '';
                    })()}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border rounded-md focus:ring-blue-500 focus:border-blue-500"
                    required
                    disabled={!!currentCollection._id} // Disable for existing collections
                  >
                    <option value="">Select Delivery Boy</option>
                    {deliveryBoys && deliveryBoys.length > 0 ? (
                      deliveryBoys.map((boy) => (
                        <option key={boy._id} value={boy._id}>
                          {boy.name} {boy.phone ? `(${boy.phone})` : ''}
                        </option>
                      ))
                    ) : (
                      <option value="" disabled>No delivery boys available</option>
                    )}
                  </select>
                </div>
        
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Order ID *</label>
                  <input
                    type="text"
                    name="orderId"
                    value={currentCollection.orderId || ''}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border rounded-md focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Amount (₹) *</label>
                  <div className="relative">
                    <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                      <span className="text-gray-500 sm:text-sm">₹</span>
                    </div>
                    <input
                      type="number"
                      name="amount"
                      value={currentCollection.amount || 0}
                      onChange={handleInputChange}
                      className="block w-full pl-7 pr-12 sm:text-sm border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
                      placeholder="0.00"
                      step="0.01"
                      min="0"
                      required
                    />
                  </div>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Collection Date *</label>
                  <input
                    type="date"
                    name="collectionDate"
                    value={currentCollection.collectionDate ? new Date(currentCollection.collectionDate).toISOString().split('T')[0] : ''}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border rounded-md focus:ring-blue-500 focus:border-blue-500"
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Status</label>
                  <select
                    name="status"
                    value={currentCollection.status || 'pending'}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border rounded-md focus:ring-blue-500 focus:border-blue-500"
                    disabled={!!currentCollection._id && currentCollection.status === 'handed_over'}
                  >
                    <option value="pending">Pending</option>
                    <option value="collected">Collected</option>
                    {currentCollection.status === 'handed_over' && (
                      <option value="handed_over">Handed Over</option>
                    )}
                  </select>
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">Notes</label>
                  <textarea
                    name="notes"
                    value={currentCollection.notes || ''}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 border rounded-md focus:ring-blue-500 focus:border-blue-500"
                    rows={3}
                  />
                </div>

                <div className="flex justify-end space-x-3 pt-4">
                  <button
                    type="button"
                    onClick={() => {
                      setIsModalOpen(false);
                      setCurrentCollection(null);
                    }}
                    className="px-4 py-2 border border-gray-300 rounded-md text-sm font-medium text-gray-700 hover:bg-gray-50"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 text-sm font-medium"
                  >
                    {currentCollection._id ? 'Update' : 'Create'} Collection
                  </button>
                </div>
              </div>
            </form>
          </div>
        </div>
      </div>
    )}
    </>
  );
};

export default CashCollections;