import React from 'react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import StatsCard from '../../components/shared/StatsCard';
import { adminSidebarItems } from './AdminLayout';
import { 
  DollarSign, 
  ShoppingCart, 
  Users, 
  Store,
  TrendingUp,
  Package,
  Truck,
  Star
} from 'lucide-react';

const AdminDashboard: React.FC = () => {
  const stats = [
    {
      title: 'Total Revenue',
      value: '$124,850',
      icon: DollarSign,
      trend: { value: 12, isPositive: true },
      color: 'green' as const
    },
    {
      title: 'Total Orders',
      value: '1,247',
      icon: ShoppingCart,
      trend: { value: 8, isPositive: true },
      color: 'blue' as const
    },
    {
      title: 'Total Customers',
      value: '856',
      icon: Users,
      trend: { value: 15, isPositive: true },
      color: 'purple' as const
    },
    {
      title: 'Active Stores',
      value: '12',
      icon: Store,
      trend: { value: 5, isPositive: true },
      color: 'orange' as const
    }
  ];

  const recentOrders = [
    { id: '#ORD-001', customer: 'John Doe', amount: '$125.50', status: 'delivered', date: '2024-01-15' },
    { id: '#ORD-002', customer: 'Jane Smith', amount: '$89.25', status: 'out_for_delivery', date: '2024-01-15' },
    { id: '#ORD-003', customer: 'Mike Johnson', amount: '$245.00', status: 'packed', date: '2024-01-14' },
    { id: '#ORD-004', customer: 'Sarah Wilson', amount: '$67.80', status: 'confirmed', date: '2024-01-14' },
  ];

  const getStatusBadge = (status: string) => {
    const statusColors = {
      delivered: 'bg-green-100 text-green-800',
      out_for_delivery: 'bg-blue-100 text-blue-800',
      packed: 'bg-yellow-100 text-yellow-800',
      confirmed: 'bg-purple-100 text-purple-800',
      pending: 'bg-gray-100 text-gray-800'
    };
    
    return `px-2 py-1 text-xs font-medium rounded-full ${statusColors[status as keyof typeof statusColors] || statusColors.pending}`;
  };

  return (
    <Layout title="Dashboard" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <StatsCard
              key={index}
              title={stat.title}
              value={stat.value}
              icon={stat.icon}
              trend={stat.trend}
              color={stat.color}
            />
          ))}
        </div>

        {/* Charts and Tables */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Recent Orders */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Recent Orders</h3>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order ID
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Amount
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {recentOrders.map((order) => (
                    <tr key={order.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                        {order.id}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.customer}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        {order.amount}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={getStatusBadge(order.status)}>
                          {order.status.replace('_', ' ')}
                        </span>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {/* Top Performing Stores */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Top Performing Stores</h3>
            </div>
            <div className="p-6">
              <div className="space-y-4">
                {[
                  { name: 'MedPlus Pharmacy', sales: '$45,230', orders: 324 },
                  { name: 'HealthCare Store', sales: '$38,150', orders: 298 },
                  { name: 'City Pharmacy', sales: '$32,890', orders: 256 },
                  { name: 'Wellness Mart', sales: '$28,540', orders: 189 }
                ].map((store, index) => (
                  <div key={index} className="flex items-center justify-between">
                    <div className="flex items-center">
                      <div className="bg-blue-100 p-2 rounded-lg mr-3">
                        <Store className="w-5 h-5 text-blue-600" />
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-900">{store.name}</p>
                        <p className="text-xs text-gray-500">{store.orders} orders</p>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-gray-900">{store.sales}</p>
                      <div className="flex items-center text-xs text-green-600">
                        <TrendingUp className="w-3 h-3 mr-1" />
                        {12 + index}%
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Quick Actions</h3>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {[
                { label: 'Add Product', icon: Package, color: 'bg-blue-500' },
                { label: 'Manage Orders', icon: ShoppingCart, color: 'bg-green-500' },
                { label: 'View Reports', icon: TrendingUp, color: 'bg-purple-500' },
                { label: 'Store Settings', icon: Store, color: 'bg-orange-500' }
              ].map((action, index) => {
                const Icon = action.icon;
                return (
                  <button
                    key={index}
                    className="flex items-center p-4 border border-gray-200 rounded-lg hover:border-gray-300 hover:shadow-sm transition-all"
                  >
                    <div className={`p-2 rounded-lg ${action.color} text-white mr-3`}>
                      <Icon className="w-5 h-5" />
                    </div>
                    <span className="text-sm font-medium text-gray-900">{action.label}</span>
                  </button>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </Layout>
  );
};

export default AdminDashboard;