import React, { useEffect, useMemo, useState } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, Plus, Edit, Trash2, Users } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';

interface Customer {
  _id?: string;
  id?: string;
  name: string;
  email: string;
  phone: string;
  role?: string | { name: string };
  status: boolean;
  createdAt?: string;
  created_at?: string;
}

const Customers: React.FC = () => {
  const [customers, setCustomers] = useState<Customer[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState<Customer | null>(null);
  const [form, setForm] = useState({ name: '', email: '', phone: '', role: 'customer', status: true, password: '' });

  const hasRows = useMemo(() => customers.length > 0, [customers]);

  const fetchCustomers = async () => {
    try {
      setLoading(true);
      const res = await api.get('/customers', { params: { search: searchTerm } });
      const rows = res.data?.data || res.data || [];
      setCustomers(rows);
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to load customers');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCustomers();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!form.name.trim() || !form.email.trim() || !form.phone.trim()) {
      toast.error('Name, email and phone are required');
      return;
    }
    if (!editing && !form.password.trim()) {
      toast.error('Password is required for new customer');
      return;
    }

    try {
      if (editing) {
        await api.put(`/customers/${editing.id || editing._id}`, {
          name: form.name,
          email: form.email,
          phone: form.phone,
          role: form.role,
          status: form.status
        });
        toast.success('Customer updated');
      } else {
        // Create via users endpoint if available, else customers controller will handle insert
        await api.post('/auth/register', {
          name: form.name,
          email: form.email,
          phone: form.phone,
          password: form.password,
          role: 'customer'
        });
        toast.success('Customer created');
      }
      setShowModal(false);
      setEditing(null);
      setForm({ name: '', email: '', phone: '', role: 'customer', status: true, password: '' });
      fetchCustomers();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to save customer');
    }
  };

  const handleEdit = (row: Customer) => {
    setEditing(row);
    setForm({
      name: row.name,
      email: row.email,
      phone: row.phone,
      role: typeof row.role === 'string' ? row.role : (row.role?.name || 'customer'),
      status: row.status,
      password: ''
    });
    setShowModal(true);
  };

  const handleDelete = async (row: Customer) => {
    if (!confirm('This will permanently delete the customer. Continue?')) return;
    try {
      await api.delete(`/customers/${row.id || row._id}`);
      toast.success('Customer deleted');
      fetchCustomers();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to delete customer');
    }
  };

  return (
    <Layout title="Customers" sidebar={<Sidebar items={adminSidebarItems} />}> 
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && fetchCustomers()}
              placeholder="Search customers..."
              className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
            />
          </div>
          <button onClick={() => { setEditing(null); setShowModal(true); }} className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg">
            <Plus className="w-4 h-4 mr-2" /> Add Customer
          </button>
        </div>

        {loading && (
          <div className="py-12 text-center">Loading...</div>
        )}

        {!loading && hasRows && (
          <div className="bg-white border rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Name</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Email</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Phone</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {customers.map((c) => (
                    <tr key={String(c.id || c._id)} className="hover:bg-gray-50">
                      <td className="px-6 py-4">{c.name}</td>
                      <td className="px-6 py-4">{c.email}</td>
                      <td className="px-6 py-4">{c.phone}</td>
                      <td className="px-6 py-4">
                        <span className={`px-2 py-1 rounded-full text-xs ${c.status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}>{c.status ? 'Active' : 'Inactive'}</span>
                      </td>
                      <td className="px-6 py-4 text-right">
                        <button onClick={() => handleEdit(c)} className="text-green-600 hover:text-green-800 mr-3"><Edit className="w-4 h-4" /></button>
                        <button onClick={() => handleDelete(c)} className="text-red-600 hover:text-red-800"><Trash2 className="w-4 h-4" /></button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}

        {!loading && !hasRows && (
          <div className="text-center py-12">
            <Users className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <div className="text-gray-600">No customers found</div>
          </div>
        )}

        {showModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-lg">
              <h3 className="text-lg font-semibold mb-4">{editing ? 'Edit Customer' : 'Add Customer'}</h3>
              <form onSubmit={handleSubmit} className="space-y-4">
                <div>
                  <label className="block text-sm font-medium mb-1">Name *</label>
                  <input value={form.name} onChange={(e) => setForm({ ...form, name: e.target.value })} className="w-full border rounded-lg px-3 py-2" required />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Email *</label>
                  <input type="email" value={form.email} onChange={(e) => setForm({ ...form, email: e.target.value })} className="w-full border rounded-lg px-3 py-2" required />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Phone *</label>
                  <input value={form.phone} onChange={(e) => setForm({ ...form, phone: e.target.value })} className="w-full border rounded-lg px-3 py-2" required />
                </div>
                {!editing && (
                  <div>
                    <label className="block text-sm font-medium mb-1">Password *</label>
                    <input type="password" value={form.password} onChange={(e) => setForm({ ...form, password: e.target.value })} className="w-full border rounded-lg px-3 py-2" required />
                  </div>
                )}
                <div>
                  <label className="block text-sm font-medium mb-1">Role</label>
                  <select value={form.role} onChange={(e) => setForm({ ...form, role: e.target.value })} className="w-full border rounded-lg px-3 py-2">
                    <option value="customer">Customer</option>
                    <option value="admin">Admin</option>
                    <option value="store_manager">Store Manager</option>
                    <option value="delivery_boy">Delivery Boy</option>
                  </select>
                </div>
                <div>
                  <label className="block text-sm font-medium mb-1">Status</label>
                  <select value={String(form.status)} onChange={(e) => setForm({ ...form, status: e.target.value === 'true' })} className="w-full border rounded-lg px-3 py-2">
                    <option value="true">Active</option>
                    <option value="false">Inactive</option>
                  </select>
                </div>
              </form>
              <div className="flex justify-end space-x-3 mt-6">
                <button onClick={() => { setShowModal(false); setEditing(null); }} className="px-4 py-2 bg-gray-100 rounded-lg">Cancel</button>
                <button onClick={handleSubmit as any} className="px-4 py-2 bg-blue-600 text-white rounded-lg">{editing ? 'Update' : 'Create'}</button>
              </div>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Customers;
