import React, { useEffect, useMemo, useState } from 'react';
import Layout from '../../../components/shared/Layout';
import Sidebar from '../../../components/shared/Sidebar';
import { adminSidebarItems } from '../AdminLayout';
import { Search, Plus, Edit, Trash2, Tag, Calendar, Percent } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '@/services/api';

interface Coupon {
  _id?: string;
  id?: string;
  code: string;
  description: string;
  discount_type: 'percentage' | 'fixed';
  discount_value: number;
  min_order_amount: number;
  max_discount: number;
  start_date: string;
  end_date: string;
  usage_limit: number;
  used_count: number;
  status: boolean;
  created_at?: string;
}

const Coupon: React.FC = () => {
  const [coupons, setCoupons] = useState<Coupon[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [showModal, setShowModal] = useState(false);
  const [editing, setEditing] = useState<Coupon | null>(null);
  const [form, setForm] = useState({
    code: '',
    description: '',
    discount_type: 'percentage' as 'percentage' | 'fixed',
    discount_value: 0,
    min_order_amount: 0,
    max_discount: 100, // Default max discount for percentage
    start_date: new Date().toISOString().split('T')[0],
    end_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
    usage_limit: 100,
    status: true
  });

  const handleDiscountTypeChange = (value: 'percentage' | 'fixed') => {
    setForm(prev => ({
      ...prev,
      discount_type: value,
      // Reset max_discount to default when switching to percentage
      max_discount: value === 'percentage' ? 100 : 0
    }));
    
    // Clear any existing max_discount error when changing type
    if (errors.max_discount) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors.max_discount;
        return newErrors;
      });
    }
  };

  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);

  const hasRows = useMemo(() => coupons.length > 0, [coupons]);

  const validateForm = (): boolean => {
    const newErrors: Record<string, string> = {};
    if (!form.code.trim()) {
      newErrors.code = 'Coupon code is required';
    } else if (form.code.length < 3) {
      newErrors.code = 'Code must be at least 3 characters';
    }
    if (!form.description.trim()) {
      newErrors.description = 'Description is required';
    }
    if (form.discount_value <= 0) {
      newErrors.discount_value = 'Discount value must be greater than 0';
    } else if (form.discount_type === 'percentage' && form.discount_value > 100) {
      newErrors.discount_value = 'Percentage discount cannot exceed 100%';
    }
    if (form.min_order_amount < 0) {
      newErrors.min_order_amount = 'Minimum order amount cannot be negative';
    }
    if (form.discount_type === 'percentage') {
      if (form.max_discount <= 0) {
        newErrors.max_discount = 'Maximum discount amount is required and must be greater than 0';
      }
    }
    if (new Date(form.end_date) <= new Date(form.start_date)) {
      newErrors.end_date = 'End date must be after start date';
    }
    if (form.usage_limit < 0) {
      newErrors.usage_limit = 'Usage limit cannot be negative';
    }
    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleInputChange = (field: string, value: any) => {
    // Clear error for field when user starts typing
    if (errors[field]) {
      setErrors(prev => {
        const newErrors = { ...prev };
        delete newErrors[field];
        return newErrors;
      });
    }
    setForm(prev => ({ ...prev, [field]: value }));
  };

  const fetchCoupons = async () => {
    try {
      setLoading(true);
      const res = await api.get('/coupons', { 
        params: { 
          search: searchTerm,
          page: 1,
          limit: 100
        } 
      });
      const rows = Array.isArray(res.data) ? res.data : res.data?.data || [];
      const formattedCoupons = rows.map((coupon: any) => ({
        _id: coupon._id,
        id: coupon._id,
        code: coupon.code || '',
        description: coupon.description || '',
        discount_type: coupon.discountType || 'percentage',
        discount_value: coupon.discountValue || 0,
        min_order_amount: coupon.minOrderAmount || 0,
        max_discount: coupon.maxDiscountAmount || 0,
        start_date: coupon.startDate ? new Date(coupon.startDate).toISOString().split('T')[0] : '',
        end_date: coupon.expiryDate ? new Date(coupon.expiryDate).toISOString().split('T')[0] : '',
        usage_limit: coupon.usageLimit || 0,
        used_count: coupon.usedCount || 0,
        status: coupon.status === 'active' || coupon.status === true,
        created_at: coupon.createdAt
      }));
      setCoupons(formattedCoupons);
    } catch (e: any) {
      console.error('Error fetching coupons:', e);
      toast.error(e.response?.data?.message || e.response?.data?.msg || 'Failed to load coupons');
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCoupons();
  }, []);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!validateForm()) {
      toast.error('Please fix the form errors before submitting');
      return;
    }
    setIsSubmitting(true);
    try {
      const requestData = {
        code: form.code,
        description: form.description,
        discount_type: form.discount_type,
        discount_value: Number(form.discount_value),
        min_order_amount: Number(form.min_order_amount) || 0,
        max_discount: form.discount_type === 'percentage' ? Number(form.max_discount) : 0,
        start_date: form.start_date,
        end_date: form.end_date,
        usage_limit: Number(form.usage_limit) || 0,
        status: form.status ? 'active' : 'inactive'
      };
      if (editing) {
        await api.put(`/coupons/${editing.id || editing._id}`, requestData);
        toast.success('Coupon updated');
      } else {
        await api.post('/coupons', requestData);
        toast.success('Coupon created');
      }
      setShowModal(false);
      setEditing(null);
      setForm({
        code: '',
        description: '',
        discount_type: 'percentage',
        discount_value: 0,
        min_order_amount: 0,
        max_discount: 0,
        start_date: new Date().toISOString().split('T')[0],
        end_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
        usage_limit: 100,
        status: true
      });
      fetchCoupons();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to save coupon');
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleEdit = (coupon: Coupon) => {
    setEditing(coupon);
    const formatDate = (dateString?: string) => {
      if (!dateString) return '';
      if (dateString.match(/^\d{4}-\d{2}-\d{2}$/)) {
        return dateString;
      }
      return dateString.split('T')[0];
    };
    setForm({
      code: coupon.code || '',
      description: coupon.description || '',
      discount_type: coupon.discount_type || 'fixed',
      discount_value: coupon.discount_value || 0,
      min_order_amount: coupon.min_order_amount || 0,
      max_discount: coupon.max_discount || 0,
      start_date: formatDate(coupon.start_date),
      end_date: formatDate(coupon.end_date),
      usage_limit: coupon.usage_limit || 100,
      status: coupon.status ?? true
    });
    setShowModal(true);
  };

  const handleDelete = async (coupon: Coupon) => {
    if (!window.confirm('This will permanently delete the coupon. Continue?')) return;
    try {
      await api.delete(`/coupons/${coupon.id || coupon._id}`);
      toast.success('Coupon deleted');
      fetchCoupons();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to delete coupon');
    }
  };

  const toggleStatus = async (coupon: Coupon) => {
    try {
      await api.patch(`/coupons/${coupon.id || coupon._id}/status`, {
        status: !coupon.status
      });
      toast.success(`Coupon ${!coupon.status ? 'activated' : 'deactivated'}`);
      fetchCoupons();
    } catch (e: any) {
      toast.error(e.response?.data?.msg || 'Failed to update coupon status');
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  // Helper function to render form field with error message
  const renderFormField = (
    field: string,
    label: string,
    type: string = 'text',
    options?: { value: string; label: string }[],
    min?: number,
    max?: number,
    step: string = '1',
    prefix?: string,
    suffix?: string,
    placeholder?: string
  ) => (
    <div>
      <label className="block text-sm font-medium mb-1">
        {label} {['code', 'description', 'discount_value'].includes(field) && <span className="text-red-500">*</span>}
      </label>
      <div className="relative">
        {prefix && (
          <span className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-500">
            {prefix}
          </span>
        )}
        {type === 'select' ? (
          <select
            value={form[field as keyof typeof form] as string}
            onChange={(e) => handleInputChange(field, e.target.value)}
            className={`w-full border ${errors[field] ? 'border-red-500' : 'border-gray-300'} rounded-lg px-3 py-2 ${prefix ? 'pl-8' : ''}`}
          >
            {options?.map((option) => (
              <option key={option.value} value={option.value}>
                {option.label}
              </option>
            ))}
          </select>
        ) : type === 'date' ? (
          <input
            type={type}
            value={form[field as keyof typeof form] as string}
            onChange={(e) => handleInputChange(field, e.target.value)}
            min={field === 'end_date' ? form.start_date : undefined}
            className={`w-full border ${errors[field] ? 'border-red-500' : 'border-gray-300'} rounded-lg px-3 py-2 ${prefix ? 'pl-8' : ''}`}
          />
        ) : (
          <input
            type={type}
            value={form[field as keyof typeof form] as string | number}
            onChange={(e) => handleInputChange(field, type === 'number' ? parseFloat(e.target.value) || 0 : e.target.value)}
            min={min}
            max={max}
            step={step}
            placeholder={placeholder}
            className={`w-full border ${errors[field] ? 'border-red-500' : 'border-gray-300'} rounded-lg px-3 py-2 ${prefix ? 'pl-8' : ''}`}
          />
        )}
        {suffix && (
          <span className="absolute right-3 top-1/2 -translate-y-1/2 text-gray-500">
            {suffix}
          </span>
        )}
      </div>
      {errors[field] && (
        <p className="mt-1 text-sm text-red-600">{errors[field]}</p>
      )}
    </div>
  );

  return (
    <Layout title="Coupons" sidebar={<Sidebar items={adminSidebarItems} />}>
      <div className="space-y-6">
        <div className="flex items-center justify-between">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              onKeyDown={(e) => e.key === 'Enter' && fetchCoupons()}
              placeholder="Search coupons..."
              className="pl-10 pr-4 py-2 border rounded-lg focus:ring-2 focus:ring-blue-500"
            />
          </div>
          <button 
            onClick={() => { 
              setEditing(null); 
              setForm({
                code: '',
                description: '',
                discount_type: 'percentage',
                discount_value: 0,
                min_order_amount: 0,
                max_discount: 0,
                start_date: new Date().toISOString().split('T')[0],
                end_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
                usage_limit: 100,
                status: true
              });
              setShowModal(true); 
            }} 
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg"
          >
            <Plus className="w-4 h-4 mr-2" /> Add Coupon
          </button>
        </div>
        {loading && (
          <div className="py-12 text-center">Loading...</div>
        )}
        {!loading && hasRows && (
          <div className="bg-white border rounded-lg overflow-hidden">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Code</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Description</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Discount</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Validity</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Usage</th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  {coupons.map((coupon) => (
                    <tr key={String(coupon.id || coupon._id)} className="hover:bg-gray-50">
                      <td className="px-6 py-4 font-medium">
                        <div className="flex items-center">
                          <Tag className="w-4 h-4 mr-2 text-blue-600" />
                          {coupon.code}
                        </div>
                      </td>
                      <td className="px-6 py-4 text-sm text-gray-600 max-w-xs truncate">{coupon.description}</td>
                      <td className="px-6 py-4">
                        <div className="flex items-center">
                          {coupon.discount_type === 'percentage' ? (
                            <Percent className="w-4 h-4 mr-1 text-green-600" />
                          ) : (
                            <span className="mr-1">₹</span>
                          )}
                          {coupon.discount_value?.toLocaleString()}
                          {coupon.discount_type === 'percentage' ? '%' : ''}
                          {coupon.min_order_amount > 0 && (
                            <span className="text-xs text-gray-500 ml-1">
                              (Min. ₹{coupon.min_order_amount?.toLocaleString()})
                            </span>
                          )}
                          {coupon.max_discount > 0 && coupon.discount_type === 'percentage' && (
                            <span className="text-xs text-gray-500 ml-1">
                              (Max. ₹{coupon.max_discount?.toLocaleString()})
                            </span>
                          )}
                        </div>
                      </td>
                      <td className="px-6 py-4 text-sm text-gray-600">
                        <div className="flex items-center">
                          <Calendar className="w-4 h-4 mr-2 text-gray-400 flex-shrink-0" />
                          <div className="flex flex-col">
                            <span>{formatDate(coupon.start_date) || 'No start date'}</span>
                            <span className="text-xs text-gray-500">to {formatDate(coupon.end_date) || 'No end date'}</span>
                            {coupon.end_date && new Date(coupon.end_date) < new Date() && (
                              <span className="text-xs text-red-500">Expired</span>
                            )}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="flex flex-col">
                          <div className="flex items-center mb-1">
                            <div className="w-24 bg-gray-200 rounded-full h-2.5 mr-2">
                              <div 
                                className="bg-blue-600 h-2.5 rounded-full" 
                                style={{
                                  width: `${coupon.usage_limit > 0 
                                    ? Math.min(100, (coupon.used_count / coupon.usage_limit) * 100) 
                                    : 0}%`
                                }}
                              ></div>
                            </div>
                            <span className="text-sm text-gray-600">
                              {coupon.used_count?.toLocaleString()}/{coupon.usage_limit?.toLocaleString() || '∞'}
                            </span>
                          </div>
                          {coupon.usage_limit > 0 && (
                            <span className="text-xs text-gray-500">
                              {Math.round((coupon.used_count / coupon.usage_limit) * 100)}% used
                            </span>
                          )}
                          {coupon.usage_limit > 0 && coupon.used_count >= coupon.usage_limit && (
                            <span className="text-xs text-red-500">Fully redeemed</span>
                          )}
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <button 
                          onClick={() => toggleStatus(coupon)}
                          className={`flex items-center px-3 py-1 rounded-full text-xs font-medium ${coupon.status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'}`}
                        >
                          {coupon.status ? (
                            <>
                              <span className="w-2 h-2 rounded-full bg-green-500 mr-2"></span>
                              Active
                            </>
                          ) : (
                            <>
                              <span className="w-2 h-2 rounded-full bg-red-500 mr-2"></span>
                              Inactive
                            </>
                          )}
                        </button>
                      </td>
                      <td className="px-6 py-4 text-right">
                        <button 
                          onClick={() => handleEdit(coupon)} 
                          className="text-blue-600 hover:text-blue-800 mr-3"
                        >
                          <Edit className="w-4 h-4" />
                        </button>
                        <button 
                          onClick={() => handleDelete(coupon)} 
                          className="text-red-600 hover:text-red-800"
                        >
                          <Trash2 className="w-4 h-4" />
                        </button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        )}
        {!loading && !hasRows && (
          <div className="text-center py-12">
            <Tag className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <div className="text-gray-600">No coupons found</div>
            <button 
              onClick={() => setShowModal(true)}
              className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-lg text-sm"
            >
              Create your first coupon
            </button>
          </div>
        )}
        {showModal && (
          <div className="fixed inset-0 bg-black bg-opacity-30 flex items-center justify-center z-50">
            <div className="bg-white rounded-lg p-6 w-full max-w-lg shadow-lg relative">
              <h2 className="text-xl font-semibold mb-4">{editing ? 'Edit Coupon' : 'Add Coupon'}</h2>
              <form onSubmit={handleSubmit} className="space-y-4">
                {renderFormField('code', 'Coupon Code', 'text')}
                {renderFormField('description', 'Description', 'text')}
                <div>
                  <label className="block text-sm font-medium mb-1">
                    Discount Type <span className="text-red-500">*</span>
                  </label>
                  <select
                    value={form.discount_type}
                    onChange={(e) => handleDiscountTypeChange(e.target.value as 'percentage' | 'fixed')}
                    className={`w-full border ${errors.discount_type ? 'border-red-500' : 'border-gray-300'} rounded-lg px-3 py-2`}
                  >
                    <option value="percentage">Percentage (%)</option>
                    <option value="fixed">Fixed Amount (₹)</option>
                  </select>
                  {errors.discount_type && (
                    <p className="mt-1 text-sm text-red-600">{errors.discount_type}</p>
                  )}
                </div>
                {renderFormField('discount_value', 'Discount Value', 'number', undefined, 1, form.discount_type === 'percentage' ? 100 : undefined)}
                {renderFormField('min_order_amount', 'Minimum Order Amount', 'number', undefined, 0)}
                {form.discount_type === 'percentage' && renderFormField('max_discount', 'Maximum Discount (₹)', 'number', undefined, 0)}
                {renderFormField('start_date', 'Start Date', 'date')}
                {renderFormField('end_date', 'End Date', 'date')}
                {renderFormField('usage_limit', 'Usage Limit', 'number', undefined, 0)}
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    checked={form.status}
                    onChange={(e) => handleInputChange('status', e.target.checked)}
                    className="mr-2"
                    id="coupon-status"
                  />
                  <label htmlFor="coupon-status" className="text-sm">Active Status</label>
                </div>
                <div className="flex justify-end space-x-3 pt-4">
                  <button 
                    type="button" 
                    onClick={() => {
                      setShowModal(false);
                      setEditing(null);
                      setErrors({});
                      setForm({
                        code: '',
                        description: '',
                        discount_type: 'percentage',
                        discount_value: 0,
                        min_order_amount: 0,
                        max_discount: 0,
                        start_date: new Date().toISOString().split('T')[0],
                        end_date: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000).toISOString().split('T')[0],
                        usage_limit: 100,
                        status: true
                      });
                    }} 
                    className="px-4 py-2 border rounded-lg hover:bg-gray-50 transition-colors"
                    disabled={isSubmitting}
                  >
                    Cancel
                  </button>
                  <button 
                    type="submit" 
                    className={`px-4 py-2 text-white rounded-lg flex items-center ${
                      isSubmitting 
                        ? 'bg-blue-400 cursor-not-allowed' 
                        : 'bg-blue-600 hover:bg-blue-700'
                    }`}
                    disabled={isSubmitting}
                  >
                    {isSubmitting ? (
                      <>
                        <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                        </svg>
                        {editing ? 'Updating...' : 'Creating...'}
                      </>
                    ) : (
                      <>{editing ? 'Update Coupon' : 'Create Coupon'}</>
                    )}
                  </button>
                </div>
              </form>
            </div>
          </div>
        )}
      </div>
    </Layout>
  );
};

export default Coupon;
