import React, { useState, useEffect } from 'react';
import { useQuery, useQueryClient, useMutation } from 'react-query';
import { toast } from 'react-hot-toast';
import { format } from 'date-fns';
import { Search, Package, CheckCircle, Clock, X, Check, Loader2 } from 'lucide-react';
import { useAuth } from '../../context/AuthContext';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import { pickerPackerSidebarItems } from './PickerPackerLayout';
import api from '../../services/api';

interface OrderItem {
  _id: string;
  product: string | { name: string; _id: string };
  quantity: number;
  price: number;
  name?: string;
  status?: string;
}

interface Order {
  _id: string;
  orderNumber: string;
  user: {
    name: string;
    email: string;
    phone?: string;
  };
  items: OrderItem[];
  totalAmount: number;
  status: string;
  store?: string | { name: string; _id: string };
  deliveryAddress?: {
    address: string;
    city: string;
    state: string;
    pincode: string;
  };
  paymentMethod?: string;
  prescription?: string;
  createdAt: string;
  updatedAt: string;
}

const OrdersToPick: React.FC = () => {
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
  const [assignedStore, setAssignedStore] = useState<string | null>(null);
  const queryClient = useQueryClient();
  const { user } = useAuth();

  // Set the store from the user object if available
  const [isLoadingPickerPacker, setIsLoadingPickerPacker] = useState(false);

  const isPickerPacker = () => {
    if (!user?.role) return false;
    return typeof user.role === 'string' 
      ? user.role === 'picker_packer' 
      : user.role.name === 'picker_packer';
  };
  
  const getUserId = (): string | null => {
    if (!user || !user.id) return null;
    return user.id.toString(); // Ensure we always return a string or null
  };

  useEffect(() => {
    if (user?.store?._id) {
      console.log('Setting store from user object:', user.store._id);
      setAssignedStore(user.store._id);
      setIsLoadingPickerPacker(false);
    } else if (isPickerPacker()) {
      // If no store in user object, try to fetch it
      const userId = getUserId();
      if (!userId) {
        console.error('No user ID available');
        toast.error('User session error. Please log in again.');
        return;
      }
      
      const fetchPickerPackerStore = async () => {
        try {
          const { data } = await api.get(`/picker-packers/user/${userId}`);
          if (data.success && data.data?.store) {
            const storeId = data.data.store._id || data.data.store;
            console.log('Fetched store ID from picker/packer:', storeId);
            setAssignedStore(storeId);
          } else {
            console.error('No store assigned to picker/packer');
            toast.error('You are not assigned to any store. Please contact your administrator.');
          }
        } catch (error) {
          console.error('Error fetching picker/packer store:', error);
          toast.error('Failed to load store information. Please try again.');
        }
      };
      fetchPickerPackerStore();
    }
  }, [user]);

  // Fetch orders for the picker/packer's assigned store
  const { data: orders, isLoading, error, refetch } = useQuery<Order[]>(
    ['picker-orders', searchTerm, assignedStore],
    async () => {
      if (!assignedStore) return [];
      try {
        const { data } = await api.get('/orders/picker-orders', {
          params: {
            search: searchTerm,
            limit: 50,
          },
        });
        return data.data || [];
      } catch (err) {
        console.error('Error fetching orders:', err);
        toast.error('Failed to load orders');
        return [];
      }
    },
    {
      enabled: !!assignedStore,
      refetchOnWindowFocus: false,
    }
  );

  // Mark order as 'picking' status
  const startPickingMutation = useMutation(
    async (orderId: string) => {
      await api.patch(`/orders/${orderId}`, {
        status: 'picking',
      });
    },
    {
      onSuccess: () => {
        queryClient.invalidateQueries('picker-orders');
        toast.success('Order marked as in picking');
      },
      onError: (error: any) => {
        console.error('Error updating order status:', error);
        toast.error(error.response?.data?.message || 'Failed to update order status');
      },
    }
  );

  const handleStartPicking = (orderId: string) => {
    if (window.confirm('Start picking this order?')) {
      startPickingMutation.mutate(orderId);
    }
  };

  const getProductName = (item: OrderItem) => {
    if (typeof item.product === 'object' && item.product !== null) {
      return item.product.name;
    }
    return item.name || 'Product';
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader2 className="w-8 h-8 animate-spin text-primary" />
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-4 text-red-600 bg-red-100 rounded-lg">
        Error loading orders. Please try again.
      </div>
    );
  }

  if (isLoadingPickerPacker || (isPickerPacker() && !assignedStore)) {
    return (
      <div className="flex items-center justify-center h-64">
        <Loader2 className="w-8 h-8 animate-spin text-primary" />
      </div>
    );
  }

  if (!assignedStore) {
    return (
      <div className="p-4 text-yellow-600 bg-yellow-50 rounded-lg">
        You are not assigned to any store. Please contact your administrator.
      </div>
    );
  }

  return (
    <>
      <Layout title="Orders to Pick" sidebar={
        <Sidebar items={pickerPackerSidebarItems} basePath="/pickerpacker" />
      }>
      <div className="px-4 sm:px-6 lg:px-8 py-6">
        <div className="mb-6 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Orders to Pick</h1>
            <p className="text-gray-600 mt-1">
              {orders?.length || 0} order{orders?.length !== 1 ? 's' : ''} ready for picking
            </p>
          </div>
          
          <div className="relative w-full sm:w-64">
            <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
              <Search className="h-5 w-5 text-gray-400" />
            </div>
            <input
              type="text"
              className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:ring-primary-500 focus:border-primary-500 sm:text-sm"
              placeholder="Search orders..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
          </div>
        </div>

        <div className="bg-white shadow overflow-hidden sm:rounded-lg">
          {orders?.length === 0 ? (
            <div className="text-center p-12">
              <Package className="mx-auto h-12 w-12 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">No orders to pick</h3>
              <p className="mt-1 text-sm text-gray-500">
                All caught up! Check back later for new orders.
              </p>
            </div>
          ) : (
            <div className="overflow-x-auto">
              <table className="min-w-full divide-y divide-gray-200">
                <thead className="bg-gray-50">
                  <tr>
                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order #
                    </th>
                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Customer
                    </th>
                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Items
                    </th>
                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Total
                    </th>
                    <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider whitespace-nowrap">
                      Date
                    </th>
                    <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider whitespace-nowrap">
                      Action
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {orders?.map((order) => (
                    <tr key={order._id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">#{order.orderNumber}</div>
                        {order.prescription && (
                          <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 text-blue-800">
                            Prescription
                          </span>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{order.user?.name || 'N/A'}</div>
                        <div className="text-sm text-gray-500">{order.user?.phone || order.user?.email || ''}</div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">
                          {order.items?.slice(0, 2).map((item, idx) => (
                            <div key={idx} className="truncate">
                              {item.quantity} × {getProductName(item)}
                            </div>
                          ))}
                          {order.items?.length > 2 && (
                            <div className="text-gray-500 text-xs mt-1">+{order.items.length - 2} more items</div>
                          )}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        ₹{order.totalAmount?.toFixed(2) || '0.00'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {order.createdAt ? format(new Date(order.createdAt), 'MMM d, yyyy h:mm a') : 'N/A'}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right">
                        <div className="flex justify-end">
                          <button
                            onClick={() => handleStartPicking(order._id)}
                            disabled={startPickingMutation.isLoading}
                            className="inline-flex items-center px-3 py-1.5 border border-transparent text-xs font-medium rounded-md shadow-sm text-white bg-primary-600 hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 disabled:opacity-50 disabled:cursor-not-allowed whitespace-nowrap"
                          >
                            {startPickingMutation.isLoading ? (
                              <>
                                <Loader2 className="mr-2 h-3 w-3 animate-spin" />
                                Processing...
                              </>
                            ) : (
                              <>
                                <Package className="mr-1.5 h-3.5 w-3.5" />
                                Start Picking
                              </>
                            )}
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          )}
        </div>
      </div>
      </Layout>

      {/* Order Details Modal */}
      {selectedOrder && (
        <div className="fixed inset-0 overflow-y-auto z-50">
          <div className="flex items-center justify-center min-h-screen pt-4 px-4 pb-20 text-center sm:block sm:p-0">
            <div 
              className="fixed inset-0 transition-opacity" 
              onClick={() => setSelectedOrder(null)}
            >
              <div className="absolute inset-0 bg-gray-500 opacity-75"></div>
            </div>
            <span className="hidden sm:inline-block sm:align-middle sm:h-screen">&#8203;</span>
            <div className="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-2xl sm:w-full">
              {selectedOrder && (
                <div className="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                  <div className="sm:flex sm:items-start">
                    <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left w-full">
                      <div className="flex justify-between items-center">
                        <h3 className="text-lg leading-6 font-medium text-gray-900">
                          Order #{selectedOrder.orderNumber}
                        </h3>
                        <button
                          onClick={() => setSelectedOrder(null)}
                          className="text-gray-400 hover:text-gray-500"
                        >
                          <X className="h-6 w-6" />
                        </button>
                      </div>
                      
                      <div className="mt-4">
                        <h4 className="font-medium text-gray-900">Customer Details</h4>
                        <p className="text-sm text-gray-700">{selectedOrder.user.name}</p>
                        <p className="text-sm text-gray-500">{selectedOrder.user.email}</p>
                        {selectedOrder.user.phone && (
                          <p className="text-sm text-gray-500">{selectedOrder.user.phone}</p>
                        )}
                      </div>

                      {selectedOrder.deliveryAddress && (
                        <div className="mt-4">
                          <h4 className="font-medium text-gray-900">Delivery Address</h4>
                          <p className="text-sm text-gray-700">
                            {selectedOrder.deliveryAddress.address}
                          </p>
                          <p className="text-sm text-gray-700">
                            {selectedOrder.deliveryAddress.city}, {selectedOrder.deliveryAddress.state} - {selectedOrder.deliveryAddress.pincode}
                          </p>
                        </div>
                      )}

                      <div className="mt-4">
                        <h4 className="font-medium text-gray-900">Order Items</h4>
                        <div className="mt-2 border-t border-gray-200">
                          {selectedOrder.items.map((item, idx) => (
                            <div key={idx} className="py-2 flex justify-between">
                              <div>
                                <p className="text-sm font-medium text-gray-900">{getProductName(item)}</p>
                                <p className="text-sm text-gray-500">Qty: {item.quantity}</p>
                              </div>
                              <p className="text-sm font-medium text-gray-900">
                                ₹{(item.price * item.quantity).toFixed(2)}
                              </p>
                            </div>
                          ))}
                        </div>
                        <div className="mt-4 pt-4 border-t border-gray-200">
                          <div className="flex justify-between text-base font-medium text-gray-900">
                            <p>Total</p>
                            <p>₹{selectedOrder.totalAmount.toFixed(2)}</p>
                          </div>
                        </div>
                      </div>

                      {selectedOrder.prescription && (
                        <div className="mt-4">
                          <h4 className="font-medium text-gray-900">Prescription</h4>
                          <div className="mt-2">
                            <img 
                              src={selectedOrder.prescription} 
                              alt="Prescription" 
                              className="h-48 w-auto object-contain border rounded-md"
                            />
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}
              <div className="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                <button
                  type="button"
                  onClick={() => handleStartPicking(selectedOrder?._id || '')}
                  disabled={startPickingMutation.isLoading}
                  className="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-primary-600 text-base font-medium text-white hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 sm:ml-3 sm:w-auto sm:text-sm disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {startPickingMutation.isLoading ? (
                    <>
                      <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                      Processing...
                    </>
                  ) : (
                    'Start Picking'
                  )}
                </button>
                <button
                  type="button"
                  onClick={() => setSelectedOrder(null)}
                  className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </>

    
  );
};

export default OrdersToPick;
