import React from 'react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import StatsCard from '../../components/shared/StatsCard';
import { pickerPackerSidebarItems } from './PickerPackerLayout';
import { 
  Package, 
  Clock,
  CheckCircle,
  BarChart3,
  AlertTriangle,
  ChevronRight,
  Clipboard,
  Calendar
} from 'lucide-react';

interface Order {
  id: string;
  items: number;
  priority: 'high' | 'medium' | 'low';
  timeReceived: string;
  estimatedTime: string;
  status: 'pending' | 'in_progress' | 'completed';
}

interface PerformanceMetric {
  metric: string;
  value: string;
  category: string;
  trend?: { value: number; isPositive: boolean };
}

const PickerPackerDashboard: React.FC = () => {
  const stats = [
    {
      title: 'Orders to Pick',
      value: '24',
      icon: Package,
      trend: { value: 5, isPositive: false },
      color: 'orange' as const
    },
    {
      title: 'In Progress',
      value: '8',
      icon: Clock,
      trend: { value: 2, isPositive: true },
      color: 'blue' as const
    },
    {
      title: 'Ready for Dispatch',
      value: '12',
      icon: CheckCircle,
      trend: { value: 15, isPositive: true },
      color: 'green' as const
    },
    {
      title: 'Performance',
      value: '4.2/5',
      icon: BarChart3,
      trend: { value: 8, isPositive: true },
      color: 'purple' as const
    }
  ];

  const ordersToPick: Order[] = [
    { id: '#ORD-1234', items: 5, priority: 'high', timeReceived: '10:15 AM', estimatedTime: '15 min', status: 'pending' },
    { id: '#ORD-1235', items: 3, priority: 'medium', timeReceived: '10:30 AM', estimatedTime: '10 min', status: 'in_progress' },
    { id: '#ORD-1236', items: 8, priority: 'high', timeReceived: '10:45 AM', estimatedTime: '25 min', status: 'pending' },
    { id: '#ORD-1237', items: 2, priority: 'low', timeReceived: '11:00 AM', estimatedTime: '5 min', status: 'pending' },
  ];

  const getPriorityBadge = (priority: string) => {
    const priorityClasses = {
      high: 'bg-red-100 text-red-800',
      medium: 'bg-yellow-100 text-yellow-800',
      low: 'bg-green-100 text-green-800'
    };
    return priorityClasses[priority as keyof typeof priorityClasses] || 'bg-gray-100 text-gray-800';
  };

  const getStatusBadge = (status: string) => {
    const statusClasses = {
      pending: 'bg-yellow-100 text-yellow-800',
      in_progress: 'bg-blue-100 text-blue-800',
      completed: 'bg-green-100 text-green-800'
    };
    return statusClasses[status as keyof typeof statusClasses] || 'bg-gray-100 text-gray-800';
  };

  const performanceMetrics: PerformanceMetric[] = [
    { 
      metric: 'Orders Picked Today', 
      value: '42',
      category: 'Daily Total',
      trend: { value: 12, isPositive: true }
    },
    { 
      metric: 'Average Pick Time', 
      value: '8.5 min',
      category: 'Efficiency',
      trend: { value: 1.2, isPositive: true }
    },
    { 
      metric: 'Accuracy Rate', 
      value: '99.4%',
      category: 'Quality',
      trend: { value: 0.5, isPositive: true }
    },
    { 
      metric: 'On-Time Picking', 
      value: '96%',
      category: 'Punctuality',
      trend: { value: 2, isPositive: true }
    }
  ];

  const quickActions = [
    { label: 'Start Picking', icon: Package, count: '24 orders', action: () => console.log('Start Picking') },
    { label: 'View Picking List', icon: Clipboard, count: '3 lists', action: () => console.log('View Picking List') },
    { label: 'Report Issue', icon: AlertTriangle, count: '2 issues', action: () => console.log('Report Issue') },
    { label: 'View Schedule', icon: Calendar, count: '5 shifts', action: () => console.log('View Schedule') }
  ];

  return (
    <Layout title="Picker & Packer Dashboard" sidebar={<Sidebar items={pickerPackerSidebarItems} />}>
      <div className="space-y-6">
        {/* Stats Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {stats.map((stat, index) => (
            <StatsCard
              key={index}
              title={stat.title}
              value={stat.value}
              icon={stat.icon}
              trend={stat.trend}
              color={stat.color}
            />
          ))}
        </div>

        {/* Main Content Grid */}
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Orders to Pick */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
              <h3 className="text-lg font-medium text-gray-900">Orders to Pick</h3>
              <button className="text-sm text-blue-600 hover:text-blue-800 flex items-center">
                View All <ChevronRight className="ml-1 h-4 w-4" />
              </button>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Order ID
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Items
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                      Time
                    </th>
                  </tr>
                </thead>
                <tbody className="bg-white divide-y divide-gray-200">
                  {ordersToPick.map((order) => (
                    <tr key={order.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">{order.id}</div>
                        <div className="text-xs text-gray-500">
                          <span className={`px-2 py-1 rounded-full ${getPriorityBadge(order.priority)}`}>
                            {order.priority} priority
                          </span>
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm text-gray-900">{order.items} {order.items === 1 ? 'item' : 'items'}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusBadge(order.status)}`}>
                          {order.status.replace('_', ' ')}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <div>Received: {order.timeReceived}</div>
                        <div>Est: {order.estimatedTime}</div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>

          {/* Performance Metrics */}
          <div className="bg-white rounded-lg shadow-sm border">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-medium text-gray-900">Performance Metrics</h3>
            </div>
            <div className="p-6">
              <div className="space-y-4">
                {performanceMetrics.map((metric, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-blue-50 border border-blue-200 rounded-lg">
                    <div className="flex-1">
                      <p className="text-sm font-medium text-gray-900">{metric.metric}</p>
                      <p className="text-xs text-gray-500">{metric.category}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-sm font-medium text-blue-600">
                        {metric.value}
                      </p>
                      {metric.trend && (
                        <p className={`text-xs ${metric.trend.isPositive ? 'text-green-600' : 'text-red-600'}`}>
                          {metric.trend.isPositive ? '↑' : '↓'} {metric.trend.value}%
                        </p>
                      )}
                    </div>
                  </div>
                ))}
              </div>
              <button className="w-full mt-4 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 transition-colors">
                View Full Analytics
              </button>
            </div>
          </div>
        </div>

        {/* Quick Actions */}
        <div className="bg-white rounded-lg shadow-sm border">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-medium text-gray-900">Quick Actions</h3>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              {quickActions.map((action, index) => {
                const Icon = action.icon;
                return (
                  <button
                    key={index}
                    onClick={action.action}
                    className="p-4 border border-gray-200 rounded-lg hover:border-blue-300 hover:bg-blue-50 transition-all text-left"
                  >
                    <div className="flex items-center mb-2">
                      <Icon className="w-5 h-5 text-blue-600 mr-2" />
                      <span className="text-sm font-medium text-gray-900">{action.label}</span>
                    </div>
                    <p className="text-xs text-gray-500">{action.count}</p>
                  </button>
                );
              })}
            </div>
          </div>
        </div>
      </div>
    </Layout>
  );
};

export default PickerPackerDashboard;