import { useEffect, useState } from 'react';
import { toast } from 'react-hot-toast';
import {
  CheckCircle,
  XCircle,
  Clock as ClockIcon,
  Search,
  FileText,
  User,
  ClipboardList,
  Loader2,
  ChevronLeft,
  ChevronRight
} from 'lucide-react';
import Layout from '../../components/shared/Layout';
import Sidebar from '../../components/shared/Sidebar';
import { pharmacistSidebarItems } from './PharmacistLayout';
import api from '@/services/api';
import { useAuth } from '@/context/AuthContext';
import { getImageUrl } from '../../utils/imageUtils';

interface OrderItem {
  _id: string;
  name: string;
  price: number;
  quantity: number;
  requiresPrescription: boolean;
  image?: string;
  brand?: string;
  category?: string;
  unit?: string;
}

interface Address {
  name: string;
  contactNumber?: string;
  address: string;
  landmark?: string;
  city: string;
  state: string;
  postalCode: string;
  country?: string;
  type?: string;
  isDefault?: boolean;
}

interface Prescription {
  _id: string;
  orderNumber: string;
  userId: string;
  orderId?: string | null;
  storeId?: string | null;
  user: {
    _id: string;
    name: string;
    email: string;
    phone?: string;
  };
  items: OrderItem[];
  address: Address;
  status: 'pending' | 'approved' | 'rejected' | 'fulfilled';
  paymentStatus: 'PENDING' | 'PAID' | 'FAILED' | 'REFUNDED' | 'PARTIALLY_REFUNDED' | string;
  paymentMethod: 'cod' | 'online' | 'wallet' | string;
  prescription: {
    _id?: string;
    doctor_name?: string;
    patient_name: string;
    patient_age?: number;
    patient_gender?: 'male' | 'female' | 'other' | string;
    diagnosis?: string;
    notes?: string;
    images: Array<{
      path: string;
      is_primary?: boolean;
      mimetype?: string;
      size?: number;
      uploaded_at?: string;
      original_name?: string;
      _id?: string;
    }>;
    status: 'pending' | 'approved' | 'rejected' | 'fulfilled';
    rejection_reason?: string;
    pharmacist_notes?: string;
    reviewed_by?: string | { _id: string; name: string };
    reviewed_at?: string;
    verification_history?: Array<{
      status: string;
      changed_by: string | { _id: string; name: string };
      notes: string;
      changed_at: string;
      _id?: string;
    }>;
  };
  createdAt: string | Date;
  updatedAt: string | Date;
  [key: string]: any; // Allow additional properties
}

const PrescriptionsPage = () => {
  const { user } = useAuth();
  const [orders, setOrders] = useState<Prescription[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<'all' | 'pending' | 'approved' | 'rejected' | 'fulfilled'>('pending');
  const [selectedOrder, setSelectedOrder] = useState<Prescription | null>(null);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [rejectionReason, setRejectionReason] = useState('');
  const [pharmacistNotes, setPharmacistNotes] = useState('');
  const [isUpdating, setIsUpdating] = useState(false);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);

  const openImageViewer = (order: Prescription, index: number) => {
    if (order?.prescription?.images?.length) {
      setImageViewerState({
        isOpen: true,
        currentIndex: index,
        images: order.prescription.images
      });
    }
  };

  const closeImageViewer = () => {
    setImageViewerState(prev => ({
      ...prev,
      isOpen: false
    }));
  };

  const goToNext = () => {
    setImageViewerState(prev => ({
      ...prev,
      currentIndex: (prev.currentIndex + 1) % prev.images.length
    }));
  };

  const goToPrevious = () => {
    setImageViewerState(prev => ({
      ...prev,
      currentIndex: (prev.currentIndex - 1 + prev.images.length) % prev.images.length
    }));
  };
  const [isImageLoading, setIsImageLoading] = useState(true);
  
  const handleStatusUpdate = async (status: 'approved' | 'rejected') => {
    if (!selectedOrder?._id) return;
    
    if (status === 'rejected' && !rejectionReason.trim()) {
      toast.error('Please provide a reason for rejection');
      return;
    }

    try {
      setIsUpdating(true);
      
      // First update the prescription status
      const updateData = {
        status,
        ...(status === 'rejected' && { rejectionReason }),
        ...(pharmacistNotes && { pharmacistNotes })
      };

      // Use the correct endpoint for updating prescription status
      await api.put(`/orders/${selectedOrder.orderId}/prescription/status`, updateData);
      
      // If prescription is approved, also update the order status to 'processing'
      if (status === 'approved' && selectedOrder.orderId) {
        try {
          // Use the correct endpoint for updating order status
          await api.put(`/api/v1/orders/${selectedOrder.orderId}/status`, {
            status: 'processing',
            notes: 'Prescription approved by pharmacist'
          });
        } catch (error) {
          console.error('Error updating order status:', error);
          // Don't fail the whole operation if order status update fails
          toast.error('Prescription approved, but failed to update order status');
        }
      }
      
      // Update the orders list with the updated prescription
      setOrders(prevOrders => 
        prevOrders.map(order => 
          order._id === selectedOrder._id 
            ? { 
                ...order, 
                status,
                ...(status === 'rejected' && { rejectionReason }),
                ...(pharmacistNotes && { pharmacistNotes }),
                ...(status === 'approved' && selectedOrder.orderId && { 
                  order: {
                    ...order.order,
                    status: 'processing',
                    updatedAt: new Date().toISOString()
                  }
                })
              } 
            : order
        )
      );
      
      // Update the selected order in the modal
      setSelectedOrder(prev => prev ? { 
        ...prev, 
        status,
        ...(status === 'rejected' && { rejectionReason }),
        ...(pharmacistNotes && { pharmacistNotes }),
        ...(status === 'approved' && prev.orderId && {
          order: {
            ...prev.order,
            status: 'processing',
            updatedAt: new Date().toISOString()
          }
        })
      } : null);
      
      toast.success(`Prescription ${status === 'approved' ? 'approved and order processing started' : 'rejected'} successfully`);
      
      // Close the modal after a short delay
      setTimeout(() => {
        setIsModalOpen(false);
        setRejectionReason('');
        setPharmacistNotes('');
      }, 1000);
      
    } catch (error) {
      console.error('Error updating prescription status:', error);
      toast.error('Failed to update prescription status');
    } finally {
      setIsUpdating(false);
    }
  };

  const [imageViewerState, setImageViewerState] = useState({
    isOpen: false,
    currentIndex: 0,
    images: [] as { path: string }[]
  });
  const [pagination, setPagination] = useState({
    page: 1,
    limit: 10,
    total: 0,
    totalPages: 1
  });

  const fetchOrders = async () => {
    try {
      setLoading(true);
      
      // Build query parameters
      const params: any = {
        page: pagination.page,
        limit: pagination.limit
      };
      
      if (statusFilter !== 'all') {
        // Map frontend status to backend status
        if (statusFilter === 'pending') {
          // Include all pending and verification statuses
          params.status = ['pending', 'PENDING_VERIFICATION', 'in_review', 'processing'];
        } else if (statusFilter === 'approved') {
          // Include all approved/confirmed statuses
          params.status = ['approved', 'confirmed', 'processing', 'shipped'];
        } else {
          params.status = statusFilter;
        }
      }
      
      if (searchTerm) {
        params.search = searchTerm;
      }
      
      // Fetch orders from the pharmacist's store - remove the leading /api/v1 since it's already included in the base URL
      const response = await api.get('/orders/pharmacist/store-orders', { params });
      
      if (response.data && response.data.success) {
        const { data, total, page, totalPages } = response.data;
        
        // Process the orders data to match our frontend interface
        const processedOrders = data.map((order: any) => ({
          _id: order._id,
          orderNumber: order.orderNumber,
          userId: order.user?._id || 'unknown',
          user: {
            _id: order.user?._id || 'unknown',
            name: order.user?.name || 'Unknown User',
            email: order.user?.email || '',
            phone: order.user?.phone || ''
          },
          orderId: order._id,
          storeId: order.store?._id || order.store || null,
          items: order.items?.map((item: any) => ({
            _id: item.item?._id || item.item || '',
            name: item.item?.name || 'Unknown Item',
            price: item.price || 0,
            quantity: item.quantity || 1,
            requiresPrescription: item.requiresPrescription || false,
            image: item.item?.image,
            brand: item.item?.brand?.name || item.item?.brand,
            category: item.item?.category?.name || item.item?.category,
            unit: item.item?.unit?.name || item.item?.unit
          })) || [],
          address: order.shippingAddress || {
            name: order.user?.name || 'N/A',
            address: order.shippingAddress?.address || 'N/A',
            city: order.shippingAddress?.city || 'N/A',
            state: order.shippingAddress?.state || 'N/A',
            postalCode: order.shippingAddress?.postalCode || 'N/A',
            contactNumber: order.user?.phone || ''
          },
          // Map various statuses to consistent UI statuses
          status: (() => {
            const status = order.status?.toLowerCase();
            if (['pending', 'pending_verification', 'in_review', 'processing'].includes(status)) {
              return 'pending';
            }
            if (['approved', 'confirmed', 'shipped'].includes(status)) {
              return 'approved';
            }
            return status as 'pending' | 'approved' | 'rejected' | 'fulfilled';
          })(),
          paymentStatus: order.paymentStatus as 'PENDING' | 'PAID' | 'FAILED' | 'REFUNDED' | 'PARTIALLY_REFUNDED',
          paymentMethod: order.paymentMethod as 'cod' | 'online' | 'wallet',
          prescription: {
            _id: order.prescription?._id,
            doctor_name: order.prescription?.doctorName || 'Dr. Unknown',
            patient_name: order.prescription?.patientName || 'Unknown Patient',
            patient_age: order.prescription?.patientAge,
            patient_gender: order.prescription?.patientGender,
            diagnosis: order.prescription?.diagnosis,
            notes: order.prescription?.notes,
            images: order.prescription?.images?.map((img: any) => ({
              path: img.path || img,
              is_primary: img.is_primary,
              mimetype: img.mimetype,
              size: img.size,
              uploaded_at: img.uploaded_at,
              original_name: img.original_name
            })) || [],
            status: (order.prescription?.status?.toLowerCase() as 'pending' | 'approved' | 'rejected' | 'fulfilled') || 'pending',
            rejection_reason: order.prescription?.rejectionReason,
            pharmacist_notes: order.prescription?.pharmacistNotes,
            reviewed_by: order.prescription?.reviewedBy,
            reviewed_at: order.prescription?.reviewedAt
          },
          createdAt: order.createdAt,
          updatedAt: order.updatedAt
        }));
        
        setOrders(processedOrders);
        setPagination({
          ...pagination,
          total,
          page,
          totalPages
        });
      } else {
        console.error('Failed to fetch orders:', response.data?.message || 'Unknown error');
        toast.error('Failed to fetch orders');
      }
    } catch (error) {
      console.error('Error fetching orders:', error);
      toast.error('An error occurred while fetching orders');
    } finally {
      setLoading(false);
    }
  };

  const handleStatusFilterChange = (status: 'all' | 'pending' | 'approved' | 'rejected' | 'fulfilled') => {
    setStatusFilter(status);
    // Reset to first page when changing status filter
    setPagination(prev => ({ ...prev, page: 1 }));
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    // Reset to first page when searching
    setPagination(prev => ({ ...prev, page: 1 }));
    fetchOrders();
  };

  const handlePageChange = (newPage: number) => {
    setPagination(prev => ({ ...prev, page: newPage }));
  };

  // Fetch orders when component mounts or when dependencies change
  useEffect(() => {
    fetchOrders();
  }, [statusFilter, pagination.page, searchTerm]);

  const handleOrderSelect = (order: Prescription) => {
    setSelectedOrder(order);
    setCurrentImageIndex(0);
    setRejectionReason(order.prescription?.rejection_reason || '');
    setPharmacistNotes(order.prescription?.pharmacist_notes || '');
    setIsModalOpen(true);
  };

  const handleVerifyPrescription = async (status: 'approved' | 'rejected') => {
    if (!selectedOrder?.prescription?._id) return;
    
    try {
      setIsUpdating(true);
      
      const response = await api.put(`/api/v1/orders/${selectedOrder._id}/verify-prescription`, {
        status,
        rejectionReason: status === 'rejected' ? rejectionReason : undefined,
        pharmacistNotes
      });
      
      if (response.data.success) {
        toast.success(`Prescription ${status} successfully`);
        fetchOrders();
        setIsModalOpen(false);
      } else {
        throw new Error(response.data.message || 'Failed to update prescription status');
      }
    } catch (error: any) {
      console.error('Error updating prescription status:', error);
      toast.error(error.response?.data?.message || error.message || 'Failed to update prescription status');
    } finally {
      setIsUpdating(false);
    }
  };

  const formatDate = (dateString: string) =>
    new Date(dateString).toLocaleString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });

  const getStatusBadge = (status: string) => {
    switch (status?.toLowerCase()) {
      case 'pending':
      case 'in_review':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
            <ClockIcon className="w-3 h-3 mr-1" />
            {status.charAt(0).toUpperCase() + status.slice(1).replace('_', ' ')}
          </span>
        );
      case 'approved':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
            <CheckCircle className="w-3 h-3 mr-1" />
            Approved
          </span>
        );
      case 'rejected':
        return (
          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
            <XCircle className="w-3 h-3 mr-1" />
            Rejected
          </span>
        );
      default:
        return <span className="text-gray-500">Unknown</span>;
    }
  };

  return (
    <Layout title="Prescription Management" sidebar={<Sidebar items={pharmacistSidebarItems} />}>
      <div className="space-y-6">
        {/* Header */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Prescription Management</h1>
            <p className="text-sm text-gray-500">
              Review, verify, or reject patient prescriptions
            </p>
          </div>
          <div className="flex flex-col space-y-2 sm:flex-row sm:space-x-3 sm:space-y-0 mt-4 md:mt-0">
            <div className="relative">
              <Search className="absolute left-3 top-2.5 h-4 w-4 text-gray-400" />
              <input
                type="text"
                placeholder="Search..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-9 pr-3 py-2 border border-gray-300 rounded-md text-sm focus:ring-blue-500 focus:border-blue-500"
              />
            </div>
            <select
              value={statusFilter}
              onChange={(e) => setStatusFilter(e.target.value as any)}
              className="py-2 pl-3 pr-10 text-sm border border-gray-300 rounded-md focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="all">All</option>
              <option value="pending">Pending</option>
              <option value="approved">Approved</option>
              <option value="rejected">Rejected</option>
            </select>
          </div>
        </div>

        {/* Content */}
        <div className="bg-white rounded-lg shadow-sm border">
          {loading ? (
            <div className="flex justify-center py-16">
              <div className="w-8 h-8 border-4 border-blue-500 border-t-transparent rounded-full animate-spin"></div>
            </div>
          ) : orders.length === 0 ? (
            <div className="flex flex-col items-center justify-center py-16 text-center">
              <ClipboardList className="w-10 h-10 text-gray-400" />
              <h3 className="mt-2 text-sm font-medium text-gray-900">
                No prescriptions found
              </h3>
              <p className="mt-1 text-sm text-gray-500">
                Try changing your filters or search term.
              </p>
            </div>
          ) : (
            <div className="overflow-hidden">
              <ul className="divide-y divide-gray-200">
                {orders.map((order) => (
                  <li key={order._id} className="hover:bg-gray-50 transition">
                    <div className="px-6 py-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-2">
                          <p className="font-medium text-blue-600">{order.prescription?.patient_name || 'N/A'}</p>
                          {order.prescription?.patient_age && (
                            <span className="text-xs bg-blue-100 text-blue-800 px-2.5 py-0.5 rounded-full">
                              {order.prescription.patient_age} yrs
                            </span>
                          )}
                          {order.prescription?.patient_gender && (
                            <span className="text-xs bg-purple-100 text-purple-800 px-2.5 py-0.5 rounded-full capitalize">
                              {order.prescription.patient_gender}
                            </span>
                          )}
                          <span className="text-xs bg-gray-100 text-gray-800 px-2.5 py-0.5 rounded-full">
                            Order: {order.orderNumber}
                          </span>
                        </div>
                        <div className="flex items-center space-x-2">
                          {getStatusBadge(order.status)}
                          <button
                            onClick={() => {
                              setSelectedOrder(order);
                              setRejectionReason(order.prescription?.rejection_reason || '');
                              setPharmacistNotes(order.prescription?.pharmacist_notes || '');
                              setIsModalOpen(true);
                            }}
                            className="px-3 py-1 text-sm font-medium text-white bg-blue-600 rounded-md hover:bg-blue-700"
                            disabled={order.status !== 'pending' && order.status !== 'in_review'}
                          >
                            {order.status === 'pending' || order.status === 'in_review' 
                              ? 'Review Prescription' 
                              : 'View Details'}
                          </button>
                        </div>
                      </div>

                      <div className="mt-2 flex flex-wrap items-center text-sm text-gray-500">
                        <div className="flex items-center mr-4">
                          <User className="h-4 w-4 text-gray-400 mr-1" />
                          {order.user?.name || 'N/A'}
                        </div>
                        <div className="flex items-center mr-4">
                          <FileText className="h-4 w-4 text-gray-400 mr-1" />
                          {order.prescription?.images?.length || 0} file(s)
                        </div>
                        <div className="flex items-center">
                          <ClockIcon className="h-4 w-4 text-gray-400 mr-1" />
                          {formatDate(order.createdAt)}
                        </div>
                      </div>
                    </div>
                  </li>
                ))}
              </ul>
            </div>
          )}
        </div>

        {/* Modal */}
        {isModalOpen && selectedOrder && (
          <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/50 p-4">
            <div className="bg-white rounded-lg shadow-xl w-full max-w-3xl p-6">
              <div className="flex justify-between items-start mb-4">
                <div>
                  <h2 className="text-xl font-semibold text-gray-900">
                    Prescription Details
                  </h2>
                  <p className="text-sm text-gray-600">
                    Order: {selectedOrder.orderNumber}
                  </p>
                  <p className="text-sm text-gray-600">
                    Patient: {selectedOrder.prescription?.patient_name || 'N/A'}{' '}
                    {selectedOrder.prescription?.patient_age && `(${selectedOrder.prescription.patient_age} yrs${selectedOrder.prescription.patient_gender ? `, ${selectedOrder.prescription.patient_gender}` : ''})`}
                  </p>
                  <p className="text-sm text-gray-600">
                    Doctor: {selectedOrder.prescription?.doctor_name || 'Not specified'}
                  </p>
                </div>
                <div className="text-right">
                  <p className="text-sm text-gray-500">Status</p>
                  {getStatusBadge(selectedOrder.status)}
                </div>
              </div>

              {(selectedOrder.prescription?.notes || selectedOrder.prescription?.diagnosis) && (
                <div className="mb-4 space-y-2">
                  {selectedOrder.prescription?.diagnosis && (
                    <div>
                      <h3 className="text-sm font-medium text-gray-500">Diagnosis</h3>
                      <p className="text-sm text-gray-800 mt-1">{selectedOrder.prescription.diagnosis}</p>
                    </div>
                  )}
                  {selectedOrder.prescription?.notes && (
                    <div>
                      <h3 className="text-sm font-medium text-gray-500">Doctor's Notes</h3>
                      <p className="text-sm text-gray-800 mt-1">{selectedOrder.prescription.notes}</p>
                    </div>
                  )}
                </div>
              )}

              <div className="mt-4">
                <h3 className="text-sm font-medium text-gray-500 mb-2">Prescription Images</h3>
                <div className="grid sm:grid-cols-2 gap-3">
                  {selectedOrder.prescription?.images?.length > 0 ? (
                    selectedOrder.prescription.images.map((img, i) => (
                      <div 
                        key={i} 
                        className="border rounded-lg overflow-hidden bg-gray-50 hover:shadow-md transition-shadow cursor-pointer"
                        onClick={() => openImageViewer(selectedOrder, i)}
                      >
                        <div className="relative w-full h-48 flex items-center justify-center p-2">
                          <img
                            src={getImageUrl(img.path)}
                            alt={`Prescription ${i + 1}`}
                            className="max-h-full max-w-full object-contain"
                            onLoad={() => setIsImageLoading(false)}
                            onError={(e) => {
                              console.error('Error loading image:', img.path);
                              const target = e.target as HTMLImageElement;
                              target.src = '/images/placeholder-prescription.jpg';
                              toast.error('Failed to load image');
                            }}
                          />
                          {isImageLoading && (
                            <div className="absolute inset-0 flex items-center justify-center bg-white bg-opacity-50">
                              <Loader2 className="animate-spin h-8 w-8 text-blue-500" />
                            </div>
                          )}
                        </div>
                        {img.is_primary && (
                          <div className="text-xs text-center py-1 bg-blue-50 text-blue-700 border-t">
                            Primary Image
                          </div>
                        )}
                      </div>
                    ))
                  ) : (
                    <div className="col-span-2 py-8 text-center text-gray-500">
                      No prescription images available
                    </div>
                  )}
                </div>
              </div>

              {(selectedOrder.prescription?.status === 'pending' || selectedOrder.prescription?.status === 'in_review') && (
                <div className="mt-5 space-y-3">
                  <div className="space-y-2">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">
                        Pharmacist Notes
                      </label>
                      <textarea
                        placeholder="Add any notes or instructions for this prescription..."
                        className="w-full border rounded-md p-2 text-sm focus:ring-blue-500 focus:border-blue-500"
                        rows={3}
                        value={pharmacistNotes}
                        onChange={(e) => setPharmacistNotes(e.target.value)}
                      />
                    </div>
                    <div className="flex flex-col sm:flex-row sm:space-x-3 space-y-2 sm:space-y-0">
                      <button
                        onClick={() => handleStatusUpdate('approved')}
                        disabled={isUpdating}
                        className="flex-1 flex items-center justify-center px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700"
                      >
                        <CheckCircle className="w-5 h-5 mr-2" />
                        {isUpdating ? 'Approving...' : 'Approve'}
                      </button>
                      <div className="flex flex-col space-y-2">
                        <div className="flex w-full">
                          <input
                            placeholder="Reason for rejection (required)"
                            type="text"
                            className="flex-1 min-w-0 block w-full px-3 py-2 rounded-l-md border border-gray-300 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm"
                            value={rejectionReason}
                            onChange={(e) => setRejectionReason(e.target.value)}
                            disabled={isUpdating}
                          />
                          <button
                            onClick={() => handleStatusUpdate('rejected')}
                            disabled={isUpdating || !rejectionReason.trim()}
                            className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-r-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 disabled:opacity-50 disabled:cursor-not-allowed"
                          >
                            <XCircle className="w-5 h-5 mr-2" />
                            {isUpdating ? 'Rejecting...' : 'Reject'}
                          </button>
                        </div>
                        <p className="text-xs text-gray-500">
                          Please provide a clear reason for rejection that will be shared with the customer.
                        </p>
                      </div>
                    </div>
                  </div>
                </div>
              )}

              <div className="mt-6 flex justify-end">
                <button
                  onClick={() => setIsModalOpen(false)}
                  className="px-4 py-2 text-sm font-medium bg-gray-100 rounded-md hover:bg-gray-200"
                >
                  Close
                </button>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Image Viewer Modal */}
      {imageViewerState.isOpen && imageViewerState.images.length > 0 && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/90 p-4">
          <button 
            onClick={closeImageViewer}
            className="absolute top-4 right-4 text-white hover:text-gray-300"
            aria-label="Close"
          >
            <XCircle className="h-8 w-8" />
          </button>
          
          <div className="relative w-full max-w-4xl">
            <button 
              onClick={goToPrevious}
              className="absolute left-4 top-1/2 -translate-y-1/2 bg-black/50 text-white p-2 rounded-full hover:bg-black/70 transition-colors"
              aria-label="Previous image"
            >
              <ChevronLeft className="h-6 w-6" />
            </button>
            
            <div className="flex justify-center items-center max-h-[80vh]">
              <img
                src={getImageUrl(imageViewerState.images[imageViewerState.currentIndex]?.path)}
                alt={`Prescription ${imageViewerState.currentIndex + 1}`}
                className="max-h-full max-w-full object-contain"
              />
            </div>
            
            <button 
              onClick={goToNext}
              className="absolute right-4 top-1/2 -translate-y-1/2 bg-black/50 text-white p-2 rounded-full hover:bg-black/70 transition-colors"
              aria-label="Next image"
            >
              <ChevronRight className="h-6 w-6" />
            </button>
            
            <div className="absolute bottom-4 left-0 right-0 text-center text-white text-sm">
              {imageViewerState.currentIndex + 1} / {imageViewerState.images.length}
            </div>
          </div>
        </div>
      )}
    </Layout>
  );
};

export default PrescriptionsPage;
