import { useState, useEffect, useCallback } from 'react';
import { toast } from 'react-hot-toast';
import api from '../services/api';

interface Store {
  _id: string;
  name: string;
  storeName: string;
  distanceKm?: number;
  address?: {
    city?: string;
    state?: string;
    country?: string;
    formattedAddress?: string;
  };
  store?: Store; // For nested store objects
}

interface Location {
  lat: number;
  lng: number;
  address?: string;
  city?: string;
  state?: string;
  country?: string;
  pincode?: string;
  formattedAddress?: string;
}

interface UseLocationProps {
  onLocationChange?: (location: Location) => void;
  onError?: (error: Error) => void;
}

const useLocation = ({ onLocationChange, onError }: UseLocationProps = {}) => {
  const [location, setLocation] = useState<Location | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<Error | null>(null);
  const [nearbyStores, setNearbyStores] = useState<Store[]>([]);
  const [selectedStore, setSelectedStore] = useState<Store | null>(null);

  // Load saved location from localStorage on mount
  useEffect(() => {
    const savedLocation = localStorage.getItem('userLocation');
    const savedStore = localStorage.getItem('selectedStore');
    
    if (savedLocation) {
      try {
        const parsedLocation = JSON.parse(savedLocation);
        setLocation(parsedLocation);
        onLocationChange?.(parsedLocation);
      } catch (e) {
        console.error('Failed to parse saved location', e);
      }
    }

    if (savedStore) {
      try {
        setSelectedStore(JSON.parse(savedStore));
      } catch (e) {
        console.error('Failed to parse saved store', e);
      }
    }
  }, [onLocationChange]);

  // Detect user's current location using browser geolocation
  const detectCurrentLocation = useCallback(async () => {
    if (!navigator.geolocation) {
      const err = new Error('Geolocation is not supported by your browser');
      setError(err);
      onError?.(err);
      return null;
    }

    setLoading(true);
    setError(null);

    return new Promise<Location>((resolve, reject) => {
      navigator.geolocation.getCurrentPosition(
        async (position) => {
          try {
            const { latitude: lat, longitude: lng } = position.coords;
            const locationData = await reverseGeocode(lat, lng);
            
            setLocation(locationData);
            localStorage.setItem('userLocation', JSON.stringify(locationData));
            onLocationChange?.(locationData);
            
            // Find nearby stores and auto-select the nearest one
            const stores = await findNearbyStores(lat, lng);
            if (stores.length > 0) {
              const nearestStore = stores.sort((a, b) => 
                (a.distanceKm || Number.MAX_SAFE_INTEGER) - (b.distanceKm || Number.MAX_SAFE_INTEGER)
              )[0];
              selectStore(nearestStore);
            }
            
            resolve(locationData);
          } catch (err) {
            const error = err instanceof Error ? err : new Error('Failed to get location');
            setError(error);
            onError?.(error);
            reject(error);
          } finally {
            setLoading(false);
          }
        },
        (error) => {
          const err = new Error(
            error.code === error.PERMISSION_DENIED
              ? 'Location access was denied. Please enable location services.'
              : 'Unable to retrieve your location'
          );
          setError(err);
          onError?.(err);
          setLoading(false);
          reject(err);
        },
        {
          enableHighAccuracy: true,
          timeout: 10000,
          maximumAge: 0
        }
      );
    });
  }, [onLocationChange, onError]);

  // Reverse geocode coordinates to address
  const reverseGeocode = async (lat: number, lng: number): Promise<Location> => {
    try {
      const response = await api.get('/location/geocode', {
        params: { lat, lng }
      });

      const { data } = response.data;
      
      return {
        lat,
        lng,
        address: data.formatted,
        city: data.city,
        state: data.state,
        country: data.country,
        pincode: data.postalCode,
        formattedAddress: data.formatted
      };
    } catch (error) {
      console.error('Reverse geocoding failed:', error);
      throw new Error('Could not determine your address. Please try again or enter it manually.');
    }
  };

  // Find stores near a location and automatically select the nearest one
  const findNearbyStores = async (lat: number, lng: number, maxDistance = 3000): Promise<Store[]> => {
    try {
      const response = await api.get<{ data: Store[] }>('/location/nearby-stores', {
        params: { lat, lng, maxDistance }
      });

      const stores = response.data?.data || [];
      setNearbyStores(stores);
      
      // If there are stores, automatically select the nearest one
      if (stores.length > 0) {
        const nearestStore = [...stores].sort((a: Store, b: Store) => 
          (a.distanceKm ?? Number.MAX_SAFE_INTEGER) - (b.distanceKm ?? Number.MAX_SAFE_INTEGER)
        )[0];
        
        // Only update if we don't have a selected store or if the nearest store is different
        if (!selectedStore || selectedStore._id !== nearestStore._id) {
          selectStore(nearestStore);
        }
      }
      
      return stores;
    } catch (error) {
      console.error('Failed to find nearby stores:', error);
      toast.error('Failed to find nearby stores');
      return [];
    }
  };

  // Select a store
  const selectStore = (store: Store | null) => {
    if (!store) return;
    
    const storeData = 'store' in store ? store.store : store; // Handle both store objects and direct store data
    if (!storeData) return;
    
    setSelectedStore(storeData);
    localStorage.setItem('selectedStore', JSON.stringify(storeData));
    
    // Show success message if not in auto-selection mode
    if (storeData.storeName) {
      toast.success(`Selected store: ${storeData.storeName}`);
    }
  };

  // Update location manually
  const updateLocation = async (newLocation: Partial<Location>) => {
    try {
      setLoading(true);
      setError(null);
      
      // If we have coordinates, use them to get full address
      if (newLocation.lat && newLocation.lng) {
        const locationData = await reverseGeocode(newLocation.lat, newLocation.lng);
        setLocation(locationData);
        localStorage.setItem('userLocation', JSON.stringify(locationData));
        onLocationChange?.(locationData);
        await findNearbyStores(newLocation.lat, newLocation.lng);
        return locationData;
      }
      
      // Otherwise, just update with provided data
      const updatedLocation = { ...location, ...newLocation } as Location;
      setLocation(updatedLocation);
      localStorage.setItem('userLocation', JSON.stringify(updatedLocation));
      onLocationChange?.(updatedLocation);
      return updatedLocation;
    } catch (error) {
      const err = error instanceof Error ? error : new Error('Failed to update location');
      setError(err);
      onError?.(err);
      throw err;
    } finally {
      setLoading(false);
    }
  };

  // Clear location
  const clearLocation = () => {
    setLocation(null);
    setSelectedStore(null);
    setNearbyStores([]);
    localStorage.removeItem('userLocation');
    localStorage.removeItem('selectedStore');
  };

  return {
    location,
    loading,
    error,
    nearbyStores,
    selectedStore,
    detectCurrentLocation,
    updateLocation,
    clearLocation,
    selectStore,
    findNearbyStores
  };
};

export default useLocation;
