import { createContext, useContext, useEffect, useState, useCallback } from 'react';
import { useLocation } from 'react-router-dom';
import api from '../services/api';

interface WebSettings {
  siteName: string;
  siteLogo: string;
  siteStoreLogo: string;
  maintenanceMode: boolean;
  siteDescription?: string;
  siteEmail?: string;
  sitePhone?: string;
  siteAddress?: string;
}

interface WebSettingsContextType {
  settings: WebSettings | null;
  loading: boolean;
  error: Error | null;
  refreshSettings: () => Promise<void>;
  isMaintenanceMode: boolean;
}

const WebSettingsContext = createContext<WebSettingsContextType | undefined>(undefined);

export const WebSettingsProvider: React.FC<{children: React.ReactNode}> = ({ children }) => {
  const [settings, setSettings] = useState<WebSettings | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<Error | null>(null);
  const location = useLocation();

  const fetchSettings = useCallback(async () => {
    try {
      setLoading(true);
      const response = await api.get('/web-settings');
      setSettings(response.data.data);
      setError(null);
    } catch (err) {
      console.error('Failed to fetch web settings:', err);
      setError(err instanceof Error ? err : new Error('Failed to load settings'));
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    fetchSettings();
  }, [fetchSettings]);

  // Check if maintenance mode is enabled and current route is not in the allowed list
  const isMaintenanceMode = useCallback(() => {
    if (!settings?.maintenanceMode) return false;
    
    // Routes that are allowed during maintenance
    const allowedRoutes = [
      '/admin',
      '/admin/*',
      '/login',
      '/api/*',
      '/maintenance',
    ];

    // Check if current path matches any allowed route
    return !allowedRoutes.some(route => {
      const regex = new RegExp(`^${route.replace('*', '.*')}$`);
      return regex.test(location.pathname);
    });
  }, [settings?.maintenanceMode, location.pathname]);

  return (
    <WebSettingsContext.Provider 
      value={{
        settings,
        loading,
        error,
        refreshSettings: fetchSettings,
        isMaintenanceMode: isMaintenanceMode()
      }}
    >
      {children}
    </WebSettingsContext.Provider>
  );
};

export const useWebSettings = (): WebSettingsContextType => {
  const context = useContext(WebSettingsContext);
  if (context === undefined) {
    throw new Error('useWebSettings must be used within a WebSettingsProvider');
  }
  return context;
};

export default WebSettingsContext;
