import { createContext, useContext, useState, useEffect, ReactNode } from 'react';

interface Store {
  _id: string;
  storeName: string;
  address?: string;
  city?: string;
  state?: string;
  pincode?: string;
  location?: {
    type: string;
    coordinates: [number, number];
  };
  distanceKm?: number;
  [key: string]: any;
}

interface StoreContextType {
  selectedStore: Store | null;
  setSelectedStore: (store: Store | null) => void;
  loading: boolean;
  error: string | null;
  updateStoreLocation: (lat: number, lng: number) => Promise<void>;
  clearStore: () => void;
}

const StoreContext = createContext<StoreContextType | undefined>(undefined);

export const StoreProvider = ({ children }: { children: ReactNode }) => {
  const [selectedStore, setSelectedStoreState] = useState<Store | null>(null);
  const [loading, setLoading] = useState<boolean>(false);
  const [error, setError] = useState<string | null>(null);

  // Load store from localStorage on initial render
  useEffect(() => {
    const loadStore = () => {
      try {
        const storedStore = localStorage.getItem('selectedStore');
        if (storedStore) {
          setSelectedStoreState(JSON.parse(storedStore));
        }
      } catch (err) {
        console.error('Failed to load store from localStorage:', err);
        localStorage.removeItem('selectedStore');
      }
    };

    loadStore();
  }, []);

  // Update localStorage when selectedStore changes
  useEffect(() => {
    if (selectedStore) {
      localStorage.setItem('selectedStore', JSON.stringify(selectedStore));
    } else {
      localStorage.removeItem('selectedStore');
    }
  }, [selectedStore]);

  const updateStoreLocation = async (lat: number, lng: number) => {
    setLoading(true);
    setError(null);
    
    try {
      // Find nearest store
      const response = await fetch(`/api/stores/nearby?lat=${lat}&lng=${lng}`);
      const data = await response.json();

      if (!response.ok) {
        throw new Error(data.message || 'Failed to find nearby stores');
      }

      if (data.stores && data.stores.length > 0) {
        // Select the nearest store (first in the array)
        const nearestStore = data.stores[0];
        setSelectedStoreState(nearestStore);
      } else {
        throw new Error('No stores found near your location');
      }
    } catch (err) {
      console.error('Error updating store location:', err);
      setError(err instanceof Error ? err.message : 'Failed to update store location');
      throw err; // Re-throw to allow components to handle the error
    } finally {
      setLoading(false);
    }
  };

  const clearStore = () => {
    setSelectedStoreState(null);
  };

  const setSelectedStore = (store: Store | null) => {
    setSelectedStoreState(store);
  };

  return (
    <StoreContext.Provider 
      value={{
        selectedStore,
        setSelectedStore,
        loading,
        error,
        updateStoreLocation,
        clearStore
      }}
    >
      {children}
    </StoreContext.Provider>
  );
};

export const useStore = (): StoreContextType => {
  const context = useContext(StoreContext);
  if (context === undefined) {
    throw new Error('useStore must be used within a StoreProvider');
  }
  return context;
};

export default StoreContext;
