import React, { createContext, useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '../services/api';
import { getDashboardRoute } from '../utils/routes';

// Define the StoreManager type first
type StoreManager = {
  _id?: string;
  $oid?: string;
  // Add other store manager fields as needed
};

interface User {
  id: number;
  email: string;
  name: string;
  _id?: string;  // MongoDB _id
  store_id?: string;
  storeManager?: string | StoreManager; // Can be string ID or object with _id/$oid
  role: {
    id: number;
    name: string;
    permissions: string[];
  };
  phone?: string;
  address?: string;
  isEmailVerified?: boolean;
  status: 'active' | 'inactive';
  created_at: string;
  updatedAt?: string;
  avatar?: string | { public_id?: string; url?: string };
  // Add other fields that might be present in the user object
  [key: string]: any; // Allow additional properties
}

interface AuthContextType {
  user: User | null;
  login: (email: string, password: string) => Promise<void>;
  register: (userData: RegisterData) => Promise<{ user: User; token: string }>;
  sendOtp: (phone: string) => Promise<{ 
    success: boolean; 
    message: string; 
    otp?: string; 
  }>;
  verifyOtp: (phone: string, otp: string, userData?: Partial<RegisterData>) => Promise<{ 
    user: User; 
    token: string;
    isNewUser: boolean;
  }>;
  logout: () => Promise<void>;
  loading: boolean;
  error: string | null;
  clearError: () => void;
}

interface RegisterData {
  name: string;
  email: string;
  password: string;
  passwordConfirm: string;
  phone?: string;
  address?: string;
}

export const AuthContext = createContext<AuthContextType | null>(null);

export const useAuth = () => {
  const context = React.useContext(AuthContext);
  if (context === null) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};

export const AuthProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  // Initialize user state from localStorage if available (for HMR persistence)
  const [user, setUser] = useState<User | null>(() => {
    const savedUser = localStorage.getItem('user');
    return savedUser ? JSON.parse(savedUser) : null;
  });
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const navigate = useNavigate();

  // Load user on mount and preserve state during HMR
  useEffect(() => {
    const checkAuth = async () => {
      const token = localStorage.getItem('token');
      if (!token) {
        // Clear any stale user data if no token
        localStorage.removeItem('user');
        setUser(null);
        setLoading(false);
        return;
      }

      // If we already have user data from localStorage and we're in development,
      // skip the auth check to prevent logout during HMR
      const isDevelopment = import.meta.env.DEV;
      if (isDevelopment && user) {
        setLoading(false);
        return;
      }

      try {
        const response = await api.get('/auth/me');
        
        if (!response.data) {
          throw new Error('No response data received');
        }
        
        // The backend returns the user directly, not nested in a user property
        const user = response.data.data || response.data.user || response.data;
        
        if (!user) {
          throw new Error('Invalid user data received');
        }
        
        // Ensure the user is active (if status field exists)
        if (user.status && user.status !== 'active') {
          throw new Error('Your account is not active');
        }
        
        console.log('AuthContext - Setting user:', user);
        setUser(user);
        // Save user to localStorage for HMR persistence
        localStorage.setItem('user', JSON.stringify(user));
      } catch (err: any) {
        console.error('Auth check failed:', err);
        
        // Don't remove token during development if it's a network error
        // Only remove token for actual auth failures (401, 403)
        if (err.response?.status === 401 || err.response?.status === 403) {
          localStorage.removeItem('token');
          localStorage.removeItem('user');
          setUser(null);
        } else {
          // For network errors or other issues, keep the token but show error
          console.warn('Auth check failed due to network/server error, keeping token');
        }
        
        // Only show error if it's not a 401 (unauthorized) error
        if (err.response?.status !== 401) {
          setError(err.response?.data?.message || 'Failed to verify authentication');
        }
      } finally {
        setLoading(false);
      }
    };

    checkAuth();
  }, []);
  const sendOtp = async (phone: string) => {
    setLoading(true);
    setError(null);
    try {
      console.log('Sending OTP to:', phone);
      const response = await api.post('auth/send-otp', { phone });
      
      console.log('OTP Response:', response.data);
      
      if (!response.data) {
        throw new Error('No response received from server');
      }
      
      // In development, log the OTP for testing
      if (import.meta.env.DEV && response.data.otp) {
        console.log(`[DEV] OTP for ${phone}: ${response.data.otp}`);
      }
      
      return {
        success: true,
        message: response.data.message || 'OTP sent successfully',
        // Pass through the OTP in development for testing
        ...(import.meta.env.DEV && { otp: response.data.otp })
      };
    } catch (err: any) {
      console.error('Error in sendOtp:', err);
      const errorMessage = err.response?.data?.message || 
                         err.message || 
                         'Failed to send OTP. Please try again.';
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const verifyOtp = async (phone: string, otp: string, userData?: Partial<RegisterData>) => {
    setLoading(true);
    setError(null);
    try {
      console.log('Verifying OTP for phone:', phone);
      
      const payload = { phone, otp };
      
      // Include user data if provided (for registration)
      if (userData) {
        Object.assign(payload, userData);
        console.log('Including user data in OTP verification:', userData);
      }
      
      console.log('Sending OTP verification request with payload:', { ...payload, otp: '******' });
      
      const response = await api.post('auth/verify-otp', payload);
      console.log('OTP verification response:', response.data);
      
      if (!response.data || !response.data.token) {
        throw new Error('Invalid response from server');
      }
      
      const { token, user, isNewUser = false } = response.data;
      
      if (!user) {
        throw new Error('No user data received');
      }
      
      // Store token and user data
      localStorage.setItem('token', token);
      localStorage.setItem('user', JSON.stringify(user));
      
      // Update auth state
      setUser(user);
      
      // Redirect to dashboard based on user role
      const dashboardPath = getDashboardRoute(user.role?.name || 'customer');
      console.log('Redirecting to:', dashboardPath);
      navigate(dashboardPath);
      
      return { 
        user, 
        token, 
        isNewUser 
      };
    } catch (err: any) {
      const errorMessage = err.response?.data?.message || 'OTP verification failed. Please try again.';
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const register = async (userData: RegisterData) => {
    setLoading(true);
    setError(null);
    try {
      // Check if passwords match
      if (userData.password !== userData.passwordConfirm) {
        throw new Error('Passwords do not match');
      }

      // Send all user data including passwordConfirm
      const response = await api.post('/auth/register', userData);
      
      if (!response.data || !response.data.token) {
        throw new Error('Registration failed. Please try again.');
      }

      const { token, user } = response.data;
      
      localStorage.setItem('token', token);
      localStorage.setItem('user', JSON.stringify(user));
      
      setUser(user);
      
      // Redirect to dashboard based on user role
      const dashboardPath = getDashboardRoute(user.role?.name || 'customer');
      navigate(dashboardPath);
      
      return { user, token };
    } catch (err: any) {
      const errorMessage = err.response?.data?.message || 'Registration failed. Please try again.';
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const logout = async () => {
    try {
      // Remove tokens and user data from storage
      localStorage.removeItem('token');
      localStorage.removeItem('user');
      sessionStorage.removeItem('token');
      
      // Clear the user state
      setUser(null);
      
      // Optional: Call backend to invalidate the token
      try {
        await api.post('/auth/logout');
      } catch (err) {
        console.warn('Logout API call failed, but continuing with client-side cleanup', err);
      }
      
      // Redirect to home page or login page
      navigate('/');
    } catch (err) {
      console.error('Error during logout:', err);
      // Even if there's an error, we still want to clear the local state
      localStorage.removeItem('token');
      localStorage.removeItem('user');
      sessionStorage.removeItem('token');
      setUser(null);
      navigate('/');
    }
  };

  const login = async (email: string, password: string) => {
    setLoading(true);
    setError(null);
    try {
      console.log('Attempting login with:', { email });
      
      // Clear any existing tokens to ensure a clean state
      localStorage.removeItem('token');
      sessionStorage.removeItem('token');
      
      // Make the login request
      const response = await api.post('/auth/login', { email, password });
      
      console.log('Login response received');
      
      if (!response.data) {
        throw new Error('No response data received from server');
      }
      
      // Log the full response data for debugging (excluding sensitive info)
      const responseToLog = { ...response.data };
      if (responseToLog.token) responseToLog.token = '***';
      if (responseToLog.accessToken) responseToLog.accessToken = '***';
      console.log('Login response data:', JSON.stringify(responseToLog, null, 2));
      
      // Handle different response formats
      let token: string | undefined;
      let user: any;
      
      // Check for different response formats
      if (response.data.token) {
        // Format 1: { token: '...', user: {...} }
        token = response.data.token;
        user = response.data.user || response.data.data;
      } else if (response.data.accessToken) {
        // Format 2: { accessToken: '...', user: {...} }
        token = response.data.accessToken;
        user = response.data.user || response.data.data;
      } else if (response.data.data?.token) {
        // Format 3: { data: { token: '...', user: {...} } }
        token = response.data.data.token;
        user = response.data.data.user;
      } else {
        throw new Error('No authentication token received in response');
      }
      
      if (!token) {
        throw new Error('Authentication token is empty');
      }
      
      if (!user) {
        throw new Error('User data not found in response');
      }
      
      console.log('Login successful, user role:', user?.role?.name || 'unknown');
      
      // Store the token securely
      try {
        localStorage.setItem('token', token);
        console.log('Token stored in localStorage');
      } catch (storageError) {
        console.warn('Failed to store token in localStorage, falling back to sessionStorage', storageError);
        sessionStorage.setItem('token', token);
      }
      
      // Store user data
      localStorage.setItem('user', JSON.stringify(user));
      setUser(user);
      
      // Redirect to dashboard based on user role
      const dashboardPath = getDashboardRoute(user.role?.name || 'customer');
      navigate(dashboardPath);
      
    } catch (err: any) {
      const errorMessage = err.response?.data?.message || 'Login failed. Please check your credentials.';
      setError(errorMessage);
      throw new Error(errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const clearError = () => {
    setError(null);
  };

  return (
    <AuthContext.Provider
      value={{
        user,
        login,
        register,
        sendOtp,
        verifyOtp,
        logout,
        loading,
        error,
        clearError,
      }}
    >  {children}
    </AuthContext.Provider>
  );
};