import React from 'react';
import { Link, useLocation } from 'react-router-dom';

export interface SidebarItem {
  name: string;
  href: string;
  icon: React.ComponentType<{ className?: string }>;
  current?: boolean;
  children?: SidebarItem[];
}

interface SidebarProps {
  items: SidebarItem[];
}

const Sidebar: React.FC<SidebarProps> = ({ items }) => {
  const location = useLocation();

  const isCurrentPath = (href: string) => {
    // Normalize paths to handle trailing slashes
    const currentPath = location.pathname.endsWith('/') 
      ? location.pathname.slice(0, -1) 
      : location.pathname;
    
    const normalizedHref = href.endsWith('/') ? href.slice(0, -1) : href;
    
    // Check for exact match or if current path starts with the href (for nested routes)
    if (currentPath === normalizedHref || 
        (currentPath.startsWith(normalizedHref) && 
         (currentPath[normalizedHref.length] === '/' || currentPath[normalizedHref.length] === undefined))) {
      return true;
    }
    
    // Check if any child path matches
    if (items.some(item => 
      item.children?.some(child => 
        child.href === currentPath || 
        (child.href.endsWith('/') ? child.href.slice(0, -1) : child.href) === currentPath
      )
    ) && normalizedHref === '/admin/delivery') {
      return true;
    }
    
    return false;
  };

  return (
    <div className="flex flex-col h-full">
      <nav className="flex-1 px-4 py-6 space-y-2">
        {items.map((item) => {
          const isCurrent = isCurrentPath(item.href);
          const Icon = item.icon;
          
          return (
            <div key={item.name}>
              <Link
                to={item.href}
                className={`flex items-center px-3 py-2 text-sm font-medium rounded-md transition-colors ${
                  isCurrent
                    ? 'bg-blue-100 text-blue-700 font-semibold'
                    : 'text-gray-600 hover:bg-gray-100 hover:text-gray-900'
                }`}
              >
                <Icon className="w-5 h-5 mr-3" />
                {item.name}
              </Link>
              
              {item.children && (isCurrent || 
                (item.href === '/admin/delivery' && location.pathname.startsWith('/admin/delivery/'))) && (
                <div className="ml-8 mt-2 space-y-1">
                  {item.children.map((child) => {
                    const isChildCurrent = isCurrentPath(child.href);
                    const ChildIcon = child.icon;
                    
                    return (
                      <Link
                        key={child.name}
                        to={child.href}
                        className={`flex items-center px-3 py-1.5 text-sm rounded transition-colors ${
                          isChildCurrent
                            ? 'bg-blue-50 text-blue-600 font-medium'
                            : 'text-gray-600 hover:bg-gray-50 hover:text-gray-900'
                        }`}
                      >
                        <ChildIcon className="w-4 h-4 mr-2" />
                        {child.name}
                      </Link>
                    );
                  })}
                </div>
              )}
            </div>
          );
        })}
      </nav>
    </div>
  );
};

export default Sidebar;