import React, { useState, useEffect, useCallback } from 'react';
import { X, MapPin, Loader2, Check } from 'lucide-react';
import { toast } from 'react-hot-toast';
import api from '../../services/api';

interface LocationData {
  _id?: string;
  name: string;
  address: {
    display: string;
    line1: string;
    line2?: string;
    city: string;
    state: string;
    country: string;
    postalCode: string;
    coordinates: [number, number]; // [lng, lat]
  };
  deliveryRadius: number; // in km
  openingHours?: any;
  isActive: boolean;
  phone?: string;
  email?: string;
  createdAt?: string;
  updatedAt?: string;
}

interface SearchResult {
  display_name: string;
  address: {
    city?: string;
    state?: string;
    country?: string;
    postcode?: string;
    road?: string;
    suburb?: string;
  };
  lat: string;
  lon: string;
}

interface LocationModalProps {
  isOpen: boolean;
  onClose: () => void;
  onLocationSelect: (location: any) => void;
}

const LocationModal: React.FC<LocationModalProps> = ({ isOpen: initialIsOpen = false, onClose, onLocationSelect }) => {
    const [internalIsOpen, setInternalIsOpen] = useState(initialIsOpen);

  // Sync internal state with prop
  useEffect(() => {
    setInternalIsOpen(initialIsOpen);
  }, [initialIsOpen]);
  const [nearbyStores, setNearbyStores] = useState<Array<{
    _id: string;
    store: {
      storeName: string;
    };
    distanceKm: number;
  }>>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState<SearchResult[]>([]);
  const [selectedStore, setSelectedStore] = useState<LocationData | null>(null);
  const [error, setError] = useState<string | null>(null);
  
  // Check for saved location on mount and when modal opens
  useEffect(() => {
    if (!initialIsOpen) return;
    
    const savedLocation = localStorage.getItem('userLocation');
    const savedStore = localStorage.getItem('selectedStore');
    
    if (savedLocation && savedStore) {
      try {
        const store = JSON.parse(savedStore);
        setSelectedStore(store);
        // Don't call onLocationSelect here as it might close the modal
      } catch (e) {
        console.error('Error parsing saved store:', e);
      }
    }
    
    // Always show the modal when the location section is clicked
    setInternalIsOpen(true);
  }, [onLocationSelect, initialIsOpen]);

  const findStoresNearby = useCallback(async (lat: number, lng: number) => {
    try {
      console.log('Fetching stores near:', { lat, lng });
      const response = await api.get(`/stores/find?lat=${lat}&lng=${lng}`);
      console.log('API Response:', response);
      
      // Check if response.data exists and has a data property
      const stores = response?.data?.data || [];
      console.log('Stores found:', stores);

      if (stores.length > 0) {
        const store = stores[0];
        console.log('Nearest store:', store);
        
        // Get store location details
        const storeLocation = store.store?.location || {};
        const address = {
          display: storeLocation.formattedAddress || 
                   [storeLocation.address, storeLocation.city, storeLocation.state, storeLocation.country, storeLocation.pincode]
                     .filter(Boolean)
                     .join(', '),
          line1: storeLocation.address || '',
          city: storeLocation.city || '',
          state: storeLocation.state || '',
          country: storeLocation.country || '',
          postalCode: storeLocation.pincode || '',
          coordinates: storeLocation.coordinates || [lng, lat]
        };

        // Create the store object with the required structure
        const storeData = {
          ...store,
          address,
          display: store.store?.storeName || store.name || 'Store',
          deliveryRadius: store.store?.deliveryRadius || 5000,
          isActive: store.store?.isActive !== false,
          coordinates: storeLocation.coordinates || [lng, lat]
        };

        // Save to local storage
        const userLocation = {
          lat: storeLocation.coordinates?.[1] || lat,
          lng: storeLocation.coordinates?.[0] || lng,
          address: address.display,
          city: address.city,
          state: address.state,
          country: address.country,
          pincode: address.postalCode,
          formattedAddress: address.display
        };
        console.log('Saving to localStorage - userLocation:', userLocation);
        console.log('Saving to localStorage - selectedStore:', storeData);
        
        localStorage.setItem('userLocation', JSON.stringify(userLocation));
        localStorage.setItem('selectedStore', JSON.stringify(storeData));

        // Update state
        setSelectedStore(storeData);
        onLocationSelect?.({
          ...storeData,
          coordinates: storeLocation.coordinates || [lng, lat],
          address: userLocation
        });

        // Close the modal after a short delay
        setTimeout(() => {
          onClose();
        }, 1000);
      } else {
        console.log('No stores found in response');
        setError('No stores found near your location. Please try a different location.');
      }
    } catch (err) {
      console.error('Error finding nearby stores:', err);
      setError('Failed to find nearby stores. Please try again.');
    } finally {
      setIsLoading(false);
    }
  }, [onClose, onLocationSelect]);

  const selectStore = (store: any) => {
    setSelectedStore(store);
    localStorage.setItem('selectedStore', JSON.stringify(store));
    onLocationSelect(store);
    onClose();
  };

  const handleClose = () => {
    if (selectedStore) {
      onClose();
    } else {
      toast.error('Please select a location to continue');
    }
  };

  const handleDetectLocation = () => {
    if (!navigator.geolocation) {
      setError('Geolocation is not supported by your browser');
      return;
    }

    setIsLoading(true);
    setError(null);

    navigator.geolocation.getCurrentPosition(
      async (position) => {
        const { latitude, longitude } = position.coords;
        await findStoresNearby(latitude, longitude);
      },
      (err) => {
        console.error('Error getting location:', err);
        setError('Could not access your location. Please allow location access or enter manually.');
        setIsLoading(false);
      },
      { enableHighAccuracy: true, timeout: 10000, maximumAge: 0 }
    );
  };

  const handleSearch = async (e?: React.FormEvent) => {
    e?.preventDefault();
    if (!searchQuery.trim()) return;

    setIsLoading(true);
    setError(null);

    try {
      const response = await fetch(
        `https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(searchQuery)}`
      );

      const results = await response.json();

      if (results && results.length > 0) {
        setSearchResults(results.slice(0, 5));
      } else {
        setError('No locations found. Please try a different search term.');
        setSearchResults([]);
      }
    } catch (err) {
      console.error('Search error:', err);
      setError('Failed to search for location. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleSelectSearchResult = async (result: SearchResult) => {
    try {
      const lat = parseFloat(result.lat);
      const lng = parseFloat(result.lon);
      const addr = result.address || {};

      const address = {
        display: result.display_name || '',
        line1: addr.road || addr.suburb || '',
        line2: addr.suburb || '',
        city: addr.city || addr.suburb || '',
        state: addr.state || '',
        country: addr.country || '',
        postalCode: addr.postcode || '',
        coordinates: [lng, lat] as [number, number]
      };

      localStorage.setItem('userAddress', JSON.stringify(address));
      await findStoresNearby(lat, lng);
    } catch (error) {
      console.error('Error processing location selection:', error);
      setError('Failed to process the selected location. Please try again.');
    }
  };

  if (!internalIsOpen) return null;
  
  // Get saved address from localStorage
  const savedAddress = localStorage.getItem('userAddress');
  const parsedAddress = savedAddress ? JSON.parse(savedAddress) : null;

  return (
    <div className="fixed inset-0 z-50 overflow-y-auto bg-black bg-opacity-50 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg w-full max-w-md overflow-hidden shadow-xl">
        <form onSubmit={handleSearch}>
          <div className="p-4 border-b border-gray-200 flex justify-between items-center">
            <h3 className="text-lg font-medium text-gray-900">Select your location</h3>
            <button
              type="button"
              onClick={() => {
              setInternalIsOpen(false);
              onClose();
            }}
              className="text-gray-400 hover:text-gray-500"
            >
              <X className="h-5 w-5" />
            </button>
          </div>

          <div className="p-4">
            {/* Show current location if available */}
            {parsedAddress && (
              <div className="mb-4 p-3 bg-blue-50 rounded-lg border border-blue-100">
                <h4 className="font-medium text-blue-800 mb-1">Current Location</h4>
                <p className="text-sm text-blue-700">{parsedAddress.display}</p>
                <div className="mt-2">
                  <button
                    onClick={() => {
                      if (selectedStore) {
                        onLocationSelect(selectedStore);
                        onClose();
                      }
                    }}
                    className="text-sm text-blue-600 hover:text-blue-800 font-medium"
                  >
                    Use this location
                  </button>
                </div>
              </div>
            )}
            
            {error && (
              <div className="mb-4 p-3 text-sm text-red-700 bg-red-100 rounded-md">
                {error}
              </div>
            )}

            <div className="flex mb-4">
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search for area, street name..."
                className="flex-1 border border-gray-300 rounded-l-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
              <button
                type="submit"
                disabled={isLoading}
                className="bg-blue-600 text-white px-4 py-2 rounded-r-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 disabled:opacity-50"
              >
                {isLoading ? <Loader2 className="h-5 w-5 animate-spin" /> : 'Search'}
              </button>
            </div>

            {searchResults.length > 0 && (
              <div className="mb-4 max-h-60 overflow-y-auto border rounded-md">
                {searchResults.map((result, index) => (
                  <div
                    key={index}
                    onClick={() => handleSelectSearchResult(result)}
                    className="p-3 hover:bg-gray-50 cursor-pointer border-b last:border-b-0"
                  >
                    <div className="font-medium">{
                      result.display_name?.split(',').slice(0, 2).join(',')
                    }</div>
                    <div className="text-sm text-gray-500">
                      {[result.address?.city, result.address?.suburb, result.address?.state]
                        .filter(Boolean)
                        .join(', ')}
                    </div>
                  </div>
                ))}
              </div>
            )}

            <div className="relative my-6">
              <div className="absolute inset-0 flex items-center">
                <div className="w-full border-t border-gray-300"></div>
              </div>
              <div className="relative flex justify-center text-sm">
                <span className="px-2 bg-white text-gray-500">OR</span>
              </div>
            </div>

              {nearbyStores && nearbyStores.length > 0 && (
                <div className="mt-4">
                  <h4 className="text-sm font-medium text-gray-700 mb-2">Nearby Stores</h4>
                  <div className="space-y-2">
                    {nearbyStores.map((store) => (
                      <div
                        key={store._id}
                        className={`flex items-center justify-between p-3 border rounded-md cursor-pointer ${
                          selectedStore?._id === store._id
                            ? 'border-indigo-500 bg-indigo-50'
                            : 'border-gray-200 hover:border-indigo-300'
                        }`}
                        onClick={() => {
                          selectStore(store);
                          onLocationSelect?.({
                            ...location,
                            storeId: store._id
                          });
                        }}
                      >
                        <div>
                          <div className="font-medium">{store.store.storeName}</div>
                          <div className="text-sm text-gray-500">
                            {store.distanceKm} km away
                          </div>
                        </div>
                        {selectedStore?._id === store._id && (
                          <Check className="h-5 w-5 text-indigo-600" />
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {error && (
                <div className="mt-4 p-3 bg-red-50 text-red-700 text-sm rounded-md">
                  {error}
                </div>
              )}
            </div>
          </form>
        </div>
      </div>
  );
};

export default LocationModal;
