import { useState, useEffect, useRef, useCallback } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { useBrowsingHistory } from '../../hooks/useBrowsingHistory';
import {
  ChevronLeft, Share2, Star,
  ChevronDown, MapPin, ChevronRight, ShoppingCart
} from 'lucide-react';
import Header from './Header';
import Footer from './Footer';
import { itemService } from '../../services/api';
import { useCart } from '../../context/CartContext';
import { toast } from 'react-toastify';

interface Product {
  _id: string;
  id: string;
  name: string;
  brand: string;
  price: number;
  discount?: {
    _id: string;
    name: string;
    type: 'percentage' | 'fixed';
    value: number;
  };
  discount_id?: string | {
    _id: string;
    name: string;
    type: 'percentage' | 'fixed';
    value: number;
  };
  rating: number;
  reviews: number;
  availability: string;
  category: string;
  tags: string[];
  highlights: string[];
  images: string[] | Array<{ url: string; is_primary?: boolean; position?: number }>;
  variants?: string[];
  specifications?: Array<{ name: string; value: string }>;
  status?: number;
  sales_price?: number;
  minimum_qty?: number;
  inStock?: boolean;
  originalPrice?: number;
  code?: string;
  color?: string;
  size?: string;
  material?: string;
  description: string;
  warranty?: string;
  shipping?: string;
  returnPolicy?: string;
  additionalInfo?: string;
  relatedProducts?: string[];
  store_id?: string | { _id: string; name: string };
  store_name?: string;
  createdAt?: string;
  updatedAt?: string;
  unit_id?: string | { _id: string; name: string };
  tax_type?: 'inclusive' | 'exclusive';
  tax_id?: string | { _id: string; name: string; percentage: number };
  quantity?: number;
}

const defaultProduct: Product = {
  _id: '',
  id: '',
  name: 'Product not found',
  brand: '',
  price: 0,
  discount: 0,
  rating: 0,
  reviews: 0,
  availability: '',
  category: '',
  tags: [],
  highlights: [],
  images: ['/placeholder-product.png'],
  inStock: true,
  description: ''
};

const ProductDetails: React.FC = () => {
  const { id } = useParams();
  const navigate = useNavigate();
  const { addToCart, cart } = useCart();
  
  // Initialize browsing history
  const { addToBrowsingHistory, getRecentItems } = useBrowsingHistory();
  const [recentItems, setRecentItems] = useState<Array<{
    id: string;
    name: string;
    price: number;
    image: string;
    slug?: string;
  }>>([]);

  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [safeProduct, setSafeProduct] = useState<Product>(defaultProduct);
  const [selectedImage, setSelectedImage] = useState(0);
  const [isWishlisted, setIsWishlisted] = useState(false);
  const [variants, setVariants] = useState<string[]>(['Standard']);
  const [selectedVariant, setSelectedVariant] = useState<string>('Standard');
  const [quantity] = useState(1);
  const [selectedSize] = useState('M');
  const [selectedColor] = useState('Black');
  const [relatedProducts, setRelatedProducts] = useState<Product[]>([]);
  const scrollRef = useRef<HTMLDivElement>(null);

  // Scroll functionality for the carousel
  const scroll = useCallback((direction: 'left' | 'right') => {
    if (!scrollRef.current) return;
    
    const scrollAmount = 300;
    const currentScroll = scrollRef.current.scrollLeft;
    
    scrollRef.current.scrollTo({
      left: direction === 'left' 
        ? Math.max(0, currentScroll - scrollAmount) 
        : currentScroll + scrollAmount,
      behavior: 'smooth'
    });
  }, []);

  const products = [
    {
      id: 1,
      name: "Foley Catheter",
      price: "₹102",
      oldPrice: "₹102",
      discount: "60% OFF",
      image: "https://via.placeholder.com/120x120?text=Product",
    },
    {
      id: 2,
      name: "Foley Catheter",
      price: "₹102",
      oldPrice: "₹102",
      discount: null,
      image: "https://via.placeholder.com/120x120?text=Product",
    },
    {
      id: 3,
      name: "Foley Catheter",
      price: "₹102",
      oldPrice: "₹102",
      discount: null,
      image: "https://via.placeholder.com/120x120?text=Product",
    },
    {
      id: 4,
      name: "Foley Catheter",
      price: "₹102",
      oldPrice: "₹102",
      discount: null,
      image: "https://via.placeholder.com/120x120?text=Product",
    },
  ];

  // Scroll functionality is already defined with useCallback above

  useEffect(() => {
    const fetchProductData = async () => {
      setLoading(true);
      try {
        if (!id) {
          setError('Invalid product ID');
          return;
        }
        const response = await itemService.getItemById(id.trim());
        if (!response?.data) throw new Error('No product data received');
        setSafeProduct(response.data);
      } catch (err) {
        setError('Failed to fetch product details');
        console.error('Error fetching product:', err);
      } finally {
        setLoading(false);
      }
    };

    fetchProductData();
  }, [id]);

  const getImageUrl = (imagePath: string | string[] | undefined | null): string => {
    console.log('Original image path:', imagePath);
    
    // Handle null/undefined/empty
    if (!imagePath || (Array.isArray(imagePath) && imagePath.length === 0)) {
      console.log('No image path provided, using placeholder');
      return '/placeholder-product.png';
    }
    
    // If it's an array, get the first image
    if (Array.isArray(imagePath)) {
      console.log('Image path is an array, using first image');
      return getImageUrl(imagePath[0]);
    }
    
    // Handle object with url property
    if (imagePath && typeof imagePath === 'object' && imagePath.url) {
      console.log('Image is an object with url property');
      return getImageUrl(imagePath.url);
    }
    
    // If it's already a full URL, return as is
    if (typeof imagePath === 'string' && (imagePath.startsWith('http://') || imagePath.startsWith('https://'))) {
      console.log('Image is already a full URL:', imagePath);
      return imagePath;
    }
    
    // Handle local file paths
    if (typeof imagePath === 'string') {
      // Remove any leading slashes and backslashes
      let normalizedPath = imagePath.replace(/^[\\/]+/, '');
      
      // Remove any 'uploads/items/' prefix if it exists
      normalizedPath = normalizedPath.replace(/^uploads\/items\//, '');
      
      // Construct the full URL to the backend
      const baseUrl = import.meta.env.VITE_API_URL || 'http://localhost:5000';
      const imageUrl = `${baseUrl}/api/uploads/items/${normalizedPath}`;
      console.log('Generated image URL:', imageUrl);
      return imageUrl;
    }
    
    console.log('No valid image path found, using placeholder');
    return '/placeholder-product.png';
  };

  const handleAddToCart = async (e: React.MouseEvent) => {
    e.stopPropagation();
    
    try {
      if (!safeProduct._id) {
        throw new Error('Product ID is missing');
      }
      
      setIsAdding(true);
      
      // Get the first image URL
      const getImageUrl = () => {
        if (!safeProduct.images) return '';
        if (Array.isArray(safeProduct.images) && safeProduct.images.length > 0) {
          const firstImage = safeProduct.images[0];
          return typeof firstImage === 'string' ? firstImage : firstImage?.url || '';
        }
        return '';
      };
      
      // Get store info
      const getStoreInfo = () => {
        if (!safeProduct.store_id) return { name: 'Default Store', id: 'default-store' };
        
        if (typeof safeProduct.store_id === 'object' && safeProduct.store_id !== null) {
          return {
            name: safeProduct.store_id.name || 'Default Store',
            id: safeProduct.store_id._id || 'default-store',
            store: safeProduct.store_id.name || 'Default Store'
          };
        }
        
        return {
          name: safeProduct.store_name || 'Default Store',
          id: safeProduct.store_id as string || 'default-store',
          store: safeProduct.store_name || 'Default Store'
        };
      };
      
      const { name: storeName, id: storeId, store } = getStoreInfo();
      const price = safeProduct.sales_price || safeProduct.price || 0;
      
      // Prepare cart item with store information
      const cartItem = {
        id: safeProduct._id,
        _id: safeProduct._id,
        name: safeProduct.name || 'Unnamed Product',
        price: typeof price === 'string' ? parseFloat(price) : price,
        image: getImageUrl(),
        storeId,
        storeName: storeName,
        store: store, // Add store name to both storeName and store for backward compatibility
        quantity: 1,
        ...(safeProduct.brand && { brand: safeProduct.brand }),
        ...(safeProduct.category && { category: safeProduct.category }),
        ...(safeProduct.inStock !== undefined && { inStock: safeProduct.inStock }),
        originalPrice: safeProduct.originalPrice || (typeof price === 'string' ? parseFloat(price) : price),
        ...(safeProduct.discount && { discount: safeProduct.discount }),
        ...(safeProduct.discount_id && { discount: safeProduct.discount_id }),
        ...(safeProduct.unit_id && { 
          unit: typeof safeProduct.unit_id === 'object' ? safeProduct.unit_id.name : safeProduct.unit_id 
        }),
        requiresPrescription: safeProduct.requiresPrescription || false
      };
      
      console.log('Adding to cart:', cartItem);
      
      await addToCart(cartItem);
      toast.success(`${safeProduct.name || 'Product'} added to cart`);
    } catch (error) {
      console.error('Error adding to cart:', error);
      toast.error('Failed to add to cart');
    } finally {
      setIsAdding(false);
    }
  };
  const handleBuyNow = () => {
    handleAddToCart();
    navigate('/checkout');
  };

  const handleShare = async () => {
    const productUrl = window.location.href;

    try {
      if (navigator.share) {
        // Use Web Share API if available (mobile devices)
        await navigator.share({
          title: safeProduct.name,
          text: `Check out ${safeProduct.name}`,
          url: productUrl,
        });
        toast.success('Product shared successfully!');
      } else {
        // Fallback to clipboard copy for desktop
        await navigator.clipboard.writeText(productUrl);
        toast.success('Product link copied to clipboard!');
      }
    } catch (error) {
      console.error('Error sharing product:', error);
      // Fallback if clipboard API fails
      try {
        const textArea = document.createElement('textarea');
        textArea.value = productUrl;
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        toast.success('Product link copied to clipboard!');
      } catch (fallbackError) {
        console.error('Fallback copy failed:', fallbackError);
        toast.error('Unable to share product. Please copy the URL manually.');
      }
    }
  };

  const renderStars = (rating: number) => {
    const stars = [];
    for (let i = 1; i <= 5; i++) {
      stars.push(
        <Star
          key={i}
          className={`w-4 h-4 ${i <= rating ? 'text-yellow-400 fill-yellow-400' : 'text-gray-300'}`}
        />
      );
    }
    return <div className="flex items-center">{stars}</div>;
  };


  if (loading) return (
    <div className="min-h-screen flex flex-col items-center justify-center bg-white">
      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-800"></div>
      <p className="mt-4 text-gray-600">Loading product details...</p>
    </div>
  );

  if (error || !safeProduct) return (
    <div className="min-h-screen flex flex-col items-center justify-center bg-white">
      <X className="w-12 h-12 text-red-500 mb-4" />
      <p className="text-red-600 text-lg">{error || 'Product not found'}</p>
      <button 
        onClick={() => navigate(-1)}
        className="mt-4 px-6 py-2 bg-blue-800 text-white rounded-md hover:bg-blue-900 transition-colors"
      >
        Go Back
      </button>
    </div>
  );

  return (
    <>
      <div className="min-h-screen bg-white flex flex-col">
        <Header />
      
      {/* Main Content */}
      <div className="container mx-auto px-4 py-8 flex-1">
        {/* Breadcrumb */}
        <nav className="flex items-center text-sm text-gray-500 mb-8">
          <button 
            onClick={() => navigate(-1)} 
            className="flex items-center text-blue-800 hover:text-blue-800 transition-colors"
          >
            <ChevronLeft className="w-4 h-4 mr-1" /> Back
          </button>
          <span className="mx-2">/</span>
          <span className="text-gray-500">Products</span>
          <span className="mx-2">/</span>
          <span className="text-gray-700 font-medium">{safeProduct.name}</span>
        </nav>

        {/* Product Grid - Three Column Layout */}
        <div className="grid grid-cols-1 md:grid-cols-12 gap-8">
          {/* Left Column - Product Images (3/12 width) */}
          <div className="md:col-span-3">
            <div className="bg-white rounded-lg overflow-hidden mb-4 border border-gray-200">
              <img 
                src={getImageUrl(safeProduct.images[selectedImage])}
                alt={safeProduct.name}
                className="w-full h-auto max-h-[500px] object-contain p-8"
                onError={(e) => {
                  // Fallback to placeholder if image fails to load
                  const target = e.target as HTMLImageElement;
                  target.src = '/placeholder-product.png';
                }}
              />
            </div>
            <div className="flex space-x-3 overflow-x-auto pb-2">
              {(safeProduct.images || []).map((img, i) => (
                <button 
                  key={i} 
                  onClick={() => setSelectedImage(i)}
                  className={`flex-shrink-0 w-16 h-16 border-2 rounded-md overflow-hidden ${
                    selectedImage === i ? "border-blue-800" : "border-gray-200"
                  }`}
                >
                  <img 
                    src={getImageUrl(img)} 
                    alt={`Thumb ${i + 1}`} 
                    className="w-full h-full object-cover" 
                  />
                </button>
              ))}
            </div>
          </div>

          {/* Middle Column - Product Info (5/12 width) */}
          <div className="md:col-span-5 pr-6">
            <div className="w-full p-5 bg-white text-gray-800 space-y-3">
              {/* Title + Share Icon */}
              <div className="flex justify-between items-start">
                <h2 className="text-lg sm:text-xl font-bold leading-tight">
                  {safeProduct.name}
                </h2>
                <Share2 
                  className="w-6 h-6 text-gray-600 cursor-pointer hover:text-gray-800 mt-1" 
                  onClick={handleShare}
                />
              </div>

              {/* Store Link */}
              {safeProduct.store_id && (
                <a
                  href={`/store/${typeof safeProduct.store_id === 'string' ? safeProduct.store_id : safeProduct.store_id?._id}`}
                  className="text-blue-800 text-base font-medium mt-2 inline-block hover:underline"
                >
                  Visit {typeof safeProduct.store_id === 'object' ? safeProduct.store_id.name : safeProduct.store_name || 'Store'}
                </a>
              )}

              {/* Ratings */}
              <div className="flex items-center gap-2 mt-2">
                {renderStars(safeProduct.rating)}
                <span className="text-sm text-gray-500">({safeProduct.reviews} ratings)</span>
              </div>

              {/* Offer */}
              <p className="text-sm text-gray-600 mt-2">*Offer applicable on order above ₹1000</p>

              <hr className="my-4 border-gray-200" />
              {/* Variant selection */}
              <div className="space-y-3">
                <h3 className="text-base font-semibold text-gray-800">
                  Select Available Variant
                </h3>
                <p className="text-sm text-gray-600">Variant</p>

                <div className="flex gap-3 flex-wrap">
                  {variants.map((v) => (
                    <button
                      key={v}
                      onClick={() => setSelectedVariant(v)}
                      className={`px-4 py-2 rounded-md border text-base font-medium transition-colors ${ 
                        selectedVariant === v
                          ? 'bg-blue-50 text-blue-800 border-blue-800'
                          : 'bg-white text-gray-700 border-gray-300 hover:border-gray-400'
                      }`}
                    >
                      {v}
                    </button>
                  ))}
                </div>
              </div>
            </div>
          </div>

          {/* Right Column - Purchase Info (4/12 width) */}
          <div className="md:col-span-4">
             <div className="w-full border rounded-xl shadow-sm p-5 bg-white text-gray-800 space-y-4">
                {/* Price Section */}
                <div className="space-y-1">
                  <div className="flex items-baseline gap-3">
                    <span className="text-2xl font-bold text-gray-900">
                      ₹{safeProduct.sales_price ? safeProduct.sales_price.toFixed(2) : safeProduct.price?.toFixed(2)}
                    </span>
                    {safeProduct.sales_price && safeProduct.sales_price < safeProduct.price && (
                      <>
                        <span className="text-base text-gray-400 line-through">
                          ₹{safeProduct.price?.toFixed(2)}
                        </span>
                        {(safeProduct.discount || safeProduct.discount_id) && (
                          <span className="text-base font-medium text-green-600">
                            {safeProduct.discount?.value 
                              ? `${safeProduct.discount.value}% off`
                              : Math.round(((safeProduct.price - safeProduct.sales_price) / safeProduct.price) * 100) + '% off'}
                          </span>
                        )}
                      </>
                    )}
                  </div>
                  <p className="text-sm text-gray-600">
                    {typeof safeProduct.unit_id === 'object' 
                      ? safeProduct.unit_id?.name 
                      : safeProduct.unit_id || 'Unit'} • 
                    {safeProduct.tax_type === 'inclusive' ? 'Incl.' : 'Excl.'} of all taxes
                  </p>
                </div>

                {/* Location Section */}
                <div className="flex justify-between items-center border rounded-lg px-4 py-3 bg-gray-50">
                  <div className="flex items-center gap-2">
                    <MapPin className="w-5 h-5 text-gray-500" />
                    <span className="text-base font-medium text-gray-800">700150 • Huzaifa</span>
                  </div>
                  <button className="text-blue-800 text-base font-medium hover:underline whitespace-nowrap">
                    Change
                  </button>
                </div>

                {/* Dropdown */}
                <div className="mt-2">
                  <div className="relative">
                    <ChevronDown className="w-5 h-5 absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 pointer-events-none" />
                    <select className="w-full border rounded-lg px-4 py-3 text-base text-gray-800 focus:outline-none focus:ring-2 focus:ring-blue-800 appearance-none bg-white">
                      <option>1 Pack • 200ml Shampoo</option>
                      <option>2 Pack • 400ml Shampoo</option>
                    </select>
                  </div>
                </div>

                {/* Cart Button */}
                <div className="flex bg-blue-800 justify-between items-center rounded-lg">
                  <div className="text-base px-4 py-2">
                    <span className="block font-semibold text-white">{quantity} {quantity === 1 ? 'item' : 'items'}</span>
                    <span className="text-lg font-bold text-white">
                      ₹{((safeProduct.sales_price || safeProduct.price || 0) * quantity).toFixed(2)}
                    </span>
                  </div>
                  <button 
                    onClick={() => addToCart({
                      ...safeProduct,
                      quantity,
                      total: (safeProduct.sales_price || safeProduct.price || 0) * quantity
                    })}
                    className="bg-blue-800 hover:bg-blue-900 text-white text-base font-semibold px-6 py-3 rounded-lg transition-colors shadow-sm hover:shadow-md"
                  >
                    {quantity > 0 ? 'Add to Cart' : 'View Cart'}
                  </button>
                </div>
              </div>
          </div>
        </div>
        </div>

      
        {/* Frequently Bought Together */}
        <div className="container mx-auto px-4 py-6">
          <h2 className="text-2xl font-semibold text-gray-800 mb-6">
            Frequently Bought Together
          </h2>

          {loading ? (
            <div className="flex justify-center items-center h-40">
              <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-gray-900"></div>
            </div>
          ) : relatedProducts.length > 0 ? (
            <div className="relative">
              {/* Scroll buttons */}
              <button
                onClick={() => scroll("left")}
                className="absolute left-0 top-1/2 -translate-y-1/2 bg-white shadow-md rounded-full p-2 z-10 hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500"
                aria-label="Scroll left"
              >
                <ChevronLeft className="w-5 h-5 text-gray-600" />
              </button>

              <div
                ref={scrollRef}
                className="flex gap-6 overflow-x-auto scroll-smooth no-scrollbar pb-4"
              >
                {relatedProducts.map((product) => {
                  const imageUrl = Array.isArray(product.images) && product.images.length > 0
                    ? product.images[0].url || '/placeholder-product.png'
                    : '/placeholder-product.png';
                  
                  const isInCart = cart.items.some(item => item.productId === product._id);
                  
                  return (
                    <div
                      key={product._id}
                      className="min-w-[180px] sm:min-w-[200px] border border-gray-200 rounded-lg bg-white shadow-sm hover:shadow-md transition-shadow flex flex-col flex-shrink-0"
                    >
                      <div className="p-3">
                        <img
                          src={imageUrl}
                          alt={product.name}
                          className="w-full h-36 object-contain mb-3"
                          onError={(e) => {
                            const target = e.target as HTMLImageElement;
                            target.onerror = null;
                            target.src = '/placeholder-product.png';
                          }}
                        />
                        <h3 className="text-sm font-medium text-gray-800 line-clamp-2 h-10 mb-2">
                          {product.name}
                        </h3>
                        <div className="flex items-center gap-2">
                          <span className="text-base font-bold text-gray-900">
                            ₹{product.sales_price?.toFixed(2) || product.price?.toFixed(2) || '0.00'}
                          </span>
                          {product.sales_price && product.sales_price < product.price && (
                            <span className="text-sm text-gray-500 line-through">
                              ₹{product.price?.toFixed(2)}
                            </span>
                          )}
                        </div>
                        {product.rating > 0 && (
                          <div className="flex items-center mt-1">
                            <div className="flex">
                              {[...Array(5)].map((_, i) => (
                                <Star
                                  key={i}
                                  className={`w-4 h-4 ${
                                    i < Math.floor(product.rating)
                                      ? 'text-yellow-400 fill-current'
                                      : 'text-gray-300'
                                  }`}
                                />
                              ))}
                            </div>
                            <span className="text-xs text-gray-500 ml-1">
                              ({product.reviews || 0})
                            </span>
                          </div>
                        )}
                      </div>
                      <div className="mt-auto p-3 pt-0">
                        <button
                          onClick={() => addToCart({
                            productId: product._id,
                            name: product.name,
                            price: product.sales_price || product.price,
                            quantity: 1,
                            image: imageUrl,
                            stock: product.quantity || 0
                          })}
                          className={`w-full py-2 rounded-md text-sm font-medium transition-colors ${
                            isInCart
                              ? 'bg-green-100 text-green-800 hover:bg-green-200'
                              : 'bg-blue-800 text-white hover:bg-blue-900'
                          }`}
                        >
                          {isInCart ? 'Added to Cart' : 'Add to Cart'}
                        </button>
                      </div>
                    </div>
                  );
                })}
              </div>

              <button
                onClick={() => scroll("right")}
                className="absolute right-0 top-1/2 -translate-y-1/2 bg-white shadow-md rounded-full p-2 z-10 hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-blue-500"
                aria-label="Scroll right"
              >
                <ChevronRight className="w-5 h-5 text-gray-600" />
              </button>
            </div>
          ) : (
            <div className="text-center py-8 text-gray-500">
              No related products found.
            </div>
          )}

          {/* Description Section */}
          <div className="mt-6 border-t pt-4">
            <h3 className="text-lg font-semibold text-gray-800 mb-2">Description</h3>
            <div 
              className="text-sm text-gray-600 leading-relaxed"
              dangerouslySetInnerHTML={{ __html: safeProduct?.description || 'No description available' }}
            />
          </div>

         

           {/* Product Details Section */}
           <div className="mt-6 border-t pt-4">
            <h3 className="text-lg font-semibold text-gray-800 mb-2">Product Details</h3>
            <div className="text-sm text-gray-600 space-y-1">
              <p><strong>SKU:</strong> {safeProduct?.sku || 'N/A'}</p>
              <p><strong>HSN Code:</strong> {safeProduct?.hsn || 'N/A'}</p>
              <p><strong>Minimum Quantity:</strong> {safeProduct?.minimum_qty || 'N/A'}</p>
              {safeProduct?.expire_date && (
                <p>
                  <strong>Expiry Date:</strong>{' '}
                  {new Date(safeProduct.expire_date).toLocaleDateString()}
                </p>
              )}
              <p><strong>Price:</strong> ₹{safeProduct?.price?.toFixed(2) || '0.00'}</p>
              {safeProduct?.sales_price && (
                <p><strong>Sale Price:</strong> ₹{safeProduct.sales_price.toFixed(2)}</p>
              )}
              <p><strong>In Stock:</strong> {safeProduct?.quantity || 0} units</p>
              <p>
                <strong>Status:</strong>{' '}
                <span className={`font-medium ${safeProduct?.status === 1 ? 'text-green-600' : 'text-red-600'}`}>
                  {safeProduct?.status === 1 ? 'Active' : 'Inactive'}
                </span>
              </p>
              <p><strong>Last Updated:</strong> {new Date(safeProduct?.updatedAt).toLocaleString()}</p>
            </div>
          </div>

          {/* Previously Browsed Items */}
          {recentItems.length > 0 && (
            <div className="mt-8">
              <h3 className="text-lg font-semibold text-gray-800 mb-4">Previously Browsed Items</h3>
              <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                {recentItems.map((item) => (
                  <div 
                    key={item.id} 
                    className="border rounded-lg p-3 bg-white text-center hover:shadow-md transition-shadow cursor-pointer"
                    onClick={() => navigate(`/product/${item.slug || item.id}`)}
                  >
                    <img 
                      src={item.image || '/placeholder-product.png'} 
                      alt={item.name} 
                      className="h-24 mx-auto object-contain mb-2"
                      onError={(e) => {
                        const target = e.target as HTMLImageElement;
                        target.onerror = null;
                        target.src = '/placeholder-product.png';
                      }}
                    />
                    <h4 className="text-sm font-medium text-gray-800 line-clamp-2 h-10 flex items-center justify-center">
                      {item.name}
                    </h4>
                    <div className="mt-2">
                      <div className="flex items-center justify-center gap-2">
                        <span className="text-base font-bold text-gray-800">
                          ₹{item.price.toFixed(2)}
                        </span>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* Ratings and Reviews */}
          <div className="mt-8 border-t pt-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">Ratings and Reviews</h3>
            <div className="flex flex-col md:flex-row gap-8">
              <div className="text-center bg-gray-50 p-4 rounded-lg">
                <p className="text-4xl font-bold">4.3<span className="text-2xl text-gray-500">/5</span></p>
                {renderStars(4.3)}
                <p className="text-sm text-gray-600 mt-2">95 Ratings</p>
              </div>
              <div className="flex-1 space-y-2">
                {[
                  { stars: 5, percentage: 64, color: 'bg-green-500' },
                  { stars: 4, percentage: 21, color: 'bg-green-500' },
                  { stars: 3, percentage: 4, color: 'bg-orange-400' },
                  { stars: 2, percentage: 4, color: 'bg-orange-400' },
                  { stars: 1, percentage: 6, color: 'bg-red-500' }
                ].map(rating => (
                  <div key={rating.stars} className="flex items-center gap-2 text-sm">
                    <span>{rating.stars} stars</span>
                    <div className="w-full bg-gray-200 rounded-full h-2.5">
                      <div className={`${rating.color} h-2.5 rounded-full`} style={{ width: `${rating.percentage}%` }}></div>
                    </div>
                    <span className="w-10 text-right text-gray-600">{rating.percentage}%</span>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* Recent Reviews */}
          <div className="mt-8 border-t pt-6">
            <h3 className="text-lg font-semibold text-gray-800 mb-4">Recent Reviews</h3>
            <div className="space-y-6">
              {[
                { name: 'Ankit', rating: 1, text: '"Price high price"', time: '1 months ago' },
                { name: 'Divyansh Chauhan', rating: 1, text: '"Wrong product delivered scam!!"', time: '3 months ago' },
                { name: 'Giftson David', rating: 5, text: '"Pharmacy has the lowest price online."', time: '3 months ago' },
              ].map((review, index) => (
                <div key={index}>
                  <p className="font-semibold text-gray-800">{review.name}</p>
                  {renderStars(review.rating)}
                  <p className="text-gray-700 mt-1">{review.text}</p>
                  <p className="text-xs text-gray-500 mt-1">{review.time}</p>
                </div>
              ))}
            </div>
            <button className="text-blue-800 font-semibold mt-6 flex items-center gap-1">
              View All Reviews <ChevronRight className="w-4 h-4" />
            </button>
          </div>
        </div>
      </div>
      
      {/* Footer */}
      <Footer />
      
      {/* Mobile Sticky Add to Cart */}
      <div className="md:hidden fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 p-3 shadow-lg">
        <div className="flex items-center justify-between">
          <div className="flex items-center">
            <div className="relative">
              <span className="text-lg font-bold text-gray-900">${safeProduct.price.toFixed(2)}</span>
              {safeProduct.discount_id && (0 && (
                <span className="ml-1 text-xs text-gray-500 line-through">${(safeProduct.price * 1.2).toFixed(2)}</span>
              ))}
            </div>
          </div>
          <div className="flex space-x-2">
            <button 
              onClick={handleAddToCart}
              className="px-4 py-2 bg-blue-800 text-white rounded-md text-sm font-medium flex items-center"
            >
              <ShoppingCart className="w-4 h-4 mr-1" />
              Add to Cart
            </button>
            <button 
              onClick={handleBuyNow}
              className="px-4 py-2 bg-white border border-blue-800 text-blue-800 rounded-md text-sm font-medium"
            >
              Buy Now
            </button>
          </div>
        </div>
      </div>
    </>
  );
};

export default ProductDetails;
