import React, { useState } from "react";
import { ShoppingBag, Loader2 } from "lucide-react";
import { useNavigate } from 'react-router-dom';
import { useCart } from '../../context/CartContext';
import { toast } from 'react-hot-toast';

interface ProductCardProps {
  id: string;
  image: string;
  title: string;
  price: number | string;
  originalPrice?: number | string;
  discount?: string;
  storeId?: string;
  storeName?: string;
  requiresPrescription?: boolean;
}

const ProductCard: React.FC<ProductCardProps> = ({
  id,
  image,
  title,
  price,
  originalPrice,
  discount,
  storeId,
  storeName,
  requiresPrescription,
}) => {
  const navigate = useNavigate();
  const { addToCart, getItemQuantity } = useCart();
  const [isAdding, setIsAdding] = useState(false);
  const quantityInCart = getItemQuantity(id);

  const handleCardClick = (e: React.MouseEvent) => {
    // Don't navigate if the click was on the add to cart button
    if ((e.target as HTMLElement).closest('button')) {
      return;
    }
    navigate(`/product/${id}`);
  };

  const handleAddToCart = async (e: React.MouseEvent) => {
    e.stopPropagation();
    
    try {
      setIsAdding(true);
      
      const cartItem = {
        id,
        _id: id, // Include _id for backward compatibility
        name: title,
        price: typeof price === 'string' ? parseFloat(price) : price,
        image: Array.isArray(image) ? image[0]?.url || '' : image,
        storeId: storeId || 'default-store',
        storeName: storeName || 'Default Store',
        store: storeId || 'default-store',
        quantity: 1,
        ...(originalPrice && { 
          originalPrice: typeof originalPrice === 'string' 
            ? parseFloat(originalPrice) 
            : originalPrice 
        }),
        ...(discount && { discount: parseFloat(discount) }),
        requiresPrescription: requiresPrescription || false
      };
      
      await addToCart(cartItem);
      toast.success(`${title} added to cart`);
    } catch (error) {
      console.error('Error adding to cart:', error);
      toast.error('Failed to add to cart');
    } finally {
      setIsAdding(false);
    }
  };

  // Format price to display with Indian Rupee symbol
  const formatPrice = (value: number | string | undefined): string => {
    if (value === undefined) return "";
    const numValue = typeof value === "string" ? parseFloat(value) : value;
    if (isNaN(numValue)) return "₹0";
    return `₹${numValue.toFixed(2).replace(/\.00$/, "")}`;
  };

  return (
    <div 
      onClick={handleCardClick}
      className="w-full max-w-xs mx-auto bg-white rounded-2xl border border-gray-100 overflow-hidden shadow-sm hover:shadow-lg hover:-translate-y-1 transition-all duration-300 cursor-pointer"
    >
      {/* Image Section */}
      <div className="flex justify-center items-center p-4 bg-gray-50 h-48">
        <img
          src={image}
          alt={title}
          className="h-full w-full object-contain"
          onError={(e) => {
            const target = e.target as HTMLImageElement;
            target.src = "/placeholder-product.png";
          }}
        />
      </div>

      {/* Product Info */}
      <div className="text-center space-y-3">
        <div className="px-4">
          <h3 className="text-[15px] font-semibold text-gray-800 leading-tight line-clamp-2 h-10">
            {title}
          </h3>

          {/* Price Section */}
          <div className="flex items-center justify-center gap-3 mt-2">
            <div className="flex items-baseline gap-2">
              <span className="text-lg font-bold text-gray-900">
                {formatPrice(price)}
              </span>
              {originalPrice && (
                <span className="text-sm line-through text-gray-400">
                  {formatPrice(originalPrice)}
                </span>
              )}
            </div>
            {discount && (
              <span className="text-xs bg-green-100 text-green-700 font-semibold px-2 py-0.5 rounded-md whitespace-nowrap">
                {discount} OFF
              </span>
            )}
          </div>
        </div>

        {/* Add to Cart Button */}
        <button
          onClick={handleAddToCart}
          disabled={isAdding}
          className={`w-full py-2.5 mt-2 flex items-center justify-center gap-2 font-medium transition-all duration-200 ${
            isAdding || quantityInCart > 0
              ? 'bg-green-600 hover:bg-green-700 text-white'
              : 'bg-blue-600 hover:bg-blue-700 text-white'
          }`}
          style={{
            borderRadius: '0 0 1rem 1rem' // Match the card's border radius
          }}
        >
          {isAdding ? (
            <>
              <Loader2 className="h-4 w-4 animate-spin" />
              <span>Adding...</span>
            </>
          ) : quantityInCart > 0 ? (
            <span>✓ Added to Cart</span>
          ) : (
            <>
              <span>Add to Cart</span>
              <ShoppingBag size={16} />
            </>
          )}
        </button>
      </div>
    </div>
  );
};

export default ProductCard;
